/*
 * Created on Nov 14, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.actionconfig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.jbpm.gd.common.model.GenericElement;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

/**
 * Utility class that provides methods to read the configured Arguments for each of Action Class
 * that is specified in a properties file.
 * 
 * @author Imran Naqvi
 * @author Vijay Silva
 */
class ActionConfigurationReader
{
	private static final Logger logger = Logger.getLogger(ActionConfigurationReader.class);

	private static final String RESOURCE_NAME = "/ActionClasses.xml";

	private static final String ACTION_CLASSES_XPATH = "//class/@name";

	private IValidationService validationService;

	/**
	 * Constructor
	 */
	public ActionConfigurationReader()
	{
		this.validationService = ValidationServiceFactory.createFactory().createValidationService();
	}

	/**
	 * Read the Action Configuration from a file. Returns a list of all the Action Configuration
	 * objects.
	 * 
	 * @return List of Action Configuration objects
	 * 
	 * @throws ActionConfigurationException
	 *             Exception when reading the configuration.
	 */
	public Map<String, ActionClassConfiguration> read() throws ActionConfigurationException
	{
		Map<String, ActionClassConfiguration> actionClasses = new HashMap<String, ActionClassConfiguration>();

		try
		{
			DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			Document document = builder.parse(this.getClass().getResourceAsStream(RESOURCE_NAME));

			XPath xpath = XPathFactory.newInstance().newXPath();
			NodeList nodes = (NodeList) xpath.evaluate(ACTION_CLASSES_XPATH, document,
					XPathConstants.NODESET);

			int nodeCount = (nodes != null) ? nodes.getLength() : 0;
			for (int i = 0; i < nodeCount; i++)
			{
				Node node = nodes.item(i);
				String className = node.getNodeValue();

				// Skip over blank action classes
				if (className == null || className.trim().length() == 0)
					continue;

				String expression = "//class[@name='" + className + "']/variable";
				NodeList variableNodes = (NodeList) xpath.evaluate(expression, document,
						XPathConstants.NODESET);
				List<ActionVariableConfiguration> variables = new ArrayList<ActionVariableConfiguration>();
				List<String> variableNames = new ArrayList<String>();
				int variableNodeCount = (variableNodes != null) ? variableNodes.getLength() : 0;
				for (int j = 0; j < variableNodeCount; j++)
				{
					Node variableNode = variableNodes.item(j);
					if (!(variableNode instanceof Element))
						continue;

					Element element = (Element) variableNode;
					String name = element.getAttribute("name");

					// Skip over invalid variable names
					if (this.isInvalidVariableName(name))
					{
						String msg = "The Action Class with name: " + className
								+ " contains a variable with an invalid name: " + name
								+ ", skipping variable.";
						logger.warn(msg);
						continue;
					}

					// Skip over duplicate names
					if (variableNames.contains(name))
					{
						String msg = "The Action Class with name: " + className
								+ " contains a more than one variable with name: " + name
								+ ", skipping duplicate variables.";
						logger.warn(msg);
						continue;
					}

					boolean required = Boolean.parseBoolean(element.getAttribute("required"));
					String validator = element.getAttribute("validationRule");

					variables.add(new ActionVariableConfiguration(name, required, validator));
				}

				actionClasses.put(className, new ActionClassConfiguration(className, variables));
			}
		}
		catch (Exception ex)
		{
			String msg = "Failed to load Action Class configuration from file: " + RESOURCE_NAME;
			throw new ActionConfigurationException(msg, ex);
		}

		return actionClasses;
	}

	/*
	 * Validate the Variable Name.
	 */
	private boolean isInvalidVariableName(String name)
	{
		// Create a 'fake' Generic Element
		GenericElement element = new GenericElement();
		element.setName(name);

		List<ValidationMessage> messages = this.validationService.validateActionVariableName(element);
		return (messages != null && messages.size() > 0);
	}
}
