package com.tandbergtv.watchpoint.studio.ui.deployment;

import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.EXPORT_DIR;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClassFile;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJarEntryResource;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.FormColors;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.jbpm.gd.jpdl.Logger;
import org.jbpm.gd.jpdl.deployment.IncludeClassesTreeContentProvider;
import org.jbpm.gd.jpdl.deployment.IncludeFilesTreeContentProvider;
import org.jbpm.gd.jpdl.deployment.IncludeInDeploymentTreeViewer;
import org.jbpm.gd.jpdl.util.ProcessDeployer;

import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.UI;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateGraphPage;
import com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.JavaClassNameCollector;

/**
 * Class based on jbpm's org.jbpm.gd.jpdl.deployment.DeploymentForm.
 * 
 * @author <a href="francisco.neto@venturus.org.br">xfranet Francisco Bento da Silva Neto</a>
 * @see org.jbpm.gd.jpdl.deployment.DeploymentForm
 */
public class DeploymentForm {

    public static final String TEMPLATES_FOLDER_NAME = "templates";
    public static final int NONE = 0;
    public static final int EXPRESSION = 1;
    public static final int HANDLER = 2;

    private final FormToolkit toolkit;
    private final Composite composite;
    private final IFolder processFolder;
    private final WatchPointTemplateEditor editor;

    private ScrolledForm form;
    private Text locationText;
    private Button exportImageButton;
    private Button exportGroupsButton;
    private Button saveButton;
    private Button locationButton;

    private IncludeInDeploymentTreeViewer includeFilesTreeViewer;
    private IncludeInDeploymentTreeViewer includeClassesTreeViewer;

    public DeploymentForm(FormToolkit toolkit, Composite composite, IFolder processFolder,
            WatchPointTemplateEditor editor) {
        this.toolkit = toolkit;
        this.composite = composite;
        this.processFolder = processFolder;
        this.editor = editor;
    }

    public void create() {
        createMainForm();
        createIncludeFilesSection();
        createIncludeClassesSection();
        createLocalSaveSection();
        updateSaveAndDeployButtons();
        toolkit.createForm(form.getBody()); // Create an empty grid cell
    }

    private void createMainForm() {
        form = toolkit.createScrolledForm(composite);
        GridData layoutData = new GridData(GridData.FILL_BOTH);
        form.setLayoutData(layoutData);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        layout.makeColumnsEqualWidth = true;
        form.getBody().setLayout(layout);
        form.getBody().setLayoutData(new GridData(GridData.FILL_BOTH));
    }

    private ProcessDeployer createProcessDeployer() {
        ProcessDeployer result = new ProcessDeployer();
        String location = null;
        if (saveButton.isEnabled()) {
            location = locationText.getText();
        }
        result.setTargetLocation(location);
        result.setShell(form.getShell());
        result.setProcessFolder(processFolder);

        result.setFilesAndFolders(getIncludedFiles());
        result.setClassesAndResources(getClassesAndResources());
        return result;
    }

    private ArrayList<Object> getIncludedFiles() {
        ArrayList<Object> result = new ArrayList<Object>();
        Object[] objects = includeFilesTreeViewer.getCheckedElements();
        for (int i = 0; i < objects.length; i++) {
            result.add(objects[i]);
        }
        return result;
    }

    private ArrayList<String> getClassesAndResources() {
        ArrayList<String> result = new ArrayList<String>();
        Object[] objects = includeClassesTreeViewer.getCheckedElements();
        for (int i = 0; i < objects.length; i++) {
            if (objects[i] instanceof ICompilationUnit) {
                ICompilationUnit compilationUnit = (ICompilationUnit) objects[i];
                IResource resource = compilationUnit.getResource();

                try {
                    String parentFolder = getResourceName(compilationUnit.getResource().getParent());
                    for (IType type : compilationUnit.getTypes()) {
                        String className = parentFolder + "/" + type.getElementName() + ".class";
                        result.add(className);

                        // Workaround to export inner classes
                        IJavaProject project = JavaCore.create(resource.getProject());
                        // Get the project's output location folder
                        IFolder outputLocation = ResourcesPlugin.getWorkspace().getRoot()
                                .getFolder(project.getOutputLocation());
                        /*
                         * Output location usually is: - Project/bin. Append the parent folder (class package) to the to
                         * get the folder where the inner classes are stored - Project/bin/com/ericsson/
                         */
                        IPath classOutputLocation = outputLocation.getLocation().append(new Path(parentFolder));
                        File folder = new File(classOutputLocation.toPortableString());
                        final String innerClassName = type.getElementName() + "$";
                        /*
                         * inner classes are denoted by MyClass$1.class Search the project output location for MyClass$*
                         * files
                         */
                        File[] innerClasses = folder.listFiles(new FileFilter() {
                            @Override
                            public boolean accept(File pathname) {
                                return pathname.getName().startsWith(innerClassName);
                            }
                        });
                        String packageName = getResourceName(resource.getParent());
                        for (File innerClass : innerClasses) {
                            result.add(packageName + "/" + innerClass.getName());
                        }
                    }
                } catch (JavaModelException e) {
                    // ignore as its just for optimization
                }
            } else if (objects[i] instanceof IFile) {
                result.add(getResourceName((IFile) objects[i]));
            } else if (objects[i] instanceof IClassFile) {
                IClassFile cf = (IClassFile) objects[i];
                String name = cf.getType().getFullyQualifiedName();
                // returns com.ericsson.action.ActionHandler
                // has to be com/ericsson/action/ActionHandler.class pattern.
                name = name.replace(".", "/");
                name += ".class";
                result.add(name);
            } else if (objects[i] instanceof IJarEntryResource) {
                IJarEntryResource jer = (IJarEntryResource) objects[i];
                String name = jer.getFullPath().toPortableString();
                // returns /com/ericsson/action/conf.properties
                // has to be com/ericsson/action/conf.properties
                name = name.substring(1);
                result.add(name);
            }
        }
        return result;
    }

    private String getResourceName(IResource resource) {
        IPackageFragmentRoot root = getPackageFragmentRoot(resource);
        if (root == null) {
            return null;
        } else {
            int index = root.getResource().getProjectRelativePath().toString().length() + 1;
            return resource.getProjectRelativePath().toString().substring(index);
        }
    }

    private IPackageFragmentRoot getPackageFragmentRoot(IResource resource) {
        IPackageFragmentRoot root = null;
        IResource r = resource;
        while (r != null) {
            IJavaElement javaElement = JavaCore.create(r);
            if (javaElement != null && javaElement instanceof IPackageFragmentRoot) {
                root = (IPackageFragmentRoot) javaElement;
                break;
            }
            r = r.getParent();
        }
        return root;
    }

    private void createLocalSaveSection() {
        Composite localSaveFormClient = createLocalSaveFormClient();
        createSaveLocationField(localSaveFormClient);
        createExportGroupsButton(localSaveFormClient);
        createSaveButton(localSaveFormClient);
    }

    private Composite createIncludeFilesSection() {
        Section includeFilesDetails = toolkit.createSection(form.getBody(), Section.TITLE_BAR | Section.DESCRIPTION);
        includeFilesDetails.marginWidth = 5;
        includeFilesDetails.setText("Files and Folders");
        includeFilesDetails.setLayoutData(new GridData(GridData.FILL_BOTH));

        Composite includeFilesFormClient = toolkit.createComposite(includeFilesDetails);
        includeFilesDetails.setClient(includeFilesFormClient);
        includeFilesDetails.setDescription("Select the files and folders to include in the template archive.");
        toolkit.paintBordersFor(includeFilesFormClient);

        GridLayout layout = new GridLayout();
        layout.marginWidth = 2;
        layout.marginHeight = 2;
        layout.numColumns = 1;
        includeFilesFormClient.setLayout(layout);

        Tree tree = toolkit.createTree(includeFilesFormClient, SWT.CHECK);
        tree.setLayoutData(new GridData(GridData.FILL_BOTH));

        includeFilesTreeViewer = new IncludeInDeploymentTreeViewer(tree);
        includeFilesTreeViewer.setContentProvider(new IncludeFilesTreeContentProvider());
        includeFilesTreeViewer.setLabelProvider(new WorkbenchLabelProvider());
        includeFilesTreeViewer.setInput(processFolder);
        includeFilesTreeViewer.addFilter(new ViewerFilter() {

            @Override
            public boolean select(Viewer viewer, Object parent, Object element) {
                return element instanceof IFile
                        && !StringUtils.equals(((IFile) element).getName(),
                                ((WatchPointTemplateGraphPage) editor.getGraphPage()).getImageSavePath().lastSegment());
            }
        });
        tree.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                includeFilesTreeViewer.setCheckedElements(getElementsToCheckFor(processFolder).toArray());
            }
        });

        final Button includeFilesDefaultButton = toolkit.createButton(includeFilesFormClient, "Reset Defaults",
                SWT.PUSH);
        includeFilesDefaultButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
        includeFilesDefaultButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                includeFilesDefaultButton.getDisplay().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        includeFilesTreeViewer.setCheckedElements(getElementsToCheckFor(processFolder).toArray());
                    }
                });
            }
        });

        return includeFilesFormClient;
    }

    private ArrayList<IResource> getElementsToCheckFor(IFolder folder) {
        ArrayList<IResource> list = new ArrayList<IResource>();
        try {
            IResource[] members = folder.members();
            for (int i = 0; i < members.length; i++) {
                list.add(members[i]);
                if (members[i] instanceof IFolder) {
                    list.addAll(getElementsToCheckFor((IFolder) members[i]));
                }
            }
        } catch (CoreException e) {
            Logger.logError(e);
        }
        return list;
    }

    private Set<ICompilationUnit> getElementsToCheckFor(IJavaProject project) {
        Set<ICompilationUnit> result = new HashSet<ICompilationUnit>();
        try {
            if (project != null) {
                Set<?> javaClassNames = JavaClassNameCollector.getJavaClassNames(editor.getProcessDefinition());
                Iterator<?> iterator = javaClassNames.iterator();
                while (iterator.hasNext()) {
                    IType type = project.findType((String) iterator.next());
                    if (type != null) {
                        result.add(type.getCompilationUnit());
                    }
                }
            }
        } catch (JavaModelException e) {
            Logger.logError(e);
        }
        return result;
    }

    private Composite createIncludeClassesSection() {
        Section includeClassesDetails = toolkit.createSection(form.getBody(), Section.TITLE_BAR | Section.DESCRIPTION);
        includeClassesDetails.marginWidth = 5;
        includeClassesDetails.setText("Java Classes and Resources");
        includeClassesDetails.setLayoutData(new GridData(GridData.FILL_BOTH));

        Composite includeClassesFormClient = toolkit.createComposite(includeClassesDetails);
        includeClassesDetails.setClient(includeClassesFormClient);
        includeClassesDetails
        .setDescription("Select the Java classes and resources to include in the template archive.");
        toolkit.paintBordersFor(includeClassesFormClient);

        GridLayout layout = new GridLayout();
        layout.marginWidth = 2;
        layout.marginHeight = 2;
        layout.numColumns = 1;
        includeClassesFormClient.setLayout(layout);

        Tree tree = toolkit.createTree(includeClassesFormClient, SWT.CHECK);
        tree.setLayoutData(new GridData(GridData.FILL_BOTH));

        includeClassesTreeViewer = new IncludeInDeploymentTreeViewer(tree);
        includeClassesTreeViewer.setContentProvider(new IncludeClassesTreeContentProvider());
        includeClassesTreeViewer.setLabelProvider(new WorkbenchLabelProvider());
        final IJavaProject project = JavaCore.create(processFolder.getProject());
        if (project != null) {
            includeClassesTreeViewer.setInput(project);
        }
        composite.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                includeClassesTreeViewer.setCheckedElements(getElementsToCheckFor(project).toArray());
            }
        });

        final Button includeClassesDefaultButton = toolkit.createButton(includeClassesFormClient, "Reset Defaults",
                SWT.PUSH);
        includeClassesDefaultButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
        includeClassesDefaultButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                composite.getDisplay().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        includeClassesTreeViewer.setCheckedElements(getElementsToCheckFor(project).toArray());
                    }
                });
            }
        });

        return includeClassesFormClient;
    }

    private Composite createLocalSaveFormClient() {
        Section httpInfoDetails = toolkit.createSection(form.getBody(), Section.TITLE_BAR | Section.DESCRIPTION);
        httpInfoDetails.marginWidth = 5;
        httpInfoDetails.setText("Local Save Settings");
        GridData gridData = new GridData(GridData.FILL_BOTH);
        gridData.verticalAlignment = GridData.BEGINNING;
        gridData.horizontalSpan = 2;
        httpInfoDetails.setLayoutData(gridData);

        Composite detailClient = toolkit.createComposite(httpInfoDetails);
        httpInfoDetails.setClient(detailClient);
        httpInfoDetails.setDescription("Choose the location where the process archive will be stored.");
        toolkit.paintBordersFor(detailClient);

        GridLayout layout = new GridLayout();
        layout.marginWidth = 2;
        layout.marginHeight = 2;
        layout.numColumns = 3;
        detailClient.setLayout(layout);
        return detailClient;
    }

    private void updateSaveAndDeployButtons() {
        if (notEmpty(locationText)) {
            saveButton.setEnabled(true);
        } else {
            saveButton.setEnabled(false);
        }
    }

    private boolean notEmpty(Text text) {
        String string = text.getText();
        return string != null && !"".equals(string);
    }

    private String getSuggestedName() {
        IPreferenceStore prefs = WatchPointStudioPlugin.getDefault().getPreferenceStore();
        String suggestedName = prefs.getString(EXPORT_DIR);

        if (suggestedName != null && !suggestedName.equals("")) {
            suggestedName = suggestedName.trim();
            if (!suggestedName.endsWith(File.separator)) {
                suggestedName += File.separator;
            }
            suggestedName += TEMPLATES_FOLDER_NAME + File.separator;
        } else {
            suggestedName = "";
        }
        suggestedName += editor.getPartName() + WorkflowTemplateDTO.DEFAULT_TEMPLATE_PACKAGE_EXTENSION;
        return suggestedName;
    }

    @SuppressWarnings("deprecation")
    private void createSaveLocationField(Composite localSaveFormclient) {
        Label locationLabel = toolkit.createLabel(localSaveFormclient, "Location:");
        locationLabel.setForeground(toolkit.getColors().getColor(FormColors.TITLE));

        locationText = toolkit.createText(localSaveFormclient, getSuggestedName());
        locationText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        locationText.addKeyListener(new KeyAdapter() {
            @Override
            public void keyReleased(KeyEvent e) {
                updateSaveAndDeployButtons();
            }
        });
        locationButton = toolkit.createButton(localSaveFormclient, "Search...", SWT.PUSH);
        locationButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                searchLocation();
            }
        });
    }

    private void createExportGroupsButton(Composite localSaveFormclient) {
        exportImageButton = toolkit.createButton(localSaveFormclient, UI.Exportpage_ExportImage, SWT.CHECK);
        GridData data = new GridData(GridData.FILL_HORIZONTAL);
        data.horizontalSpan = 3;
        exportImageButton.setLayoutData(data);
        exportImageButton.setSelection(WatchPointStudioPlugin.getDefault().getPreferenceStore()
                .getBoolean(PreferenceConstants.EXPORT_IMAGE));

        exportGroupsButton = toolkit
                .createButton(localSaveFormclient, UI.ExportPage_AutoExportResourceGroup, SWT.CHECK);
        data = new GridData(GridData.FILL_HORIZONTAL);
        data.horizontalSpan = 3;
        exportGroupsButton.setLayoutData(data);
        exportGroupsButton.setSelection(WatchPointStudioPlugin.getDefault().getPreferenceStore()
                .getBoolean(PreferenceConstants.AUTO_EXPORT_RESOURCE_GROUPS));
    }

    private void searchLocation() {
        FileDialog dialog = new FileDialog(form.getShell(), SWT.OPEN);
        String result = dialog.open();
        if (result != null) {
            locationText.setText(result);
            updateSaveAndDeployButtons();
        }
    }

    private void createSaveButton(Composite localSaveFormClient) {
        saveButton = toolkit.createButton(localSaveFormClient, "Save...", SWT.PUSH);
        GridData gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = SWT.BEGINNING;
        saveButton.setLayoutData(gridData);
        saveButton.setEnabled(false);
        saveButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (!editor.isTemplateProject()) {
                    String message = "Template does not belong to a Template Project and thus can not be exported.";
                    MessageDialog.openError(Display.getCurrent().getActiveShell(), "Export error", message);
                } else {
                    if (cancelOrSaveAndContinue() && directoryExists()) {
                        WorkflowTemplateDTO template = null;
                        try {
                            template = export();
                            ProcessDeployer processDeployer = createProcessDeployer();
                            if (exportImageButton.getSelection()) {
                                try {
                                    ((WatchPointTemplateGraphPage) editor.getGraphPage()).saveImage();
                                    ArrayList<Object> includedFiles = getIncludedFiles();
                                    includedFiles.add(getImageFile());
                                    processDeployer.setFilesAndFolders(includedFiles);
                                } catch (Throwable err) {
                                    Utility.reportError(
                                            "Cannot generate .jpg file for template, with export .par without image.",
                                            err);
                                }
                            }
                            processDeployer.saveWithoutDeploying();
                            Utility.reportOperationCompleted("Template Export");
                        } catch (ExportFailureException e1) {
                            Utility.reportError("Template Export", e1);
                        }

                        if (template != null) {
                            try {
                                exportResourceGroups(template);
                            } catch (ExportFailureException e1) {
                                Utility.reportError("ResourceGroup Export", e1);
                            }
                        }
                    }
                }
            }
        });
    }

    private IFile getImageFile() {
        String filename = ((WatchPointTemplateGraphPage) editor.getGraphPage()).getImageSavePath().lastSegment();
        return processFolder.getFile(filename);
    }

    private void exportResourceGroups(WorkflowTemplateDTO template) throws ExportFailureException {
        ExportFailureException exception = null;

        if (exportGroupsButton.getSelection()) {
            IResourceGroupService resourceGroupService = ServiceFactory.createFactory().createResourceGroupService();

            Collection<ResourceGroup> resourceGroups = resourceGroupService.getResourceGroupsByTemplateUsage(template
                    .getPath());
            for (ResourceGroup group : resourceGroups) {
                try {
                    Map<String, Object> exportInputAttrs = new HashMap<String, Object>();
                    File pluginsFolder = new File(locationText.getText());
                    /*
                     * If Template export location is /opt/tandbergtv/cms/workflow/templates/MyTemplate.par resource
                     * group should be: /opt/tandbergtv/cms/workflow/plugins/groups/MyGroup/
                     */
                    pluginsFolder = new File(pluginsFolder.getParentFile().getParentFile(), "plugins/groups/"
                            + group.getName());
                    // Create destination folder
                    if (!pluginsFolder.exists()) {
                        pluginsFolder.mkdirs();
                    }

                    exportInputAttrs.put(ExporterInputAttributes.EXPORT_FOLDER_PATH, pluginsFolder.getAbsolutePath());
                    exportInputAttrs.put(ExporterInputAttributes.PLUGIN_CLASSES_FILE_PATHS, Collections.EMPTY_LIST);
                    exportInputAttrs.put(ExporterInputAttributes.PLUGIN_RESOURCE_FILE_PATHS, Collections.EMPTY_LIST);
                    exportInputAttrs.put(ExporterInputAttributes.PLUGIN_JAR_FILE_PATHS, Collections.EMPTY_LIST);

                    resourceGroupService.exportResourceGroup(group, exportInputAttrs);
                } catch (ExportFailureException e) {
                    // Save the exception to throw it later...
                    exception = e;
                    // Error exporting group, ignore...
                    Logger.logError("Error while trying to auto-export ResourceGroup", e);
                }
            }
        }
        if (exception != null) {
            // had exception, throw-it
            throw exception;
        }

    }

    private WorkflowTemplateDTO export() throws ExportFailureException {
        WorkflowTemplateDTO template = null;
        Object[] objects = includeFilesTreeViewer.getCheckedElements();
        for (Object o : objects) {
            IFile file = (IFile) o;
            if (file.getFileExtension().equals(WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME)
                    || file.getName().equals(WorkflowTemplateDTO.TEMPLATE_RUNTIME_FILE_NAME)) {
                IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
                template = service.exportTemplate(editor);
                break;
            }

        }
        return template;
    }

    public void refresh(final ArrayList<?> objectsToRefresh) {
        form.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                refreshIncludeClassesTreeViewer(objectsToRefresh);
                refreshIncludeFilesTreeViewer(objectsToRefresh);
            }
        });
    }

    private void refreshIncludeFilesTreeViewer(ArrayList<?> objectsToRefresh) {
        Object[] elements = includeFilesTreeViewer.getCheckedElements();
        includeFilesTreeViewer.refresh();
        includeFilesTreeViewer.setCheckedElements(elements);
        IWorkspaceRoot root = processFolder.getWorkspace().getRoot();
        for (int i = 0; i < objectsToRefresh.size(); i++) {
            IPath path = (IPath) objectsToRefresh.get(i);
            if (root.getFile(path).exists()) {
                includeFilesTreeViewer.setChecked(root.getFile(path), true);
            } else if (root.getFolder(path).exists()) {
                includeFilesTreeViewer.setChecked(root.getFolder(path), true);
            }
        }
    }

    private void refreshIncludeClassesTreeViewer(ArrayList<?> objectsToRefresh) {
        Set<?> referencedJavaClassNames = null;
        Object[] elements = includeClassesTreeViewer.getCheckedElements();
        includeClassesTreeViewer.refresh();
        includeClassesTreeViewer.setCheckedElements(elements);
        IWorkspaceRoot root = processFolder.getWorkspace().getRoot();
        for (int i = 0; i < objectsToRefresh.size(); i++) {
            IPath path = (IPath) objectsToRefresh.get(i);
            IJavaElement javaElement = JavaCore.create(root.getFile(path));
            if (javaElement != null && javaElement instanceof ICompilationUnit) {
                if (referencedJavaClassNames == null) {
                    referencedJavaClassNames = JavaClassNameCollector.getJavaClassNames(editor.getProcessDefinition());
                }
                String name = getTypeName((ICompilationUnit) javaElement);
                boolean checkNeeded = referencedJavaClassNames.contains(name);
                includeClassesTreeViewer.setChecked(javaElement, checkNeeded);
            }
        }
    }

    private String getTypeName(ICompilationUnit unit) {
        try {
            IType[] types = unit.getTypes();
            if (types.length > 0) {
                return types[0].getFullyQualifiedName();
            }
        } catch (JavaModelException e) {
            Logger.logError(e);
        }
        return null;
    }

    private boolean cancelOrSaveAndContinue() {
        IEditorPart editor = getEditorPart();
        boolean result = true;
        if (editor.isDirty()) {
            int saveProceedCancel = openSaveProceedCancelDialog();
            if (saveProceedCancel == 2) {
                result = false;
            } else if (saveProceedCancel == 0) {
                editor.doSave(null);
            }
        }
        return result;
    }

    private boolean directoryExists() {
        File folder = new File(locationText.getText()).getParentFile();
        boolean result = folder.isDirectory();

        if (result == false) {
            result = MessageDialog.openConfirm(form.getShell(), "Create Directory",
                    "The specified location does not exist. Create it now ?");
            if (result == true) {
                result = folder.mkdirs();
                if (result == false) {
                    MessageDialog.openError(form.getShell(), "Create Directory",
                            "Failed to create the specified location.");
                }
            }
        }
        return result;
    }

    private int openSaveProceedCancelDialog() {
        MessageDialog dialog = new MessageDialog(getWorkBenchWindow().getShell(), "Save Resource", null, "'"
                + processFolder.getName() + "' has been modified. Save changes before deploying?",
                MessageDialog.QUESTION, new String[] { IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL,
            IDialogConstants.CANCEL_LABEL }, 0);
        return dialog.open();

    }

    private IWorkbenchWindow getWorkBenchWindow() {
        return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
    }

    private IEditorPart getEditorPart() {
        return getWorkBenchWindow().getActivePage().getActiveEditor();
    }

    public void refresh() {
        if (composite.isDisposed() || !processFolder.exists()) {
            return;
        }
        composite.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                if (!includeClassesTreeViewer.getTree().isDisposed()) {
                    includeClassesTreeViewer.refresh();
                }
            }
        });
        composite.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                if (!includeFilesTreeViewer.getTree().isDisposed()) {
                    includeFilesTreeViewer.refresh();
                    includeFilesTreeViewer.setCheckedElements(getElementsToCheckFor(processFolder).toArray());
                }
            }
        });
    }

}
