package com.tandbergtv.watchpoint.studio.ui.editor;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.draw2d.RangeModel;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.ui.actions.Clipboard;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IEditorPart;
import org.jbpm.gd.common.model.NamedElement;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.notation.Node;
import org.jbpm.gd.common.notation.NodeContainer;
import org.jbpm.gd.common.notation.RootContainer;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;
import org.jbpm.gd.jpdl.notation.JpdlEdge;

import com.tandbergtv.watchpoint.studio.ui.model.WPTransition;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.util.SemanticElementCloneUtil;

public class TemplateEditorPasteCommand {
	
	private WatchPointTemplateEditor templateEditor;
	private int horizontalMoveFactor = 0;
	private int verticalMoveFactor = 0;
	List<Object> clipboardItems = null;
	List<NodeElement> newSemanticElementsList = null;
	List<Node> newNotationElementsList = null;
	List<Node> copiedNotationElementsList = null;
	WorkflowTemplate template;

	@SuppressWarnings("unchecked")
	public void execute() {
			clipboardItems = (List<Object>) Clipboard.getDefault().getContents();
			if (clipboardItems == null || clipboardItems.isEmpty()) {
				return;
			}
			
			/* Calculate the factor used to move the element being pasted */
			calculateDisplacementFactor();
			/* Initialize the semantic elements and notation elements list */
			initializeElementsLists();
			/* Create the items to paste based on the copied items */
			createItemsToPaste();
			/* Create transitions between the nodes being paste */
			createTransitions();
			/* Paste the items in the editor */
			pasteItemsInEditor();
			/* Add transitions listener and handle the editor */
			postProcess();
			
	}
	
	private void postProcess(){
		addTransitionsListener(newNotationElementsList);

		templateEditor.setDirty(true);
		templateEditor.getGraphicalViewer().flush();
	}
	
	private void pasteItemsInEditor(){
		for (NodeElement nodeElement : newSemanticElementsList) {
			RootContainer templateRootContainer = templateEditor.getRootContainer();
			template = (WorkflowTemplate) templateRootContainer.getSemanticElement();
			
			// If the template has a node with the same name, set a different name to the new node
			if ( template.getNodeElementByName(nodeElement.getName()) != null ){
				String oldName = nodeElement.getName();
				String newName = getNextAvailableName( nodeElement.getName() );
				nodeElement.setName(newName);
				
				fixTransitionsToAttribute(nodeElement, oldName);
			} 
			
			template.addNodeElement(nodeElement);
		}
	}
	
	private void createItemsToPaste(){
		for (Object clipboardItem : clipboardItems) {
			EditPart copiedElementGraphicalEditPart = (EditPart)clipboardItem;
			Node copiedElementModel = (Node)copiedElementGraphicalEditPart.getModel();
			copiedNotationElementsList.add(copiedElementModel);

			Node newNotationElement = createNotationElementCopy(copiedElementModel);
			
			// Set notation element position
			setNotationElementPosition(copiedElementModel, newNotationElement);
			
			newNotationElementsList.add(newNotationElement);
		}
	}
	
	private void initializeElementsLists(){
		/* List of semantic elements created to paste in the target editor */
		newSemanticElementsList = new ArrayList<NodeElement>();
		/* List of notation elements created to paste in the target editor */
		newNotationElementsList = new ArrayList<Node>();
		/* List of notation elements from the clipboard */
		copiedNotationElementsList = new ArrayList<Node>();
	}
	
	private void addTransitionsListener(List<Node> notationElements){
		for (Node node : notationElements) {
			for (Object edgeObject : node.getLeavingEdges()) {
				WPJpdlEdge edge = (WPJpdlEdge) edgeObject;
				edge.getSemanticElement().addPropertyChangeListener(edge);
			}
			if(node instanceof NodeContainer){
				NodeContainer container = (NodeContainer) node;
				addTransitionsListener(container.getNodes());
			}
		}
	}
	
	private void fixTransitionsToAttribute(NodeElement nodeElement, String oldName){
		for (NodeElement nodeElementInList : newSemanticElementsList) {
			for (Transition transition : nodeElementInList.getTransitions()) {
				if ( transition.getTo().equals(oldName) ){
					transition.setTo(nodeElement.getName());
				}
			}
		}
	}
	
	private Node createNotationElementCopy(Node copiedElementModel){
		// Create a new semantic element
		SemanticElement newSemanticElement = createSemanticElement(copiedElementModel);
		// Set the right semantic element source in the transitions
		fixSemanticElementTransitionsSource((NodeElement)newSemanticElement);
		
		newSemanticElementsList.add((NodeElement) newSemanticElement);
		// Create the new notation element
		Node notationElement = createNotationElement(copiedElementModel);
		notationElement.setSemanticElement(newSemanticElement);
		// Register the notation element
		registerNewNotationElement(notationElement);
		
		return notationElement;
	}
	
	private void registerNewNotationElement(Node notationElement){
		
		if(notationElement instanceof NodeContainer){
			NodeContainer container = (NodeContainer) notationElement;
			for (Object nodeObject : container.getNodes()) {
				Node node = (Node) nodeObject;
				registerNewNotationElement(node);
			} 
		}
		notationElement.register();
		notationElement.getSemanticElement().addPropertyChangeListener(notationElement);
	}
	
	private Node createNotationElement(Node element){
		Node notationElement = null;

		if(element instanceof WPNodeContainer){
			notationElement = createContainerNotationElement(element);
		} else {
			notationElement = (Node) templateEditor.getRootContainer().getFactory().create("org.jbpm.gd.jpdl.node");
		}
		
		return notationElement;
	}
	
	private Node createContainerNotationElement(Node element){
		Node containerNotationElement = (Node) templateEditor.getRootContainer().getFactory().create("org.jbpm.gd.jpdl.container");
		if(element instanceof WPNodeContainer){
			NodeContainer container = (NodeContainer)element;
			for (Object node : container.getNodes()) {
				Node notationElement = createNotationElement( (Node)node );
				((WPNodeContainer)containerNotationElement).addNode(notationElement);
			}
		}
		return containerNotationElement;
	}
	
	private void createTransitions(){
		// For each new node(notation element)
		for (Node newNode : newNotationElementsList) {
			// Create the leaving edges for the given node
			createLeavingEdges(newNode);
			// Set the target to the recently created edges
			setEdgesTarget(newNode, copiedNotationElementsList, newNotationElementsList);
		}
		
	}
	
	private void createLeavingEdges(Node newNode){
		NodeElement newNodeElement = (NodeElement)newNode.getSemanticElement();
		for (Transition transition : newNodeElement.getTransitions()) {
			WPJpdlEdge newEdge = createEdge();
			
			newEdge.setSource(newNode);
			newEdge.setSemanticElement(transition);
			
			newEdge.register();
			newNode.addLeavingEdge(newEdge);
		}
		if(newNode instanceof NodeContainer){
			NodeContainer nodeContainer = (NodeContainer) newNode;
			for (Object node : nodeContainer.getNodes()) {
				createLeavingEdges((Node)node);
			}
		}
	}
	
	private void setEdgesTarget(Node newNode, List<Node> references, List<Node> newNodes){
		// Get the reference node(notation element)
		Node referenceNode = getNodeFromListByName(newNode, references);
		// Look at the leaving edges of the reference node and search for the target node
		for (Object referenceEdgeObject : referenceNode.getLeavingEdges()) {
			WPJpdlEdge referenceEdge = (WPJpdlEdge) referenceEdgeObject;
			// Get the target node in the new created nodes
			Node newTargetNode = getNodeFromListByName(referenceEdge.getTarget(), newNodes);
			if(newTargetNode != null){
				WPJpdlEdge newEdge = getNewEdge(newNode, referenceEdge);
				newEdge.setTarget(newTargetNode);
				
				newTargetNode.addArrivingEdge(newEdge);
			} else {
				// If there is no new target node, remove this node transition
				WPJpdlEdge newEdge = getNewEdge(newNode, referenceEdge);
				removeNodeTransition(newNode, newEdge);
			}
			if( newNode.getContainer() == null || !(newNode.getContainer() instanceof WPNodeContainer) ){
				newNode.getSemanticElement().removePropertyChangeListener(newNode);
			}
		} 
		if(newNode instanceof NodeContainer){
			NodeContainer newNodeContainer = (NodeContainer) newNode;
			NodeContainer referenceNodeContainer = (NodeContainer) referenceNode;
			for (Object node : newNodeContainer.getNodes()) {
				setEdgesTarget((Node)node, referenceNodeContainer.getNodes(), newNodeContainer.getNodes());
			}
		}
	}
	
	private void removeNodeTransition(Node node, JpdlEdge edgeToRemove){
		NodeElement nodeElement = (NodeElement)node.getSemanticElement();
		List leavingEdgesCopy = new ArrayList(node.getLeavingEdges()); 
		
		for (Object edgeObject : leavingEdgesCopy) {
			WPJpdlEdge edge = (WPJpdlEdge) edgeObject;
			if(edge == edgeToRemove){
				nodeElement.removeTransition( (WPTransition)edge.getSemanticElement() );
				node.removeLeavingEdge(edge);
			}
		}
	}
	
	
	private WPJpdlEdge createEdge(){
		WPJpdlEdge newEdge = (WPJpdlEdge) templateEditor.getRootContainer().getFactory().create("org.jbpm.gd.jpdl.edge");
		return newEdge;
	}
	
	private void fixSemanticElementTransitionsSource(NodeElement nodeElement){
		if(nodeElement instanceof NodeElementContainer){
			NodeElementContainer nodeContainer = (NodeElementContainer) nodeElement;
			for (NodeElement containerNodeElement : nodeContainer.getNodeElements()) {
				fixSemanticElementTransitionsSource(containerNodeElement);
			}
			
		}
		for (Transition transition : nodeElement.getTransitions()) {
			transition.setSource(nodeElement);
		}
	}
	
	private WPJpdlEdge getNewEdge(Node newNode, WPJpdlEdge referenceEdge){
		WPTransition referenceTransition = (WPTransition) referenceEdge.getSemanticElement();
		for (Object newEdgeObject : newNode.getLeavingEdges()) {
			WPJpdlEdge newEdge = (WPJpdlEdge) newEdgeObject;
			WPTransition newTransition = (WPTransition) newEdge.getSemanticElement();
			if( referenceTransition.getTo().equals(newTransition.getTo()) ){
				return newEdge;
			}
		}
		return null;
	}
	
	private Node getNodeFromListByName(Node node, List<Node> list){
		NamedElement newNodeNamed = (NamedElement)node.getSemanticElement();
		for (Node referenceNode : list) {
			NamedElement referenceNodeNamed = (NamedElement)referenceNode.getSemanticElement();
			if(newNodeNamed.getName().equals(referenceNodeNamed.getName())){
				return referenceNode;
			}
		}
		return null;
	}
	
	private String generateNewName(String currentName){
		String newName = "";
		Pattern digitsPattern = Pattern.compile("\\d+");
		Matcher digitsMatcher = digitsPattern.matcher(currentName);
		
		if(digitsMatcher.find()){
			digitsMatcher.reset();
			while (digitsMatcher.find()) {
			    if (currentName.endsWith(digitsMatcher.group())) {
					String prefix = currentName.split(digitsMatcher.group())[0];
					newName = prefix + (Integer.parseInt( digitsMatcher.group() )+1 );
				}
			}
		}else{
			newName = currentName + 1;
		}
		return newName;
	}
	
	private String getNextAvailableName(String nodeName) {
		String result = nodeName;
		while (true) {
			if (template.getNodeElementByName(result) != null) {
				result = generateNewName(result);
			}else{
				break;
			}
		}
		return result;
	}
	
	private SemanticElement createSemanticElement(Node node) {
		SemanticElement element = null;
		if(node.getSemanticElement() != null){
			element = SemanticElementCloneUtil.cloneSemanticElement(node.getSemanticElement());
		} 
		
		return element;
	}

	private void setNotationElementPosition(Node reference, Node newNode){
		if(newNode instanceof NodeContainer){
			setContainerChildElementPosition(reference, newNode);
		}
		Rectangle referenceConstraint = reference.getConstraint(); 
		setConstraintValues(newNode, 
				referenceConstraint.x + horizontalMoveFactor, referenceConstraint.y + verticalMoveFactor, 
				referenceConstraint.width, referenceConstraint.height
		);
	}
	
	private void setContainerChildElementPosition(Node reference, Node newNode){
		NodeContainer newNodeContainer = (NodeContainer) newNode;
		NodeContainer referenceContainer = (NodeContainer) reference;
		int index = 0;
		for (Object objectNode : newNodeContainer.getNodes()) {
			Node newNodeChild = (Node) objectNode;
			Node referenceNode = (Node) referenceContainer.getNodes().get(index);
			Rectangle referenceNodeConstraint = referenceNode.getConstraint(); 
			setConstraintValues(newNodeChild, 
					referenceNodeConstraint.x, referenceNodeConstraint.y, 
					referenceNodeConstraint.width, referenceNodeConstraint.height
			);
			index++;
		}
	}
	
	private void setConstraintValues(Node node, int x, int y, int width, int height){
		Rectangle nodeConstraint = node.getConstraint();
		nodeConstraint.x = x;
		nodeConstraint.y = y;
		nodeConstraint.width = width;
		nodeConstraint.height = height;
	}
	
	/**
	 * Calculate the displacement factor for the node elements<br>
	 * This allow to paste the elements keeping their relative position. 
	 */
	private void calculateDisplacementFactor(){
		// Use this to calculate where to paste the objects
		RangeModel horizontal = templateEditor.getGraphPage().getDesignerModelViewer().getFigureCanvas().getViewport().getHorizontalRangeModel();
		RangeModel vertical = templateEditor.getGraphPage().getDesignerModelViewer().getFigureCanvas().getViewport().getVerticalRangeModel();
		
		int xPosition = horizontal.getValue() + (horizontal.getExtent()/2);
		int yPosition = vertical.getValue() + (vertical.getExtent()/2);
		
		// Calculate notation elements displacement factor
		Rectangle constraint = getFirstElementConstraint((EditPart)clipboardItems.get(0)) ;
		
		int yValue = constraint.y;
		int xValue = constraint.x;
		
		horizontalMoveFactor = xPosition - xValue;
		verticalMoveFactor = yPosition - yValue;
		
		horizontalMoveFactor -= 200;
		verticalMoveFactor -= 50;
	}
	
	public Rectangle getFirstElementConstraint(EditPart part){
		Node node = (Node)part.getModel();
		return node.getConstraint();
	}
	
	public void selectionChanged(IAction action, ISelection selection) {
		action.setEnabled(true);

	}

	public void setActiveEditor(IAction action, IEditorPart targetEditor) {
		if(targetEditor != null && targetEditor instanceof WatchPointTemplateEditor){
			this.templateEditor = (WatchPointTemplateEditor) targetEditor;
		}
	}

	public WatchPointTemplateEditor getTemplateEditor() {
		return templateEditor;
	}

	public void setTemplateEditor(WatchPointTemplateEditor templateEditor) {
		this.templateEditor = templateEditor;
	}
	
}
