package com.tandbergtv.watchpoint.studio.ui.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.jbpm.gd.common.model.AbstractSemanticElement;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.jpdl.model.AbstractNode;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.Event;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.application.StudioRuntimeException;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.ui.util.NameValuePair;
import com.tandbergtv.watchpoint.studio.util.SemanticElementCloneUtil;
import com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.nodedefinition.IRequiredVariableFinder;
import com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.nodedefinition.TemplateRequiredVariableFinder;

/**
 * This model represents either a Superstate or a Message Node. It depends on what the user drags in at the UI.
 * 
 * If Superstate is dragged, then a NodeGroup model is created and set to the "node" property of this model. In this case,
 * the NodeGroup will host the semantic elements of this super state and its variables will be inside this NodeDefinition model.
 * 
 * If a Message Node is dragged, then an Automatic Task Node is created and set to the "node" property of this model. In this case,
 * the incoming, outgoing and compositekey variables of the Automatic Task Node are used.
 * 
 * @author Imran Naqvi
 * 
 */
public class NodeDefinition extends WPAbstractNode implements IResourceGroupElement,
		IMessageNodeContainer, IDueDateElement, NodeElementContainer {

    public static final String DEPRECATED_ANNOTATION = "@deprecated";

    private long id;

	private AbstractNode node;

	private String definitionName;

	private long resourceType;

	private NodeDefinitionType nodeType;

	private final List<WPVariable> variables;
	
	private boolean incoming;

    private boolean highlight;
    
	/**
	 * Constructor.
	 */
	public NodeDefinition() {
		super();
		this.variables = new ArrayList<WPVariable>();
	}

	/**
	 * @return the id
	 */
	public long getId() {
		return id;
	}

	/**
	 * @param newId
	 *            the id to set
	 */
	public void setId(long newId) {
		long oldId = this.id;
		this.id = newId;
		firePropertyChange("id", oldId, newId);
	}
	
	public boolean isIncoming() {
		return incoming;
	}
	
	public void setIncoming(boolean incoming) {
		this.incoming = incoming;
	}

	/**
	 * @see org.jbpm.gd.common.model.AbstractNamedElement#setName(java.lang.String)
	 */
	@Override
	public void setName(String newName) {
		super.setName(newName);
		/* We keep the name of the internal node the same as that of that node definition */
		if (this.node != null) {
            this.node.setName(newName);
        }
	}

	/**
	 * @return the node
	 */
	public AbstractNode getNode() {
		return node;
	}

	/**
	 * @param newNode
	 *            the node to set
	 */
	public void setNode(AbstractNode newNode) {
		AbstractNode oldNode = this.node;
		this.node = newNode;
		if (newNode instanceof AutomaticTaskNode) {
		    ((AutomaticTaskNode) newNode).setParent(this);
			firePropertyChange("auto-task", oldNode, newNode);
		} else {
		    ((WPAbstractNode)newNode).setParent(this);
			firePropertyChange("super-state", oldNode, newNode);
		}
	}

	/**
	 * @return the definitionName
	 */
	public String getDefinitionName() {
		return definitionName;
	}

	/**
	 * @param newDefinitionName
	 *            the definitionName to set
	 */
	public void setDefinitionName(String newDefinitionName) {
		String oldDefinitionName = this.definitionName;
		this.definitionName = newDefinitionName;
		firePropertyChange("definitionName", oldDefinitionName, newDefinitionName);
	}

	/**
	 * @return the nodeType
	 */
	public NodeDefinitionType getNodeType() {
		if (this.nodeType != null) {
            return nodeType;
        }
		if (this.node instanceof NodeElementContainer) {
            return NodeDefinitionType.SuperState;
        }
		return NodeDefinitionType.MessageNode;
	}

	/**
	 * @param nodeType
	 *            the nodeType to set
	 */
	public void setNodeType(NodeDefinitionType nodeType) {
		this.nodeType = nodeType;
	}

	/**
	 * @return the resourceType
	 */
	@Override
    @Deprecated
	public long getResourceType() {
		return resourceType;
	}

	/**
	 * @param newResourceType
	 *            the resourceType id to set
	 */
	public void setResourceType(long newResourceType) {
		String oldResourceType = this.definitionName;
		this.resourceType = newResourceType;
		firePropertyChange("resourceType", oldResourceType, newResourceType);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement#getResourceGroup()
	 */
	@Override
    public ResourceGroup getResourceGroup() {
		if (this.node == null) {
            return null;
        }

		return ((IResourceGroupElement) this.node).getResourceGroup();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement#setResourceGroup(ResourceGroup)
	 */
	@Override
    public void setResourceGroup(ResourceGroup group) {
		((IResourceGroupElement) this.node).setResourceGroup(group);
	}

	public Collection<NodeElement> findNodes(long id) {
		Collection<NodeElement> nodes = new ArrayList<NodeElement>();
		NodeElement [] elements = getNodeElements();
		
		for (NodeElement element : elements) {
			if (element instanceof NodeDefinition && NodeDefinition.class.cast(element).getId() == id) {
                nodes.add(element);
            }
		}
		
		return nodes;
	}
	
	// ========================================================================
	// ===================== IMessageNodeContainer Methods
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getUid()
	 */
	@Override
    public String getUid() {
		if (this.node instanceof AutomaticTaskNode) {
			return ((AutomaticTaskNode) this.node).getUid();
		}
		return null;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#setUid(java.lang.String)
	 */
	@Override
    public void setUid(String uid) {
		if (this.node instanceof AutomaticTaskNode) {
			((AutomaticTaskNode) this.node).setUid(uid);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getUid()
	 */
	public String getRollbackUid() {
		if (this.node instanceof AutomaticTaskNode) {
			return ((AutomaticTaskNode) this.node).getRollbackUid();
		}
		return null;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getUid()
	 */
	public String getRollbackDefinitionName() {
		if (this.node instanceof AutomaticTaskNode) {
			return ((AutomaticTaskNode) this.node).getRollbackDefinitionName();
		}
		return null;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#setUid(java.lang.String)
	 */
	public void setRollbackNode(NodeDefinition rollbackNode) {
		if (this.node instanceof AutomaticTaskNode) {
		    ((AutomaticTaskNode) this.node).setRollbackNode(rollbackNode);
		}
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#removeRollbackNode()
	 */
	public void removeRollbackNode() {
		if (this.node instanceof AutomaticTaskNode) {
		    ((AutomaticTaskNode) this.node).removeRollbackNode();
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#getVariables()
	 */
	@Override
    public List<WPVariable> getVariables() {
		if (NodeDefinitionType.SuperState.equals(getNodeType())) {
            return this.variables;
        } else {
			if (this.node == null) {
				return new ArrayList<WPVariable>();
			}
			IMessageContainer messageContainer = (IMessageContainer) this.node;
			List<WPVariable> vars = messageContainer.getIncomingVariables();
			vars.addAll(messageContainer.getOutgoingVariables());
			vars.addAll(getNodeCompositeKeys());
			return vars;
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#getVariables()
	 */
	public List<WPVariable> getRollbackVariables() {
		List<WPVariable> vars = new ArrayList<WPVariable>();
		if (this.node instanceof AutomaticTaskNode) {
            vars.addAll(((AutomaticTaskNode) this.node).getRollbackIncomingVariables());
		    vars.addAll(((AutomaticTaskNode) this.node).getRollbackOutgoingVariables());
		}
		return vars;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
    public boolean hasVariable(String name) {
		for (WPVariable variable : getVariables()) {
			if (variable.getMappedName().equals(name)) {
                return true;
            }
		}
		
		return false;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean hasRollbackVariable(String name) {
		for (WPVariable variable : getRollbackVariables()) {
			if (variable.getMappedName().equals(name)) {
                return true;
            }
		}
		
		return false;
	}
	
	/**
	 * This method should only be called for super-state node definitions. FIXME WHY???????
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#addVariable(com.tandbergtv.watchpoint.studio.ui.model.WPVariable)
	 */
	@Override
    public void addVariable(WPVariable variable) {
		if (this.node instanceof NodeElementContainer) {
			if (hasVariable(variable.getMappedName())) {
                return;
            }
			
			this.variables.add(variable);
	
			firePropertyChange("variableAdd", null, variable);
		}
	}
	
	/**
	 * This method should only be called for super-state node definitions.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#removeVariable(java.lang.String)
	 */
	@Override
    public WPVariable removeVariable(String name) {
		WPVariable variable = null;
		if (this.node instanceof NodeElementContainer) {
			for (WPVariable var : this.variables) {
				if (!(var.getMappedName().equals(name))) {
                    continue;
                }

				variable = this.variables.remove(this.variables.indexOf(var));
				firePropertyChange("variableRemove", var, null);
				break;
			}
		}
		return variable;
	}
	
	/**
	 * Returns the variable whose mapped name matches the specified value
	 * @param name
	 * @return
	 */
	public WPVariable getVariable(String name) {
		for (WPVariable variable : getVariables()) {
			
			if (variable.getMappedName().equals(name)) {
                return variable;
            }
		}
		
		return null;
	}
	
	/**
	 * @param name
	 */
	public void removeVariableRecursively(String name) {
		if (!(this.node instanceof NodeElementContainer)) {
            return;
        }
		
		removeVariable(name);
		
		for (NodeElement element : getNodeElements()) {
			if (element instanceof NodeDefinition) {
				NodeDefinition child = (NodeDefinition) element;
				List<WPVariable> innerVariables = child.getVariables();
				
				if (innerVariables == null) {
                    continue;
                }
				
				IMessageContainer container = (IMessageContainer) child.getNode();
				
				for (WPVariable innerVariable : innerVariables) {
					if (name.equals(innerVariable.getName())) {
                        container.removeVariable(innerVariable.getMappedName());
                    }
				}
			}
		}
	}
	
	/**
	 * Gets a child node definition using the specified definition name. This call only applies
	 * if this is a superstate node definition.
	 * 
	 * @param definitionName
	 * @return
	 */
	public NodeDefinition getNodeDefinitionByDefinitionName(String definitionName) {
		if (!(this.node instanceof NodeElementContainer)) {
            return null;
        }
		
		for (NodeElement element : getNodeElements()) {
			if (!(element instanceof NodeDefinition)) {
                continue;
            }

			NodeDefinition child = NodeDefinition.class.cast(element);

			if (definitionName.equals(child.getDefinitionName())) {
                return child;
            }
		}
		
		return null;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getConstants()
	 */
	@Override
    public List<NameValuePair> getConstants() {
		List<NameValuePair> constants = new ArrayList<NameValuePair>();
		if (this.node == null) {
            return constants;
        }

		if (this.node instanceof NodeElementContainer) {
			NodeElementContainer superState = (NodeElementContainer) this.node;
			for (NodeElement nodeElement : superState.getNodeElements()) {
				if (!(nodeElement instanceof IMessageNodeContainer)) {
                    continue;
                }

				IMessageNodeContainer messageNodeContainer = (IMessageNodeContainer) nodeElement;
				constants.addAll(messageNodeContainer.getConstants());
			}
		}
		else {
			IMessageContainer messageContainer = (IMessageContainer) this.node;
			for (String key : messageContainer.getConstants().keySet()) {
				constants.add(new NameValuePair(key, messageContainer.getConstants().get(
						key)));
			}
		}
		return constants;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
    public String getConstantValue(String name) {
		/* Superstates don't have constants */
		if (this.node instanceof NodeElementContainer) {
            return null;
        }
		
		return IMessageContainer.class.cast(this.node).getConstantValue(name);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#setConstantValue(java.lang.String,
	 *      java.lang.String)
	 */
	@Override
    public void setConstantValue(String varName, String val) {
		if (this.node instanceof NodeElementContainer) {
			throw new StudioRuntimeException("Can not set constant values for super-states.");
		} else {
            ((IMessageContainer) this.node).setConstantValue(varName, val);
        }
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getCompositeKey(java.lang.String)
	 */
	@Override
    public String getCompositeKey(String varName) {
		if (this.node instanceof NodeElementContainer) {
            return null;
        } else {
            return ((IMessageContainer) this.node).getCompositeKey(varName);
        }
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#setCompositeKey(java.lang.String,
	 *      java.lang.String)
	 */
	@Override
    public void setCompositeKey(String varName, String val) {
		if (NodeDefinitionType.SuperState.equals(getNodeType())) {
			throw new StudioRuntimeException("Can not set composite key for a super-state.");
		}
		
		((IMessageContainer) this.node).setCompositeKey(varName, val);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getCompositeKeys()
	 */
	@Override
    public Map<String, String> getCompositeKeys() {
		Map<String, String> keys = new HashMap<String, String>();

		if (this.node == null) {
            return keys;
        }

		if (!(this.node instanceof NodeElementContainer)) {
			IMessageContainer messageContainer = (IMessageContainer) this.node;
			keys.putAll(messageContainer.getCompositeKeys());
		}
		return keys;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getIncomingVariables()
	 */
	@Override
    public List<WPVariable> getIncomingVariables() {
		List<WPVariable> incomingVars = new ArrayList<WPVariable>();
		if (!(this.node instanceof NodeElementContainer)) {
			IMessageContainer messageContainer = (IMessageContainer) this.node;
			addIncomingVariablesFromNode(incomingVars, messageContainer);
		}
		return incomingVars;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageNodeContainer#getOutgoingVariables()
	 */
	@Override
    public List<WPVariable> getOutgoingVariables() {
		List<WPVariable> outgoingVars = new ArrayList<WPVariable>();
		if (!(this.node instanceof NodeElementContainer)) {
			IMessageContainer messageContainer = (IMessageContainer) this.node;
			addOutgoingVariablesFromNode(outgoingVars, messageContainer);
		}
		return outgoingVars;
	}

	// ========================================================================
	// ===================== END IMessageNodeContainer Methods
	// ========================================================================
	
	/**
	 * @param node
	 * @return
	 */
	public boolean matchUids(IMessageNodeContainer node) {
		if (this.node instanceof NodeElementContainer) {
            return true;
        }
		
		return getUid().equals(node.getUid());
	}
	

	// ========================================================================
	// ===================== IDueDateElement Methods
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IDueDateElement#getDueDate()
	 */
	@Override
    public String getDueDate() {
		if (this.node == null) {
            return "";
        }
		return ((IDueDateElement) this.node).getDueDate();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IDueDateElement#setDueDate(java.lang.String)
	 */
	@Override
    public void setDueDate(String dueDate) {
		if (this.node != null) {
            ((IDueDateElement) this.node).setDueDate(dueDate);
        }
	}

	// ========================================================================
	// ===================== END IDueDateElement Methods
	// ========================================================================

	// ========================================================================
	// ===================== NodeElementContainer Methods
	// ========================================================================

	@Override
    public void addNodeElement(NodeElement node) {
		if (this.node instanceof NodeGroup) {
			((NodeGroup) this.node).addNodeElement(node);
			firePropertyChange("nodeElementAdd", null, node);
		}
	}

	@Override
    public boolean canAdd(NodeElement node) {
		if (this.node instanceof NodeGroup) {
            return node.isPossibleChildOf(this);
        }
		return false;
	}

	@Override
    public NodeElement getNodeElementByName(String nodeName) {
		if (this.node instanceof NodeGroup) {
            return ((NodeGroup) this.node).getNodeElementByName(nodeName);
        } else if (this.node.getName().equals(nodeName)) {
            return this.node;
        }
		return null;
	}

	@Override
    public NodeElement[] getNodeElements() {
		if (this.node instanceof NodeGroup) {
            return ((NodeGroup) this.node).getNodeElements();
        } else {
            return new NodeElement[] { this.node };
        }
	}

	@Override
    public void removeNodeElement(NodeElement node) {
		if (this.node instanceof NodeGroup) {
			((NodeGroup) this.node).removeNodeElement(node);
			firePropertyChange("nodeElementRemove", node, null);
		}
	}

	// ========================================================================
	// ===================== END NodeElementContainer METHODS
	// ========================================================================

	/*
	 * @param incomingVars @param node
	 */
	private void addIncomingVariablesFromNode(List<WPVariable> incomingVars, IMessageContainer node) {
		if (node == null) {
            return;
        }
		incomingVars.addAll(node.getIncomingVariables());
	}

	/*
	 * @param outgoingVars @param node
	 */
	private void addOutgoingVariablesFromNode(List<WPVariable> outgoingVars, IMessageContainer node) {
		if (node == null) {
            return;
        }
		outgoingVars.addAll(node.getOutgoingVariables());
	}

	private List<WPVariable> getNodeCompositeKeys() {
		List<WPVariable> compositeKeys = new ArrayList<WPVariable>();
		if (this.node == null) {
            return compositeKeys;
        }

		if (!(this.node instanceof NodeElementContainer)) {

			IMessageContainer messageContainer = (IMessageContainer) this.node;
			for (String key : messageContainer.getCompositeKeys().keySet()) {
				WPVariable var = this.createWPVariable();
				var.setMappedName(key);
				var.setName(messageContainer.getCompositeKeys().get(key));
				var.setCompositeKey(true);
				compositeKeys.add(var);
			}
		}
		return compositeKeys;
	}

	/*
	 * Creates a new instance of WPVariable
	 * 
	 * @return a new instance of WPVariable
	 */
	private WPVariable createWPVariable() {
		return (WPVariable) this.getFactory().createById(
				SemanticElementConstants.VARIABLE_SEID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return getLabel() + " " + getName();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
            return true;
        }
		if (obj == null) {
            return false;
        }
		if (getClass() != obj.getClass()) {
            return false;
        }
		NodeDefinition other = (NodeDefinition) obj;
		if (definitionName == null) {
			if (other.definitionName != null) {
                return false;
            }
		} else if (!definitionName.equals(other.definitionName)) {
            return false;
        }
		if (id != other.id) {
            return false;
        }
		if (incoming != other.incoming) {
            return false;
        }
		if (nodeType != other.nodeType) {
            return false;
        }
		if (resourceType != other.resourceType) {
            return false;
        }
		if (variables == null) {
			if (other.variables != null) {
                return false;
            }
		} else if (!variables.equals(other.variables)) {
            return false;
        }
        if (!StringUtils.equals(getName(), other.getName())) {
            return false;
		}
		return true;
	}
	
	public NodeDefinition cloneNodeDefinition(){
		return SemanticElementCloneUtil.cloneNodeDefinition(this);
	}

    public boolean hasSetResourceAction() {
        String eventType = getEventTypeForSetResourceAction();
        
        if (node == null) {
            return false;
        }

        for (Event event : node.getEvents()) {
            if (event.getType().equals(eventType)) {
                for (ActionElement action : event.getActionElements()) {
                    if (action instanceof Action && ((Action)action).getRefName().equals("set-resource")) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public void toggleSetResourceAction() {
        if (hasSetResourceAction()) {
            removeSetResourceAction();
        } else {
            addSetResourceAction();
        }
    }

    private void removeSetResourceAction() {
        String eventType = getEventTypeForSetResourceAction();

        for (Event event : node.getEvents()) {
            if (event.getType().equals(eventType)) {
                for (ActionElement action : event.getActionElements()) {
                    if (action instanceof Action && ((Action)action).getRefName().equals("set-resource")) {
                        event.removeActionElement(action);
                        break;
                    }
                }
            }
        }
    }

    private void addSetResourceAction() {
        String eventType = getEventTypeForSetResourceAction();

        Event event = null;
        if (node.getEvents().length > 0) {
            for (int i = 0; i < node.getEvents().length; i++) {
                if (node.getEvents()[i].getType().equals(eventType)) {
                    event = node.getEvents()[i];
                    break;
                }
            }
        } else {
            event = (Event) getFactory().createById(SemanticElementConstants.EVENT_SEID);
            event.setType(eventType);
            node.addEvent(event);
        }

        Action action = (Action) getFactory().createById(SemanticElementConstants.ACTION_SEID);
        action.setRefName("set-resource");
        event.addActionElement(action);
    }

    private String getEventTypeForSetResourceAction() {
        String eventType = Event.PREDEFINED_EVENT_TYPES[5];
        if (getNodeType().equals(NodeDefinitionType.SuperState)) {
            eventType = Event.PREDEFINED_EVENT_TYPES[7];
        }
        return eventType;
    }

    public void setHighlight(boolean highlight) {
        this.highlight = highlight;
    }

    public boolean isHighlight() {
        return highlight;
    }

    @Override
    public void accept(SemanticElementVisitor visitor) {
        if (NodeDefinitionType.SuperState.equals(getNodeType())) {
            for (SemanticElement semanticElement : ((NodeGroup)node).getNodeElements()) {
                ((AbstractSemanticElement)semanticElement).accept(visitor);
            }
        }
        visitor.visit(this);
    }

    public void updateSuperStateRequiredVariables() {
        IRequiredVariableFinder variableFinder = new TemplateRequiredVariableFinder();
        Set<String> requiredNames = variableFinder.getRequiredVariableNames(this);

        for (WPVariable var : variables) {
            var.addSuperStateInfo();

            boolean required = (requiredNames != null && requiredNames.contains(var.getMappedName()));
            var.getSuperStateInfo().setRequired(required);
        }
    }

    public WPVariable addEmptySuperstateVariable() {
        WPVariable variable = (WPVariable) getFactory().createById(SemanticElementConstants.VARIABLE_SEID);
        variable.setMappedName(""); 
        variable.setType(DataType.STRING.name());
        variable.setAccess("read,write");

        addVariable(variable);

        return variable;
    }
}