/**
 * SynchronizationContext.java
 * Created Apr 17, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.util;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;

import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.MessageDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI;
import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.jpa.JPAPersistenceContext;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.ResourceTypeParameter;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;

/**
 * The context used during synchronization of a resource type
 * 
 * @author Sahil Verma
 */
public class SynchronizationContext implements ISynchronizationContext {
	
	private MessageDAI messageDAO;
	
	private NodeDefinitionDTODAI nodeDAO;
	
	private ResourceTypeDAI resourceTypeDAO;
	
	private WorkflowTemplateDTODAI templateDAO;
	
	private IPersistenceContext context;
	
	private DataAccessFactory factory;
	
	/**
	 * @param context
	 * @param factory
	 */
	public SynchronizationContext(IPersistenceContext context, DataAccessFactory factory) {
		super();
		this.context = context;
		this.factory = factory;
	}

	/**
	 * @return the messageDAO
	 */
	public MessageDAI getMessageDAO() {
		return messageDAO;
	}

	/**
	 * @return the nodeDAO
	 */
	public NodeDefinitionDTODAI getNodeDAO() {
		return nodeDAO;
	}

	/**
	 * @return the resourceTypeDAO
	 */
	public ResourceTypeDAI getResourceTypeDAO() {
		return resourceTypeDAO;
	}

	/**
	 * @return the templateDAO
	 */
	public WorkflowTemplateDTODAI getTemplateDAO() {
		return templateDAO;
	}

	/**
	 * {@inheritDoc}
	 */
	public void begin() {
		this.context.initialize();
		this.context.beginTransaction();
		
		/* Create DAOs */
		messageDAO = this.factory.getMessageDAO(context);
		nodeDAO = this.factory.getNodeDefinitionDTODAO(context);
		resourceTypeDAO = this.factory.getResourceTypeDAO(context);
		templateDAO = this.factory.getWorkflowTemplateDTODAO(context);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void end() {
		this.context.commitTransaction();
		this.context.close();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void abort() {
		if (this.context.getCurrentTransaction() != null) {
			EntityTransaction tx = EntityTransaction.class.cast(context.getCurrentTransaction());
			
			if (tx.isActive())
				this.context.rollbackTransaction();
		}
	}

	public void addMessage(Message message) {
		messageDAO.create(message);
	}
	
	public void deleteMessage(Message message) {
		messageDAO.delete(message.getId());
	}
	
	public void updateMessage(Message message) {
		messageDAO.update(message);
	}
	
	public void createNode(NodeDefinitionDTO node) {
		nodeDAO.create(node);
	}
	
	public void deleteNode(NodeDefinitionDTO node) {
		ResourceType rt = node.getResourceType();
		
		rt.removeNode(node);
		
		nodeDAO.delete(node.getId());
	}
	
	public void updateNode(NodeDefinitionDTO node) {
		nodeDAO.update(node);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public NodeDefinitionDTO mergeNode(NodeDefinitionDTO node) {
		return getEntityManager().merge(node);
	}

	/**
	 * {@inheritDoc}
	 */
	public void createResourceType(ResourceType type) {
		resourceTypeDAO.create(type);
	}

	public void updateResourceType(ResourceType type) {
		getEntityManager().merge(type);
	}

	/**
	 * {@inheritDoc}
	 */
	public void deleteParameter(ResourceTypeParameter parameter) {
		ResourceTypeParameter p = getEntityManager().merge(parameter);
		
		getEntityManager().remove(p);
	}

	/**
	 * {@inheritDoc}
	 */
	public void updateTemplate(WorkflowTemplateDTO template) {
		templateDAO.update(template);
	}
	
	protected EntityManager getEntityManager() {
		return JPAPersistenceContext.class.cast(this.context).getEntityManager();
	}
}
