package com.tandbergtv.watchpoint.studio.ui.view.resourcetype;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.navigator.IDescriptionProvider;
import org.eclipse.ui.statushandlers.StatusManager;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.NameValuePair;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;
/**
 * The Label Provider for the Explorer
 * @author Patrik Araujo
 */
public class ResourceTypeProjectExplorerLabelProvider extends LabelProvider implements IDescriptionProvider{
	
	private static final String RESOURCE_TYPE_IMAGE_PATH = "resource_type.png";

	private static final String INCOMING_MESSAGE_IMAGE_PATH = "message_incoming.png";
	
	private static final String OUTGOING_MESSAGE_IMAGE_PATH = "message_outgoing.png";

	private static final String MESSAGE_VARIABLE_IMAGE_PATH = "icon_node.png";
	
	
	/**
	 * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
	 */
	@Override
	public String getText(Object element) {
		String text = "";
		if(element instanceof ResourceType){
			ResourceType resourceType = (ResourceType)element;
			text = "Resource Type : " + resourceType.getName();
		}
		
		if (element instanceof NodeDefinitionDTO) {
			NodeDefinitionDTO nodeDefinition = (NodeDefinitionDTO)element;
			text = nodeDefinition.getName();
			if(nodeDefinition.getType() == NodeDefinitionType.MessageNode){
				text += " (Message)";
			}else{
				text += " (SuperState)";
			}
		}
		
		if(element instanceof WPVariable){
			WPVariable var = (WPVariable) element;
			text = var.getMappedName();
			String readorwrite = (var.getAccess() != null)? var.getAccess():"";
			if (readorwrite.equals("write"))
				text += " : in";
			else if (readorwrite.equals("read"))
				text += " : out";
			
			if (var.isCompositeKey())
				text += " : key";
		}
		
		if(element instanceof NameValuePair){
			NameValuePair var = (NameValuePair) element;
			text = var.getName() + " : out (Constant)";
		}
		
		return text;
	}

	/**
	 * @see org.eclipse.jface.viewers.LabelProvider#getImage(java.lang.Object)
	 */
	@Override
	public Image getImage(Object element) {
		// TODO Cache the images or share them. Not tested with images
		Image image = null;
		
		if (element instanceof ResourceType) {
			image = this.createImage(RESOURCE_TYPE_IMAGE_PATH);
		}
		
		if (element instanceof NodeDefinitionDTO) {
			NodeDefinitionDTO nodeDefinition = (NodeDefinitionDTO)element;
			if(nodeDefinition.getType() == NodeDefinitionType.SuperState){
				image = this.createImage(OUTGOING_MESSAGE_IMAGE_PATH);
			}else{
				String uid = getNodeDefinitionUID(nodeDefinition.getXml());
				if(!uid.startsWith("01")){
					image = this.createImage(INCOMING_MESSAGE_IMAGE_PATH);
				}else{
					image = this.createImage(OUTGOING_MESSAGE_IMAGE_PATH);
				}
			}
		}
		
		if (element instanceof WPVariable) {
			image = this.createImage(MESSAGE_VARIABLE_IMAGE_PATH);
		}
		
		if (element instanceof NameValuePair){
			image = this.createImage(MESSAGE_VARIABLE_IMAGE_PATH);
		}
		
		return image;
	}

	/*
	 * Create a new Image from the relative path, or null if the path is
	 * null.
	 */
	private Image createImage(String path) {
		if (path == null)
			return null;

		ImageDescriptor descriptor = Utility.getImageDescriptor(path);
		return new Image(Display.getCurrent(), descriptor.getImageData());
	}
	
	private String getNodeDefinitionUID(String xmlString) {
		String uid = null;
		try {
			Document doc = XMLDocumentUtility.loadXml(xmlString);

			XPath xpath = XPathFactory.newInstance().newXPath();
			String sendNodeUID = (String) xpath.evaluate("//action/send", doc.getDocumentElement(), XPathConstants.STRING);
			sendNodeUID = sendNodeUID.trim();

			String receiveNodeUID = (String) xpath.evaluate("//action/receive",	doc.getDocumentElement(), XPathConstants.STRING);
			receiveNodeUID = receiveNodeUID.trim();

			uid = sendNodeUID.length() == 0 ? receiveNodeUID : sendNodeUID;
		} catch (Exception e) {
			String errorMessage = "Error fetching the Node Definition UID";
			Status status = new Status(IStatus.ERROR, "WatchPoint Studio",IStatus.ERROR, errorMessage, e);
			StatusManager.getManager().handle(status, StatusManager.SHOW);
		}
		return uid;
	}

	@Override
	public String getDescription(Object element) {
		String text = "";
		if(element instanceof ResourceType){
			ResourceType resourceType = (ResourceType)element;
			text = "Resource Type : " + resourceType.getName() +
				   " | System ID : " + resourceType.getSystemId() +
				   " | Connection Type : " + resourceType.getConnectionType().getName() +
				   " | Version : " + resourceType.getVersion();
		}
		
		if (element instanceof NodeDefinitionDTO) {
			NodeDefinitionDTO nodeDefinition = (NodeDefinitionDTO)element;
			text = nodeDefinition.getName();
			if(nodeDefinition.getType() == NodeDefinitionType.MessageNode){
				text += " (Message)";
				text += " | UID : " + getNodeDefinitionUID(nodeDefinition.getXml());
			}else{
				text += " (SuperState)";
			}
		}
		
		if(element instanceof WPVariable){
			WPVariable var = (WPVariable) element;
			text = "Parameter - " + var.getMappedName();
			String readorwrite = (var.getAccess() != null) ? var.getAccess() : "";
			if (readorwrite.equals("write"))
				text += " : in";
			else if (readorwrite.equals("read"))
				text += " : out";
			
			if(var.getType() != null && !var.getType().isEmpty())
				text += " | Type : " + var.getType();
			
			if( var.isCompositeKey() )
				text += " | Composite Key";
		}
		
		if(element instanceof NameValuePair){
			NameValuePair var = (NameValuePair) element;
			text = "Parameter - " + var.getName() + " : out (Constant)" +
			       " | Constant Value : " + var.getValue();
		}
		
		return text;
	}
}