/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.wizard;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;

import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * @author Raj Prakash
 *
 */
public class ResourceGroupWizard extends Wizard implements INewWizard {

	/**
	 * Maximum file size (in bytes) the readFile(String) method supports.
	 */
	private static final int MAX_IMAGE_FILE_SIZE = 50 * 1024;	//50 KB

	/**
	 * Maximum number of bytes read from the stream in one call.
	 */
	private static final int READ_SIZE = 1024;	//1 KB
	
	private ResourceGroupWizardPage page;
	private IResourceGroupService rgService;
	private ResourceGroupWizardPage.Mode mode;
	private ResourceGroup group;
	private ResourceType rtSelected;
	
	public ResourceGroupWizard() {
		this(ResourceGroupWizardPage.Mode.CREATE, null, null);
	}
	
	public ResourceGroupWizard(ResourceGroupWizardPage.Mode mode, ResourceGroup group, ResourceType rtSelected) {
		this.mode = mode;
		this.group = group;
		setWindowTitle(mode.getTitle());
		this.rgService = ServiceFactory.createFactory().createResourceGroupService();
		this.rtSelected = rtSelected;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench, org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	public void addPages() {
		page = new ResourceGroupWizardPage(this.mode, this.group, this.rtSelected);
		addPage(page);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	@Override
	public boolean performFinish() {
		switch(this.mode) {
		case CREATE:
			try {
				if(this.rgService.isNameExists(((ResourceGroupWizardPage)this.page).getResourceGroupName())) {
					this.page.setErrorMessage("A Resource Group with this name already exists");
					return false;
				}
			}
			catch(Exception e) {
				Utility.reportError("Error while checking if Name is unique", e);
				return false;
			}
			return createResourceGroup();
		case EDIT:
			return editResourceGroup();
		default:
			return true;
		}
	}

	
	//////////////////////////////////////////////////////////
	//					HELPER METHODS
	//////////////////////////////////////////////////////////

	private boolean createResourceGroup() {
		ResourceGroup newGroup = new ResourceGroup();
		
		try {
			fillEntityFromPage(newGroup, (ResourceGroupWizardPage) this.page);
			this.rgService.saveResourceGroup(newGroup);
		}
		catch(Exception e) {
			Utility.reportError("Resource Group Creation Failed.", e);
			return false;
		}
		
		return true;
	}
	
	private boolean editResourceGroup() {
		try {
			fillEntityFromPage(this.group, (ResourceGroupWizardPage) this.page);
			this.rgService.saveResourceGroup(this.group);
		}
		catch(Exception e) {
			Utility.reportError("Resource Group Edit Failed.", e);
			return false;
		}

		return true;
	}
	
	private void fillEntityFromPage(ResourceGroup group, ResourceGroupWizardPage rgPage) {
		group.setName(rgPage.getResourceGroupName());
		group.setResourceType(rgPage.getResourceType());
		group.setFunctionalType(convertBlankToNull(rgPage.getFunctionalType()));
		group.setInternallyAcquired(rgPage.isAllocationStyleInternal());
		group.setAllocationStrategyClassName(convertBlankToNull(rgPage.getAllocationStrategy()));
		if(this.mode != ResourceGroupWizardPage.Mode.CREATE && rgPage.isClearCurrentImage()) {
			group.setImageContent(null);
			group.setImageFileName(null);
		}
		if(rgPage.getImageFile() != null && rgPage.getImageFile().length() > 0) {
			group.setImageContent(readFile(rgPage.getImageFile()));
			group.setImageFileName((new File(rgPage.getImageFile())).getName());
		}
		group.setProtectionKey(rgPage.getProtectionKey());
		group.setVersion(1);
	}
	
	private String convertBlankToNull(String str) {
		if(str != null && str.trim().length() > 0) {
			return str;	
		}
		else {
			return null;
		}
	}
	
	/**
	 * Reads the file specified by filePath and returns a byte array of the file contents.
	 * 
	 * @param filePath The file to read
	 * @return	the file contents
	 * @throws UIException - Any IOException is wrapped as UIException and thrown.
	 */
	private byte[] readFile(String filePath) {
		File file = new File(filePath);
		int fileLengthInt = (int) file.length();
		
		if(fileLengthInt > MAX_IMAGE_FILE_SIZE)
			throw new UIException("File Too Big | Image File Max Size (bytes): " + MAX_IMAGE_FILE_SIZE);

		int arraySize = fileLengthInt;
		byte[] fileContents = new byte[arraySize];
		BufferedInputStream bis = null;
		try {
			bis = new BufferedInputStream(new FileInputStream(file));
			int offset = 0;
			while(true) {
				int remainingArraySize = arraySize - offset;
				int numBytesToRead = (READ_SIZE > remainingArraySize) ? remainingArraySize : READ_SIZE;
				int bytesRead = bis.read(fileContents, offset, numBytesToRead);
				if(bytesRead == -1)
					break;
				offset += bytesRead;
				if(offset == arraySize)
					break;
			}
			return fileContents;
		}
		catch(Exception ioe) {
			String message = "Exception while reading the image file: " + filePath;
			throw new UIException(message, ioe);
		}
		finally {
			if(bis != null) {
				try {
					bis.close();
				}
				catch(IOException ioe) {
					String message = "Exception while closing the image file: " + filePath;
					throw new UIException(message, ioe);
				}
			}
		}
	}
}
