/**
 * ResourceTypeImportService.java
 * Created Feb 22, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.resource;

import static com.tandbergtv.watchpoint.studio.ui.util.Utility.PLUGIN_ID;

import java.util.Collection;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;

import com.tandbergtv.watchpoint.studio.application.StudioRuntimeException;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.service.impl.ServiceImpl;
import com.tandbergtv.watchpoint.studio.ui.command.ICommand;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.DefaultTree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.MutableTreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.SimpleTreeVisitor;
import com.tandbergtv.watchpoint.studio.ui.sync.DiffKind;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiffCalculator;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiffModel;
import com.tandbergtv.watchpoint.studio.ui.sync.IUpdateElement;
import com.tandbergtv.watchpoint.studio.ui.sync.compare.ResourceTypeComparer;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.Diff;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.DiffModel;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.DiffTreeVisitor;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.ui.sync.util.SynchronizationContext;
import com.tandbergtv.watchpoint.studio.ui.wizard.imports.Importer;

/**
 * Imports resource types
 * 
 * @author Sahil Verma
 */
public class ResourceTypeImportService implements Importer {

	private IResourceTypeService service;
	
	private static final Logger logger = Logger.getLogger(ResourceTypeImportService.class);
	
	/**
	 * Ctor
	 */
	public ResourceTypeImportService() {
		super();
		this.service = ServiceFactory.createFactory().createResourceTypeService();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDiffModel getDiffModel(Collection<? extends IWatchPointDTO> items, IProgressMonitor pm) {
		ITreeNode<IDiff> root = new MutableTreeNode<IDiff>(Diff.BLANK);
		ITree<IDiff> tree = new DefaultTree<IDiff>(root);
		IDiffModel diffModel = new DiffModel(tree);

		for (IWatchPointDTO item : items) {
			ResourceType type = ResourceType.class.cast(item);
			
			try {
				ResourceType local = findResourceType(type.getSystemId());

				if (local == null) {
					root.addChild(new MutableTreeNode<IDiff>(new Diff(type, DiffKind.ADD)));
				} else {
					ITree<IDiff> difftree = getComparer().compare(local, type, new SubProgressMonitor(pm, 1));

					root.addChild(difftree.getRoot());
				}
			} catch (Exception e) {
				logger.warn("Problem generating diff for " + type.getName(), e);
				org.jbpm.gd.jpdl.Logger.log(new Status(IStatus.ERROR, PLUGIN_ID, 
					"Problem generating diff for " + type.getName(), e));
			}
		}
		
		return diffModel;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void importItems(IDiffModel diffModel, IProgressMonitor monitor) {
		ISynchronizationContext context = getSynchronizationContext();
		MultiStatus status = new MultiStatus(PLUGIN_ID, IStatus.INFO, "Imported items", null);
		ITree<IDiff> tree = diffModel.getDifferences();
		ITreeNode<IDiff> root = tree.getRoot();
		monitor.beginTask("Importing", root.getChildren().size());
		
		walk(tree);
		
		try {
			context.begin();
			for (ITreeNode<IDiff> node : root.getChildren()) {
				IStatus s = importItem(node, new SubProgressMonitor(monitor, 1), context);
				status.add(s);
			}
			context.end();
		} catch (Exception e) {
			context.abort();
			status.add(new Status(IStatus.ERROR, PLUGIN_ID, e.getMessage(), e));
			throw new StudioRuntimeException(e);
		} finally {
			monitor.done();
			org.jbpm.gd.jpdl.Logger.log(status);
		}
	}
	
	/**
	 * Imports the item contained in the given tree node
	 * 
	 * @param node
	 * @param pm
	 * @param context
	 * @return the status of the operation
	 */
	protected IStatus importItem(ITreeNode<IDiff> node, IProgressMonitor pm, ISynchronizationContext context) {
		IDiff diff = node.getData();
		ResourceType type = null;
		String message = null;
		MultiStatus status = null;
		
		try {
			pm.beginTask("Calculating changes for resource type", 3);

			if (diff instanceof IUpdateElement) {
				type = (ResourceType) IUpdateElement.class.cast(diff).getLocalModel();
				ResourceType remote = (ResourceType) diff.getModel();
				message = "Updated resource type " + remote.getName() + " - v" + remote.getVersion();
			} else {
				type = (ResourceType) diff.getModel();
				message = "Imported resource type " + type.getName() + " - v" + type.getVersion();
			}

			status = new MultiStatus(PLUGIN_ID, IStatus.INFO, message, null);
			DiffTreeVisitor visitor = new DiffTreeVisitor(type, context);

			ITree<IDiff> difftree = new DefaultTree<IDiff>(node);
			
			difftree.accept(visitor, node);

			pm.worked(1);

			executeCommands(visitor.getCommands(), new SubProgressMonitor(pm, 2), status);
			
			logger.info(message);
		} finally {
			pm.done();
		}
		
		return status;
	}
	
	protected void executeCommands(Collection<ICommand> commands, IProgressMonitor pm, MultiStatus status) {
		pm.beginTask("Executing commands", commands.size());

		try {
			for (ICommand command : commands) {
				if (command.isEnabled()) {
					command.execute();
					logger.debug(command);
					status.add(new Status(IStatus.INFO, PLUGIN_ID, command.toString()));
				}
				pm.worked(1);
			}
		} finally {
			pm.done();
		}
	}
	
	protected void walk(ITree<IDiff> tree) {
		tree.accept(new SimpleTreeVisitor<IDiff>(), tree.getRoot());
	}
	
	private ISynchronizationContext getSynchronizationContext() {
		ServiceImpl si = ServiceImpl.class.cast(this.service);
		IPersistenceContext context = si.getPersistenceContextFactory().createPersistenceContext();
		
		return new SynchronizationContext(context, si.getDataAccessFactory());
	}
	
	/**
	 * @param systemId
	 * @return
	 */
	private ResourceType findResourceType(String systemId) {
		return this.service.getResourceTypeBySystemId(systemId);
	}
	
	protected IDiffCalculator getComparer() {
		return new ResourceTypeComparer();
	}
}
