/*
 * Created on Nov 9, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement;

import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.VARIABLE_ACCESS_INVALID;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.VARIABLE_VALUE_INVALID;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Rule that performs validation of the Node Variables.
 * 
 * @param <NE>
 *            The Type of the Node Element
 * 
 * @author Vijay Silva
 */
public abstract class NodeVariablesRule<NE extends NodeElement> extends ValidationRule<NE>
{
	private static final String NAME_REGEX = "^.+$";

	private static final String MAPPED_NAME_REGEX = "^.+$";

	private static final String TYPE_REGEX = "^.+$";

	private static final Pattern NAME_PATTERN = Pattern.compile(NAME_REGEX);

	private static final Pattern MAPPED_NAME_PATTERN = Pattern.compile(MAPPED_NAME_REGEX);

	private static final Pattern TYPE_PATTERN = Pattern.compile(TYPE_REGEX);

	/**
	 * Validates the variables of a Node Element.
	 * 
	 * @param target
	 *            The Start State node to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(NE target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		List<Variable> variables = this.getVariables(target);
		if (variables != null)
		{
			/* Perform any group validations */
			this.validateVariables(target, variables, messages);

			/* Perform validation of each variable */
			for (Variable variable : variables)
			{
				this.validateVariableProperties(target, variable, messages);
			}
		}

		return messages;
	}

	/**
	 * Get the list of variables in the Node that needs to be validated.
	 * 
	 * @param node
	 *            The Node to validate
	 * 
	 * @return The list of variables in the node
	 */
	protected abstract List<Variable> getVariables(NE node);

	/**
	 * Perform validation of the Node Variables that generate validation messages specific to the
	 * node. Validates if the node contains any variables with blank names or blank mapped names.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variables
	 *            The list of node variables.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateVariables(NE node, List<Variable> variables,
			List<ValidationMessage> messages)
	{
		this.validateBlankNames(node, variables, messages);
		this.validateBlankMappedNames(node, variables, messages);
	}

	/**
	 * Validates all the variable properties. Validates the name, mapped name and type.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateVariableProperties(NE node, Variable variable,
			List<ValidationMessage> messages)
	{
		/* Validate the Name */
		this.validateNameBlank(node, variable, messages);
		this.validateNameValue(node, variable, messages);

		/* Validate the Mapped Name */
		this.validateMappedNameBlank(node, variable, messages);
		this.validateMappedNameValue(node, variable, messages);

		/* Validate the Type */
		this.validateTypeBlank(node, variable, messages);
		this.validateTypeValue(node, variable, messages);
		
		this.validateAccess(node, variable, messages);
		
		/* Values must match the type */
		this.validateValue(node, variable, messages);
	}

	// ========================================================================
	// ===================== GROUP VARIABLE VALIDATIONS
	// ========================================================================

	/**
	 * Validation to check if any of the node variables have a blank name. In case of a blank name,
	 * add a single validation message indicating that the node contains one or more variables with
	 * blank names.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variables
	 *            The list of node variables.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateBlankNames(NE node, List<Variable> variables,
			List<ValidationMessage> messages)
	{
		if (!this.isValidatingBlankNames(node))
			return;

		if (variables != null)
		{
			boolean hasBlanks = false;

			for (Variable variable : variables)
			{
				String name = variable.getName();
				if (this.isBlank(name))
				{
					hasBlanks = true;
					break;
				}
			}

			if (hasBlanks)
			{
				ValidationMessageCode code = this.getBlankNamesCode(node);
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code);
			}
		}
	}

	/**
	 * Validation to check if any of the node variables have a blank mapped name. In case of a blank
	 * mapped name, add a single validation message indicating that the node contains one or more
	 * variables with blank mapped names.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variables
	 *            The list of node variables.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateBlankMappedNames(NE node, List<Variable> variables,
			List<ValidationMessage> messages)
	{
		if (!this.isValidatingBlankMappedNames(node))
			return;

		if (variables != null)
		{
			boolean hasBlanks = false;

			for (Variable variable : variables)
			{
				String mappedName = variable.getMappedName();
				if (this.isBlank(mappedName))
				{
					hasBlanks = true;
					break;
				}
			}

			if (hasBlanks)
			{
				ValidationMessageCode code = this.getBlankMappedNamesCode(node);
				ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code);
			}
		}
	}

	// ========================================================================
	// ===================== VARIABLE PROPERTY VALIDATIONS
	// ========================================================================

	/**
	 * Validates that the Variable Name is not blank.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The node variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateNameBlank(NE node, Variable variable, List<ValidationMessage> messages)
	{
		if (!this.isValidatingNameBlank(node))
			return;

		if (this.isBlank(variable.getName()))
		{
			ValidationMessageCode code = this.getNameBlankCode(node);
			List<String> parameters = this.generateParameters(node, variable);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}

	/**
	 * Validates the value of a non-blank Variable name.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The node variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateNameValue(NE node, Variable variable, List<ValidationMessage> messages)
	{
		if (!this.isValidatingNameValue(node))
			return;

		// TODO: Update Regular Expression to exclude invalid format / characters.
		String name = variable.getName();
		if ((!this.isBlank(name)) && (!this.matchesPattern(NAME_PATTERN, name)))
		{
			ValidationMessageCode code = this.getNameInvalidCode(node);
			List<String> parameters = this.generateParameters(node, variable);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}

	/**
	 * Validates that the Variable Mapped Name is not blank.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The node variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateMappedNameBlank(NE node, Variable variable,
			List<ValidationMessage> messages)
	{
		if (!this.isValidatingMappedNameBlank(node))
			return;

		if (this.isBlank(variable.getMappedName()))
		{
			ValidationMessageCode code = this.getMappedNameBlankCode(node);
			List<String> parameters = this.generateParameters(node, variable);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}

	/**
	 * Validates the value of a non-blank Variable mapped name.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The node variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateMappedNameValue(NE node, Variable variable,
			List<ValidationMessage> messages)
	{
		if (!this.isValidatingMappedNameValue(node))
			return;

		// TODO: Update Regular Expression to exclude invalid format / characters.
		String mappedName = variable.getMappedName();
		if ((!this.isBlank(mappedName)) && (!this.matchesPattern(MAPPED_NAME_PATTERN, mappedName)))
		{
			ValidationMessageCode code = this.getMappedNameInvalidCode(node);
			List<String> parameters = this.generateParameters(node, variable);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}

	/**
	 * Validates that the Variable Type is not blank.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The node variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateTypeBlank(NE node, Variable variable, List<ValidationMessage> messages)
	{
		if (!this.isValidatingTypeBlank(node))
			return;

		String type = (variable instanceof WPVariable) ? ((WPVariable) variable).getType() : null;
		if (this.isBlank(type))
		{
			ValidationMessageCode code = this.getTypeBlankCode(node);
			List<String> parameters = this.generateParameters(node, variable);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}

	/**
	 * Validates the value of a non-blank Variable type.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * @param variable
	 *            The node variable being validated.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateTypeValue(NE node, Variable variable, List<ValidationMessage> messages)
	{
		if (!this.isValidatingTypeValue(node))
			return;

		// TODO: Update Regular Expression to include only allowed types.
		String type = (variable instanceof WPVariable) ? ((WPVariable) variable).getType() : null;
		if ((!this.isBlank(type)) && (!this.matchesPattern(TYPE_PATTERN, type)))
		{
			ValidationMessageCode code = this.getTypeInvalidCode(node);
			List<String> parameters = this.generateParameters(node, variable);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}

	/**
	 * Ensures that the access permissions on a variable are sane
	 * 
	 * @param node
	 * @param variable
	 * @param messages
	 */
	protected void validateAccess(NE node, Variable variable, List<ValidationMessage> messages)
	{
		if (variable.isReadable() || variable.isWritable())
			return;
		
		List<String> parameters = new ArrayList<String>();

		parameters.add((variable.getName() != null) ? variable.getName() : "");
		ValidationMessageAdder.getInstance().addValidationMessage(messages, node, VARIABLE_ACCESS_INVALID, parameters);
	}
	
	/**
	 * Checks a variable's value against its type
	 * 
	 * @param node
	 * @param variable
	 * @param messages
	 */
	protected void validateValue(NE node, Variable variable, List<ValidationMessage> messages)
	{
		WPVariable v = WPVariable.class.cast(variable);
		DataType datatype = DataType.STRING;
		
		if (v.getType() != null)
			datatype = DataType.valueOf(v.getType());
		
		String value = v.getValue();
		
		/* It's optional... */
		if (value == null || value.trim().length() == 0)
			return;
		
		if (!datatype.match(value))
		{
			List<String> parameters = new ArrayList<String>();

			parameters.add((variable.getName() != null) ? variable.getName() : "");
			parameters.add(value);
			
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, VARIABLE_VALUE_INVALID, parameters);
		}
	}

	// ========================================================================
	// ===================== REQUIRED VALIDATIONS
	// ========================================================================

	/**
	 * Checks if group validation for blank Variable Names is required. Default implementation
	 * returns true.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingBlankNames(NE node)
	{
		return true;
	}

	/**
	 * Checks if group validation of blank Variable Mapped Names is required. Default implementation
	 * returns true.
	 * 
	 * @param node
	 *            The Node being validated.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingBlankMappedNames(NE node)
	{
		return true;
	}

	/**
	 * Checks if the validation for a blank name for each variable in the node is required. Default
	 * implementation returns true.
	 * 
	 * @param node
	 *            The Node being validated.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingNameBlank(NE node)
	{
		return true;
	}

	/**
	 * Checks if validation is required for each variable in the node for the value of the name.
	 * Default implementation returns true.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingNameValue(NE node)
	{
		return true;
	}

	/**
	 * Checks if the validation for a blank mapped name for each variable in the node is required.
	 * Default implementation returns true.
	 * 
	 * @param node
	 *            The Node being validated.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingMappedNameBlank(NE node)
	{
		return true;
	}

	/**
	 * Checks if validation is required for each variable in the node for the value of the mapped
	 * name. Default implementation returns true.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingMappedNameValue(NE node)
	{
		return true;
	}

	/**
	 * Checks if validation is required for each variable for blank type. Default implementation
	 * returns true.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingTypeBlank(NE node)
	{
		return true;
	}

	/**
	 * Checks if validation is required for each variable for the value of the type. Default
	 * implementation returns true.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingTypeValue(NE node)
	{
		return true;
	}

	// ========================================================================
	// ===================== VALIDATION CODES
	// ========================================================================

	/**
	 * Get the Validation Message Code for a Node that contains one or more blank variable names.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLES_BLANK_NAMES}
	 */
	protected ValidationMessageCode getBlankNamesCode(NE node)
	{
		return ValidationMessageCode.NODE_VARIABLES_BLANK_NAMES;
	}

	/**
	 * Get the Validation Message Code for a Node that contains one or more blank variable mapped
	 * names.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLES_BLANK_MAPPED_NAMES}
	 */
	protected ValidationMessageCode getBlankMappedNamesCode(NE node)
	{
		return ValidationMessageCode.NODE_VARIABLES_BLANK_MAPPED_NAMES;
	}

	/**
	 * Get the Validation Message Code for a Variable in the Node with a blank name.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#VARIABLE_NAME_BLANK}
	 */
	protected ValidationMessageCode getNameBlankCode(NE node)
	{
		return ValidationMessageCode.VARIABLE_NAME_BLANK;
	}

	/**
	 * Get the Validation Message Code for a Variable in the Node with an invalid non-blank value
	 * for the name.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#VARIABLE_NAME_INVALID}
	 */
	protected ValidationMessageCode getNameInvalidCode(NE node)
	{
		return ValidationMessageCode.VARIABLE_NAME_INVALID;
	}

	/**
	 * Get the Validation Message Code for a Variable in the Node with a blank mapped name.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#VARIABLE_MAPPED_NAME_BLANK}
	 */
	protected ValidationMessageCode getMappedNameBlankCode(NE node)
	{
		return ValidationMessageCode.VARIABLE_MAPPED_NAME_BLANK;
	}

	/**
	 * Get the Validation Message Code for a Variable in the Node with an invalid non-blank value
	 * for the mapped name.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#VARIABLE_MAPPED_NAME_INVALID}
	 */
	protected ValidationMessageCode getMappedNameInvalidCode(NE node)
	{
		return ValidationMessageCode.VARIABLE_MAPPED_NAME_INVALID;
	}

	/**
	 * Get the Validation Message Code for a Variable in the Node with a blank type.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#VARIABLE_TYPE_BLANK}
	 */
	protected ValidationMessageCode getTypeBlankCode(NE node)
	{
		return ValidationMessageCode.VARIABLE_TYPE_BLANK;
	}

	/**
	 * Get the Validation Message Code for a Variable in the Node with an invalid non-blank value
	 * for the type.
	 * 
	 * @param node
	 *            The Node being validated
	 * 
	 * @return {@link ValidationMessageCode#VARIABLE_TYPE_INVALID}
	 */
	protected ValidationMessageCode getTypeInvalidCode(NE node)
	{
		return ValidationMessageCode.VARIABLE_TYPE_INVALID;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/* Checks if the value is blank, where null is also considered blank. */
	private boolean isBlank(String value)
	{
		return (value == null || value.trim().length() == 0);
	}

	/* Validates that the value specified matched the regular expression specified */
	private boolean matchesPattern(Pattern pattern, String input)
	{
		return pattern.matcher(input).matches();
	}

	/*
	 * Generates a list of String parameters, where the first value is the variable name, and the
	 * second is the variable mapped name. If the values are null, they are replaced with empty
	 * string.
	 */
	private List<String> generateParameters(NE node, Variable variable)
	{
		List<String> parameters = new ArrayList<String>();

		parameters.add((variable.getName() != null) ? variable.getName() : "");
		parameters.add((variable.getMappedName() != null) ? variable.getMappedName() : "");

		return parameters;
	}
}
