/*
 * Created on Nov 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.nodedefinition;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.util.DecisionExprValidationUtil;

/**
 * Strategy implementation used to determine the variables present in a node definition that require
 * mapping to a process variable in the Workflow template.
 * 
 * @author Vijay Silva
 */
public class TemplateRequiredVariableFinder implements IRequiredVariableFinder {

    /**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.nodedefinition.IRequiredVariableFinder#getRequiredVariableNames(com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition)
	 */
    public Set<String> getRequiredVariableNames(NodeDefinition nodeDefinition) {
        Set<String> variableNames = new HashSet<String>();

        if (nodeDefinition.getNodeType() == NodeDefinitionType.MessageNode) {
            variableNames.addAll(getRequiredVariableNames((AutomaticTaskNode) nodeDefinition.getNode(), false));

        } else if (nodeDefinition.getNodeType() == NodeDefinitionType.SuperState) {
            NodeGroup superState = (NodeGroup) nodeDefinition.getNode();
            if (superState == null) {
                return variableNames;
            }

            for (NodeElement nodeElement : superState.getNodeElements()) {
                if (nodeElement instanceof NodeDefinition) {
                    NodeDefinition containedDefinition = (NodeDefinition) nodeElement;
                    variableNames.addAll(getRequiredVariableNames((AutomaticTaskNode) containedDefinition.getNode(), true));

                } else if (nodeElement instanceof Decision) {
                    getRequiredVariableNames((Decision) nodeElement);
                }
            }
        }

		return variableNames;
	}

    private Set<String> getRequiredVariableNames(AutomaticTaskNode node, boolean inSuperState) {
        Set<String> result = new HashSet<String>();
		if (node == null) {
			return result;
		}

        List<WPVariable> outgoingVariables = node.getOutgoingVariables();
        for (WPVariable variable : outgoingVariables) {
            String name = (inSuperState) ? variable.getName() : variable.getMappedName();
            if (name != null && name.trim().length() > 0) {
                result.add(name);
            }
        }

        Map<String, String> compositeKeys = node.getCompositeKeys();
        if (compositeKeys != null) {
            Collection<String> names = (inSuperState) ? compositeKeys.values() : compositeKeys.keySet();
            for (String name : names) {
                if (name != null && name.trim().length() > 0) {
                    result.add(name);
                }
            }
        }

        return result;
    }

    private Set<String> getRequiredVariableNames(Decision decision) {
        Set<String> result = new HashSet<String>();
        result.addAll(getRequiredVariableNames(decision.getExpression()));

        Transition[] transitions = decision.getTransitions();
        if (transitions != null) {
            for (Transition transition : transitions) {
                Condition condition = transition.getCondition();
                String expression = (condition != null) ? condition.getExpression() : null;
                result.addAll(getRequiredVariableNames(expression));
            }
        }

        return result;
    }

    private Set<String> getRequiredVariableNames(String expression) {
        Set<String> result = new HashSet<String>();
        if (expression != null && expression.trim().length() > 0) {
            Map<Integer, String> operands = DecisionExprValidationUtil.getOperands(expression);
            if (operands != null) {
                result.addAll(operands.values());
            }
        }
        return result;
    }
}