/*
 * Created on Jul 18, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

import edu.uci.ics.jung.graph.Vertex;

/**
 * Validates the types of the Nodes that are present in the Template.
 * 
 * @param <G>
 *            The type / class of the Graph element.
 * 
 * @author Vijay Silva
 */
public abstract class AllowedNodeTypesRule<G extends IWatchPointGraph> extends ValidationRule<G>
{
	/**
	 * Validates that the Graph element contains only nodes of the specified types.
	 * 
	 * @param graph
	 *            The object being validated
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateRule(G graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		Set<Vertex> vertices = graph.getVertices();
		for (Vertex vertex : vertices)
		{
			this.validateNodeType(WatchPointGraphUtils.getWTVertexElement(vertex), messages);
		}

		return messages;
	}

	/**
	 * Validates that the Node is of a allowed type.
	 * 
	 * @param node
	 *            The Node Element for which the node type needs to be validated.
	 * @param messages
	 *            The List of validation messages
	 */
	protected void validateNodeType(NodeElement node, final List<ValidationMessage> messages)
	{
		Class<? extends NodeElement> nodeType = node.getClass();
		List<Class<? extends NodeElement>> allowedNodeTypes = this.getAllowedNodeTypes();

		/* Check if the Node Type is allowed */
		if ((allowedNodeTypes != null) && (!allowedNodeTypes.contains(nodeType)))
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.INVALID_NODE_TYPE);
		}
	}

	/**
	 * Gets the list of allowed Node Element types/classes that can be present in the graph.
	 * 
	 * @return The List of Node Element implementations allowed to be present as a vertex.
	 */
	protected abstract List<Class<? extends NodeElement>> getAllowedNodeTypes();
}
