/*
 * Created on Jul 17, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.GraphFactory;
import com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph;

import edu.uci.ics.jung.algorithms.connectivity.BFSDistanceLabeler;
import edu.uci.ics.jung.graph.Vertex;
import edu.uci.ics.jung.utils.GraphUtils;

/**
 * Validation rule that ensures that every node in the Graph is 'reachable'. A node is considered
 * reachable if a path exists from a root node to that node, and a path exists from that node to a
 * sink node.
 * 
 * @param <G>
 *            The type of the graph
 * 
 * @author Vijay Silva
 */
public abstract class NodeReachableRule<G extends IWatchPointGraph>  extends ValidationRule<G>
{
	/**
	 * Validates that all the nodes in the graph can be reached from the root node and can reach the
	 * sink node.
	 * 
	 * @param graph
	 *            The graph to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateRule(G graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		/* Find the root and sink node vertices */
		Set<Vertex> rootVertices = this.getRootVertices(graph);
		Set<Vertex> sinkVertices = this.getSinkVertices(graph);

		/* Visitor to perform BFS search on the graph */
		BFSDistanceLabeler visitor = new BFSDistanceLabeler("distance");

		/* Do a BFS on the graph using the Root Nodes as the root */
		if (rootVertices != null && rootVertices.size() > 0)
		{
			IWatchPointGraph graphCopy = (IWatchPointGraph) graph.copy();
			visitor.labelDistances(graphCopy, rootVertices);

			Set<Vertex> unreachableVertices = visitor.getUnivistedVertices();
			this.validateNodesUnreachableFromRoot(unreachableVertices, messages);
		}

		/* Reverse the graph and do a BFS on the graph using the Sink Nodes as the root */
		if (sinkVertices.size() > 0)
		{
			GraphFactory factory = GraphFactory.createFactory();
			IWatchPointGraph reversedGraph = factory.createReverseGraph(graph);
			Set<Vertex> vertices = GraphUtils.getEqualVertices(sinkVertices, reversedGraph);
			visitor.labelDistances(reversedGraph, vertices);

			Set<Vertex> unreachableVertices = visitor.getUnivistedVertices();
			this.validateNodesUnreachableToSink(unreachableVertices, messages);
		}

		return messages;
	}

	/**
	 * Get the set of vertices that are the root of the graph. All vertices should be reachable from
	 * at least one of the root vertices.
	 * 
	 * @param graph
	 *            The graph
	 * 
	 * @return The set of root vertices.
	 */
	protected abstract Set<Vertex> getRootVertices(G graph);

	/**
	 * Get the set of vertices that are the sink of the graph. All vertices should be able to reach
	 * at least one of the sink vertices.
	 * 
	 * @param graph
	 *            The graph
	 * 
	 * @return The set of sink vertices.
	 */
	protected abstract Set<Vertex> getSinkVertices(G graph);

	/**
	 * Create the validation messages for the vertices that cannot be reached from the Root nodes.
	 * 
	 * @param vertices
	 *            The set of vertices not reachable from the root node.
	 * @param messages
	 *            The list of validation messages to add to.
	 */
	protected abstract void validateNodesUnreachableFromRoot(Set<Vertex> vertices,
			final List<ValidationMessage> messages);

	/**
	 * Create the validation messages for the vertices that cannot reach from the Sink nodes.
	 * 
	 * @param vertices
	 *            The set of vertices that cannot reach from the sink node.
	 * @param messages
	 *            The list of validation messages to add to.
	 */
	protected abstract void validateNodesUnreachableToSink(Set<Vertex> vertices,
			final List<ValidationMessage> messages);
}
