/*
 * Created on Jul 24, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.template;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement;
import com.tandbergtv.watchpoint.studio.ui.model.ProtectionKey;
import com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validation Rule that verifies that the Protection Keys of the Template and the Protection Keys of
 * the Resource Groups used by the Template match up.
 * 
 * @author Vijay Silva
 */
public class ProtectionKeyMatchRule extends ValidationRule<WorkflowTemplate>
{
	/**
	 * Check that the Protection Keys specified for the Template match up to all the Protection Keys
	 * of the Resource Groups used by the Template.
	 * 
	 * @param target
	 *            The WorkflowTemplate to validate
	 * 
	 * @return The List of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(WorkflowTemplate target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		Set<ProtectionKey> protectionKeys = target.getProtectionKeys();
		if (protectionKeys == null)
			protectionKeys = new HashSet<ProtectionKey>();

		/* Validate that the template does not have protection keys with the same name */
		Set<String> protectionKeyNames = new HashSet<String>();
		Set<String> duplicateNames = new HashSet<String>();
		for (ProtectionKey protectionKey : protectionKeys)
		{
			String name = protectionKey.getName();
			if (name == null)
				name = "";

			if (protectionKeyNames.contains(name))
			{
				if (!duplicateNames.contains(name))
				{
					duplicateNames.add(name);
					this.validateDuplicateProtectionKey(name, messages, target);
				}
			}
			else
			{
				protectionKeyNames.add(name);
			}
		}

		/*
		 * Collect all Resource Groups (by name) used in the template, and the list of nodes using
		 * each Resource Group. Make one DB call for the Protection Key per Resource Group instead
		 * of per Template Node.
		 */
		Map<String, Set<NodeElement>> nodesUsingGroup = new HashMap<String, Set<NodeElement>>();
		for (NodeElement node : target.getNodeElements())
		{
			if (node instanceof IResourceGroupElement)
			{
				IResourceGroupElement element = (IResourceGroupElement) node;
				ResourceGroup resourceGroupModel = element.getResourceGroup();
				if (resourceGroupModel != null)
				{
					String name = resourceGroupModel.getName();
					if (name == null || name.length() == 0)
					{ // Skip empty names
						continue;
					}

					Set<NodeElement> nodes = nodesUsingGroup.get(name);
					if (nodes == null)
					{
						nodes = new HashSet<NodeElement>();
						nodesUsingGroup.put(name, nodes);
					}

					nodes.add(node);
				}
			}
		}

		// Validate that the Protection Key of the Resource Group used is allowed
		for (String groupName : nodesUsingGroup.keySet())
		{
			IResourceGroupService resourceGroupService = ServiceFactory.createFactory().createResourceGroupService();
			List<com.tandbergtv.watchpoint.studio.dto.ResourceGroup> resourceGroups = resourceGroupService.getResourceGroupByName(groupName);
			if (resourceGroups.size() == 0) {
				// ResourceGroup is not present on workspace
				ValidationMessageCode code = ValidationMessageCode.TEMPLATE_NODE_RESOURCE_GROUP_NOT_FOUND;
				List<String> params = Arrays.asList(new String[] {groupName});
				Set<NodeElement> nodes = nodesUsingGroup.get(groupName);
				for (NodeElement node : nodes)
				{
				    ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, params);
				}
			} else {
				com.tandbergtv.watchpoint.studio.dto.ResourceGroup resourceGroup = resourceGroups.get(0);
				if (resourceGroups.size() > 1 || resourceGroup.isHasErrors()) {
					// ResourceGroup is invalid.
					ValidationMessageCode code = ValidationMessageCode.TEMPLATE_NODE_RESOURCE_GROUP_INVALID;
					List<String> params = Arrays.asList(new String[] {groupName});
					Set<NodeElement> nodes = nodesUsingGroup.get(groupName);
					for (NodeElement node : nodes)
					{
					    ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, params);
					}
				} 
				// Validate protection key
				String pkName = null;
				com.tandbergtv.watchpoint.studio.dto.ProtectionKey pk = resourceGroup.getProtectionKey();
				if (pk != null) {
					pkName = pk.getName();
				}
				Set<NodeElement> nodes = nodesUsingGroup.get(groupName);
				this.validateProtectionKeyForNodes(pkName, protectionKeyNames, messages, nodes);					
			}
		}

		return messages;
	}

	/*
	 * Create the validation messages required when validating duplicate protection key names for
	 * the Template.
	 */
	private void validateDuplicateProtectionKey(String name, List<ValidationMessage> messages,
			WorkflowTemplate template)
	{
		ValidationMessageCode code = ValidationMessageCode.DUPLICATE_TEMPLATE_PROTECTION_KEY;
		List<String> parameters = new ArrayList<String>();
		parameters.add(name);
		ValidationMessageAdder.getInstance().addValidationMessage(messages, template, code, parameters);
	}

	/*
	 * Create the validation messages required when a Template node uses a resource group who's
	 * protection key is not part of the template protection keys.
	 */
	private void validateProtectionKeyForNodes(String keyName, Set<String> protectionKeyNames,
			List<ValidationMessage> messages, Set<NodeElement> nodes)
	{
		if (keyName == null || !protectionKeyNames.contains(keyName))
		{
			ValidationMessageCode code = ValidationMessageCode.TEMPLATE_NODE_PROTECTION_KEY_MISMATCH;
			List<String> parameters = new ArrayList<String>();
			parameters.add(keyName);

			for (NodeElement node : nodes)
			{
			    ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
			}
		}
	}
}
