package com.tandbergtv.metadatamanager.specimpl.mstv2;

import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.tandbergtv.metadatamanager.exception.TranslationException;
import com.tandbergtv.metadatamanager.spec.ITranslator;
import com.tandbergtv.metadatamanager.util.ResourceResolver;
import com.tandbergtv.metadatamanager.util.XmlUtil;

public class FromTTVTranslator implements ITranslator {

	private InputStream xslStream;
	private Transformer transformer;
	private ClassLoader classLoader;
	
	private static String SHOWTIME_ELEMENT = "showtime";
	private static String START_TIME_ATTR = "startTimeInUTC";
	private static String ADDITIONAL_INFO_ELEMENT = "additionalInfo";
	private static String ADDITIONAL_INFO_NAME_ATTR_VALUE = "unModifiedStarttimeFromGLF";
	private static String ADDITIONAL_INFO_NAME_ATTR = "name";
	private static String ADDITIONAL_INFO_VALUE_ELEMENT = "value";
	
	private static String TZ_HOUR_MINUTE_DELIMITER = ":";
	private static String MINUS_DATETIME_TZ_DELIMITER = "-";
	private static String PLUS_DATETIME_TZ_DELIMITER = "+";
	
	private static String UTC_TIME_ZONE_CODE = "Z";
	
	private static String VALID_DATE_TIME_TZ_FORMAT_1 = "yyyy-mm-ddTHH:mm:ss-hhss"; 
	private static String VALID_DATE_TIME_TZ_FORMAT_2 = "yyyy-mm-ddTHH:mm:ss-hh:ss";
	private static String INCOMING_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ssZ";
	private static String UTC_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";
	
	protected final Logger logger = Logger.getLogger(this.getClass());
	
	public void setXslStream(InputStream xslStream) {
		this.xslStream = xslStream;
		try {
			TransformerFactory tf = XmlUtil.createTransformerFactory(getClass().getClassLoader());
			tf.setURIResolver(new ResourceResolver(classLoader));
			transformer = tf.newTransformer(new StreamSource(this.xslStream));
		} catch (TransformerConfigurationException e) {
			throw new RuntimeException(e);
		} catch (TransformerFactoryConfigurationError e) {
			throw new RuntimeException(e);
		}
	}

	public Document translate(Document doc) throws TranslationException {
		DocumentBuilder dBuilder;
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			dBuilder = factory.newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		}

		Document docOut = dBuilder.newDocument();
		DOMResult result = new DOMResult(docOut);
		try {
			transformer.transform(new DOMSource(doc), result);
		} catch (TransformerException e) {
			throw new RuntimeException(e);
		}
		
		modifyFields(docOut);
		
		return docOut;
	}

	private void modifyFields(Document docOut) {
		// start time in utc 
		NodeList showTimes = docOut.getElementsByTagName(SHOWTIME_ELEMENT);
		if(showTimes.getLength() > 0) {
			Element showTime = (Element) showTimes.item(0);
			String startTime = showTime.getAttribute(START_TIME_ATTR);
			/* only if the value is of the expected format try to get the utc equivalent */
			if (startTime != null
					&& (startTime.length() == VALID_DATE_TIME_TZ_FORMAT_1.length() 
							|| startTime.length() == VALID_DATE_TIME_TZ_FORMAT_2.length())) {
				int tzPos = startTime.lastIndexOf(PLUS_DATETIME_TZ_DELIMITER);
				if(tzPos == -1) {
					tzPos = startTime.lastIndexOf(MINUS_DATETIME_TZ_DELIMITER);
				}
				String time = startTime.substring(0, tzPos);
				String timezone = startTime.substring(tzPos);
				timezone = fixTimeZone(timezone);
				startTime = time + timezone;
				String utcTime = getUtcTime(startTime);
				showTime.setAttribute(START_TIME_ATTR, utcTime);
			}
		}
		
		// unModifiedStarttimeFromGLF
		NodeList additionalInfo = docOut.getElementsByTagName(ADDITIONAL_INFO_ELEMENT);
		if(additionalInfo.getLength() > 0) {
			Element aInfo = (Element) additionalInfo.item(0);
			String nameAttr = aInfo.getAttribute(ADDITIONAL_INFO_NAME_ATTR);
			if(nameAttr != null && !nameAttr.isEmpty() && nameAttr.equals(ADDITIONAL_INFO_NAME_ATTR_VALUE)) {
				NodeList values = aInfo.getElementsByTagName(ADDITIONAL_INFO_VALUE_ELEMENT);
				if(values.getLength() > 0) {
					String glfStartTime = values.item(0).getTextContent();
					/* only if the value is of the expected format try to get the utc equivalent */
					if(glfStartTime != null && 
							(glfStartTime.length() == VALID_DATE_TIME_TZ_FORMAT_1.length() 
							|| glfStartTime.length() == VALID_DATE_TIME_TZ_FORMAT_2.length())) {
						int tzPos = glfStartTime.lastIndexOf(PLUS_DATETIME_TZ_DELIMITER);
						if(tzPos == -1) {
							tzPos = glfStartTime.lastIndexOf(MINUS_DATETIME_TZ_DELIMITER);
						}
						String time = glfStartTime.substring(0, tzPos);
						String timezone = glfStartTime.substring(tzPos);
						timezone = fixTimeZone(timezone);
						glfStartTime = time + timezone;
						String utcGlfTime = getUtcTime(glfStartTime);
						values.item(0).setTextContent(utcGlfTime);
					}
				}
			}
		}
	}

	private String fixTimeZone(String timezone) {
		if(timezone.contains(TZ_HOUR_MINUTE_DELIMITER)) {
			timezone = timezone.replace(TZ_HOUR_MINUTE_DELIMITER, "");
		}		
		return timezone;
	}

	private String getUtcTime(String time) {
		SimpleDateFormat sdf = new SimpleDateFormat(INCOMING_DATE_FORMAT);
		
		SimpleDateFormat utcSdf = new SimpleDateFormat(UTC_DATE_FORMAT);
		utcSdf.setTimeZone(TimeZone.getTimeZone("UTC"));
		try {
			Date localDate = sdf.parse(time);
			return utcSdf.format(localDate) + UTC_TIME_ZONE_CODE;
		} catch (ParseException e) {
			logger.error("Failed to convert: " + time + " to utc equivalent.", e);
		}
		return time;
	}

	@Override
	public void setClassLoader(ClassLoader loader) {
		this.classLoader = loader;
		
	}
}
