#!/bin/sh
#######################################################################################
##                                                                                   ##
##     Script Name : cmsdb_inplace_upgrade_3.1.010.sh                                ##
##     Purpose     : Script to upgrade CMS database OS from 3.0 or 3.1 to 3.1.010    ##
##     Created by  : Matthew Kruer & Suresh Neravati                                 ##
##     Created on  : 01/06/2014                                                      ##
##                                                                                   ##
#######################################################################################
DBUPGRADE_DIR="$(pwd)"
alias mkdir='mkdir -p'
YUM_REPO="/etc/yum.repos.d/upgrade.repo"
YUM_REPO_SOURCE=""
if [ `whoami` != "root" ]; then
   echo "************************************************************************************"
   echo " Only root user can run this script. Please log in as a root user and try again...!" 
   echo "************************************************************************************"
   exit 1
fi

SILENT=0

while getopts :s flag ; do
    case ${flag} in 
        s)
            SILENT=1
            ;;
        ?)
            echo "invalid option"
            ;;
    esac
done

# Backup Config Files
CONFIG_FILES=(
    "/etc/modprobe.d/bonding.conf"
    "/usr/local/platform/backup/bin/commonbackup_oracle_full_back.sh"
    "/usr/local/platform/backup/bin/commonbackup_oracle_inc_back.sh"
    "/usr/local/platform/backup/conf/commonbackup.cfg"
    "/var/spool/cron/root"
    "/etc/sysctl.conf"                                        
    "/etc/security/limits.conf"                               
    "/etc/oratab"                                             
    "/etc/init.d/oracle"                                      
    "/etc/oraInst.loc"                                        
    "/home/oracle/.bash_profile"                              
    "/opt/oracle/product/11.2.0/db_1/sqlplus/admin/glogin.sql"
    "/opt/oracle/backup/scripts/rman_backup.ksh"              
    "/opt/oracle/backup/scripts/rman_backup_arch.ksh"
)

## Application Packages
APP_PACKAGES="  info-cms
                info-oracle
                oracle-admin
                oracleEE
                oracle-installcms        
                oracle-installdw           
                oracle-installepgmgr       
                oracle-installrcs          
                oracle-installrm 
                oracle-rdbms-install-scripts"

## Base OS Packages installed as part of the Kickstart
BASE_PACKAGES=" @Core
                bc
                cmsstaging
                commonbackup
                dmidecode
                dos2unix
                eject
                file
                jdk
                man
                mutt
                net-snmp
                net-snmp-utils
                nfs-utils
                ntp
                sysstat
                system-config-network-tui
                tcpdump
                zip
                pdksh
                warmstandby
                hpacucli"

# Settings
START=$(date +%s)
TIMESTAMP=`date --utc -d "@$(( $START ))" +%Y%m%d_%H%M%S`
LOGPATH="${DBUPGRADE_DIR}/cms_direct_inplace_upgrade"
FULL_LOG="${DBUPGRADE_DIR}/cms_direct_inplace_upgrade/$TIMESTAMP.update.log"
BACKUP_DIR="${DBUPGRADE_DIR}/cms_direct_inplace_upgrade/upgrade_pre_backup"
Color_Off='\e[0m'       # Text Reset
Black='\e[0;30m'        # Black
Red='\e[0;31m'          # Red
Green='\e[0;32m'        # Green
Yellow='\e[0;33m'       # Yellow
Blue='\e[0;34m'         # Blue
Purple='\e[0;35m'       # Purple
Cyan='\e[0;36m'         # Cyan
White='\e[0;37m'        # White

## Abort Loop
die() {
    printf -v f "\n\n%65s";printf "%s\n" "${f// /#}" 
    printf "Error: $1\n"
    printf "Please Review Log File $FULL_LOG\n"
    printf -v f "%65s";printf "%s\n" "${f// /#}" 
    printf "%-15s${Red}%-5s${Color_Off}\n\n\n" "Update Status: " "Failed!"
    exit 1
}

# Create log folder if it does exists
function create_log_folder
{
    if [[ ! -d "$LOGPATH" ]]; then
        mkdir $LOGPATH
        echo "Creating $LOGPATH"
        printf "${Green}%-5s${Color_Off}\n" "Done."    
    fi
}

function change_archive_mode
{
	REQUIRED_MODE="$1"
	if [ "$REQUIRED_MODE" = "" ]; then
		echo "*************************************************************************"
		echo "The database archive mode ARCHIVELOG or NOARCHIVELOG is mandatory to run "
		echo "the script. Please try running the script with either of this parameter"
		echo "*************************************************************************"
		exit 1
	fi
	
	ARCH_MODE=$(su - oracle -c "source /home/oracle/.bash_profile;
	sqlplus -S /nolog<<SQL1
		connect / as sysdba
		set pages 0
		set feedback off
		SELECT LOG_MODE FROM SYS.V\$DATABASE;
		exit
SQL1"
	)
	if [ "${REQUIRED_MODE}" = "${ARCH_MODE}" ]; then
		echo -e "\nDatabase is already running in ${REQUIRED_MODE} mode."    
	else
		echo -e "\nDatabase is changing to ${REQUIRED_MODE} mode, please wait....!\n"
		su - oracle -c "source /home/oracle/.bash_profile;
		sqlplus -S /nolog<<EOFSQL
		connect / as sysdba
		shut immediate
		startup mount
		alter database ${REQUIRED_MODE};
		alter database open;
		exit
EOFSQL"
		echo -e "\nDatabase has been changed to ${REQUIRED_MODE} mode successfully..."
	fi
}

function printline
{
   echo -e "********************************************************************************\n"
}

# Oracle Prep
function ora_prep
{    

    change_archive_mode ARCHIVELOG
    su - oracle -c "/opt/oracle/backup/scripts/rman_backup.ksh DISK"
    
    # unset Path
    unset PATH
    PATH=/opt/oracle/product/11.2.0/db_1/bin:/opt/oracle/product/11.2.0/db_1/OPatch:/usr/local/bin:/bin:/usr/bin:/usr/local/sbin:/usr/sbin:/sbin:
    export PATH

    service oracle stop
    
    cd ${DBUPGRADE_DIR}

    echo "Backingup crontabs ...."
    su - oracle -c "crontab -l > /home/oracle/oracle_cron.bak"
    cp /home/oracle/oracle_cron.bak ${BACKUP_DIR}/oracle_cron.bak
    crontab -l > ${BACKUP_DIR}/root_cron.bak    
    
}

# Backup configuration files
function backup_config_files 
{
    printf "%-7s\n" "Backing up config Files:"
    for c in "${CONFIG_FILES[@]}"
    do
        if [[ -e "$c" ]]; then 
            printf "%-9s%s\n" "  [ OK ]" "Backing up $c to $BACKUP_DIR/$c"
            DIRNAME=`dirname "${c}"`
            BASENAME=`basename "${c}"`
            if [ ! -d "$BACKUP_DIR"/"$DIRNAME" ] ; then
               mkdir -p "$BACKUP_DIR"/"$DIRNAME"
            fi
            cp -p "$c" "$BACKUP_DIR"/"$c"
        else 
            printf "%-9s%s\n" "  [WARN]" "Missing file $c"
        fi
    done
    printf "${Green}%-5s${Color_Off}\n" "Done."
}
function ora_post_steps
{
    cp -f ${BACKUP_DIR}/etc/oratab /etc/oratab
    cp -f /home/oracle/scripts/dbca/oracle_profile        /home/oracle/.bash_profile
    cp -f /home/oracle/scripts/dbca/glogin.sql            /opt/oracle/product/11.2.0/db_1/sqlplus/admin/glogin.sql
    cp -f /home/oracle/scripts/dbca/rman_backup.ksh       /opt/oracle/backup/scripts/rman_backup.ksh 
    cp -f /home/oracle/scripts/dbca/rman_backup_arch.ksh  /opt/oracle/backup/scripts/rman_backup_arch.ksh
    chown oracle:oinstall /opt/oracle/product/11.2.0/db_1/sqlplus/admin/glogin.sql
    chown oracle:oinstall /opt/oracle/backup/scripts/rman_backup*.ksh
    chown oracle:oinstall /home/oracle/.bash_profile
    chmod 755 /opt/oracle/backup/scripts/rman_backup*.ksh
    cp -f /home/oracle/scripts/dbca/oracle_service /etc/init.d/oracle
    chown root:root /etc/init.d/oracle
    chmod 755 /etc/init.d/oracle
}   
# If rPath found remove it
function remove_rpath
{
    if [ -x /usr/bin/conary ] ; then
        printf "%-45s" "Removing Conary:"
        conary erase iconfig
        conary erase conary-cim-configuration
        conary erase conary-cim rpath-tools
        conary erase conary
        printf "${Green}%-5s${Color_Off}\n" "Done."
    else
        echo "Conary Not Installed"
    fi
}

# Create upgrade repository if it does not exists
function upg_repos
{
	if [ $SILENT -eq 0 ]; then
    	# Create upgrade repository if it does not exists
	    while [[ $CORRECT != "y" ]]
	    do
	        printf "Checking for Upgrade Repository\n"
	        if [ ! -f ${YUM_REPO} ] ; then
	            printf "%-50s\n" "Upgrade Repository not found, Please Enter Repository URL"
	            printf "%-50s\n" "Examples:"
	            printf "%-50s\n" "  http://www.my-repository.com/software/"
	            printf "%-50s\n" "  ftp://www.my-repository.com/software/"
	            printf "%-50s\n" "  file:///mnt/cdrom/"
	            printf           "Enter Repository URL: "
	        
	            while [[ -z $YUM_REPO_SOURCE ]] ; do
	                read YUM_REPO_SOURCE
	                regex='(https?|ftp|file)://[-A-Za-z0-9\+&@#/%?=~_|!:,.;]*[-A-Za-z0-9\+&@#/%=~_|]'
	                if [[ $YUM_REPO_SOURCE =~ $regex ]] ; then
	                    YUM_REPO_SOURCE=$YUM_REPO_SOURCE
	                else [[ $YUM_REPO_SOURCE == "" ]]
	                    printf "Invalid Repository URL, Please enter again:"
	                    YUM_REPO_SOURCE=""
	                fi
	            done
	        else
	            YUM_REPO_SOURCE=`cat /etc/yum.repos.d/upgrade.repo |grep "baseurl" | awk -F= '{print $2}'`
	        fi
	        printf "%-50s\n" "Repository URL is $YUM_REPO_SOURCE"
	        printf "Is this correct? [y/n] "
	        read CORRECT
	        if [[ $CORRECT != "y" ]] ; then
	            YUM_REPO_SOURCE=""
	            if [[ -f $YUM_REPO ]] ; then
	                rm  $YUM_REPO
	            fi
	        fi
	    done
	
	    printf "%-45s\n" "Creating Upgrade $YUM_REPO"
	    echo "[Upgrade_Repository]"     >> ${YUM_REPO}
	    echo "name=Upgrade Repository"  >> ${YUM_REPO}
	    echo "baseurl=$YUM_REPO_SOURCE" >> ${YUM_REPO}
	    echo "enabled=1"                >> ${YUM_REPO}
	    echo "gpgcheck=0"               >> ${YUM_REPO}
	else
		echo "The -s switch was passed.  Assuming the yum repo has already been configured."
	fi
}

function chk_repos
{
    yum clean all
    yum deplist ${BASE_PACKAGES} ${APP_PACKAGES}   
    if [ $? -ne 0 ]; then
       echo -e "\nInvalid repository: Missing Dependencies\n"
       exit 1
    else
       echo -e "\nValid Repository, please wait upgrade is continuing...\n"
    fi
}

# Updating Existing System Packages
function updt_existing_packs 
{
    printf "%-45s\n" "Updating Existing System Packages:"
    yum update --assumeyes                   2>&1 || die "Missing: System RPMs Not Found In Repository"
    printf "${Green}%-5s${Color_Off}\n" "Done."
}

# Installing New System and Appliance Packages
function install_appli_packs
{
    printf "%-45s\n" "Installing New System and Appliance Packages:"
    yum install --assumeyes ${BASE_PACKAGES} 2>&1 || die "Missing: System RPMs Not Found In Repository"
    yum install --assumeyes ${APP_PACKAGES}  2>&1 || die "Missing: Appliance RPMs Not Found In Repository"
    printf "${Green}%-5s${Color_Off}\n" "Done."
}

create_log_folder;

(
DB_ROLE=$(su - oracle -c "sqlplus -S /nolog<<PQR
  connect / as sysdba
  set pages 0
  set feedback off
  select trim(database_role) from v\\\$database;
  exit
PQR"
)
mkdir $BACKUP_DIR
if [ $? -ne 0 ]; then
    echo -e "\nYou can't run the upgrade script. This database server is already upgraded...!"
    printline
    exit 1
else
    if [ "${DB_ROLE}" = "PRIMARY" ]; then

DBVER=$(su - oracle -c "sqlplus -S /nolog<<SQ1
  connect / as sysdba
  set pages 0
  set feedback off
  SELECT setting_value FROM WFS.SYSTEM_SETTINGS WHERE setting_type = 'RELEASE';
  exit
SQ1"
)
      if [ "${DBVER}" = "3.3.20" ] || [ "${DBVER}" = "3.4.12" ]; then
        printline
        echo -e "Script started at $(date)"
        upg_repos
        chk_repos
        ora_prep
        backup_config_files
        remove_rpath
        yum erase ksh --assumeyes
        updt_existing_packs
        install_appli_packs
        ora_post_steps
        service oracle start
	su - oracle -c "cd /home/oracle/scripts/install/installcms; sh cmsdb_upgrade3_31010.sh"
        printline
      else 
       echo -e "You can't run the upgrade script on this server. This is not a cms 3.x database.\n"
       exit 1
      fi

     else 
        echo -e "\nThis is not a Primary Database. Upgrade script can run only on the Primary Database........\n"
        exit 1
    fi
fi
) 2>&1 | tee -a ${FULL_LOG}   

exit 0
