/* 
================================================================================
Database name:  ttv   
Schema Name:    WFS
Version:                                  
DBMS name:      ORACLE Version 11.2.0.2                    
Script purpose: Create the Contract Management objects:
		 		        tables, views, sequences and constraints etc.

Created by:     Suresh Neravati 06/07/2011
NOTE:		        Must be logged on as sys user and WFS Schema must also exists.

Modifications:
SwVer   DbVer     ModifiedBy             ModifiedDate    Details  
-----   -----     -------------          ------------   -------
1.0     1.0.0     Daniel Paranhos Silva  06/07/2011     Changed cm_sm_asset table ID_LICENSE column to nullable.  
1.0     1.0.0     Amit Seth              06/16/2011     Changed ID_OWNERSHIP and METADATA_FORMAT columns on CM_DISTRIBUTION_MANAGEMENT nullable.
1.0     1.0.0     Murtaza and Suresh     06/17/2011     Removed the Warehouse columns from the tables.
1.0     1.0.1     Amit Seth              12/12/2011     Removed the DB objects drop procedure, also added the createdate and updatedate columns.
1.0     1.0.1     Amit Seth              12/20/2011     Amit request to change CM_LICENSE.CONTENT_PROVIDER column to null
1.0     1.0.1     Amit Seth              12/21/2011     Amit request to change CM_LICENSE.NOTES column to null
1.5     1.0.2     Amit Seth              01/19/2012     Amit request to update cm_license table all_assets column.
1.5     1.0.2     Flavio Pereira Orfano  01/20/2012     Added cm_audit table. 
1.5     1.0.2     Gunnar                 01/26/2012     Added delivery_location_id column to cm_license_details table.
1.5     1.0.3     Flavio Pereira Orfano  02/17/2012     Updated cm_audit table audit_type values. 
1.5     1.0.3     Gunnar                 02/17/2012     Updated active column of CM_CONTRACT and CM_DEAL tables to Y.
1.5     1.0.4     Suneeta                04/12/2012     If CM had created a license in 2.010 or 2.020 for a SEFLF or src partner, we want to blank out the content provider.
================================================================================
*/
SET serveroutput ON
SET verify OFF
SET echo OFF
 
DEFINE schema_name='&1'

CREATE OR REPLACE PACKAGE SYS.CREATECMDB AS
  PROCEDURE INSTALL_CM;
  install_failed exception;
  vschema VARCHAR2(100) := '&schema_name';
  vsql varchar2(32767);
END;
/

CREATE OR REPLACE PACKAGE BODY SYS.CREATECMDB AS
  /* Procedure: CreateObject. Executes a dynamic SQL statement and logs error in install_table*/
  PROCEDURE CREATEOBJECT(vsql varchar2,errmsg varchar2, mask varchar2 default 'N') IS
    verrmsg varchar2(32000);
  BEGIN
    execute immediate vsql;
  exception
    when others then
          verrmsg := substr(errmsg ||' '||sqlerrm,1,4000);
      if upper(mask) != 'Y' then
          execute immediate 'INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')';
          commit;
          raise;
      end if;
  END CREATEOBJECT;
  
/* Procedure: CreateSeq. Creates a sequence on a primary key of a table.
              Sequence name is <tablename>SEQ.*/
  procedure createseq(vtabname varchar2, vstartwith number default 1, vcache number default null) is
    vcolname dba_cons_columns.column_name%TYPE;
  BEGIN
    select dcc.column_name 
    into vcolname
    from dba_constraints dc, dba_cons_columns dcc
    where dc.owner=vschema
    and dc.owner = dcc.owner
    and dc.table_name = dcc.table_name
    and dc.table_name=upper(vtabname)
    and dc.constraint_type='P'
    and dc.constraint_name = dcc.constraint_name
    and rownum < 2;
    
		-- Create Sequence for Table
      vsql :=  'CREATE SEQUENCE '||vschema||'.'||vtabname||'_SEQ INCREMENT BY 1 START WITH '||vstartwith||' '||
                CASE WHEN (vcache IS NULL) THEN 'NOCACHE' ELSE 'CACHE '||TO_CHAR(vcache) END;
      CREATEOBJECT(vsql,'Create Sequence'||vtabname||'_SEQ','Y');

    -- Create before insert trigger to populate the PK column  
      vsql := 'CREATE OR REPLACE TRIGGER '||vschema||'.'||vtabname||'TRG BEFORE INSERT ON '||vschema||'.'||vtabname||' FOR EACH ROW
            BEGIN
              if (:new.'||vcolname||' is null) then
                select '||vschema||'.'||vtabname||'_SEQ.nextval into :new.'||vcolname||'  from dual;
              end if;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'TRG');
   END CREATESEQ;

/* Procedure: CreateUpdTimeTRG. Updates the UpdateTime column with the current date and time when any
              update statement is issued against the table.*/
  PROCEDURE CREATEUPDATETRG(vtabname varchar2) IS
    vcount NUMBER := 0;
  BEGIN
    select count(*)
    into vcount
    from dba_tables dt, dba_tab_columns dtc
    where dt.owner=vschema
    and dt.owner = dtc.owner
    and dt.table_name = dtc.table_name
    and dt.table_name=upper(vtabname)
    and dtc.column_name = 'UPDATEDATE';
    
	  if ( vcount > 0 ) then
      vsql := 'CREATE OR REPLACE TRIGGER '||VSCHEMA||'.'||SUBSTR(VTABNAME,1,26)||
                  'UPD BEFORE UPDATE ON '||vschema||'.'||vtabname||' FOR EACH ROW
            BEGIN
              select systimestamp into :new.updatedate from dual;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'UPD');
    end if;

   END CREATEUPDATETRG;

/*----------------------------------------------------------------------------
--   Procedure to DROP the Contract Management DB Objects.
-----------------------------------------------------------------------------*/
   PROCEDURE DROPCM IS
     wk_message	varchar2(500);
     err_msg		varchar2(500);
     wk_count		number := 0;
   BEGIN
     WK_MESSAGE := '';   
    
     vsql := 'DROP TABLE '||vschema||'.CM_SM_LANGUAGES'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_SM_LANGUAGES','Y');  

     vsql := 'DROP TABLE '||vschema||'.CM_SM_ASSETS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_SM_ASSETS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_REVENUE_PRICING'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_REVENUE_PRICING','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_LICENSE_DOCUMENTS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_LICENSE_DOCUMENTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_FIELD_DROPDOWN'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_FIELD_DROPDOWN','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DISTRIBUTION_MANAGEMENT'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DISTRIBUTION_MANAGEMENT','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DEAL_DOCUMENTS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DEAL_DOCUMENTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DEAL_CONTACTS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DEAL_CONTACTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_CUSTOM_LIC_FIELD_VALUE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_CUSTOM_LIC_FIELD_VALUE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_LICENSE_DETAILS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_LICENSE_DETAILS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DEVICE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DEVICE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_COPY_HOLDER_DOCUMENTS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_COPY_HOLDER_DOCUMENTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_COPY_HOLDER_CONTACTS';
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_COPY_HOLDER_CONTACTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_CONTRACT_DOCUMENTS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_CONTRACT_DOCUMENTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_CONTRACT_CONTACTS'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_CONTRACT_CONTACTS','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_LICENSE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_LICENSE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DEAL';
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DEAL','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_CONTRACT'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_CONTRACT','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_PLATFORM'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_PLATFORM','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_OWNERSHIP'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_OWNERSHIP','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_MEDIA'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_MEDIA','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_LICENSE_TYPE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_LICENSE_TYPE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DELIVERY_METHOD'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DELIVERY_METHOD','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_LANGUAGE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_LANGUAGE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_FIELD'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_FIELD','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_DEAL_TYPE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_DEAL_TYPE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_CURRENCY'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_CURRENCY','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_COPYRIGHT_HOLDER'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_COPYRIGHT_HOLDER','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_CONTRACT_TYPE'; 
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_CONTRACT_TYPE','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_FREQUENCY';
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_FREQUENCY','Y');  
    
     vsql := 'DROP TABLE '||vschema||'.CM_SYSTEM_SETTINGS';
     CREATEOBJECT(vsql,'Unable to DROP TABLE '||vschema||'.CM_SYSTEM_SETTINGS','Y');  
    
     vsql := 'DECLARE
                 CURSOR C_SEQ IS SELECT ''DROP SEQUENCE WFS.''||SEQUENCE_NAME STR FROM ALL_SEQUENCES 
                               WHERE SEQUENCE_NAME LIKE ''CM_%'' AND SEQUENCE_NAME NOT LIKE ''CMS%'';
                 VSEQ VARCHAR2(1000):='';
              BEGIN
                 FOR VSEQ IN C_SEQ LOOP
                    EXECUTE IMMEDIATE VSEQ.STR;  
                 END LOOP;
             END;';
    CREATEOBJECT(vsql,'Unable to DROP SEQUENCES','Y');   

    <<bypass_install>>  
    COMMIT;          
    IF wk_message = 'ERROR' THEN
       RAISE CREATECMDB.INSTALL_FAILED;
    ELSE
       EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values
           (''Dropping ContractManagement completed SUCCESSFULLY'')';
    END IF;
    
   END DROPCM;
   
/*----------------------------------------------------------------------------
--   Procedure to create the Contract Management DB Objects.
-----------------------------------------------------------------------------*/
  PROCEDURE CREATECM IS
     wk_message	varchar2(500);
     err_msg		varchar2(500);
     wk_count		number := 0;
  BEGIN
    wk_message := '';

  /* ---------------------------------------------------------------------------
  --              CREATING ALL TABLES AND PRIMARY KEYS
  --------------------------------------------------------------------------- */
    -- creating table CM_SYSTEM_SETTINGS
    vsql :=  'CREATE TABLE '||vschema||'.CM_SYSTEM_SETTINGS (
                   SETTING_TYPE  VARCHAR2(25), 
                   SETTING_VALUE VARCHAR2(1024))';
    CREATEOBJECT(vsql,'Unable to create table '||vschema||'.CM_SYSTEM_SETTINGS');

    vsql := 'CREATE TABLE '||vschema||'.CM_CONTRACT_TYPE ( 
     ID_TYPE NUMBER  NOT NULL , 
     TYPE VARCHAR2 (50),
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_CONTRACT_TYPE PRIMARY KEY ( ID_TYPE ),
     CONSTRAINT UNQ_CONTRACT_TYPE UNIQUE ( TYPE ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_CONTRACT_TYPE');
    CREATESEQ('CM_CONTRACT_TYPE');
    CREATEUPDATETRG('CM_CONTRACT_TYPE');

    vsql := 'CREATE TABLE '||vschema||'.CM_COPYRIGHT_HOLDER ( 
     ID_COPYRIGHTHOLDER NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     ADDRESS VARCHAR2 (300)  NOT NULL , 
     AFFILIATION VARCHAR2 (50)  NOT NULL,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_COPYRIGHT_HOLDER PRIMARY KEY ( ID_COPYRIGHTHOLDER ),
     CONSTRAINT UNQ_COPYRIGHT_HOLDER UNIQUE ( NAME ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_COPYRIGHT_HOLDER');
    CREATESEQ('CM_COPYRIGHT_HOLDER');
    CREATEUPDATETRG('CM_COPYRIGHT_HOLDER');

    vsql := 'CREATE TABLE '||vschema||'.CM_CURRENCY ( 
     ID_CURRENCY NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     SYMBOL VARCHAR2 (50)  NOT NULL,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_CURRENCY PRIMARY KEY ( ID_CURRENCY ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_CURRENCY');
    CREATESEQ('CM_CURRENCY');
    CREATEUPDATETRG('CM_CURRENCY');

    vsql := 'CREATE TABLE '||vschema||'.CM_DEAL_TYPE ( 
     ID_TYPE NUMBER  NOT NULL, 
     TYPE VARCHAR2 (50),
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DEAL_TYPE PRIMARY KEY ( ID_TYPE ),
     CONSTRAINT UNQ_DEAL_TYPE UNIQUE ( TYPE ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_DEAL_TYPE');
    CREATESEQ('CM_DEAL_TYPE');
    CREATEUPDATETRG('CM_DEAL_TYPE');

    vsql := 'CREATE TABLE '||vschema||'.CM_FIELD ( 
     ID_FIELD NUMBER  NOT NULL , 
     TYPE VARCHAR2 (50)  NOT NULL CHECK ( TYPE IN (''Alphanumeric'', ''Currency'', 
                 ''Date'', ''Frequency'', ''Number'', ''Percentage'', ''Text'')), 
     NAME VARCHAR2 (50)  NOT NULL , 
     SCREEN_NAME VARCHAR2 (50)  NOT NULL CHECK ( SCREEN_NAME IN (''ASSET_ENHANCEMENT'', 
                 ''CONTRACT'', ''COPYRIGHTHOLDER'', ''DEAL'', ''DISTRIBUTION'', 
                 ''LICENSE'', ''PRICING'', ''STORAGE_METADATA'')), 
     DROPDOWN CHAR (1) , 
     REQUIRED CHAR (1)  NOT NULL , 
     DEFAULT_VALUE VARCHAR2 (50) , 
     MIN_CHARS NUMBER , 
     MAX_CHARS NUMBER , 
     F_ORDER NUMBER ,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_FIELD PRIMARY KEY ( ID_FIELD ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_FIELD');
    CREATESEQ('CM_FIELD');
    CREATEUPDATETRG('CM_FIELD');

    vsql := 'CREATE TABLE '||vschema||'.CM_FREQUENCY ( 
     ID_FREQUENCY NUMBER  NOT NULL , 
     DESCRIPTION VARCHAR2 (50)  NOT NULL ,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_FREQUENCY PRIMARY KEY ( ID_FREQUENCY ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_FREQUENCY');
    CREATESEQ('CM_FREQUENCY');
    CREATEUPDATETRG('CM_FREQUENCY');

    vsql := 'CREATE TABLE '||vschema||'.CM_LANGUAGE ( 
     ID_LANGUAGE NUMBER  NOT NULL , 
     LANGUAGE VARCHAR2 (50)  NOT NULL,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_LANGUAGE PRIMARY KEY ( ID_LANGUAGE ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_LANGUAGE');
    CREATESEQ('CM_LANGUAGE');
    CREATEUPDATETRG('CM_LANGUAGE');

    vsql := 'CREATE TABLE '||vschema||'.CM_DELIVERY_METHOD ( 
     ID_DELIVERY_METHOD NUMBER  NOT NULL, 
     DELIVERY_METHOD VARCHAR2 (50)  NOT NULL,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DELIVERY_METHOD PRIMARY KEY ( ID_DELIVERY_METHOD ),
     CONSTRAINT UNQ_DELIVERY_METHOD UNIQUE ( DELIVERY_METHOD ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_DELIVERY_METHOD');
    CREATESEQ('CM_DELIVERY_METHOD');
    CREATEUPDATETRG('CM_DELIVERY_METHOD');

    vsql := 'CREATE TABLE '||vschema||'.CM_LICENSE_TYPE ( 
     ID_TYPE NUMBER  NOT NULL , 
     TYPE VARCHAR2 (50)  NOT NULL,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_LICENSE_TYPE PRIMARY KEY ( ID_TYPE ),
     CONSTRAINT UNQ_LICENSE_TYPE UNIQUE ( TYPE ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_LICENSE_TYPE');
    CREATESEQ('CM_LICENSE_TYPE');
    CREATEUPDATETRG('CM_LICENSE_TYPE');

    vsql := 'CREATE TABLE '||vschema||'.CM_MEDIA ( 
     ID_MEDIA NUMBER  NOT NULL , 
     MEDIA_TYPE VARCHAR2 (50)  NOT NULL ,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_MEDIA PRIMARY KEY ( ID_MEDIA ) ,
     CONSTRAINT UNQ_MEDIA UNIQUE ( MEDIA_TYPE ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_MEDIA');
    CREATESEQ('CM_MEDIA');
    CREATEUPDATETRG('CM_MEDIA');

    vsql := 'CREATE TABLE '||vschema||'.CM_OWNERSHIP ( 
     ID_OWNERSHIP NUMBER  NOT NULL , 
     OWNERSHIP VARCHAR2 (50)  NOT NULL ,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_OWNERSHIP PRIMARY KEY ( ID_OWNERSHIP ),
     CONSTRAINT UNQ_OWNERSHIP UNIQUE ( OWNERSHIP ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_OWNERSHIP');
    CREATESEQ('CM_OWNERSHIP');
    CREATEUPDATETRG('CM_OWNERSHIP');

    vsql := 'CREATE TABLE '||vschema||'.CM_PLATFORM ( 
     ID_PLATFORM NUMBER  NOT NULL , 
     DESCRIPTION VARCHAR2 (50)  NOT NULL,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_PLATAFORM PRIMARY KEY ( ID_PLATFORM ),
     CONSTRAINT UNQ_PLATFORM UNIQUE ( DESCRIPTION ))';
    CREATEOBJECT(VSQL,'Unable to create table CM_PLATFORM');
    CREATESEQ('CM_PLATFORM');
    CREATEUPDATETRG('CM_PLATFORM');
--
    vsql := 'CREATE TABLE '||vschema||'.CM_CONTRACT ( 
     ID_CONTRACT NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     ACTIVE CHAR (1)  NOT NULL , 
     ID_TYPE NUMBER  NOT NULL , 
     CREATION_DATE TIMESTAMP (0)  NOT NULL , 
     EFFECTIVE_DATE TIMESTAMP (0)  NOT NULL , 
     EXPIRY_DATE TIMESTAMP (0)  NOT NULL , 
     DEAL_EXPIRY_REMINDER NUMBER , 
     ID_COPYRIGHTHOLDER NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_CONTRACT PRIMARY KEY ( ID_CONTRACT ),
     CONSTRAINT UNQ_CONTRACT UNIQUE ( NAME ),
     CONSTRAINT FK_CONTRACT_CONTRACT_TYPE FOREIGN KEY (ID_TYPE) REFERENCES '||vschema||'.CM_CONTRACT_TYPE (ID_TYPE),
     CONSTRAINT FK_CONTRACT_COPYRIGHT_HOLDER FOREIGN KEY (ID_COPYRIGHTHOLDER) 
                 REFERENCES '||vschema||'.CM_COPYRIGHT_HOLDER(ID_COPYRIGHTHOLDER))';
    CREATEOBJECT(vsql,'Unable to create table CM_CONTRACT');
    CREATESEQ('CM_CONTRACT');
    CREATEUPDATETRG('CM_CONTRACT');

    vsql := 'CREATE TABLE '||vschema||'.CM_DEAL ( 
     ID_DEAL NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     ACTIVE CHAR (1)  NOT NULL , 
     ID_TYPE NUMBER  NOT NULL , 
     CREATION_DATE TIMESTAMP (0)  NOT NULL , 
     EFFECTIVE_DATE TIMESTAMP (0)  NOT NULL , 
     EXPIRY_DATE timestamp (0)  not null ,
     LICENSE_EXPIRY_REMINDER number default 0 , 
     ID_CONTRACT NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DEAL PRIMARY KEY ( ID_DEAL ),
     CONSTRAINT UNQ_DEAL UNIQUE ( NAME ),
     CONSTRAINT FK_DEAL_CONTRACT FOREIGN KEY (ID_CONTRACT) REFERENCES '||vschema||'.CM_CONTRACT (ID_CONTRACT),
     CONSTRAINT FK_DEAL_DEAL_TYPE FOREIGN KEY (ID_TYPE) REFERENCES '||vschema||'.CM_DEAL_TYPE (ID_TYPE))';
    CREATEOBJECT(vsql,'Unable to create table CM_DEAL');
    CREATESEQ('CM_DEAL');
    CREATEUPDATETRG('CM_DEAL');

    vsql := 'CREATE TABLE '||vschema||'.CM_LICENSE ( 
     ID_LICENSE NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     ACTIVE CHAR (1)  NOT NULL , 
     ID_TYPE NUMBER  NOT NULL , 
     CREATION_DATE TIMESTAMP (0)  NOT NULL , 
     EFFECTIVE_DATE TIMESTAMP (0)  NOT NULL , 
     EXPIRY_DATE TIMESTAMP (0)  NOT NULL , 
     NOTES VARCHAR2 (250) NULL ,
     CONTENT_PROVIDER VARCHAR2 (500) NULL , 
     ID_DEAL NUMBER  NOT NULL , 
     ALL_ASSETS CHAR (1)  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_LICENSE PRIMARY KEY (ID_LICENSE),
     CONSTRAINT FK_LICENSE_DEAL FOREIGN KEY (ID_DEAL) REFERENCES '||vschema||'.CM_DEAL (ID_DEAL),
     CONSTRAINT FK_LICENSE_LICENSE_TYPE FOREIGN KEY (ID_TYPE) REFERENCES '||vschema||'.CM_LICENSE_TYPE (ID_TYPE))';
    CREATEOBJECT(vsql,'Unable to create table CM_LICENSE');
    CREATESEQ('CM_LICENSE');
    CREATEUPDATETRG('CM_LICENSE');

    vsql := 'CREATE TABLE '||vschema||'.CM_CONTRACT_CONTACTS ( 
     ID_CONTACT NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     EMAIL VARCHAR2 (150)  NOT NULL , 
     OFFICE VARCHAR2 (20) , 
     WORK VARCHAR2 (20) , 
     MOBILE VARCHAR2 (20) , 
     NOTIFY CHAR (1) , 
     ID_CONTRACT NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_CONTRACT_CONTACTS PRIMARY KEY ( ID_CONTACT ),
     CONSTRAINT FK_CONTRACT_CONTACTS_CONTRACT FOREIGN KEY (ID_CONTRACT) REFERENCES '||vschema||'.CM_CONTRACT (ID_CONTRACT))';
    CREATEOBJECT(vsql,'Unable to create table CM_CONTRACT_CONTACTS');
    CREATESEQ('CM_CONTRACT_CONTACTS');
    CREATEUPDATETRG('CM_CONTRACT_CONTACTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_CONTRACT_DOCUMENTS ( 
     ID_DOCUMENT NUMBER  NOT NULL , 
     ID_CONTRACT NUMBER  NOT NULL , 
     file_name varchar2 (50) , 
     CONTRACT_FILE BLOB, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_CONTRACT_DOCUMENTS PRIMARY KEY ( ID_DOCUMENT ),
     CONSTRAINT FK_CONTRACT_DOCUMENTS_CONTRACT FOREIGN KEY (ID_CONTRACT) REFERENCES '||vschema||'.CM_CONTRACT (ID_CONTRACT))';
    CREATEOBJECT(vsql,'Unable to create table CM_CONTRACT_DOCUMENTS');
    CREATESEQ('CM_CONTRACT_DOCUMENTS');
    CREATEUPDATETRG('CM_CONTRACT_DOCUMENTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_COPY_HOLDER_CONTACTS ( 
     ID_CONTACT NUMBER  NOT NULL , 
     NAME VARCHAR2 (150)  NOT NULL , 
     EMAIL VARCHAR2 (150)  NOT NULL , 
     OFFICE VARCHAR2 (20) , 
     WORK VARCHAR2 (20) , 
     MOBILE VARCHAR2 (20) , 
     NOTIFY CHAR (1) , 
     ID_COPYRIGHTHOLDER NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_COPY_HOLDER_CONTACTS PRIMARY KEY ( ID_CONTACT ),
     CONSTRAINT FK_C_HOLDER_CONTACTS_C_HOLDER FOREIGN KEY (ID_COPYRIGHTHOLDER) REFERENCES '||vschema||'.CM_COPYRIGHT_HOLDER (ID_COPYRIGHTHOLDER))';
    CREATEOBJECT(vsql,'Unable to create table CM_COPY_HOLDER_CONTACTS');
    CREATESEQ('CM_COPY_HOLDER_CONTACTS');
    CREATEUPDATETRG('CM_COPY_HOLDER_CONTACTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_COPY_HOLDER_DOCUMENTS ( 
     ID_DOCUMENT NUMBER  NOT NULL , 
     ID_COPYRIGHTHOLDER NUMBER  NOT NULL , 
     FILE_NAME VARCHAR2 (50) , 
     COPY_HOLDER_FILE BLOB, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_COPY_HOLDER_DOCUMENTS PRIMARY KEY ( ID_DOCUMENT ),
     CONSTRAINT FK_C_HOLDER_DOCUMENTS_C_HOLDER FOREIGN KEY (ID_COPYRIGHTHOLDER) REFERENCES '||vschema||'.CM_COPYRIGHT_HOLDER ( ID_COPYRIGHTHOLDER))';
    CREATEOBJECT(vsql,'Unable to create table CM_COPY_HOLDER_DOCUMENTS');
    CREATESEQ('CM_COPY_HOLDER_DOCUMENTS');
    CREATEUPDATETRG('CM_COPY_HOLDER_DOCUMENTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_DEVICE ( 
     ID_DEVICE NUMBER  NOT NULL, 
     DEVICE_NAME VARCHAR2 (250), 
     ID_PLATFORM NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DEVICE PRIMARY KEY ( ID_DEVICE ),
     CONSTRAINT FK_DEVICE_PLATFORM FOREIGN KEY (ID_PLATFORM) REFERENCES '||vschema||'.CM_PLATFORM (ID_PLATFORM))';     
    CREATEOBJECT(vsql,'Unable to create table CM_DEVICE');
    CREATESEQ('CM_DEVICE');
    CREATEUPDATETRG('CM_DEVICE');

    vsql := 'CREATE TABLE '||vschema||'.CM_LICENSE_DETAILS ( 
     ID_LICENSEDETAILS NUMBER  NOT NULL , 
     LICENSE_ID_LICENSE NUMBER  NOT NULL , 
     DELIVERY_LOCATION VARCHAR2 (2000) , 
     LICENSE_ID_PLATFORM NUMBER , 
     LICENSE_ID_DEVICE NUMBER , 
     SM_POSTER_ART_REQUIRED CHAR (1)  NOT NULL , 
     SM_PREVIEW_REQUIRED CHAR (1)  NOT NULL , 
     SM_EFFECTIVE_DATE TIMESTAMP (0)  NOT NULL , 
     SM_EXPIRY_DATE TIMESTAMP (0)  NOT NULL , 
     AE_WATERMARKING CHAR (1) , 
     AE_FINGERPRINTING CHAR (1) , 
     AE_TRANSCODING CHAR (1) , 
     AE_AD_REMOVAL CHAR (1) , 
     AE_CONTENT_EDIT CHAR (1) , 
     AE_QUALITY_CHECK CHAR (1) , 
     AE_CONTENT_SECURITY CHAR (1) , 
     PR_GUARANTEED_SUBSCRIBERS NUMBER , 
     PR_GUARANTEED_BUY_RATE FLOAT , 
     PR_DEEMED_RETAIL_PRICE FLOAT , 
     PR_ACTUAL_RETAIL_PRICE FLOAT , 
     PR_LOST_BUYS FLOAT , 
     PR_REVENUE_SHARE FLOAT , 
     PR_ACTUAL_LICENSE_FEE FLOAT , 
     PR_MINIMUM_LICENSE_FEE FLOAT , 
     PR_UPFRONT_PAYMENT_REQUIRED CHAR (1) , 
     PR_UPFRONT_PAYMENT FLOAT , 
     PR_OVERAGE_REQUIRED CHAR (1) , 
     PR_OVERAGE FLOAT , 
     PR_INSTALLMENT_PLAN_REQUIRED CHAR (1) , 
     PR_INSTALLMENT_PLAN FLOAT , 
     PR_TIERED_PRICING_REQUIRED CHAR (1) , 
     PR_REVENUE_SHARE_REQUIRED CHAR (1) , 
     PR_ID_CURRENCY NUMBER,
     DELIVERY_LOCATION_ID NUMBER,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_LICENSE_DETAILS PRIMARY KEY ( ID_LICENSEDETAILS ),
     CONSTRAINT FK_LICENSE FOREIGN KEY (LICENSE_ID_LICENSE) REFERENCES '||vschema||'.CM_LICENSE(ID_LICENSE),
     CONSTRAINT FK_PRICING_MANAGEMENT_CURRENCY FOREIGN KEY (PR_ID_CURRENCY) REFERENCES '||vschema||'.CM_CURRENCY (ID_CURRENCY),
     CONSTRAINT FK_LICENSE_DETAILS_DEVICE FOREIGN KEY (LICENSE_ID_DEVICE) REFERENCES '||vschema||'.CM_DEVICE (ID_DEVICE),
     CONSTRAINT FK_LICENSE_DETAILS_PLATFORM FOREIGN KEY (LICENSE_ID_PLATFORM) REFERENCES '||vschema||'.CM_PLATFORM (ID_PLATFORM))';
    CREATEOBJECT(vsql,'Unable to create table CM_LICENSE_DETAILS');
    CREATESEQ('CM_LICENSE_DETAILS');
    CREATEUPDATETRG('CM_LICENSE_DETAILS');

    vsql := 'CREATE TABLE '||vschema||'.CM_CUSTOM_LIC_FIELD_VALUE ( 
     ID_VALUE NUMBER  NOT NULL , 
     ID_FIELD NUMBER  NOT NULL , 
     VALUE VARCHAR2 (50) , 
     ID_COPYRIGHTHOLDER NUMBER , 
     ID_CONTRACT NUMBER , 
     ID_DEAL NUMBER , 
     ID_LICENSE number , 
     ID_LICENSEDETAILS NUMBER, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_CUSTOM_FIELD_VALUE PRIMARY KEY ( ID_VALUE ),
     CONSTRAINT FK_CUSTOM_FIELD_VALUE_CONTRACT FOREIGN KEY (ID_CONTRACT) REFERENCES '||vschema||'.CM_CONTRACT (ID_CONTRACT),
     CONSTRAINT FK_CUSTOM_FIELD_VALUE_C_HOLDER FOREIGN KEY (ID_COPYRIGHTHOLDER) REFERENCES '||vschema||'.CM_COPYRIGHT_HOLDER (ID_COPYRIGHTHOLDER),
     CONSTRAINT FK_CUSTOM_FIELD_VALUE_DEAL FOREIGN KEY (ID_DEAL) REFERENCES '||vschema||'.CM_DEAL (ID_DEAL),
     CONSTRAINT FK_CUSTOM_FIELD_VALUE_FIELD FOREIGN KEY (ID_FIELD) REFERENCES '||vschema||'.CM_FIELD (ID_FIELD),
     CONSTRAINT FK_CUSTOM_FIELD_VALUE_LIC FOREIGN KEY (ID_LICENSE) REFERENCES '||vschema||'.CM_LICENSE (ID_LICENSE),
     CONSTRAINT FK_CUST_FIELD_VALUE_LIC_DETAIL FOREIGN KEY (ID_LICENSEDETAILS) REFERENCES '||vschema||'.CM_LICENSE_DETAILS (ID_LICENSEDETAILS))';
    CREATEOBJECT(vsql,'Unable to create table CM_CUSTOM_LIC_FIELD_VALUE');
    CREATESEQ('CM_CUSTOM_LIC_FIELD_VALUE');
    CREATEUPDATETRG('CM_CUSTOM_LIC_FIELD_VALUE');

    vsql := 'CREATE TABLE '||vschema||'.CM_DEAL_CONTACTS ( 
     ID_CONTACT NUMBER  NOT NULL , 
     NAME VARCHAR2 (50)  NOT NULL , 
     EMAIL VARCHAR2 (150)  NOT NULL , 
     OFFICE VARCHAR2 (20) , 
     WORK VARCHAR2 (20) , 
     MOBILE VARCHAR2 (20) , 
     NOTIFY CHAR (1) , 
     ID_DEAL NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DEAL_CONTACTS PRIMARY KEY ( ID_CONTACT ),
     CONSTRAINT FK_DEAL_CONTACTS_DEAL FOREIGN KEY (ID_DEAL) REFERENCES '||vschema||'.CM_DEAL (ID_DEAL))';
    CREATEOBJECT(vsql,'Unable to create table CM_DEAL_CONTACTS');
    CREATESEQ('CM_DEAL_CONTACTS');
    CREATEUPDATETRG('CM_DEAL_CONTACTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_DEAL_DOCUMENTS ( 
     ID_DOCUMENT NUMBER  NOT NULL , 
     ID_DEAL NUMBER  NOT NULL , 
     FILE_NAME VARCHAR2 (50) , 
     DEAL_FILE BLOB, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DEAL_DOCUMENTS PRIMARY KEY ( ID_DOCUMENT ),
     CONSTRAINT FK_DEAL_DOCUMENTS_DEAL FOREIGN KEY (ID_DEAL) REFERENCES '||vschema||'.CM_DEAL (ID_DEAL))';
    CREATEOBJECT(vsql,'Unable to create table CM_DEAL_DOCUMENTS');
    CREATESEQ('CM_DEAL_DOCUMENTS');
    CREATEUPDATETRG('CM_DEAL_DOCUMENTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_DISTRIBUTION_MANAGEMENT ( 
     ID_DISTRIBUTION_MANAGEMENT NUMBER  NOT NULL , 
     DELIVERY_DATE TIMESTAMP (0) , 
     DUE_DATE TIMESTAMP (0) , 
     DELIVERY_COST FLOAT , 
     RETURN_DATE TIMESTAMP (0) , 
     ID_MEDIA NUMBER , 
     ID_DELIVERY_METHOD NUMBER , 
     ID_OWNERSHIP NUMBER,
     METADATA_FORMAT VARCHAR2 (1000), 
     ID_LICENSEDETAILS NUMBER  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_DISTRIBUTION_MANAGEMENT PRIMARY KEY (ID_DISTRIBUTION_MANAGEMENT),
     CONSTRAINT FK_DM_DELIVERY_METHOD FOREIGN KEY (ID_DELIVERY_METHOD) REFERENCES '||vschema||'.CM_DELIVERY_METHOD (ID_DELIVERY_METHOD),
     CONSTRAINT FK_DM_LICENSE_DETAILS FOREIGN KEY (ID_LICENSEDETAILS) REFERENCES '||vschema||'.CM_LICENSE_DETAILS (ID_LICENSEDETAILS),
     CONSTRAINT FK_DM_MEDIA FOREIGN KEY (ID_MEDIA) REFERENCES '||vschema||'.CM_MEDIA (ID_MEDIA),
     CONSTRAINT FK_DM_OWNERSHIP FOREIGN KEY (ID_OWNERSHIP) REFERENCES '||vschema||'.CM_OWNERSHIP (ID_OWNERSHIP))';
    CREATEOBJECT(vsql,'Unable to create table CM_DISTRIBUTION_MANAGEMENT');
    CREATESEQ('CM_DISTRIBUTION_MANAGEMENT');
    CREATEUPDATETRG('CM_DISTRIBUTION_MANAGEMENT');

    vsql := 'CREATE TABLE '||vschema||'.CM_FIELD_DROPDOWN ( 
     ID_FIELD_DROPDOWN NUMBER  NOT NULL , 
     ID_FIELD NUMBER  NOT NULL , 
     VALUE VARCHAR2 (50)  NOT NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_FIELD_DROPDOWN PRIMARY KEY (ID_FIELD_DROPDOWN),
     CONSTRAINT FK_FIELD_DROPDOWN_FIELD FOREIGN KEY (ID_FIELD) REFERENCES '||vschema||'.CM_FIELD (ID_FIELD))';
    CREATEOBJECT(vsql,'Unable to create table CM_FIELD_DROPDOWN');
    CREATESEQ('CM_FIELD_DROPDOWN');
    CREATEUPDATETRG('CM_FIELD_DROPDOWN');

    vsql := 'CREATE TABLE '||vschema||'.CM_LICENSE_DOCUMENTS ( 
     ID_DOCUMENT NUMBER  NOT NULL , 
     ID_LICENSE NUMBER  NOT NULL , 
     FILE_NAME VARCHAR2 (50) , 
     LICENSE_FILE BLOB , 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_LICENSE_DOCUMENTS PRIMARY KEY (ID_DOCUMENT),
     CONSTRAINT FK_LICENSE_DOCUMENTS_LICENSE FOREIGN KEY (ID_LICENSE) REFERENCES '||vschema||'.CM_LICENSE (ID_LICENSE))';
    CREATEOBJECT(vsql,'Unable to create table CM_LICENSE_DOCUMENTS');
    CREATESEQ('CM_LICENSE_DOCUMENTS');
    CREATEUPDATETRG('CM_LICENSE_DOCUMENTS');

    vsql := 'CREATE TABLE '||vschema||'.CM_REVENUE_PRICING ( 
     ID_REVENUE_PRICING NUMBER  NOT NULL , 
     ID_LICENSEDETAILS NUMBER  NOT NULL , 
     THRESHOLD FLOAT , 
     FROM_DT TIMESTAMP (0) , 
     TO_DT TIMESTAMP (0) , 
     PRICE FLOAT  NOT NULL , 
     ID_CURRENCY NUMBER , 
     PENALTY FLOAT ,  
     TYPE VARCHAR2 (2) CHECK ( TYPE IN (''RS'', ''TP'')) ,
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_PRICING_PARAMETERS PRIMARY KEY (ID_REVENUE_PRICING),
     CONSTRAINT FK_REV_PRICING_LIC_DETAILS FOREIGN KEY (ID_LICENSEDETAILS) REFERENCES '||vschema||'.CM_LICENSE_DETAILS (ID_LICENSEDETAILS),
     CONSTRAINT FK_TIERED_PRICING_CURRENCY FOREIGN KEY (ID_CURRENCY) REFERENCES '||vschema||'.CM_CURRENCY (ID_CURRENCY))';
    CREATEOBJECT(vsql,'Unable to create table CM_REVENUE_PRICING');
    CREATESEQ('CM_REVENUE_PRICING');
    CREATEUPDATETRG('CM_REVENUE_PRICING');

    vsql := 'CREATE TABLE '||vschema||'.CM_SM_ASSETS ( 
     ID_ASSET VARCHAR2 (4000)  NOT NULL , 
     ID_LICENSE NUMBER NULL, 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_STORE_ASSETS PRIMARY KEY ( ID_ASSET, ID_LICENSE ))';
    CREATEOBJECT(vsql,'Unable to create table CM_SM_ASSETS');

    vsql := 'CREATE TABLE '||vschema||'.CM_SM_LANGUAGES ( 
     ID_LANGUAGE NUMBER  NOT NULL , 
     ID_LICENSEDETAILS NUMBER  NOT NULL , 
     CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_SM_LANGUAGES PRIMARY KEY ( ID_LANGUAGE, ID_LICENSEDETAILS ),
     CONSTRAINT FK_SM_LANGUAGES_LANGUAGE FOREIGN KEY (ID_LANGUAGE) REFERENCES '||vschema||'.CM_LANGUAGE (ID_LANGUAGE),
     CONSTRAINT FK_SM_LANGUAGES_LIC_DETAILS FOREIGN KEY (ID_LICENSEDETAILS) REFERENCES '||vschema||'.CM_LICENSE_DETAILS (ID_LICENSEDETAILS))';
    CREATEOBJECT(vsql,'Unable to create table CM_SM_LANGUAGES');

    -- Flavio added cm_audit table on 01/20/2012.
    vsql := 'CREATE TABLE '||vschema||'.CM_AUDIT (	
     AUDIT_ID      NUMBER, 
     AUDIT_MESSAGE VARCHAR2(2000 CHAR), 
     USER_NAME     VARCHAR2(100 CHAR), 
     AUDIT_TYPE    NUMBER, 
     AUDIT_ACTION  NUMBER,
     CREATEDATE    TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     UPDATEDATE    TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
     CONSTRAINT PK_AUDIT PRIMARY KEY (AUDIT_ID)
     )';
    CREATEOBJECT(VSQL,'Unable to create table '||vschema||'.CM_AUDIT.');
    CREATESEQ('CM_AUDIT');
    CREATEUPDATETRG('CM_AUDIT');

    vsql := 'CREATE UNIQUE INDEX '||vschema||'.IDX_LICENSE_DETAILS ON '||vschema||'.CM_LICENSE_DETAILS
              (LICENSE_ID_LICENSE, DELIVERY_LOCATION, LICENSE_ID_PLATFORM, LICENSE_ID_DEVICE)';    
    CREATEOBJECT(vsql,'Unable to create table '||vschema||'.IDX_LICENSE_DETAILS');
    
    -- Inserting into table 'CM_SYSTEM_SETTINGS'
    BEGIN      
       EXECUTE IMMEDIATE 'INSERT INTO '||vschema||'.CM_SYSTEM_SETTINGS values (''RELEASE'',''1.0.4'')';        
    EXCEPTION 
       WHEN OTHERS THEN        
          wk_message := 'ERROR';
		      err_msg :=SUBSTR(SQLERRM, 1, 500);
          EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values
             (''error inserting CM_SYSTEM_SETTINGS data'')';
          EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values ('''||err_msg||''')';
          GOTO bypass_install;               
    END;
  
    <<bypass_install>>  
    COMMIT;          
    IF wk_message = 'ERROR' THEN
       RAISE CREATECMDB.INSTALL_FAILED;
    ELSE
       EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values
           (''Installation of ContractManagement completed SUCCESSFULLY'')';
    END IF;
 
   END CREATECM;

  /*****************************************************************************
   --------------------------- ConvertCM for upgrade ---------------------------
   ****************************************************************************/
   PROCEDURE ConvertCM IS
     wk_message	VARCHAR2(500);
     wk_release	varchar2 (25);	
		 wk_count		number := 0;	
		 err_msg		VARCHAR2(500);
   BEGIN
   /*---------------------------------------------------------------------------------------
     Start of Convert release 1.0.0 (CM 1.0.000.00) to 1.0.1 (CM 1.0.000.xx) (CMS 2.0.020.9)
    ---------------------------------------------------------------------------------------*/
        EXECUTE IMMEDIATE 'SELECT setting_value
                             FROM '||vschema||'.CM_SYSTEM_SETTINGS
                             WHERE setting_type = ''RELEASE'''
                    INTO wk_release;
        
        if wk_release =  '1.0.0' then 
          BEGIN
             vsql := 'DECLARE
                         CURSOR C_TBL IS SELECT  TABLE_NAME CM_TABLE_NAME
                                           FROM ALL_TABLES 
                                          WHERE TABLE_NAME LIKE ''CM_%'' AND TABLE_NAME NOT LIKE ''CMS%'' AND OWNER=upper('''||vschema||''')  
                                            AND TABLE_NAME NOT IN (''CM_SYSTEM_SETTINGS'');
                         VTBL VARCHAR2(1000):='''';
                     BEGIN
                        FOR VTBL IN C_TBL LOOP
                           BEGIN
                              EXECUTE IMMEDIATE ''ALTER TABLE '||vschema||'.''||VTBL.CM_TABLE_NAME||'' ADD (CREATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL, UPDATEDATE TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL)'';  
                              EXECUTE IMMEDIATE ''CREATE OR REPLACE TRIGGER '||vschema||'.''||SUBSTR(VTBL.CM_TABLE_NAME,1,26)||''UPD BEFORE UPDATE ON '||vschema||'.''||VTBL.CM_TABLE_NAME||'' FOR EACH ROW
                                                     BEGIN  select systimestamp into :new.updatedate from dual; END;'';
                           EXCEPTION
                             WHEN OTHERS THEN NULL;
                           END;
                        END LOOP;
                     END;';
             CREATEOBJECT(vsql,'Unable to ADD date columns and Update Trigger to CM Tables ','Y');
          END; 
          -- Amit request on 12/20/2011 to change CM_LICENSE.CONTENT_PROVIDER column to null
          vsql := 'ALTER TABLE '||vschema||'.CM_LICENSE MODIFY (CONTENT_PROVIDER NULL)';
          CREATEOBJECT(vsql,'Unable to alter the table CM_LICENSE to modify CONTENT_PROVIDER to null','Y');

          -- Amit request on 12/21/2011 to change CM_LICENSE.NOTES column to null
          vsql := 'ALTER TABLE '||vschema||'.CM_LICENSE MODIFY (NOTES NULL)';
          CREATEOBJECT(vsql,'Unable to alter the table CM_LICENSE to modify NOTES to null','Y');
           
          BEGIN
            EXECUTE IMMEDIATE 'UPDATE '||vschema||'.CM_SYSTEM_SETTINGS SET setting_value=''1.0.1'' WHERE setting_type = ''RELEASE''';  
            EXCEPTION 
          WHEN OTHERS THEN
            wk_message := 'ERROR';
            err_msg :=substr(sqlerrm, 1, 500);
            dbms_output.put_line('convertCM error:'||err_msg);
            execute immediate 'INSERT INTO install_table(install_message) values(''error in upgrade of application database objects:'||
                            'Unable update '||vschema||'.CM_SYSTEM_SETTINGS from 1.0.0 to 1.0.1'')';
                EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values ('''||err_msg||''')';
              GOTO BYPASS_INSTALL; 
          END;

       END IF;

   /*-----------------------------------------------------------------------------------------------------
     Start of Convert release 1.0.1 (CM 1.0.000.00)(CMS 2.0.020.9) to 1.0.2 (CM 1.5.000.00)(CMS 2.5.000.6) 
    ------------------------------------------------------------------------------------------------------*/
        EXECUTE IMMEDIATE 'SELECT setting_value
                             FROM '||vschema||'.CM_SYSTEM_SETTINGS
                             WHERE setting_type = ''RELEASE'''
                    INTO wk_release;
        
        IF WK_RELEASE =  '1.0.1' THEN 
          -- Amit request on 01/19/2012 to update cm_license table all_assets column.
          vsql := 'UPDATE '||vschema||'.CM_LICENSE set ALL_ASSETS = ''N'' where ALL_ASSETS = ''Y''';
          CREATEOBJECT(vsql,'Unable to UPDATE '||vschema||'.CM_LICENSE table.','Y');

          -- Flavio added cm_audit table on 01/20/2012.
          vsql := 'CREATE TABLE '||vschema||'.CM_AUDIT (	
           AUDIT_ID      NUMBER, 
           AUDIT_MESSAGE VARCHAR2(2000 CHAR), 
           USER_NAME     VARCHAR2(100 CHAR), 
           AUDIT_TYPE    NUMBER, 
           AUDIT_ACTION  NUMBER,
           CREATEDATE    TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
           UPDATEDATE    TIMESTAMP(6) DEFAULT (SYSTIMESTAMP) NOT NULL,
           CONSTRAINT PK_AUDIT PRIMARY KEY (AUDIT_ID)
           )';
          CREATEOBJECT(VSQL,'Unable to create table '||VSCHEMA||'.CM_AUDIT.','Y');
          CREATESEQ('CM_AUDIT');
          CREATEUPDATETRG('CM_AUDIT');

          -- Gunnar request on 01/26/2012 to add delivery_location_id column to cm_license_details table.
          VSQL := 'ALTER TABLE '||VSCHEMA||'.cm_license_details ADD (delivery_location_id NUMBER)';
          CREATEOBJECT(vsql,'Unable to ALTER TABLE '||VSCHEMA||'.cm_license_details to add delivery_location_id column.','Y');
           
          BEGIN
            EXECUTE IMMEDIATE 'UPDATE '||vschema||'.CM_SYSTEM_SETTINGS SET setting_value=''1.0.2'' WHERE setting_type = ''RELEASE''';  
            EXCEPTION 
          WHEN OTHERS THEN
            wk_message := 'ERROR';
            err_msg :=substr(sqlerrm, 1, 500);
            dbms_output.put_line('convertCM error:'||err_msg);
            EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values(''error in upgrade of application database objects:'||
                            'Unable update '||vschema||'.CM_SYSTEM_SETTINGS from 1.0.1 to 1.0.2'')';
                EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values ('''||err_msg||''')';
              GOTO BYPASS_INSTALL; 
          END;

       END IF;

   /*-----------------------------------------------------------------------------------------------------
     Start of Convert release 1.0.2 (CM 1.5.000.00)(CMS 2.5.000.6) to 1.0.3 (CM 1.5.000.00)(CMS 2.5.010.10) 
    ------------------------------------------------------------------------------------------------------*/
        EXECUTE IMMEDIATE 'SELECT setting_value
                             FROM '||vschema||'.CM_SYSTEM_SETTINGS
                             WHERE setting_type = ''RELEASE'''
                    INTO wk_release;
        
        IF WK_RELEASE =  '1.0.2' THEN
          -- Flavio updated audit_type column of cm_audit table on 02/15/2012.
          vsql := 'update '||vschema||'.cm_audit set audit_type=10 where audit_type=1';
          CREATEOBJECT(vsql,'Unable to UPDATE '||vschema||'.cm_audit table.','Y');

          vsql := 'update '||vschema||'.cm_audit set audit_type=20 where audit_type=4';
          CREATEOBJECT(vsql,'Unable to UPDATE '||vschema||'.cm_audit table.','Y');

          vsql := 'update '||vschema||'.cm_audit set audit_type=30 where audit_type=3';
          CREATEOBJECT(vsql,'Unable to UPDATE '||vschema||'.cm_audit table.','Y');

          vsql := 'update '||vschema||'.cm_audit set audit_type=40 where audit_type=2';
          CREATEOBJECT(vsql,'Unable to UPDATE '||vschema||'.cm_audit table.','Y');

          -- Gunnar request on 02/15/2012 to updated active column of CM_CONTRACT and CM_DEAL tables to Y.
          vsql := 'UPDATE '||vschema||'.CM_CONTRACT SET ACTIVE=''Y'' WHERE ACTIVE=''N''';
          createobject(vsql,'Unable to UPDATE '||vschema||'.CM_CONTRACT of ACTIVE column.','Y');

          vsql := 'UPDATE '||vschema||'.CM_DEAL SET ACTIVE=''Y'' WHERE ACTIVE=''N''';
          createobject(vsql,'Unable to UPDATE '||vschema||'.CM_DEAL of ACTIVE column.','Y');           

          BEGIN
            EXECUTE IMMEDIATE 'UPDATE '||vschema||'.CM_SYSTEM_SETTINGS SET setting_value=''1.0.3'' WHERE setting_type = ''RELEASE''';  
            EXCEPTION 
          WHEN OTHERS THEN
            wk_message := 'ERROR';
            err_msg :=substr(sqlerrm, 1, 500);
            dbms_output.put_line('convertCM error:'||err_msg);
            EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values(''error in upgrade of application database objects:'||
                            'Unable update '||vschema||'.CM_SYSTEM_SETTINGS from 1.0.2 to 1.0.3'')';
                EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values ('''||err_msg||''')';
              GOTO BYPASS_INSTALL; 
          END;

       END IF;        

   /*-----------------------------------------------------------------------------------------------------
     Start of Convert release 1.0.3 (CM 1.5.000.00)(CMS2.5.000.10) to 1.0.4 (CM 1.5.000.00)(CMS2.5.000.x) 
    ------------------------------------------------------------------------------------------------------*/
        EXECUTE IMMEDIATE 'SELECT setting_value
                             FROM '||vschema||'.CM_SYSTEM_SETTINGS
                             WHERE setting_type = ''RELEASE'''
                    INTO wk_release;
        
        IF WK_RELEASE =  '1.0.3' THEN
          -- Suneeta request on 04/12/2012.
          vsql := 'update '||vschema||'.cm_license set content_provider = '''' where id_license in ( '||
                  'select id_license from '||vschema||'.cm_license l, '||vschema||'.pmm_partner p where l.content_provider = p.providerid and p.partnertype in (0,1))';
          createobject(vsql,'Unable to UPDATE '||vschema||'.cm_license table content_provider column to null.','Y');

          BEGIN
            EXECUTE IMMEDIATE 'UPDATE '||vschema||'.CM_SYSTEM_SETTINGS SET setting_value=''1.0.4'' WHERE setting_type = ''RELEASE''';  
            EXCEPTION 
          WHEN OTHERS THEN
            wk_message := 'ERROR';
            err_msg :=substr(sqlerrm, 1, 500);
            dbms_output.put_line('convertCM error:'||err_msg);
            EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values(''error in upgrade of application database objects:'||
                            'Unable update '||vschema||'.CM_SYSTEM_SETTINGS from 1.0.3 to 1.0.4'')';
                EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values ('''||err_msg||''')';
              GOTO BYPASS_INSTALL; 
          END;
       END IF;        

			<<bypass_install>>  
          COMMIT;          
          IF wk_message = 'ERROR' THEN          	
          	 RAISE CREATECMDB.INSTALL_FAILED;             
          ELSE          
             EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values
           (''Upgrade of Contract Management completed SUCCESSFULLY'')';
             
          END IF;
    END ConvertCM;

    PROCEDURE INSTALL_CM IS
		  error		number(1,0) := 0;
        wk_count	number;
        wk_message varchar2(255);
        err_msg		varchar2(500);
    BEGIN
      EXECUTE IMMEDIATE 'alter system set deferred_segment_creation=false';
      EXECUTE IMMEDIATE 'alter system set nls_length_semantics=char';
      EXECUTE IMMEDIATE 'alter session set nls_length_semantics=char';

		  EXECUTE IMMEDIATE 'select count(*) FROM dba_tables WHERE OWNER = '''||vschema||''' and table_name like ''CM_SYSTEM_SETTINGS''' into	wk_count ;

        IF WK_COUNT < 1 THEN 
        
          BEGIN
	        	  CREATECM;
	        EXCEPTION 
				     WHEN OTHERS THEN
					      wk_message := 'ERROR';
					      EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values('||SQLERRM||'An error occured while running the createCM procedure.'')';
	        END;

        ELSE
          BEGIN
               convertcm;
	        EXCEPTION 
				     WHEN OTHERS THEN
					      wk_message := 'ERROR';
                err_msg := REPLACE(substr(sqlerrm, 1, 500),'''','');
                dbms_output.put_line('Error Message:'||err_msg);
                EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values('''||err_msg||' An error occured while running the convertCM procedure.'')';
	        END;
        END IF;

		  <<bypass_install>>
		  COMMIT;
        IF wk_message = 'ERROR' THEN        	
        	 EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values('''||err_msg||' An error occured while installing the Contract Management procedure.'')';
        ELSE
          EXECUTE IMMEDIATE 'INSERT INTO install_table(install_message) values(''Contract Management script execution completed SUCCESSFULLY'')';
        END IF;
    END INSTALL_CM;  
    
END CREATECMDB;
/

begin
    CREATECMDB.install_cm; 
END;
/
exit
