--set echo on
--set serveroutput on

DEF EPG_DB_VERSION=&1;

CREATE OR REPLACE PACKAGE EPGMANAGERDB AS
  procedure installepgmanager;
  procedure dropepgmanager;
  procedure configxmldb;
  procedure creatcredittable;
  procedure dropcreditsandcasttables(pattern varchar2);
  vschema varchar2(100) := 'EPGMANAGER';
END EPGMANAGERDB;
/
create or replace
PACKAGE BODY EPGMANAGERDB AS

FUNCTION IS_TABLE_EXISTS (IN_OWNER VARCHAR2, IN_TABLE_NAME VARCHAR2) RETURN boolean is
counter number(5) := 0;
begin
  select count(1) into counter from dba_tables where owner=UPPER(IN_OWNER) AND TABLE_NAME=UPPER(IN_TABLE_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;



FUNCTION IS_COLUMN_EXISTS (IN_OWNER VARCHAR2, IN_TABLE_NAME VARCHAR2, IN_COLUMN_NAME VARCHAR2) RETURN boolean is
counter number(5) := 0;
begin
  select count(1) into counter from DBA_TAB_COLS where owner=UPPER(IN_OWNER) AND TABLE_NAME=UPPER(IN_TABLE_NAME) AND COLUMN_NAME=UPPER(IN_COLUMN_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;

FUNCTION IS_INDEX_EXISTS (IN_OWNER VARCHAR2, IN_INDEX_NAME VARCHAR2) RETURN boolean is

counter number(5) := 0;
begin
  select count(1) into counter from DBA_INDEXES where owner=UPPER(IN_OWNER)  AND INDEX_NAME=UPPER(IN_INDEX_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;

FUNCTION IS_COLUMN_VALUE_EXISTS (IN_OWNER VARCHAR2, IN_TABLE_NAME VARCHAR2, IN_COLUMN_NAME VARCHAR2, IN_COLUMN_VALUE varchar2, IN_DATA_TYPE VARCHAR2 DEFAULT 'VARCHAR2') RETURN boolean is

counter number(5) := 0;
vsql varchar2(1000);
begin
  IF upper(IN_DATA_TYPE) = 'NUMBER' THEN
  vsql := 'select count(1) from '||IN_OWNER||'.'||IN_TABLE_NAME||' WHERE '||IN_COLUMN_NAME||'='||IN_COLUMN_VALUE;
  ELSIF (upper(IN_DATA_TYPE) = 'VARCHAR' OR upper(IN_DATA_TYPE) = 'VARCHAR2') THEN
  vsql := 'select count(1) from '||IN_OWNER||'.'||IN_TABLE_NAME||' WHERE upper('||IN_COLUMN_NAME||')=upper('''||IN_COLUMN_VALUE||''')';
  END IF;
  EXECUTE IMMEDIATE vsql INTO counter;

  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;

FUNCTION IS_CONSTRAINT_EXISTS (IN_OWNER VARCHAR2, IN_TABLE_NAME VARCHAR2, IN_CONSTRAINT_NAME VARCHAR2) RETURN boolean is

counter number(5) := 0;
begin
  select count(1) into counter from  dba_constraints where owner =UPPER(IN_OWNER) and table_name=UPPER(IN_TABLE_NAME) AND CONSTRAINT_NAME=UPPER(IN_CONSTRAINT_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;

FUNCTION GET_COLUMN_DATA_TYPE (IN_OWNER VARCHAR2, IN_TABLE_NAME VARCHAR2, IN_COLUMN_NAME VARCHAR2) RETURN VARCHAR2 is

v_data_type varchar2(200) ;
begin
  SELECT DATA_TYPE INTO v_data_type FROM DBA_TAB_COLS WHERE OWNER=UPPER(IN_OWNER) AND TABLE_NAME = UPPER(IN_TABLE_NAME) AND COLUMN_NAME=UPPER(IN_COLUMN_NAME);
  return v_data_type;
end;

FUNCTION IS_COLUMN_ALLOW_NULLS (IN_OWNER VARCHAR2, IN_TABLE_NAME VARCHAR2, IN_COLUMN_NAME VARCHAR2) RETURN boolean is

v_nullable varchar2(200) ;
begin
  SELECT NULLABLE INTO v_nullable FROM DBA_TAB_COLS WHERE OWNER=UPPER(IN_OWNER) AND TABLE_NAME = UPPER(IN_TABLE_NAME) AND COLUMN_NAME=UPPER(IN_COLUMN_NAME);
  IF v_nullable = 'Y' then
    return true;
  else
    return false;
 end if;
end;

 FUNCTION IS_SEQUENCE_EXISTS (IN_SEQUENCE_OWNER VARCHAR2, IN_SEQUENCE_NAME VARCHAR2) RETURN boolean is

counter number(5) := 0;
begin
  select count(1) into counter from dba_sequences where sequence_owner=upper(IN_SEQUENCE_OWNER) AND sequence_name=upper(IN_SEQUENCE_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;

FUNCTION IS_VIEW_EXISTS (IN_OWNER VARCHAR2, IN_VIEW_NAME VARCHAR2) RETURN boolean is

counter number(5) := 0;
begin
  select count(1) into counter from dba_views where owner=upper(IN_OWNER) AND VIEW_NAME=upper(IN_VIEW_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;

PROCEDURE CONFIGXMLDB AS
    newconfig XMLType;
    success boolean := false;
  BEGIN
    SELECT
      updateXML(
        DBMS_XDB.cfg_get(),
        '/xdbconfig/sysconfig/protocolconfig/ftpconfig/session-timeout/text()',
        999999999,
       '/xdbconfig/sysconfig/protocolconfig/ftpconfig/ftp-port/text()',
       2100
        )
      INTO newconfig
      FROM DUAL;

    DBMS_XDB.cfg_update(newconfig);

    SELECT
      updateXML(
        DBMS_XDB.cfg_get(),
        '/xdbconfig/sysconfig/call-timeout/text()',
        999999999,
        '/xdbconfig/sysconfig/default-lock-timeout/text()',
        999999999)
      INTO newconfig
      FROM DUAL;

    DBMS_XDB.cfg_update(newconfig);

    SELECT
      updateXML(
        DBMS_XDB.cfg_get(),
        '/xdbconfig/sysconfig/protocolconfig/common/session-timeout/text()',
        999999999,
       '/xdbconfig/sysconfig/persistent-sessions/text()',
       'true')
      INTO newconfig
      FROM DUAL;

    DBMS_XDB.cfg_update(newconfig);

    SELECT
      updateXML(
        DBMS_XDB.cfg_get(),
        '/xdbconfig/sysconfig/protocolconfig/ftpconfig/buffer-size/text()',
        1048496)
      INTO newconfig
      FROM DUAL;

    DBMS_XDB.cfg_update(newconfig);
    
    -- create FTP folder /public/epgmanager
    if not DBMS_XDB.existsresource('/public/epgmanager') then
    	success := DBMS_XDB.createfolder('/public/epgmanager');
    end if;
        
  END CONFIGXMLDB;


/* Procedure: SetDbCreateFileDest. Sets the default directory for Oracle datafiles so the path
              need not be given when creating tablespaces. */
  PROCEDURE SETDBCREATEFILEDEST(vpath varchar2) IS
  BEGIN
    execute immediate 'alter system set db_create_file_dest='''||vpath||''' scope=both';
  END SETDBCREATEFILEDEST;

   /* Procedure: CreateObject. Executes a dynamic SQL statement and logs error in install_table*/
  PROCEDURE CREATEOBJECT(vsql varchar2,errmsg varchar2, mask varchar2 default 'N') IS
    verrmsg varchar2(32000);
  BEGIN
    execute immediate vsql;
  exception
    when others then
          verrmsg := errmsg ||' '||sqlerrm;
          select cast(verrmsg as varchar2(4000))
          into verrmsg
          from dual;
      if upper(mask) != 'Y' then
          execute immediate 'INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')';
          commit;
          raise;
      end if;
  END CREATEOBJECT;

  procedure dropepgmanager is
    vsql varchar2(32767);
  begin
    vsql := 'drop table provider';
    CREATEOBJECT(vsql,'Drop table provider','Y');
  end;
  
/* Procedure: InsertEntityMapping. inserts a new record into the EntityMappping table */
  PROCEDURE INSERTENTITYMAPPING(mapname varchar2, maptable varchar2, mapcolumn varchar2, maptype varchar2, mapcriteria number, mappredicate number) IS
  	vsql varchar2(2000);
  BEGIN
	  vsql :='INSERT INTO '||vschema||'.ENTITY_MAPPING(ID, NAME, TABLE_NAME, COLUMN_NAME, DATA_TYPE, ALLOW_AS_CRITERIA, ALLOW_AS_PREDICATE) values
	 ('||vschema||'.ENTITY_MAPPING_SEQ.NEXTVAL,'''||mapname||''', '''||maptable||''','''||mapcolumn||''','''||maptype||''','||mapcriteria||','||mappredicate||')';
	 CREATEOBJECT(vsql,'Insert standard EntityMapping');
	  
  END INSERTENTITYMAPPING;
  

  PROCEDURE setInstalledDBVersion(newVersion varchar2) IS
  	vsql varchar2(32767);
  BEGIN
      	--Version information will be updated by installer.
    	vsql := ' update '||vschema||'.config_setting set VALUE = '''||newVersion||''', UPDATE_DATE = current_timestamp where KEY = ''epgmanagerversion''';
		CREATEOBJECT(vsql,' Updating the EPGMGR db version ');

  END;

  

/* Procedure: CreateUpdTimeTRG. Updates the UpdateTime column with the current date and time when any
              update statement is issued against the table.*/
  PROCEDURE CREATEUPDATETRG(vtabname varchar2) IS
    vcount number := 0;
    vsql varchar2(32000);
  BEGIN
    select count(*)
    into vcount
    from dba_tables dt, dba_tab_columns dtc
    where dt.table_name = dtc.table_name
    and dt.table_name=upper(vtabname)
    and dt.owner = dtc.owner
    and dt.owner = vschema
    and dtc.column_name = 'UPDATE_DATE';

	  if ( vcount > 0 ) then
      vsql := 'CREATE OR REPLACE TRIGGER '||vschema||'.'||vtabname||'UPD BEFORE UPDATE ON '||vschema||'.'||vtabname||' FOR EACH ROW
            DECLARE
              vcurrent_timestamp timestamp with time zone;
            BEGIN
				if (:new.update_date is null) then
	                select current_timestamp into vcurrent_timestamp from dual;
	                :new.update_date := vcurrent_timestamp;
				end if;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'UPD');
    end if;

    select count(*)
    into vcount
    from dba_tables dt, dba_tab_columns dtc
    where dt.table_name = dtc.table_name
    and dt.table_name=upper(vtabname)
    and dt.owner = dtc.owner
    and dt.owner = vschema
    and dtc.column_name = 'CREATE_DATE';

	  if ( vcount > 0 ) then
      vsql := 'CREATE OR REPLACE TRIGGER '||vschema||'.'||vtabname||'CRT BEFORE INSERT ON '||vschema||'.'||vtabname||' FOR EACH ROW
            DECLARE
              vcurrent_timestamp timestamp with time zone;
            BEGIN
              if (:new.create_date is null or :new.update_date is null) then
                select current_timestamp into vcurrent_timestamp from dual;
                :new.create_date := vcurrent_timestamp;
                :new.update_date := vcurrent_timestamp;
              end if;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'UPD');
    end if;

   END CREATEUPDATETRG;

/* Procedure: CreateSeq. Creates a sequence on a primary key of a table.
              Sequence name is <tablename>_SEQ.*/
  PROCEDURE CREATESEQ(vtabname varchar2, vstartwith number default 1) IS
    vcolname dba_cons_columns.column_name%type;
    vsql varchar2(32000);
  BEGIN
    select dcc.column_name
    into vcolname
    from dba_constraints dc, dba_cons_columns dcc
    where dc.owner=vschema
    and dc.owner = dcc.owner
    and dc.table_name = dcc.table_name
    and dc.table_name=upper(vtabname)
    and dc.constraint_type='P'
    and dc.constraint_name = dcc.constraint_name
    and rownum < 2;

		-- Create Sequence for Table
      vsql :=  'CREATE SEQUENCE '||vschema||'.'||vtabname||'_SEQ INCREMENT BY 1 START WITH '||vstartwith||' NOCACHE';
      CREATEOBJECT(vsql,'Create Sequence'||vtabname||'_SEQ');

    -- Create before insert trigger to populate the PK column
      vsql := 'CREATE OR REPLACE TRIGGER '||vschema||'.'||vtabname||'TRG BEFORE INSERT ON '||vschema||'.'||vtabname||' FOR EACH ROW
            DECLARE
              vseq number;
            BEGIN
              if (:new.'||vcolname||' is null) then
                select '||vschema||'.'||vtabname||'_SEQ.nextval into vseq from dual;
                :new.'||vcolname||' := vseq;
              end if;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'TRG');
   END CREATESEQ;


  procedure createepgmanager IS
    vsql varchar2(32767);
  BEGIN


   /********************** CREATE EPGManager USER AND TABLESPACES ******************/

    -- Set db_create_file_dest to default directory
    select substr(file_name,1,instr(file_name,decode(platform.platformname,'windows','\','/'),-1) -1)
    INTO vsql
    from dba_data_files, (select case when lower(platform_name) like '%windows%' then 'windows' else 'unix' end as platformname from v$database )  platform
    WHERE tablespace_name = 'SYSTEM';

    --SETDBCREATEFILEDEST(vsql);

     -- Drop epgmanger_data tablespace
      vsql :='drop tablespace epgmanager_data including contents and datafiles';
			CREATEOBJECT(vsql,'DROP TABLESPACE epgmanager','Y');

      -- Creating epgmanager_data tablespace
      vsql :='create tablespace epgmanager_data datafile size 500M AUTOEXTEND ON NEXT 250M MAXSIZE UNLIMITED EXTENT MANAGEMENT LOCAL AUTOALLOCATE SEGMENT SPACE MANAGEMENT AUTO';
			CREATEOBJECT(vsql,'CREATE TABLESPACE epgmanager_data', 'Y');

	  -- Set Default temp tablespace to temp
	  --THROWS ERROR TEMP TABLESPACE ALREADY SET! hence commenting
      --vsql :=  'ALTER DATABASE DEFAULT TEMPORARY TABLESPACE temp';
      --CREATEOBJECT(vsql,'Default Tablespace temp','Y');

		--DROP EPGMANAGER USER
      vsql :=  'drop user EPGMANAGER cascade';
      CREATEOBJECT(vsql,'DROP USER EPGMANAGER','Y');

		--CREATE EPGMANAGER USER
      vsql :=  'create user EPGMANAGER identified by epgmanager default tablespace epgmanager_data';
      CREATEOBJECT(vsql,'CREATE USER EPGMANAGER');

		--Grant privs to EPGMANAGER user
      vsql :=  'grant connect,resource,CREATE SESSION,create any view,CREATE TABLE,CREATE TRIGGER,CREATE TYPE,ALTER SESSION,xdbadmin,CREATE SEQUENCE to EPGMANAGER';
      CREATEOBJECT(vsql,'Unable to grant resource, connect, and create session to the EPGMANAGER user.');

      vsql := 'grant all on dbms_errlog to EPGMANAGER';
      CREATEOBJECT(vsql,'Unable to grant all on dbms_errlog to EPGMANAGER');

      vsql := 'GRANT debug connect session, debug any procedure TO EPGMANAGER';
      CREATEOBJECT(vsql,'Unable to grant debug connect session, debug any procedure to EPGMANAGER');
      
      vsql := 'grant all on dbms_xmlschema to EPGMANAGER';
      CREATEOBJECT(vsql,'Unable to grant all on dbms_xmlschema to EPGMANAGER');
      
      -- create output directory object
      vsql :='create or replace directory EPGPUBLISH as ''/home/oracle/epgpublish''';
      CREATEOBJECT(vsql, 'unable to create replace epg publish folder');
      
      -- grant epgmanager access
      vsql :='grant all on directory EPGPUBLISH to EPGMANAGER';
      CREATEOBJECT(vsql, 'unable to grant epgmanager access to EPGPUBLISH directory');

    /********************** XMLDB configuration ******************************/
          --Recreate directory epgmanagerdir
      vsql := 'drop directory epgmanagerdir';
      CREATEOBJECT(vsql,'Unable to drop directory epgmanagerdir','Y');
         --epgmanagerdir is physcially created by calling script..here we make Oracle aware of it
      vsql := 'create directory epgmanagerdir as ''/home/oracle/epgmanager''';
      CREATEOBJECT(vsql,'Unable to create directory epgmanagerdir');

      vsql := 'grant all on directory epgmanagerdir to public,epgmanager';
      CREATEOBJECT(vsql,'Unable to grant all on directory epgmanagerdir');

      --Register Schema ETV_EPG.xsd
      -- creates GUIDE_TAB table internally (Structured XML Type storage)
      -- After the schema is registered any ETV xml FTPed to Oracle as user EPGManager will be available as a row in GUIDE_TAB table
      vsql := 'BEGIN
                DBMS_XMLSCHEMA.registerSchema(
                  SCHEMAURL => ''ETV_EPG.xsd'',
                  SCHEMADOC => bfilename(''EPGMANAGERDIR'',''ETV_EPG.xsd''),
                  OWNER => ''EPGMANAGER'',
                  CSID => nls_charset_id(''AL32UTF8'')
                  );
               END;';
      CREATEOBJECT(vsql,'Unable to registerSchema ETV_EPG.xsd');
      
      --Once a schema is registered it is copied by Oracle to schema folder hence we will delete the epgmanagerdir folder
      vsql := 'drop directory epgmanagerdir';
      CREATEOBJECT(vsql,'Unable to drop directory epgmanagerdir','Y');
      
    /********************** EPGMANAGER STAGING VIEWS ************************/
      
      /* 
       * These views are of the raw xml data and are used by loadtstv to pull into the staging table
       *  there is one view per main table
       */
      
	--Staging Program rows
    vsql := 'create or replace view '||vschema||'.stg_program_view as
    SELECT xmlcolumns.*
    FROM (select sys_nc_rowinfo$ as res  from guide_tab ) rv,
    XMLTABLE(''/GuideData/SlicerGuideData/PROGRAM/row''
    PASSING rv.res
    COLUMNS
    DATABASE_KEY nvarchar2(100) PATH ''DATABASE_KEY'',
    LANGUAGE nvarchar2(100) PATH ''LANGUAGE'',
    TITLE nvarchar2(1000) PATH ''TITLE'',
    REDUCED_TITLE1 nvarchar2(1000) PATH ''REDUCED_TITLE[1]'',
    REDUCED_TITLE2 nvarchar2(1000) PATH ''REDUCED_TITLE[2]'',
    REDUCED_TITLE3 nvarchar2(1000) PATH ''REDUCED_TITLE[3]'',
    REDUCED_TITLE4 nvarchar2(1000) PATH ''REDUCED_TITLE[4]'',
    ALT_TITLE nvarchar2(1000) PATH ''ALT_TITLE'',
    REDUCED_DESC1 nvarchar2(1000) PATH ''REDUCED_DESC[1]'',
    REDUCED_DESC2 nvarchar2(1000) PATH ''REDUCED_DESC[2]'',
    REDUCED_DESC3 nvarchar2(1000) PATH ''REDUCED_DESC[3]'',
    REDUCED_DESC4 nvarchar2(1000) PATH ''REDUCED_DESC[4]'',
    ADVISORY_DESC1 nvarchar2(100) PATH ''ADVISORY_DESC[1]'',
    ADVISORY_DESC2 nvarchar2(100) PATH ''ADVISORY_DESC[2]'',
    ADVISORY_DESC3 nvarchar2(100) PATH ''ADVISORY_DESC[3]'',
    ADVISORY_DESC4 nvarchar2(100) PATH ''ADVISORY_DESC[4]'',
    ADVISORY_DESC5 nvarchar2(100) PATH ''ADVISORY_DESC[5]'',
    ADVISORY_DESC6 nvarchar2(100) PATH ''ADVISORY_DESC[6]'',
    /* CAST_CREDITS*/
    CAST_NAME_ROLE xmltype PATH ''CAST_NAME_ROLE'',
    /* CAST_CREDITS*/
    CREDITS_NAME_ROLE xmltype PATH ''CREDITS_NAME_ROLE'',
    GENRE_DESC1 nvarchar2(100) PATH ''GENRE_DESC[1]'',
    GENRE_DESC2 nvarchar2(100) PATH ''GENRE_DESC[2]'',
    GENRE_DESC3 nvarchar2(100) PATH ''GENRE_DESC[3]'',
    GENRE_DESC4 nvarchar2(100) PATH ''GENRE_DESC[4]'',
    GENRE_DESC5 nvarchar2(100) PATH ''GENRE_DESC[5]'',
    GENRE_DESC6 nvarchar2(100) PATH ''GENRE_DESC[6]'',
    DESCRIPTION1 nvarchar2(2000) PATH ''DESC[1]'',
    DESCRIPTION2 nvarchar2(2000) PATH ''DESC[2]'',
    YEAR_PROGRAM nvarchar2(100) PATH ''YEAR'',
    MPAA_RATING nvarchar2(1000) PATH ''MPAA_RATING'',
    STAR_RATING nvarchar2(1000) PATH ''STAR_RATING'',
    PROGRAM_RATING xmltype PATH ''PROGRAM_RATING'',
    RUN_TIME varchar2(100) PATH ''RUN_TIME'',
    COLOR_CODE nvarchar2(100) PATH ''COLOR_CODE'',
    PROGRAM_LANGUAGE nvarchar2(100) PATH ''PROGRAM_LANGUAGE'',
    AUDIO_COMPONENT xmltype PATH ''AUDIO_COMPONENT'',
    SUBTITLE_COMPONENT xmltype PATH ''SUBTITLE_COMPONENT'',
    ORG_COUNTRY nvarchar2(100) PATH ''ORG_COUNTRY'',
    MADE_FOR_TV nvarchar2(1) PATH ''MADE_FOR_TV'',
    SOURCE_TYPE nvarchar2(100) PATH ''SOURCE_TYPE'',
    SHOW_TYPE nvarchar2(200) PATH ''SHOW_TYPE'',
    HOLIDAY nvarchar2(100) PATH ''HOLIDAY'',
    SYN_EPI_NUM nvarchar2(100) PATH ''SYN_EPI_NUM'',
    ALT_SYN_EPI_NUM nvarchar2(100) PATH ''ALT_SYN_EPI_NUM'',
    EPI_TITLE nvarchar2(2000) PATH ''EPI_TITLE'',
    NET_SYN_SOURCE nvarchar2(100) PATH ''NET_SYN_SOURCE'',
    NET_SYN_TYPE nvarchar2(100) PATH ''NET_SYN_TYPE'',
    ORG_STUDIO nvarchar2(100) PATH ''ORG_STUDIO'',
    GAME_DATE varchar2(100) PATH ''GAME_DATE'',
    GAME_TIMEZONE varchar2(100) PATH ''GAME_TIME_ZONE'',
    ORG_AIR_DATE varchar2(100) PATH ''ORG_AIR_DATE'',
    UNIQUE_ID  nvarchar2(100) PATH ''UNIQUE_ID'',
    USER_DATA1	nvarchar2(1000) PATH ''USER_DATA[1]'',
    USER_DATA2	nvarchar2(1000) PATH ''USER_DATA[2]'',
    USER_DATA3	nvarchar2(1000) PATH ''USER_DATA[3]'',
    USER_DATA4	nvarchar2(1000) PATH ''USER_DATA[4]'',
    USER_DATA5	nvarchar2(1000) PATH ''USER_DATA[5]'',
    USER_DATA6	nvarchar2(1000) PATH ''USER_DATA[6]'',
    USER_DATA7	nvarchar2(1000) PATH ''USER_DATA[7]'',
    USER_DATA8	nvarchar2(1000) PATH ''USER_DATA[8]'',
    USER_DATA9	nvarchar2(1000) PATH ''USER_DATA[9]'',
    USER_DATA10	nvarchar2(1000) PATH ''USER_DATA[10]'',
    USER_DATA11	nvarchar2(1000) PATH ''USER_DATA[11]'',
    USER_DATA12	nvarchar2(1000) PATH ''USER_DATA[12]'',
    USER_DATA13	nvarchar2(1000) PATH ''USER_DATA[13]'',
    PROGRAM_GENERIC  xmltype PATH ''PROGRAM_GENERIC''
    ) xmlcolumns';
    CREATEOBJECT(vsql,'create view stg_program_view');

    --Staging HEADEND row
    vsql := 'create or replace view '||vschema||'.stg_HEADEND_view as
    SELECT xmlcolumns.*
    FROM (select sys_nc_rowinfo$ as res  from guide_tab ) rv,
    XMLTABLE(''/GuideData/SlicerGuideData/HEADEND/row''
    PASSING rv.res
    COLUMNS
    HE_HEADEND_ID	nvarchar2(100) PATH ''HE_HEADEND_ID'',
    HE_COMMUNITY_NAME		nvarchar2(200) PATH ''HE_COMMUNITY_NAME'',
    HE_COUNTY_NAME		nvarchar2(200)  PATH ''HE_COUNTY_NAME'',
    HE_COUNTY_SIZE		nvarchar2(1) PATH ''HE_COUNTY_SIZE'',
    HE_ST_COUNTY_CODE		nvarchar2(100) PATH ''HE_ST_COUNTY_CODE'',
    HE_STATE_SERVED		nvarchar2(2) PATH ''HE_STATE_SERVED'',
    HE_ZIP_CODE		nvarchar2(100) PATH ''HE_ZIP_CODE'',
    HE_DMA_CODE		nvarchar2(100) PATH ''HE_DMA_CODE'',
    HE_DMA_NAME		nvarchar2(300) PATH ''HE_DMA_NAME'',
    HE_MSO_CODE		number PATH ''HE_MSO_CODE'',
    HE_DMA_RANK		nvarchar2(10) PATH ''HE_DMA_RANK'',
    HE_HEADEND_NAME		nvarchar2(1000) PATH ''HE_HEADEND_NAME'',
    HE_HEADEND_LOCATION		nvarchar2(100)  PATH ''HE_HEADEND_LOCATION'',
    HE_MSO_NAME		nvarchar2(200) PATH ''HE_MSO_NAME'',
    HE_TIME_ZONE_CODE		nvarchar2(100)  PATH ''HE_TIME_ZONE_CODE'',
	HE_GENERIC 			xmltype PATH ''HE_GENERIC''		) xmlcolumns';
    CREATEOBJECT(vsql,'create view stg_headend_view');

    --Get LINEUP rows
    vsql := 'create or replace view '||vschema||'.stg_LINEUP_view as
    SELECT xmlcolumns.*
    FROM (select sys_nc_rowinfo$ as res  from guide_tab ) rv,
    XMLTABLE(''/GuideData/SlicerGuideData/LINEUP/row''
    PASSING rv.res
    COLUMNS
    CL_ROW_ID	nvarchar2(20) PATH ''CL_ROW_ID'',
    CL_STATION_NUM	nvarchar2(100) PATH ''CL_STATION_NUM'',
    CL_HEADEND_ID	nvarchar2(100) PATH ''CL_HEADEND_ID'',
    CL_DEVICE		nvarchar2(2) PATH ''CL_DEVICE'',
    CL_TMS_CHAN		nvarchar2(20) PATH ''CL_TMS_CHAN'',
    CL_SERVICE_TIER		nvarchar2(1) PATH ''CL_SERVICE_TIER'',
    CL_EFFECTIVE_DATE	varchar2(100) PATH ''CL_EFFECTIVE_DATE'',
    CL_EXPIRATION_DATE		varchar2(100) PATH ''CL_EXPIRATION_DATE'',
	CL_GENERIC		xmltype PATH ''CL_GENERIC'' ) xmlcolumns';
    CREATEOBJECT(vsql,'create view stg_lineup_view');

    vsql := 'create or replace view '||vschema||'.stg_SCHEDULE_view as
    SELECT xmlcolumns.*
    FROM (select sys_nc_rowinfo$ as res  from guide_tab ) rv,
    XMLTABLE(''/GuideData/SlicerGuideData/SCHEDULE/row''
    PASSING rv.res
    COLUMNS
    STATION_NUM 	NVARCHAR2 (100) PATH ''STATION_NUM'',
    DATABASE_KEY 	nvarchar2(100) PATH ''DATABASE_KEY'',
    SCHEDULE_LANGUAGE		nvarchar2(100) PATH ''LANGUAGE'',
    AIR_DATE 	varchar2(200) PATH ''AIR_DATE'',
    DURATION 		varchar2(100) PATH ''DURATION'',
    EVENT_ID		xmltype PATH ''EVENT_ID'',
    PART_NUM 		number PATH ''PART_NUM'',
    NUM_OF_PARTS 		number PATH ''NUM_OF_PARTS'',
	SERIES_ID			nvarchar2(100) PATH ''SERIES_LINK/SeriesIdentifier'',
    SERIES_LINK			xmltype PATH ''SERIES_LINK'',
    CC varchar2(1) PATH ''CC'',
    STEREO 		varchar2(1) PATH ''STEREO'',
    REPEAT 		varchar2(1) PATH ''REPEAT'',
    LIVE_TAPE_DELAY 	varchar2(100) PATH ''LIVE_TAPE_DELAY'',
    SUBTITLED varchar2(1) PATH ''SUBTITLED'',
    PREMIERE_FINALE 	varchar2(100)  PATH ''PREMIERE_FINALE'',
    JOINED_IN_PROGRESS 		varchar2(1)  PATH ''JOINED_IN_PROGRESS'',
    CABLE_IN_THE_CLASSROOM 		varchar2(1) PATH ''CABLE_IN_THE_CLASSROOM'',
    TV_RATING 		varchar2(1000)  PATH ''TV_RATING'',
    SAP varchar2(1) PATH ''SAP'',
    BLACKOUT varchar2(1) PATH ''BLACKOUT'',
    SEX_RATING varchar2(1) PATH ''SEX_RATING'',
    VIOLENCE_RATING varchar2(1) PATH ''VIOLENCE_RATING'',
    LANGUAGE_RATING varchar2(1) PATH ''LANGUAGE_RATING'',
    DIALOG_RATING varchar2(1)  PATH ''DIALOG_RATING'',
    FV_RATING varchar2(1)  PATH ''FV_RATING'',
    ENHANCED varchar2(1) PATH ''ENHANCED'',
    THREE_D varchar2(1) PATH ''THREE_D'',
    LETTERBOX varchar2(1) PATH ''LETTERBOX'',
    HDTV varchar2(1) PATH ''HDTV'',
    DOLBY nvarchar2(100) PATH ''DOLBY'',
    DVS varchar2(1) PATH ''DVS'',
    USER_DATA1	nvarchar2(1000) PATH ''USER_DATA[1]'',
    USER_DATA2	nvarchar2(1000) PATH ''USER_DATA[2]'',
    USER_DATA3	nvarchar2(1000) PATH ''USER_DATA[3]'',
    USER_DATA4	nvarchar2(1000) PATH ''USER_DATA[4]'',
    USER_DATA5	nvarchar2(1000) PATH ''USER_DATA[5]'',
    USER_DATA6	nvarchar2(1000) PATH ''USER_DATA[6]'',
    USER_DATA7	nvarchar2(1000) PATH ''USER_DATA[7]'',
    USER_DATA8	nvarchar2(1000) PATH ''USER_DATA[8]'',
    USER_DATA9	nvarchar2(1000) PATH ''USER_DATA[9]'',
    USER_DATA10	nvarchar2(1000) PATH ''USER_DATA[10]'',
    USER_DATA11	nvarchar2(1000) PATH ''USER_DATA[11]'',
    SCHEDULE_GENERIC xmltype PATH ''SCHEDULE_GENERIC'',
    REQUEST_RECORD varchar2(1) PATH ''REQUEST_RECORD'') xmlcolumns';
    CREATEOBJECT(vsql,'create view stg_schedule_view');

    vsql := 'create or replace view '||vschema||'.stg_STATION_view as
    SELECT xmlcolumns.*
    FROM (select sys_nc_rowinfo$ as res  from guide_tab ) rv,
    XMLTABLE(''/GuideData/SlicerGuideData/STATION/row''
    PASSING rv.res
    COLUMNS
    STATION_ID 	nvarchar2(100) PATH ''STATION_NUM'',
    STATION_TIME_ZONE nvarchar2(100) PATH ''STATION_TIME_ZONE'',
    STATION_NAME nvarchar2(100) PATH ''STATION_NAME'',
    STATION_CALL_SIGN nvarchar2(100) PATH ''STATION_CALL_SIGN'',
    STATION_AFFIL nvarchar2(100) PATH ''STATION_AFFIL'',
    STATION_CITY nvarchar2(100) PATH ''STATION_CITY'',
    STATION_STATE nvarchar2(100) PATH ''STATION_STATE'',
    STATION_ZIP_CODE nvarchar2(100) PATH ''STATION_ZIP_CODE'',
    STATION_COUNTRY nvarchar2(100) PATH ''STATION_COUNTRY'',
    DMA_NAME nvarchar2(100) PATH ''DMA_NAME'',
    DMA_NUM number(12) PATH ''DMA_NUM'',
    FCC_CHANNEL_NUM number(12) PATH ''FCC_CHANNEL_NUM'',
    USER_DATA1	nvarchar2(1000) PATH ''USER_DATA[1]'',
    USER_DATA2	nvarchar2(1000) PATH ''USER_DATA[2]'',
    USER_DATA3	nvarchar2(1000) PATH ''USER_DATA[3]'',
    USER_DATA4	nvarchar2(1000) PATH ''USER_DATA[4]'',
    USER_DATA5	nvarchar2(1000) PATH ''USER_DATA[5]'',
    USER_DATA6	nvarchar2(1000) PATH ''USER_DATA[6]'',
    USER_DATA7	nvarchar2(1000) PATH ''USER_DATA[7]'',
    USER_DATA8	nvarchar2(1000) PATH ''USER_DATA[8]'',
    USER_DATA9	nvarchar2(1000) PATH ''USER_DATA[9]'',
    USER_DATA10	nvarchar2(1000) PATH ''USER_DATA[10]'',
	STATION_GENERIC xmltype PATH ''STATION_GENERIC''
    ) xmlcolumns';
    CREATEOBJECT(vsql,'create view stg_station_view');
    
    /********************** EPGMANAGER Stage Tables *************************/
    
    /*
     * These tables are used for each provider to pull from the xml files views into actual tables
     * these tables are flattened out with regard to xml fields.
     * 
     * One for each main table (5) plus additional tosupport complex types of program and schedule
     */
    
   /* stage tables */
    
    vsql := 'CREATE TABLE '||vschema||'."STG_HEADEND"
    (	"FILENAME" VARCHAR2(4000 BYTE),
    "HEADEND_ID" NVARCHAR2(100),
    "COMMUNITY_NAME" NVARCHAR2(200),
    "COUNTY_NAME" NVARCHAR2(200),
    "COUNTY_SIZE" NVARCHAR2(1),
    "ST_COUNTY_CODE" NVARCHAR2(100),
    "STATE" NVARCHAR2(64),
    "ZIPCODE" NVARCHAR2(100),
    "DMA_CODE" NVARCHAR2(100),
    "DMA_NAME" NVARCHAR2(300),
    "MSO_CODE" NUMBER,
    "DMA_RANK" NVARCHAR2(10),
    "HEADEND_NAME" NVARCHAR2(1000),
    "HEADEND_LOCATION" NVARCHAR2(100),
    "MSO_NAME" NVARCHAR2(200),
    "TIME_ZONE" NVARCHAR2(100),
	CONSTRAINT PK_STG_HEADEND PRIMARY KEY ( HEADEND_ID )
    ) nologging';
    CREATEOBJECT(vsql,'create table stg_headend');

    vsql := 'BEGIN
                DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_HEADEND'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
             END;';
    CREATEOBJECT(vsql,'create table err$_stg_headend');
    
        /* stg_headend generic */
    vsql := 'CREATE TABLE '||vschema||'."STG_HEADEND_GENERIC"
       (	FILENAME VARCHAR2(4000 BYTE),
         HEADEND_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_HEADEND_GENERIC_PK   PRIMARY KEY (HEADEND_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_HEADEND_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_HEADEND_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_HEADEND_GENERIC','Y');    
 
   
    vsql := 'CREATE TABLE '||vschema||'."STG_STATION"
    (	"FILENAME" VARCHAR2(4000 BYTE),
    "STATION_ID" NVARCHAR2(100),
    "TIME_ZONE" NVARCHAR2(100),
    "STATION_NAME" NVARCHAR2(100),
    "CALL_SIGN" NVARCHAR2(100),
    "AFFILIATION" NVARCHAR2(100),
    "CITY" NVARCHAR2(100),
    "STATE" NVARCHAR2(100),
    "ZIPCODE" NVARCHAR2(100),
    "COUNTRY" NVARCHAR2(100),
    "DMA_NAME" NVARCHAR2(100),
    "DMA_NUMBER" Number(12),
    "FCC_CHANNEL_NUM" Number(12),
    "USER_DATA1" NVARCHAR2(1000),
    "USER_DATA2" NVARCHAR2(1000),
    "USER_DATA3" NVARCHAR2(1000),
    "USER_DATA4" NVARCHAR2(1000),
    "USER_DATA5" NVARCHAR2(1000),
    "USER_DATA6" NVARCHAR2(1000),
    "USER_DATA7" NVARCHAR2(1000),
    "USER_DATA8" NVARCHAR2(1000),
    "USER_DATA9" NVARCHAR2(1000),
    "USER_DATA10" NVARCHAR2(1000),
    CONSTRAINT PK_STG_STATION PRIMARY KEY ( STATION_ID )
     ) nologging';
    CREATEOBJECT(vsql,'create table stg_station');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_STATION'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_stg_station');

    /* stg_station generic */
    vsql := 'CREATE TABLE '||vschema||'."STG_STATION_GENERIC"
       (	FILENAME VARCHAR2(4000 BYTE),
         STATION_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_STATION_GENERIC_PK   PRIMARY KEY (STATION_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_STATION_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_STATION_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_STATION_GENERIC','Y');    

    
    vsql := 'CREATE TABLE '||vschema||'."STG_LINEUP"
    (	"FILENAME" VARCHAR2(4000 BYTE),
    "LINEUP_ID" nvarchar2(100),
    "STATION_ID" nvarchar2(100),
    "HEADEND_ID" nvarchar2(100),
    "DEVICE" NVARCHAR2(2),
    "TMS_CHANNEL" NVARCHAR2(20),
    "SERVICE_TIER" NVARCHAR2(1),
    "EFFECTIVE_DATE" TIMESTAMP (0)	WITH TIME ZONE,
    "EXPIRATION_DATE" TIMESTAMP (0)	WITH TIME ZONE,
     CONSTRAINT PK_STG_LINEUP PRIMARY KEY ( LINEUP_ID, HEADEND_ID, STATION_ID ),
     CONSTRAINT FK_STG_LINEUP_STATION FOREIGN KEY (STATION_ID) REFERENCES '||vschema||'.STG_STATION(STATION_ID),
     CONSTRAINT FK_STG_HEADEND_LINEUP FOREIGN KEY ( HEADEND_ID) REFERENCES '||vschema||'.STG_HEADEND(HEADEND_ID)
    ) nologging';
    CREATEOBJECT(vsql,'create table stg_lineup');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_LINEUP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_stg_lineup');

    /* stg_lineup generic */
    vsql := 'CREATE TABLE '||vschema||'."STG_LINEUP_GENERIC"
       (	FILENAME VARCHAR2(4000 BYTE),
         LINEUP_ID NVARCHAR2(100) NOT NULL, 
		 HEADEND_ID NVARCHAR2(100) NOT NULL,
		 STATION_ID NVARCHAR2(100) NOT NULL,
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_LINEUP_GENERIC_PK   PRIMARY KEY (LINEUP_ID, HEADEND_ID, STATION_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_LINEUP_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_LINEUP_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_LINEUP_GENERIC','Y'); 
    
    vsql := 'CREATE TABLE '||vschema||'."STG_PROGRAM"
    (	"FILENAME" VARCHAR2(4000 BYTE),
      "PROGRAM_ID" NVARCHAR2(100),
      "LANGUAGE" NVARCHAR2(100),
      "TITLE" NVARCHAR2(1000),
      "REDUCED_TITLE1" NVARCHAR2(1000),
      "REDUCED_TITLE2" NVARCHAR2(1000),
      "REDUCED_TITLE3" NVARCHAR2(1000),
      "REDUCED_TITLE4" NVARCHAR2(1000),
      "ALT_TITLE" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION1" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION2" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION3" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION4" NVARCHAR2(1000),
      "ADVISORY_DESC1" NVARCHAR2(100),
      "ADVISORY_DESC2" NVARCHAR2(100),
      "ADVISORY_DESC3" NVARCHAR2(100),
      "ADVISORY_DESC4" NVARCHAR2(100),
      "ADVISORY_DESC5" NVARCHAR2(100),
      "ADVISORY_DESC6" NVARCHAR2(100),
      "GENRE_DESC1" NVARCHAR2(100),
      "GENRE_DESC2" NVARCHAR2(100),
      "GENRE_DESC3" NVARCHAR2(100),
      "GENRE_DESC4" NVARCHAR2(100),
      "GENRE_DESC5" NVARCHAR2(100),
      "GENRE_DESC6" NVARCHAR2(100),
      "DESCRIPTION1" NVARCHAR2(2000),
      "DESCRIPTION2" NVARCHAR2(2000),
      "YEAR_PROGRAM" NVARCHAR2(100),
      "MPAA_RATING" NVARCHAR2(1000),
      "STAR_RATING" NVARCHAR2(1000),
      "RUN_TIME" VARCHAR2(100),
      "COLOR_CODE" NVARCHAR2(100),
      "PROGRAM_LANGUAGE" NVARCHAR2(100),
      "COUNTRY_OF_ORIGIN" NVARCHAR2(100),
      "MADE_FOR_TV" NVARCHAR2(1),
      "SOURCE_TYPE" NVARCHAR2(100),
      "SHOW_TYPE" NVARCHAR2(200),
      "HOLIDAY" NVARCHAR2(100),
      "SYNDICATE_EPISODE_NUM" NVARCHAR2(100),
      "ALT_SYNDICATE_EPI_NUM" NVARCHAR2(100),
      "EPISODE_TITLE" NVARCHAR2(1000),
      "NET_SYN_SOURCE" NVARCHAR2(100),
      "NET_SYN_TYPE" NVARCHAR2(100),
      "ORG_STUDIO" NVARCHAR2(100),
      "GAME_DATETIME" TIMESTAMP (0) WITH TIME ZONE,
      "GAME_TIMEZONE" NVARCHAR2(100),
      "ORG_AIR_DATE" TIMESTAMP (0) WITH TIME ZONE,
      "UNIQUE_ID" NVARCHAR2(100),
      "USER_DATA1" NVARCHAR2(1000),
      "USER_DATA2" NVARCHAR2(1000),
      "USER_DATA3" NVARCHAR2(1000),
      "USER_DATA4" NVARCHAR2(1000),
      "USER_DATA5" NVARCHAR2(1000),
      "USER_DATA6" NVARCHAR2(1000),
      "USER_DATA7" NVARCHAR2(1000),
      "USER_DATA8" NVARCHAR2(1000),
      "USER_DATA9" NVARCHAR2(1000),
      "USER_DATA10" NVARCHAR2(1000),
      "USER_DATA11" NVARCHAR2(1000),
      "USER_DATA12" NVARCHAR2(1000),
      "USER_DATA13" NVARCHAR2(1000),
     CONSTRAINT PK_STG_PROGRAM PRIMARY KEY ( PROGRAM_ID )
    ) nologging';
    CREATEOBJECT(vsql,'create table STG_PROGRAM');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM'', ERR_LOG_TABLE_OWNER => '''||vschema||''',skip_unsupported=>TRUE);
    END;';
    CREATEOBJECT(vsql,'create table err$_STG_PROGRAM');
    
    /* support tables for stg_program */
    
    /* stg_program_credits */    
 



   

  


    /* stg_Program Rating */
    vsql := 'CREATE TABLE '||vschema||'."STG_PROGRAM_RATING"
       (	"FILENAME" VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         AUTHORITY NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_PROGRAM_RATING   PRIMARY KEY (PROGRAM_ID, AUTHORITY) 
        ) nologging';
    CREATEOBJECT(vsql,'create table STG_PROGRAM_RATING');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM_RATING'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_stg_program_rating','Y');
     
    /* stg_program generic */
    vsql := 'CREATE TABLE '||vschema||'."STG_PROGRAM_GENERIC"
       (	FILENAME VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_PROGRAM_GENERIC_PK   PRIMARY KEY (PROGRAM_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_PROGRAM_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_PROGRAM_GENERIC','Y');    
    
   /* stg_program audio component */
    vsql := 'CREATE TABLE '||vschema||'."STG_PROGRAM_AUDIO_COMP"
       (	FILENAME VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
		 TLA NVARCHAR2(600) NOT NULL,
         TYPE NVARCHAR2(200) NULL, 
         LANGUAGE NVARCHAR2(200) NULL, 
		 ATTRIBUTE_TYPE NVARCHAR2(200) NULL,
         VALUE NVARCHAR2(200) NULL, 
         CONSTRAINT STG_PROGRAM_AUDIO_COMP_PK   PRIMARY KEY (PROGRAM_ID, TLA) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_PROGRAM_AUDIO_COMP');
    
    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM_AUDIO_COMP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_PROGRAM_AUDIO_COMP','Y');    
   
   /* stg_program subtitle component */
    vsql := 'CREATE TABLE '||vschema||'."STG_PROGRAM_SUBTITLE_COMP"
       (	FILENAME VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
		 TLA NVARCHAR2(600) NOT NULL,
         TYPE NVARCHAR2(200) NULL, 
         LANGUAGE NVARCHAR2(200) NULL, 
		 ATTRIBUTE_TYPE NVARCHAR2(200) NULL,
         VALUE NVARCHAR2(200) NULL, 
         CONSTRAINT STG_PROGRAM_SUBTITLE_COMP_PK   PRIMARY KEY (PROGRAM_ID,TLA) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_PROGRAM_SUBTITLE_COMP');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM_SUBTITLE_COMP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_PROGRAM_SUBTITLE_COMP','Y');        
     
/*    vsql := 'CREATE INDEX "IDX_STGPRG_PRGID" ON STG_PROGRAM(DATABASE_KEY)';
    CREATEOBJECT(vsql,'Create Index IDX_STGPRG_PRGID','Y');
*/
     vsql := 'CREATE TABLE '||vschema||'."STG_SCHEDULE"
       (	"FILENAME" VARCHAR2(4000 BYTE),
      "STATION_ID" NVARCHAR2 (100),
      "PROGRAM_ID" NVARCHAR2(100),
      "SCHEDULE_LANGUAGE" NVARCHAR2(100),
      "START_TIME" TIMESTAMP(0) with time zone,
      "END_TIME" timestamp(0) with time zone as (start_time + numtodsinterval(duration,''SECOND'')),
      "DURATION" NUMBER,
      "PART_NUMBER" NUMBER,
	  "SERIES_ID" NVARCHAR2(100),
      "NO_OF_PARTS" NUMBER,
      "CC" NVARCHAR2(1),
      "STEREO" NVARCHAR2(1),
      "REPEAT" NVARCHAR2(1),
      "LIVE_TAPE_DELAY" NVARCHAR2(100),
      "SUBTITLED" NVARCHAR2(1),
      "PREMIERE_FINALE" NVARCHAR2(100),
      "JOINED_IN_PROGRESS" NVARCHAR2(1),
      "CABLE_IN_CLASSROOM" NVARCHAR2(1),
      "TV_RATING" NVARCHAR2(1000),
      "SAP" NVARCHAR2(1),
      "BLACKOUT" NVARCHAR2(1),
      "SEX_RATING" NVARCHAR2(1),
      "VIOLENCE_RATING" NVARCHAR2(1),
      "LANGUAGE_RATING" NVARCHAR2(1),
      "DIALOG_RATING" NVARCHAR2(1),
      "FV_RATING" NVARCHAR2(1),
      "ENHANCED" NVARCHAR2(1),
      "THREE_D" NVARCHAR2(1),
      "LETTERBOX" NVARCHAR2(1),
      "HD_TV" NVARCHAR2(1),
      "DOLBY" NVARCHAR2(100),
      "DVS" NVARCHAR2(1),
      "USER_DATA1" NVARCHAR2(1000),
      "USER_DATA2" NVARCHAR2(1000),
      "USER_DATA3" NVARCHAR2(1000),
      "USER_DATA4" NVARCHAR2(1000),
      "USER_DATA5" NVARCHAR2(1000),
      "USER_DATA6" NVARCHAR2(1000),
      "USER_DATA7" NVARCHAR2(1000),
      "USER_DATA8" NVARCHAR2(1000),
      "USER_DATA9" NVARCHAR2(1000),
      "USER_DATA10" NVARCHAR2(1000),
      "USER_DATA11" NVARCHAR2(1000),
      "REQUEST_RECORD" VARCHAR2(1),
     CONSTRAINT FK_STG_STATION_SCHEDULE FOREIGN KEY ( STATION_ID) REFERENCES '||vschema||'.STG_STATION ( STATION_ID),
     CONSTRAINT FK_STG_SCHEDULE_PROGRAM FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.STG_PROGRAM ( PROGRAM_ID)
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_SCHEDULE');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_SCHEDULE'', ERR_LOG_TABLE_OWNER => '''||vschema||''',skip_unsupported=>TRUE);
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_SCHEDULE');
/*
    vsql := 'CREATE INDEX "IDX_STGSCH_PK" ON STG_SCHEDULE(STATION_NUM,DATABASE_KEY,AIR_DATE)';
    CREATEOBJECT(vsql,'Create Index IDX_STGSCH_PK','Y');
*/
    /* stg schedule support tables */
    
    /* stg_schedule_event_id */
    vsql := 'CREATE TABLE '||vschema||'.STG_SCHEDULE_EVENT_ID( "FILENAME" VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100), 
         IDENTIFIER_TYPE NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(200) NULL, 
         CONSTRAINT STG_SCHEDULE_EVENT_IDS_PK   PRIMARY KEY (PROGRAM_ID, IDENTIFIER_TYPE) 
		) nologging';
    CREATEOBJECT(vsql,'create table STG_SCHEDULE_EVENT_ID');
    
    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_SCHEDULE_EVENT_ID'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_STG_SCHEDULE_EVENT_ID');
 
    /* stg_schedule_series_attribute */
    vsql := 'CREATE TABLE '||vschema||'.STG_SCHEDULE_SERIES_ATTRIB("FILENAME" VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100), 
         ATTRIBUTE_TYPE NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_SCHEDULE_SERIES_ATTRIBS_PK   PRIMARY KEY (PROGRAM_ID, ATTRIBUTE_TYPE) 
		) nologging';
    CREATEOBJECT(vsql,'create table STG_SCHEDULE_SERIES_ATTRIB');
    
    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_SCHEDULE_SERIES_ATTRIB'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_STG_SCHEDULE_SERIES_ATTRIB');
    
    /* stg_schedule generic */
    vsql := 'CREATE TABLE '||vschema||'."STG_SCHEDULE_GENERIC"
       (	FILENAME VARCHAR2(4000 BYTE),
         PROGRAM_ID NVARCHAR2(100), 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT STG_SCHEDULE_GENERIC_PK   PRIMARY KEY (PROGRAM_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table STG_SCHEDULE_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_SCHEDULE_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_STG_SCHEDULE_GENERIC','Y');    
    
 /* end staging tables */
    
      
    /********************** EPGMANAGER Production tables *********************/
     vsql := 'CREATE TABLE '||vschema||'.install_Table( msgdate date default sysdate,install_message varchar2(4000))';
    CREATEOBJECT(vsql,'Create table install_table','Y');
    
    vsql := 'CREATE TABLE '||vschema||'.CONFIG_SETTING
    (
     KEY NVARCHAR2 (1024)  NOT NULL ,
     VALUE NVARCHAR2 (2000) ,
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL,
     CONSTRAINT PK_CONFIG_SETTING PRIMARY KEY ( KEY )
    )';
    CREATEOBJECT(vsql,'create TABLE CONFIG_SETTING');
    
        vsql := 'CREATE TABLE '||vschema||'.ALARM_ACTIVITYLOG
    (
      ID          	NUMBER(19,0) NOT NULL,
      SEVERITY    	NVARCHAR2(64) NOT NULL,
      COMPONENT   	NVARCHAR2(64) NOT NULL,
      DESCRIPTION 	NVARCHAR2(2000) NOT NULL,    
      CREATED_BY  	NVARCHAR2(50) NOT NULL,
      UPDATED_BY  	NVARCHAR2(50) NOT NULL,
      CREATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
      UPDATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
      CONSTRAINT PK_ALARM_ACTIVITYLOG PRIMARY KEY(ID)
      )';
    CREATEOBJECT(vsql,'create table ALARM_ACTIVITYLOG');
    CREATESEQ('ALARM_ACTIVITYLOG');

    vsql := 'CREATE INDEX '||vschema||'.IDX_AAL_COMPONENT ON '||vschema||'.ALARM_ACTIVITYLOG(COMPONENT)';
    CREATEOBJECT(vsql,'create INDEX IDX_AAL_COMPONENT');

    vsql := 'CREATE INDEX '||vschema||'.IDX_AAL_SEVERITY ON '||vschema||'.ALARM_ACTIVITYLOG(SEVERITY)';
    CREATEOBJECT(vsql,'create INDEX IDX_AAL_SEVERITY');
   
    
    /*------------------------- Rules related table creation start -----------------------------------------*/
    
    
     vsql := 'CREATE TABLE '||vschema||'.ENTITY_MAPPING 
       (  
         ID NUMERIC(19,0) NOT NULL, 
         NAME NVARCHAR2(200) NOT NULL, 
         TABLE_NAME VARCHAR2(200) NOT NULL, 
         COLUMN_NAME VARCHAR2(200) NOT NULL, 
         ALT_COLUMN_NAME1 VARCHAR2(200) NULL, 
         ALT_COLUMN_VALUE1 NVARCHAR2(200) NULL, 
         ALT_COLUMN_NAME2 VARCHAR2(200) NULL, 
         ALT_COLUMN_VALUE2 NVARCHAR2(200) NULL, 
         DATA_TYPE VARCHAR2(10) NULL, 
         ALLOW_AS_CRITERIA NUMBER(1,0) DEFAULT 1 NOT NULL,
         ALLOW_AS_PREDICATE NUMBER(1,0)DEFAULT 1 NOT NULL,
         CONSTRAINT Entity_Mapping_PK   PRIMARY KEY (ID) 
       )'; 
       CREATEOBJECT(vsql,'create table Entity_Mapping'); 
       CREATESEQ('Entity_Mapping'); 
    
    
    
    vsql := 'CREATE TABLE '||vschema||'.PREDICATE
(
ID NUMBER(19,0) NOT NULL,
PREDICATE_TYPE NVARCHAR2(64) NOT NULL,
ERROR_LEVEL NVARCHAR2(64),
LOGMESSAGE NVARCHAR2(2000),
ENTITY_MAPPING NUMBER(19,0),
VALUE_TYPE NVARCHAR2(64),
LITERAL_VALUE NVARCHAR2(2000),
MAPPED_VALUE NUMBER(19,0),
CREATED_BY  	NVARCHAR2(50) NOT NULL,
UPDATED_BY  	NVARCHAR2(50) NOT NULL,
CREATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
UPDATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
CONSTRAINT PK_PREDICATE_ID PRIMARY KEY(ID),
CONSTRAINT FK_PREDICATE_ENTITY_MAPPING FOREIGN KEY(ENTITY_MAPPING) REFERENCES EPGMANAGER.ENTITY_MAPPING(ID) ON DELETE CASCADE,
CONSTRAINT FK_PREDICATE_MAPPED_VALUE FOREIGN KEY(MAPPED_VALUE) REFERENCES EPGMANAGER.ENTITY_MAPPING(ID) ON DELETE CASCADE
)';
CREATEOBJECT(vsql,'create table PREDICATE');
CREATESEQ('PREDICATE');

vsql := 'CREATE TABLE '||vschema||'.RULE
(
ID NUMBER(19,0) NOT NULL,
NAME NVARCHAR2(200) NOT NULL,
RULE_TYPE NVARCHAR2(64) NOT NULL,
PREDICATE_ID NUMBER(19,0) NOT NULL,
ORDERING	NUMBER(19,0) NOT NULL,
CREATED_BY  	NVARCHAR2(50) NOT NULL,
UPDATED_BY  	NVARCHAR2(50) NOT NULL,
CREATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
UPDATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
CONSTRAINT PK_RULE_ID PRIMARY KEY(ID),
CONSTRAINT U_RULE_NAME UNIQUE (NAME),
CONSTRAINT FK_RULE_PREDICATE FOREIGN KEY(PREDICATE_ID) REFERENCES EPGMANAGER.PREDICATE(ID) ON DELETE CASCADE
)';
CREATEOBJECT(vsql,'create table RULE');
CREATESEQ('RULE');

vsql := 'CREATE TABLE '||vschema||'.CLAUSE
(
ID NUMBER(19,0) NOT NULL,
ENTITY_MAPPING NUMBER(19,0) NOT NULL,
COMPARE_OPERATOR NVARCHAR2(64) NOT NULL,
JOIN_PREFIX NVARCHAR2(64) NOT NULL,
CONDITION_VALUE_TYPE NVARCHAR2(64) NOT NULL,
LITERAL_VALUE NVARCHAR2(2000),
MAPPED_VALUE NUMBER(19,0),
LAYER_NUMBER NUMBER(4,0) NOT NULL,
ORDERING NUMBER(4,0) NOT NULL,
GROUP_NUMBER NUMBER(4,0) NOT NULL,
NEGATE NUMBER(1,0) NOT NULL,
IGNORECASE NUMBER(1,0) NULL,
CREATED_BY  	NVARCHAR2(50) NOT NULL,
UPDATED_BY  	NVARCHAR2(50) NOT NULL,
CREATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
UPDATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
CONSTRAINT PK_CLAUSE_ID PRIMARY KEY(ID),
CONSTRAINT FK_CLAUSE_ENTITY_MAPPING FOREIGN KEY(ENTITY_MAPPING) REFERENCES EPGMANAGER.ENTITY_MAPPING(ID) ON DELETE CASCADE,
CONSTRAINT FK_CLAUSE_MAPPED_VALUE FOREIGN KEY(MAPPED_VALUE) REFERENCES EPGMANAGER.ENTITY_MAPPING(ID) ON DELETE CASCADE
)';
CREATEOBJECT(vsql,'create table CLAUSE');
CREATESEQ('CLAUSE');

vsql := 'CREATE TABLE '||vschema||'.RULE_CLAUSE_MAPPING
(
RULE_ID NUMBER(19,0),
CLAUSE_ID NUMBER(19,0) NOT NULL,
CONSTRAINT U_CLAUSE_ID UNIQUE (CLAUSE_ID),
CONSTRAINT FK_RULE_CLAUSE_RULE FOREIGN KEY(RULE_ID) REFERENCES EPGMANAGER.RULE(ID) ON DELETE CASCADE,
CONSTRAINT FK_RULE_CLAUSE_CLAUSE FOREIGN KEY(CLAUSE_ID) REFERENCES EPGMANAGER.CLAUSE(ID) ON DELETE CASCADE
)';
 CREATEOBJECT(vsql,'create table RULE_CLAUSE_MAPPING');
 
vsql := 'CREATE TABLE '||vschema||'.NORM_RULE_GROUP
(
ID NUMBER(19,0) NOT NULL,
NAME NVARCHAR2(200) NOT NULL,
ENABLED NUMBER(1,0) NOT NULL,
GLOBAL_FOR_PROVIDERS NUMBER(1,0) NOT NULL,
GLOBAL_FOR_PUBLISHERS NUMBER(1,0) NOT NULL,
ORDERING	NUMBER(19,0) NOT NULL,
CREATED_BY  	NVARCHAR2(50) NOT NULL,
UPDATED_BY  	NVARCHAR2(50) NOT NULL,
CREATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
UPDATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
CONSTRAINT PK_NORM_RULE_GROUP_ID PRIMARY KEY(ID),
CONSTRAINT U_NORM_RULE_GROUP_NAME UNIQUE (NAME)
)';
 CREATEOBJECT(vsql,'create table NORM_RULE_GROUP');
 CREATESEQ('NORM_RULE_GROUP');


vsql := 'CREATE TABLE '||vschema||'.NORM_GROUP_RULE_MAPPING
(
NORM_RULE_GROUP_ID NUMBER(19,0),
RULE_ID NUMBER(19,0) NOT NULL,
CONSTRAINT U_NGRM_RULE_ID UNIQUE (RULE_ID),
CONSTRAINT FK_NGRM_GROUP FOREIGN KEY(NORM_RULE_GROUP_ID) REFERENCES EPGMANAGER.NORM_RULE_GROUP(ID) ON DELETE CASCADE,
CONSTRAINT FK_NGRM_RULE FOREIGN KEY(RULE_ID) REFERENCES EPGMANAGER.RULE(ID) ON DELETE CASCADE
)';
 CREATEOBJECT(vsql,'create table NORM_GROUP_RULE_MAPPING');

 
vsql := 'CREATE TABLE '||vschema||'.VALID_RULE_GROUP
(
ID NUMBER(19,0) NOT NULL,
NAME NVARCHAR2(200) NOT NULL,
ENABLED NUMBER(1,0) NOT NULL,
GLOBAL NUMBER(1,0) NOT NULL,
CREATED_BY  	NVARCHAR2(50) NOT NULL,
UPDATED_BY  	NVARCHAR2(50) NOT NULL,
CREATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
UPDATE_DATE 	TIMESTAMP(0) WITH TIME ZONE NOT NULL,
CONSTRAINT PK_VALID_RULE_GROUP_ID PRIMARY KEY(ID),
CONSTRAINT U_VALID_RULE_GROUP_NAME UNIQUE (NAME)
)';
CREATEOBJECT(vsql,'create table VALID_RULE_GROUP');
 CREATESEQ('VALID_RULE_GROUP');

vsql := 'CREATE TABLE '||vschema||'.VALID_GROUP_RULE_MAPPING
(
VALID_RULE_GROUP_ID NUMBER(19,0),
RULE_ID NUMBER(19,0) NOT NULL,
CONSTRAINT U_VGRM_RULE_ID UNIQUE (RULE_ID),
CONSTRAINT FK_VGRM_GROUP FOREIGN KEY(VALID_RULE_GROUP_ID) REFERENCES EPGMANAGER.VALID_RULE_GROUP(ID) ON DELETE CASCADE,
CONSTRAINT FK_VGRM_RULE FOREIGN KEY(RULE_ID) REFERENCES EPGMANAGER.RULE(ID) ON DELETE CASCADE
)';
CREATEOBJECT(vsql,'create table VALID_GROUP_RULE_MAPPING');

    /*------------------------- Rules related table creation end -------------------------------------------*/

    /* Epg_Format */
    vsql:= 'CREATE TABLE '||vschema||'.EPG_FORMAT(ID NUMBER(19, 0) NOT NULL, 
	  NAME VARCHAR2(50 CHAR) NOT NULL, 
	  JAR_NAME VARCHAR2(255 CHAR),
  	  TYPE VARCHAR2(50 CHAR) NOT NULL,
	  INFO VARCHAR2(255 CHAR),
	  VERSION VARCHAR2(255 CHAR),
	  CREATED_BY NVARCHAR2 (50)  NOT NULL ,
	  UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
	  CREATE_DATE TIMESTAMP  NOT NULL ,
	  UPDATE_DATE TIMESTAMP  NOT NULL, 
	  CONSTRAINT PK_EPG_FORMAT_ID PRIMARY KEY(ID))'; 
	CREATEOBJECT(vsql,'Create Table EPG_FORMAT');  
	CREATESEQ('EPG_FORMAT');
	
	 vsql := 'INSERT INTO '||vschema||'.EPG_FORMAT (ID, NAME, JAR_NAME, TYPE, INFO, VERSION, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE) VALUES 
	 ('||vschema||'.EPG_FORMAT_SEQ.NEXTVAL, ''ETV'', ''dummy.jar'',''Provider'', ''default'', ''1'', ''system'', ''system'', current_timestamp, current_timestamp)';
     CREATEOBJECT(vsql,'INSERT INTO EPG_FORMAT');
     
     vsql := 'INSERT INTO '||vschema||'.EPG_FORMAT (ID, NAME, JAR_NAME, TYPE, INFO, VERSION, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE) VALUES 
	 ('||vschema||'.EPG_FORMAT_SEQ.NEXTVAL, ''ETV'', ''dummy.jar'',''Publisher'' , ''default'', ''1'', ''system'', ''system'', current_timestamp, current_timestamp)';
     CREATEOBJECT(vsql,'INSERT INTO EPG_FORMAT');
     
     /* Plugin_Properties */
     vsql:= 'CREATE TABLE '||vschema||'.PLUGIN_PROPERTY(ID NUMBER(19, 0) NOT NULL, 
	  FORMAT_ID NUMBER(19,0) NOT NULL,
	  PROPERTY_NAME VARCHAR2(2000 CHAR) NOT NULL,
	  PROPERTY_DEFAULT_VALUE VARCHAR2(2000 CHAR),
	  CREATED_BY NVARCHAR2 (50) NOT NULL ,
	  UPDATED_BY NVARCHAR2 (50) NOT NULL ,
	  CREATE_DATE TIMESTAMP NOT NULL ,
	  UPDATE_DATE TIMESTAMP NOT NULL, 
	  CONSTRAINT PK_PLUGIN_PROPERTY_ID PRIMARY KEY(ID),
	  CONSTRAINT FK_PLUGIN_PROPERTY_EPG_FORMAT FOREIGN KEY(FORMAT_ID) REFERENCES EPGMANAGER.EPG_FORMAT(ID) ON DELETE CASCADE )'; 
	CREATEOBJECT(vsql,'Create Table PLUGIN_PROPERTY');  
	CREATESEQ('PLUGIN_PROPERTY');
	
	/* Endoint_Details */
    vsql:= 'CREATE TABLE '||vschema||'.ENDPOINT_DETAILS(ID NUMBER(19, 0) NOT NULL, 
	  NAME VARCHAR2(50 CHAR) NOT NULL, 
	  ACTIVE VARCHAR2(30 CHAR) NOT NULL, 
  	  SUPPORTS VARCHAR2(30 CHAR) NOT NULL,
	  CREATED_BY NVARCHAR2 (50)  NOT NULL ,
	  UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
	  CREATE_DATE TIMESTAMP  NOT NULL ,
	  UPDATE_DATE TIMESTAMP  NOT NULL,
	  CONSTRAINT PK_TRANSFER_LOOKUP_ID PRIMARY KEY(ID),
	  CONSTRAINT U_TRANSFER_LOOKUP_NAME UNIQUE(NAME))'; 
	CREATEOBJECT(vsql,'Create Table ENDPOINT_DETAILS');		
	
	
	/* Prepopulate endpoint_details */
 vsql := 'INSERT ALL
		INTO '||vschema||'.ENDPOINT_DETAILS (ID, NAME, ACTIVE, SUPPORTS, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE) values (1,''ftp'',''Retrieve'', ''SupportsBoth'', ''SYSTEM'', ''SYSTEM'', current_timestamp,current_timestamp)
		INTO '||vschema||'.ENDPOINT_DETAILS (ID, NAME, ACTIVE, SUPPORTS, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE) values (2,''file'',''Retrieve'', ''SupportsBoth'', ''SYSTEM'', ''SYSTEM'', current_timestamp,current_timestamp)
     select * from dual';
 CREATEOBJECT(vsql,'PreLoad ENDPOINT_DETAILS');
	
	
   /* Endpoint*/
    vsql:= 'CREATE TABLE '||vschema||'.ENDPOINT(ID NUMBER(19, 0) NOT NULL,
	  ENDPOINT_TYPE_ID NUMBER(19, 0) NOT NULL,
	  SERVER_IP VARCHAR2(255 CHAR),
	  PORT NUMBER(7),
	  FOLDER VARCHAR2(255 CHAR),
	  FILE_NAME VARCHAR2(255 CHAR),
	  USER_NAME VARCHAR2(20 CHAR),
	  PASSWORD VARCHAR2(255),
	  DONE_FILE VARCHAR2(255 CHAR),	
	  IS_SECURE NUMBER(1) DEFAULT 0,
	  CREATED_BY NVARCHAR2 (50)  NOT NULL ,
	  UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
	  CREATE_DATE TIMESTAMP  NOT NULL ,
	  UPDATE_DATE TIMESTAMP  NOT NULL,
	  CONSTRAINT PK_ENDPOINT_ID PRIMARY KEY(ID),
	  CONSTRAINT FK_ENDPOINT_TYPE_ID FOREIGN KEY(ENDPOINT_TYPE_ID) REFERENCES EPGMANAGER.ENDPOINT_DETAILS(ID) ON DELETE CASCADE,
      CONSTRAINT ENDPOINT_UK UNIQUE (SERVER_IP, PORT, FOLDER, FILE_NAME, USER_NAME))';
      CREATEOBJECT(vsql,'Create Table ENDPOINT');
  	  CREATESEQ('ENDPOINT');
    
    /* Provider */
    vsql := 'CREATE TABLE '||vschema||'.PROVIDER(ID NUMBER(19, 0) NOT NULL,
     NAME VARCHAR2(50 CHAR) NOT NULL,
     CONTACT VARCHAR2(50 CHAR),
     EMAIL VARCHAR2(100 CHAR),
     FORMAT_TYPE NUMBER(19,0) NOT NULL,
     RETRIEVAL_TYPE NUMBER(19,0) NOT NULL,     
     IS_ACTIVE NUMBER(1,0) NOT NULL,
     FREQUENCY NUMBER(20,0) NOT NULL,
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL,  
     NORM_GROUP_ID  NUMBER(19,0),
     VALID_GROUP_ID NUMBER(19,0),   
     CONSTRAINT U_PROVIDER_NAME UNIQUE (NAME), 
     CONSTRAINT PK_PROVIDER_ID PRIMARY KEY (ID),
     CONSTRAINT FK_PROVIDER_FORMAT_TYPE FOREIGN KEY(FORMAT_TYPE) REFERENCES EPGMANAGER.EPG_FORMAT(ID) ON DELETE CASCADE,
     CONSTRAINT FK_PROVIDER_RETRIEVAL_TYPE FOREIGN KEY(RETRIEVAL_TYPE) REFERENCES EPGMANAGER.ENDPOINT(ID) ON DELETE CASCADE,
	 CONSTRAINT FK_PROVIDER_NORM_GROUP_TYPE FOREIGN KEY (NORM_GROUP_ID) REFERENCES EPGMANAGER.NORM_RULE_GROUP (ID),
     CONSTRAINT FK_PROVIDER_VALID_RULE_GROUP FOREIGN KEY (VALID_GROUP_ID) REFERENCES EPGMANAGER.VALID_RULE_GROUP(ID))';
     CREATEOBJECT(vsql,'Create table PROVIDER');
     CREATESEQ('PROVIDER');
    
    /* Compression */
    vsql:= 'CREATE TABLE '||vschema||'.COMPRESSION(ID NUMBER(19, 0) NOT NULL, 
	  TYPE VARCHAR2(50 CHAR) NOT NULL,
	  CREATED_BY NVARCHAR2 (50)  NOT NULL ,
	  UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
	  CREATE_DATE TIMESTAMP  NOT NULL ,
	  UPDATE_DATE TIMESTAMP  NOT NULL,
	  CONSTRAINT PK_COMPRESS_ID PRIMARY KEY(ID))'; 
	CREATEOBJECT(vsql,'Create Table COMPRESSION');	
	
	vsql := 'INSERT INTO '||vschema||'.COMPRESSION (id, type, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE)
      VALUES(1, ''DUMMY'', ''System'', ''System'', current_timestamp, current_timestamp)';
    CREATEOBJECT(vsql,'INSERT INTO COMPRESSION');
	
	/* Publisher */
    vsql:= 'CREATE TABLE '||vschema||'.PUBLISHER(ID NUMBER(19, 0) NOT NULL, 
	  NAME VARCHAR2(50 CHAR) NOT NULL,
      CONTACT VARCHAR2(50 CHAR),
      EMAIL VARCHAR2(100 CHAR),
      FORMAT_TYPE NUMBER(19,0) NOT NULL,
      PUBLISH_TYPE NUMBER(19,0) NOT NULL,
      IS_ACTIVE NUMBER(1,0) NOT NULL,
	  FREQUENCY NUMBER(20,0) NOT NULL,
      DAYS NUMBER(5,0) NOT NULL,
	  COMPRESS_TYPE NUMBER(19,0) NOT NULL,
      CREATED_BY NVARCHAR2 (50)  NOT NULL ,
      UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
      CREATE_DATE TIMESTAMP  NOT NULL ,
      UPDATE_DATE TIMESTAMP  NOT NULL,
      NORM_GROUP_ID NUMBER(19,0),     
	  CONSTRAINT PK_PUBLISHER_ID PRIMARY KEY(ID),
      CONSTRAINT U_PUBLISHER_NAME UNIQUE (NAME), 
      CONSTRAINT FK_PUBLISHER_FORMAT_TYPE FOREIGN KEY(FORMAT_TYPE) REFERENCES EPGMANAGER.EPG_FORMAT(ID) ON DELETE CASCADE,
	  CONSTRAINT FK_PUBLISHER_COMPRESS_TYPE FOREIGN KEY(COMPRESS_TYPE) REFERENCES EPGMANAGER.COMPRESSION(ID) ON DELETE CASCADE,
      CONSTRAINT FK_PUBLISHER_PUBLISH_TYPE FOREIGN KEY(PUBLISH_TYPE) REFERENCES EPGMANAGER.ENDPOINT(ID) ON DELETE CASCADE,
      CONSTRAINT FK_PUBLISHER_NORM_RULE_GROUP FOREIGN KEY(NORM_GROUP_ID) REFERENCES EPGMANAGER.NORM_RULE_GROUP(ID))'; 
	CREATEOBJECT(vsql,'Create Table PUBLISHER');
	CREATESEQ('PUBLISHER');
	
	
	vsql := 'CREATE TABLE '||vschema||'.PUBLISHER_STATIONS
	(
	PUBLISHER_ID NUMBER(19,0),
	STATION NVARCHAR2 (100) NOT NULL,
	CONSTRAINT FK_PUBLISHER_STATION FOREIGN KEY(PUBLISHER_ID) REFERENCES EPGMANAGER.PUBLISHER(ID) ON DELETE CASCADE
	)';
 	CREATEOBJECT(vsql,'create table PUBLISHER_STATIONS');
 	     
     vsql := 'CREATE TABLE '||vschema||'.PROVIDER_PLUGIN_VALUE
    (
	 ID NUMBER(19, 0) NOT NULL,
     PROVIDER_ID NUMBER(19,0) NOT NULL,
     PLUGIN_PROPERTY_ID NUMBER(19,0),
	 VALUE NVARCHAR2(2000) NULL,
	 CONSTRAINT PK_PPV_ID PRIMARY KEY (ID),
	 CONSTRAINT FK_PPV_PROVIDER FOREIGN KEY(PROVIDER_ID) REFERENCES EPGMANAGER.PROVIDER(ID) ON DELETE CASCADE,
	 CONSTRAINT FK_PPV_PROPERTY FOREIGN KEY(PLUGIN_PROPERTY_ID) REFERENCES EPGMANAGER.PLUGIN_PROPERTY(ID) ON DELETE CASCADE
	)';
	CREATEOBJECT(vsql,'create table PROVIDER_PLUGIN_VALUE');
	CREATESEQ('PROVIDER_PLUGIN_VALUE');
	
 
	vsql := 'CREATE TABLE '||vschema||'.PUBLISHER_PLUGIN_VALUE
    (
	 ID NUMBER(19, 0) NOT NULL,
     PUBLISHER_ID NUMBER(19,0) NOT NULL,
     PLUGIN_PROPERTY_ID NUMBER(19,0),
	 VALUE NVARCHAR2(2000) NULL,
	 CONSTRAINT PK_PPUV_ID PRIMARY KEY (ID),
	 CONSTRAINT FK_PPUV_PROVIDER FOREIGN KEY(PUBLISHER_ID) REFERENCES EPGMANAGER.PUBLISHER(ID) ON DELETE CASCADE,
	 CONSTRAINT FK_PPUV_PROPERTY FOREIGN KEY(PLUGIN_PROPERTY_ID) REFERENCES EPGMANAGER.PLUGIN_PROPERTY(ID) ON DELETE CASCADE
	)';
	CREATEOBJECT(vsql,'create table PUBLISHER_PLUGIN_VALUE');
	CREATESEQ('PUBLISHER_PLUGIN_VALUE');
	
	/*--- Provider and publisher mapping to rule groups -------------------------------*/
	
	vsql := 'CREATE TABLE '||vschema||'.PROVIDER_VALID_GROUP_MAPPING
    (
     PROVIDER_ID NUMBER(19,0),
     VALID_GROUP_ID NUMBER(19,0),
	 CONSTRAINT FK_PVGM_PROVIDER FOREIGN KEY(PROVIDER_ID) REFERENCES EPGMANAGER.PROVIDER(ID) ON DELETE CASCADE,
	 CONSTRAINT FK_PVGM_VALID_GROUP FOREIGN KEY(VALID_GROUP_ID) REFERENCES EPGMANAGER.VALID_RULE_GROUP(ID) ON DELETE CASCADE
	)';
	CREATEOBJECT(vsql,'create table PROVIDER_VALID_GROUP_MAPPING');
	
	vsql := 'CREATE TABLE '||vschema||'.PROVIDER_NORM_GROUP_MAPPING
    (
     PROVIDER_ID NUMBER(19,0),
     NORM_GROUP_ID NUMBER(19,0),
	 CONSTRAINT FK_PNGM_PROVIDER FOREIGN KEY(PROVIDER_ID) REFERENCES EPGMANAGER.PROVIDER(ID) ON DELETE CASCADE,
	 CONSTRAINT FK_PNGM_VALID_GROUP FOREIGN KEY(NORM_GROUP_ID) REFERENCES EPGMANAGER.NORM_RULE_GROUP(ID) ON DELETE CASCADE
	)';
	CREATEOBJECT(vsql,'create table PROVIDER_NORM_GROUP_MAPPING');
	
	vsql := 'CREATE TABLE '||vschema||'.PUBLISHER_NORM_GROUP_MAPPING
    (
     PUBLISHER_ID NUMBER(19,0),
     NORM_GROUP_ID NUMBER(19,0),
	 CONSTRAINT FK_PUNGM_PROVIDER FOREIGN KEY(PUBLISHER_ID) REFERENCES EPGMANAGER.PUBLISHER(ID) ON DELETE CASCADE,
	 CONSTRAINT FK_PUNGM_VALID_GROUP FOREIGN KEY(NORM_GROUP_ID) REFERENCES EPGMANAGER.NORM_RULE_GROUP(ID) ON DELETE CASCADE
	)';
	CREATEOBJECT(vsql,'create table PUBLISHER_NORM_GROUP_MAPPING');
	
	/*----------------------------------------------------------------------------------*/
	/* Route Status table */
	 vsql := 'CREATE TABLE '||vschema||'.ROUTE_STATUS 
		 (  
		   ROUTE_ID NVARCHAR2(50) NOT NULL, 
		   STATE NVARCHAR2(50) NOT NULL, 
		   LAST_RUN_START_TIME TIMESTAMP NOT NULL, 
		   LAST_RUN_END_TIME TIMESTAMP,    
		   REMARKS NVARCHAR2(2000),
 		   CREATED_BY NVARCHAR2 (50)  NOT NULL,
		   CREATE_DATE TIMESTAMP  NOT NULL,
		   UPDATED_BY NVARCHAR2 (50) NOT NULL,     
		   UPDATE_DATE TIMESTAMP NOT NULL, 
		   CONSTRAINT PK_ROUTE_STATUS_ROUTE_ID PRIMARY KEY (ROUTE_ID) 
		 )'; 
		 CREATEOBJECT(vsql,'create table ROUTE_STATUS');
	/*----------------------------------------------------------------------------------*/
	
    /* Headend, lineup, station */
     vsql := 'CREATE TABLE '||vschema||'.HEADEND
    (
     HEADEND_ID NVARCHAR2(100) NOT NULL ,
     COMMUNITY_NAME NVARCHAR2 (200) ,
     COUNTY_NAME NVARCHAR2 (200) ,
     COUNTY_SIZE NVARCHAR2 (1) ,
     ST_COUNTY_CODE NVARCHAR2 (100) ,
     STATE NVARCHAR2 (64) ,
     ZIPCODE NVARCHAR2 (100),
     DMA_CODE NVARCHAR2 (100) ,
     DMA_NAME NVARCHAR2 (300) ,
     MSO_CODE NUMBER(22,0) ,
     DMA_RANK NVARCHAR2 (10) ,
     HEADEND_NAME NVARCHAR2 (1000) ,
     HEADEND_LOCATION NVARCHAR2 (100) NOT NULL ,
     MSO_NAME NVARCHAR2 (200) NOT NULL ,
     TIME_ZONE NVARCHAR2 (100) NOT NULL ,
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL,
     CONSTRAINT PK_HEADEND PRIMARY KEY ( HEADEND_ID )
    )';
    CREATEOBJECT(vsql,'create TABLE HEADEND');
    
     vsql := 'CREATE TABLE '||vschema||'.HEADEND_GENERIC 
       (  
         HEADEND_ID NVARCHAR2 (100) NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT HEADEND_GENERIC_PK   PRIMARY KEY (HEADEND_ID, ASSET, NAME),
         CONSTRAINT FK_HEADEND_GENERIC_HEADEND_ID FOREIGN KEY ( HEADEND_ID) REFERENCES '||vschema||'.HEADEND ( HEADEND_ID) ON DELETE CASCADE 
       )'; 

       CREATEOBJECT(vsql,'create table HEADEND_GENERIC'); 

    vsql := 'CREATE TABLE '||vschema||'.STATION
    (
     STATION_ID NVARCHAR2 (100)  NOT NULL ,
     STATION_NAME NVARCHAR2 (100) ,
     TIME_ZONE NVARCHAR2 (100) ,
     CALL_SIGN NVARCHAR2 (100) NOT NULL ,
     CITY NVARCHAR2 (100) ,
     STATE NVARCHAR2 (100) ,
     ZIPCODE NVARCHAR2 (100) ,
     COUNTRY NVARCHAR2 (100) ,
     DMA_NAME NVARCHAR2 (100) ,
     DMA_NUMBER NUMBER(12) ,
     FCC_CHANNEL_NUM NUMBER(12) ,
     AFFILIATION NVARCHAR2 (100) NOT NULL ,
     STATION_LOCKED NVARCHAR2(1) DEFAULT ''N'',
     USER_DATA1 NVARCHAR2(1000),
     USER_DATA2 NVARCHAR2(1000),
     USER_DATA3 NVARCHAR2(1000),
     USER_DATA4 NVARCHAR2(1000),
     USER_DATA5 NVARCHAR2(1000),
     USER_DATA6 NVARCHAR2(1000),
     USER_DATA7 NVARCHAR2(1000),
     USER_DATA8 NVARCHAR2(1000),
     USER_DATA9 NVARCHAR2(1000),
     USER_DATA10 NVARCHAR2(1000),
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL,
     CONSTRAINT PK_STATION PRIMARY KEY ( STATION_ID )
    ) ';
    CREATEOBJECT(vsql,'create table STATION');

    vsql := 'CREATE INDEX '||vschema||'.IDX_STATION_CALLSIGN ON '||vschema||'.STATION(CALL_SIGN)';
    CREATEOBJECT(vsql,'create index IDX_STATION_CALLSIGN');

    vsql := 'CREATE INDEX '||vschema||'.IDX_STATION_STATIONNAME ON '||vschema||'.STATION(STATION_NAME)';
    CREATEOBJECT(vsql,'create index IDX_STATION_STATIONNAME');
    
     vsql := 'CREATE TABLE '||vschema||'.STATION_GENERIC 
       (  
         STATION_ID NVARCHAR2 (100) NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT STATION_GENERIC_PK   PRIMARY KEY (STATION_ID, ASSET, NAME),
         CONSTRAINT FK_STATION_GENERIC_STATION_ID FOREIGN KEY ( STATION_ID) REFERENCES '||vschema||'.STATION ( STATION_ID) ON DELETE CASCADE 
       )'; 

       CREATEOBJECT(vsql,'create table STATION_GENERIC'); 


    vsql := 'CREATE TABLE '||vschema||'.LINEUP
    (
     LINEUP_ID NVARCHAR2(100) NOT NULL ,
     HEADEND_ID NVARCHAR2(100) NOT NULL ,
     STATION_ID NVARCHAR2 (100)  NOT NULL,
     DEVICE NVARCHAR2 (2) ,
     TMS_CHANNEL NVARCHAR2 (20) NOT NULL ,
     SERVICE_TIER NVARCHAR2 (1) NOT NULL ,
     EFFECTIVE_DATE TIMESTAMP(0) WITH TIME ZONE ,
     EXPIRATION_DATE TIMESTAMP(0) WITH TIME ZONE ,
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL ,
     CONSTRAINT PK_LINEUP PRIMARY KEY ( LINEUP_ID, HEADEND_ID, STATION_ID ),
     CONSTRAINT FK_LINEUP_STATION FOREIGN KEY (STATION_ID) REFERENCES '||vschema||'.STATION(STATION_ID),
     CONSTRAINT FK_HEADEND_LINEUP FOREIGN KEY ( HEADEND_ID) REFERENCES '||vschema||'.HEADEND ( HEADEND_ID)
    )';
    CREATEOBJECT(vsql,'create TABLE LINEUP');

    vsql := 'CREATE INDEX '||vschema||'.IDX_L_STATIONID_HEADENDID ON '||vschema||'.LINEUP(STATION_ID,HEADEND_ID)';
    CREATEOBJECT(vsql,'create INDEX IDX_L_STATIONID_HEADENDID');

         vsql := 'CREATE TABLE '||vschema||'.LINEUP_GENERIC 
       (  
         LINEUP_ID NVARCHAR2 (100) NOT NULL,
		 HEADEND_ID NVARCHAR2 (100) NOT NULL,
		 STATION_ID NVARCHAR2 (100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT LINEUP_GENERIC_PK   PRIMARY KEY (LINEUP_ID, HEADEND_ID, STATION_ID, ASSET, NAME),
         CONSTRAINT FK_LINEUP_GENERIC_LINEUP_ID FOREIGN KEY ( LINEUP_ID, HEADEND_ID, STATION_ID) REFERENCES '||vschema||'.LINEUP ( LINEUP_ID, HEADEND_ID, STATION_ID) ON DELETE CASCADE 
       )'; 

       CREATEOBJECT(vsql,'create table LINEUP_GENERIC'); 
    
    /* Program */
     vsql := 'CREATE TABLE '||vschema||'.PROGRAM
    (
     PROGRAM_ID NVARCHAR2 (100)  NOT NULL ,
     LANGUAGE NVARCHAR2 (100) NOT NULL ,
     TITLE NVARCHAR2 (1000) NOT NULL ,
     REDUCED_TITLE1 NVARCHAR2 (1000) NOT NULL ,
     REDUCED_TITLE2 NVARCHAR2 (1000) ,
     REDUCED_TITLE3 NVARCHAR2 (1000) ,
     REDUCED_TITLE4 NVARCHAR2 (1000) ,
     ALT_TITLE NVARCHAR2 (1000) ,
     REDUCED_DESCRIPTION1 NVARCHAR2 (1000) ,
     REDUCED_DESCRIPTION2 NVARCHAR2 (1000) ,
     REDUCED_DESCRIPTION3 NVARCHAR2 (1000) ,
     REDUCED_DESCRIPTION4 NVARCHAR2 (1000) ,
     ADVISORY_DESC1 NVARCHAR2 (100) ,
     ADVISORY_DESC2 NVARCHAR2 (100) ,
     ADVISORY_DESC3 NVARCHAR2 (100) ,
     ADVISORY_DESC4 NVARCHAR2 (100) ,
     ADVISORY_DESC5 NVARCHAR2 (100) ,
     ADVISORY_DESC6 NVARCHAR2 (100) ,
     GENRE_DESC1 NVARCHAR2 (100) ,
     GENRE_DESC2 NVARCHAR2 (100) ,
     GENRE_DESC3 NVARCHAR2 (100) ,
     GENRE_DESC4 NVARCHAR2 (100) ,
     GENRE_DESC5 NVARCHAR2 (100) ,
     GENRE_DESC6 NVARCHAR2 (100) ,
     DESCRIPTION1 NVARCHAR2 (2000) NOT NULL ,
     DESCRIPTION2 NVARCHAR2 (2000) ,
     YEAR_PROGRAM NVARCHAR2 (100) ,
     MPAA_RATING NVARCHAR2 (1000) ,
     STAR_RATING NVARCHAR2 (1000) ,
     RUN_TIME VARCHAR2 (100) ,
     COLOR_CODE NVARCHAR2 (100) ,
     PROGRAM_LANGUAGE NVARCHAR2 (100) ,
     COUNTRY_OF_ORIGIN NVARCHAR2 (100) ,
     MADE_FOR_TV NVARCHAR2 (1) ,
     SOURCE_TYPE NVARCHAR2 (100) ,
     SHOW_TYPE NVARCHAR2 (200) ,
     HOLIDAY NVARCHAR2 (100) ,
     SYNDICATE_EPISODE_NUM NVARCHAR2 (100) ,
     ALT_SYNDICATE_EPI_NUM NVARCHAR2 (100) ,
     EPISODE_TITLE NVARCHAR2 (1000) ,
     NET_SYN_SOURCE NVARCHAR2 (100) ,
     NET_SYN_TYPE NVARCHAR2 (100) ,
     ORG_STUDIO NVARCHAR2 (100) NOT NULL ,
     GAME_DATETIME TIMESTAMP(0) WITH TIME ZONE ,
     GAME_TIMEZONE NVARCHAR2 (100) ,
     ORG_AIR_DATE TIMESTAMP(0)  WITH TIME ZONE,
     UNIQUE_ID NVARCHAR2 (100) ,
     USER_DATA1 NVARCHAR2(1000),
     USER_DATA2 NVARCHAR2(1000),
     USER_DATA3 NVARCHAR2(1000),
     USER_DATA4 NVARCHAR2(1000),
     USER_DATA5 NVARCHAR2(1000),
     USER_DATA6 NVARCHAR2(1000),
     USER_DATA7 NVARCHAR2(1000),
     USER_DATA8 NVARCHAR2(1000),
     USER_DATA9 NVARCHAR2(1000),
     USER_DATA10 NVARCHAR2(1000),
     USER_DATA11 NVARCHAR2(1000),
     USER_DATA12 NVARCHAR2(1000),
     USER_DATA13 NVARCHAR2(1000),
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL,
     CONSTRAINT PK_PROGRAM PRIMARY KEY ( PROGRAM_ID )
    )';
    CREATEOBJECT(vsql,'create TABLE PROGRAM');

    /* support tables for program */
        
    
     
      vsql := 'CREATE TABLE '||vschema||'.PROGRAM_SUBTITLE_COMP 
       (  
         PROGRAM_ID NVARCHAR2 (100) NOT NULL, 
		 TLA NVARCHAR2(600) NOT NULL,
         TYPE NVARCHAR2(200) NULL, 
         LANGUAGE NVARCHAR2(200) NULL, 
 		 ATTRIBUTE_TYPE NVARCHAR2(200) NULL,
         VALUE NVARCHAR2(200) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT PROGRAM_SUBTITLE_COMP_PK   PRIMARY KEY (PROGRAM_ID, TLA),
         CONSTRAINT FK_PROGRAM_SUBTITLE_PROGRAM_ID FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID) ON DELETE CASCADE

       )'; 
       CREATEOBJECT(vsql,'create table PROGRAM_SUBTITLE_COMP'); 

       vsql := 'CREATE TABLE '||vschema||'.PROGRAM_AUDIO_COMP 
       (  
         PROGRAM_ID NVARCHAR2 (100) NOT NULL, 
		 TLA NVARCHAR2(600) NOT NULL,
         TYPE NVARCHAR2(200) NULL, 
         LANGUAGE NVARCHAR2(200) NULL, 
		 ATTRIBUTE_TYPE NVARCHAR2(200) NULL,
         VALUE NVARCHAR2(200) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT PROGRAM_AUDIO_COMP_PK   PRIMARY KEY (PROGRAM_ID, TLA),
         CONSTRAINT FK_PROGRAM_AUDIO_PROGRAM_ID FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID) ON DELETE CASCADE
       )'; 
       CREATEOBJECT(vsql,'create table PROGRAM_AUDIO_COMP'); 

       vsql := 'CREATE TABLE '||vschema||'.PROGRAM_GENERIC 
       (  
         PROGRAM_ID NVARCHAR2 (100) NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT PROGRAM_GENERIC_PK   PRIMARY KEY (PROGRAM_ID, ASSET, NAME),
         CONSTRAINT FK_PROGRAM_GENERIC_PROGRAM_ID FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID) ON DELETE CASCADE 
       )'; 

       CREATEOBJECT(vsql,'create table PROGRAM_GENERIC'); 

       vsql := 'CREATE TABLE '||vschema||'.PROGRAM_RATING 
       (  
         PROGRAM_ID NVARCHAR2(100) NULL, 
         AUTHORITY NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CREATED_BY NVARCHAR2 (50) NOT NULL, 
         UPDATED_BY NVARCHAR2 (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT PROGRAM_RATING_PK   PRIMARY KEY (PROGRAM_ID, AUTHORITY),
         CONSTRAINT FK_PROGRAM_RATING_PROGRAM_ID FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID) ON DELETE CASCADE
       )'; 
       CREATEOBJECT(vsql,'create table PROGRAM_RATING'); 
       
    /* Schedule */
     vsql := 'CREATE TABLE '||vschema||'.SCHEDULE
    (
     SCHEDULE_ID NUMBER(19,0) NOT NULL ,
     SCHEDULE_LANGUAGE NVARCHAR2 (100) NOT NULL ,
     STATION_ID NVARCHAR2 (100)  NOT NULL ,
     PROGRAM_ID NVARCHAR2 (100)  NOT NULL ,
     START_TIME TIMESTAMP (0) WITH TIME ZONE  NOT NULL ,
     END_TIME TIMESTAMP (0) WITH TIME ZONE ,
     DURATION NUMBER (10,0) NOT NULL ,
     TV_RATING NVARCHAR2 (1000) ,
     HD_TV NVARCHAR2 (1) ,
     PART_NUMBER NUMBER (10,0) ,
     NO_OF_PARTS NUMBER (10,0) ,
	 SERIES_ID NVARCHAR2(100) , 
     CC NVARCHAR2 (1) NOT NULL ,
     STEREO NVARCHAR2 (1) ,
     LIVE_TAPE_DELAY NVARCHAR2 (100) ,
     SUBTITLED NVARCHAR2 (1) ,
     PREMIERE_FINALE NVARCHAR2 (100) ,
     JOINED_IN_PROGRESS NVARCHAR2 (1) ,
     CABLE_IN_CLASSROOM NVARCHAR2 (1) ,
     SAP NVARCHAR2 (1) ,
     SEX_RATING NVARCHAR2 (1) ,
     VIOLENCE_RATING NVARCHAR2 (1) ,
     LANGUAGE_RATING NVARCHAR2 (1) ,
     DIALOG_RATING NVARCHAR2 (1) ,
     FV_RATING NVARCHAR2 (1) ,
     ENHANCED NVARCHAR2 (1) ,
     THREE_D NVARCHAR2 (1) ,
     LETTERBOX NVARCHAR2 (1) ,
     DVS NVARCHAR2 (1) ,
     REPEAT NVARCHAR2 (1) ,
     BLACKOUT NVARCHAR2 (1) ,
     DOLBY NVARCHAR2 (100) ,
     REQUEST_RECORD VARCHAR2(1),
  	 USER_DATA1 NVARCHAR2(1000),
     USER_DATA2 NVARCHAR2(1000),
     USER_DATA3 NVARCHAR2(1000),
     USER_DATA4 NVARCHAR2(1000),
     USER_DATA5 NVARCHAR2(1000),
     USER_DATA6 NVARCHAR2(1000),
     USER_DATA7 NVARCHAR2(1000),
     USER_DATA8 NVARCHAR2(1000),
     USER_DATA9 NVARCHAR2(1000),
     USER_DATA10 NVARCHAR2(1000),
     USER_DATA11 NVARCHAR2(1000),
     CREATED_BY NVARCHAR2 (50)  NOT NULL ,
     UPDATED_BY NVARCHAR2 (50)  NOT NULL ,
     CREATE_DATE TIMESTAMP  NOT NULL ,
     UPDATE_DATE TIMESTAMP  NOT NULL,
     CONSTRAINT PK_SCHEDULE PRIMARY KEY ( SCHEDULE_ID ),
     CONSTRAINT FK_STATION_SCHEDULE FOREIGN KEY ( STATION_ID) REFERENCES '||vschema||'.STATION ( STATION_ID),
     CONSTRAINT FK_SCHEDULE_PROGRAM FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID)
    ) ';
    CREATEOBJECT(vsql,'create table SCHEDULE');
    CREATESEQ('SCHEDULE');

    vsql := 'COMMENT ON COLUMN '||vschema||'.SCHEDULE.DURATION IS ''Duration in seconds''';
    CREATEOBJECT(vsql,'create table COMMENT ON COLUMN SCHEDULE.DURATION ');

    vsql := 'CREATE UNIQUE INDEX '||vschema||'.IDX_SCHED_UNIQ_IND ON '||vschema||'.SCHEDULE (PROGRAM_ID,STATION_ID,START_TIME)';
    CREATEOBJECT(vsql,'create index IDX_SCHED_UNIQ_IND');

    vsql := 'CREATE INDEX '||vschema||'.IDX_SCHED_START_END_TIME ON '||vschema||'.SCHEDULE (START_TIME,END_TIME)';
    CREATEOBJECT(vsql,'create index IDX_SCHED_START_END_TIME');

    vsql := 'CREATE INDEX '||vschema||'.IDX_SCH_STATION_ID ON '||vschema||'.SCHEDULE(STATION_ID)';
    CREATEOBJECT(VSQL,'create index IDX_SCH_STATION_ID');
    
    /* schedule support tables */
    vsql := 'CREATE TABLE '||vschema||'.SCHEDULE_EVENT_ID 
    (  
         PROGRAM_ID NVARCHAR2(100) NULL, 
       IDENTIFIER_TYPE NVARCHAR2(200) NULL, 
       VALUE NVARCHAR2(200) NULL, 
       CREATED_BY NVARCHAR2 (50) NOT NULL, 
       UPDATED_BY NVARCHAR2 (50) NOT NULL,  
       CREATE_DATE TIMESTAMP NOT NULL, 
       UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT SCHEDULE_EVENT_IDS_PK   PRIMARY KEY (PROGRAM_ID, IDENTIFIER_TYPE) 
       )'; 
    CREATEOBJECT(vsql,'create table SCHEDULE_EVENT_ID');   
    
    vsql := 'CREATE TABLE '||vschema||'.SCHEDULE_SERIES_ATTRIB 
    (  
         PROGRAM_ID NVARCHAR2(100) NULL, 
       ATTRIBUTE_TYPE NVARCHAR2(200) NULL, 
       VALUE NVARCHAR2(2000) NULL, 
       CREATED_BY NVARCHAR2 (50) NOT NULL, 
       UPDATED_BY NVARCHAR2 (50) NOT NULL,  
       CREATE_DATE TIMESTAMP NOT NULL, 
       UPDATE_DATE TIMESTAMP NOT NULL, 
        CONSTRAINT SCHEDULE_SERIES_ATTRIB_PK   PRIMARY KEY (PROGRAM_ID, ATTRIBUTE_TYPE) 
     )'; 
     CREATEOBJECT(vsql,'create table SCHEDULE_SERIES_ATTRIB'); 

     vsql := 'CREATE TABLE '||vschema||'.SCHEDULE_GENERIC 
     (  
         PROGRAM_ID NVARCHAR2(100) NULL, 
       ASSET NVARCHAR2(50) NULL, 
       NAME NVARCHAR2(200) NULL, 
       VALUE NVARCHAR2(2000) NULL, 
       CREATED_BY NVARCHAR2 (50) NOT NULL, 
       UPDATED_BY NVARCHAR2 (50) NOT NULL,  
       CREATE_DATE TIMESTAMP NOT NULL, 
       UPDATE_DATE TIMESTAMP NOT NULL, 
         CONSTRAINT SCHEDULE_GENERIC_PK   PRIMARY KEY (PROGRAM_ID, ASSET, NAME) 
     )'; 
     CREATEOBJECT(vsql,'create table SCHEDULE_GENERIC');
     
    /********************** EPGMANAGER Export Tables *************************/
    
    /*
     * These tables are used for each publisher to push data from the production table on the way to external xml file
     * to be translated and pushed to final destinate.
     * 
     * One for each main table (5) plus additional to support complex types of program and schedule
     */
    
   /* export tables */
    
    vsql := 'CREATE TABLE '||vschema||'."EXP_HEADEND"
    (
    "HEADEND_ID" NVARCHAR2(100),
    "COMMUNITY_NAME" NVARCHAR2(200),
    "COUNTY_NAME" NVARCHAR2(200),
    "COUNTY_SIZE" NVARCHAR2(1),
    "ST_COUNTY_CODE" NVARCHAR2(100),
    "STATE" NVARCHAR2(64),
    "ZIPCODE" NVARCHAR2(100),
    "DMA_CODE" NVARCHAR2(100),
    "DMA_NAME" NVARCHAR2(300),
    "MSO_CODE" NUMBER,
    "DMA_RANK" NVARCHAR2(10),
    "HEADEND_NAME" NVARCHAR2(1000),
    "HEADEND_LOCATION" NVARCHAR2(100),
    "MSO_NAME" NVARCHAR2(200),
    "TIME_ZONE" NVARCHAR2(100)
    ) nologging';
    CREATEOBJECT(vsql,'create table exp_headend');

    vsql := 'BEGIN
                DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_HEADEND'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
             END;';
    CREATEOBJECT(vsql,'create table err$_exp_headend');
    
       /* exp_HEADEND generic */
    vsql := 'CREATE TABLE '||vschema||'."EXP_HEADEND_GENERIC"
       (	
         HEADEND_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_HEADEND_GENERIC_PK   PRIMARY KEY (HEADEND_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_HEADEND_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_HEADEND_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_HEADEND_GENERIC','Y');    


    vsql := 'CREATE TABLE '||vschema||'."EXP_LINEUP"
    (	
    "LINEUP_ID" nvarchar2(100),
    "STATION_ID" nvarchar2(100),
    "HEADEND_ID" nvarchar2(100),
    "DEVICE" NVARCHAR2(2),
    "TMS_CHANNEL" NVARCHAR2(20),
    "SERVICE_TIER" NVARCHAR2(1),
    "EFFECTIVE_DATE" TIMESTAMP (0)	WITH TIME ZONE,
    "EXPIRATION_DATE" TIMESTAMP (0)	WITH TIME ZONE
    ) nologging';
    CREATEOBJECT(vsql,'create table exp_lineup');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_LINEUP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_exp_lineup');
    
           /* exp_LINEUP generic */
    vsql := 'CREATE TABLE '||vschema||'."EXP_LINEUP_GENERIC"
       (	
         LINEUP_ID NVARCHAR2(100) NOT NULL, 
		 HEADEND_ID NVARCHAR2(100) NOT NULL,
		 STATION_ID NVARCHAR2(100) NOT NULL,
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_LINEUP_GENERIC_PK   PRIMARY KEY (LINEUP_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_LINEUP_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_LINEUP_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_LINEUP_GENERIC','Y');    


    vsql := 'CREATE TABLE '||vschema||'."EXP_STATION"
    (	
    "STATION_ID" NVARCHAR2(100),
    "TIME_ZONE" NVARCHAR2(100),
    "STATION_NAME" NVARCHAR2(100),
    "CALL_SIGN" NVARCHAR2(100),
    "AFFILIATION" NVARCHAR2(100),
    "CITY" NVARCHAR2(100),
    "STATE" NVARCHAR2(100),
    "ZIPCODE" NVARCHAR2(100),
    "COUNTRY" NVARCHAR2(100),
    "DMA_NAME" NVARCHAR2(100),
    "DMA_NUMBER" NUMBER(12),
    "FCC_CHANNEL_NUM" NUMBER(12),
    "USER_DATA1" NVARCHAR2(1000),
    "USER_DATA2" NVARCHAR2(1000),
    "USER_DATA3" NVARCHAR2(1000),
    "USER_DATA4" NVARCHAR2(1000),
    "USER_DATA5" NVARCHAR2(1000),
    "USER_DATA6" NVARCHAR2(1000),
    "USER_DATA7" NVARCHAR2(1000),
    "USER_DATA8" NVARCHAR2(1000),
    "USER_DATA9" NVARCHAR2(1000),
    "USER_DATA10" NVARCHAR2(1000)
     ) nologging';
    CREATEOBJECT(vsql,'create table exp_station');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_STATION'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_exp_station');
    
           /* exp_STATION generic */
    vsql := 'CREATE TABLE '||vschema||'."EXP_STATION_GENERIC"
       (	
         STATION_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_STATION_GENERIC_PK   PRIMARY KEY (STATION_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_STATION_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_STATION_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_STATION_GENERIC','Y');      

    vsql := 'CREATE TABLE '||vschema||'."EXP_PROGRAM"
    (	
      "PROGRAM_ID" NVARCHAR2(100),
      "LANGUAGE" NVARCHAR2(100),
      "TITLE" NVARCHAR2(1000),
      "REDUCED_TITLE1" NVARCHAR2(1000),
      "REDUCED_TITLE2" NVARCHAR2(1000),
      "REDUCED_TITLE3" NVARCHAR2(1000),
      "REDUCED_TITLE4" NVARCHAR2(1000),
      "ALT_TITLE" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION1" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION2" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION3" NVARCHAR2(1000),
      "REDUCED_DESCRIPTION4" NVARCHAR2(1000),
      "ADVISORY_DESC1" NVARCHAR2(100),
      "ADVISORY_DESC2" NVARCHAR2(100),
      "ADVISORY_DESC3" NVARCHAR2(100),
      "ADVISORY_DESC4" NVARCHAR2(100),
      "ADVISORY_DESC5" NVARCHAR2(100),
      "ADVISORY_DESC6" NVARCHAR2(100),
      "GENRE_DESC1" NVARCHAR2(100),
      "GENRE_DESC2" NVARCHAR2(100),
      "GENRE_DESC3" NVARCHAR2(100),
      "GENRE_DESC4" NVARCHAR2(100),
      "GENRE_DESC5" NVARCHAR2(100),
      "GENRE_DESC6" NVARCHAR2(100),
      "DESCRIPTION1" NVARCHAR2(2000),
      "DESCRIPTION2" NVARCHAR2(2000),
      "YEAR_PROGRAM" NVARCHAR2(100),
      "MPAA_RATING" NVARCHAR2(1000),
      "STAR_RATING" NVARCHAR2(1000),
      "RUN_TIME" VARCHAR2(100),
      "COLOR_CODE" NVARCHAR2(100),
      "PROGRAM_LANGUAGE" NVARCHAR2(100),
      "COUNTRY_OF_ORIGIN" NVARCHAR2(100),
      "MADE_FOR_TV" NVARCHAR2(1),
      "SOURCE_TYPE" NVARCHAR2(100),
      "SHOW_TYPE" NVARCHAR2(200),
      "HOLIDAY" NVARCHAR2(100),
      "SYNDICATE_EPISODE_NUM" NVARCHAR2(100),
      "ALT_SYNDICATE_EPI_NUM" NVARCHAR2(100),
      "EPISODE_TITLE" NVARCHAR2(1000),
      "NET_SYN_SOURCE" NVARCHAR2(100),
      "NET_SYN_TYPE" NVARCHAR2(100),
      "ORG_STUDIO" NVARCHAR2(100),
      "GAME_DATETIME" TIMESTAMP (0) WITH TIME ZONE,
      "GAME_TIMEZONE" NVARCHAR2(100),
      "ORG_AIR_DATE" TIMESTAMP (0) WITH TIME ZONE,
      "UNIQUE_ID" NVARCHAR2(100),
      "USER_DATA1" NVARCHAR2(1000),
      "USER_DATA2" NVARCHAR2(1000),
      "USER_DATA3" NVARCHAR2(1000),
      "USER_DATA4" NVARCHAR2(1000),
      "USER_DATA5" NVARCHAR2(1000),
      "USER_DATA6" NVARCHAR2(1000),
      "USER_DATA7" NVARCHAR2(1000),
      "USER_DATA8" NVARCHAR2(1000),
      "USER_DATA9" NVARCHAR2(1000),
      "USER_DATA10" NVARCHAR2(1000),
      "USER_DATA11" NVARCHAR2(1000),
      "USER_DATA12" NVARCHAR2(1000),
      "USER_DATA13" NVARCHAR2(1000)
    ) nologging';
    CREATEOBJECT(vsql,'create table EXP_PROGRAM');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM'', ERR_LOG_TABLE_OWNER => '''||vschema||''',skip_unsupported=>TRUE);
    END;';
    CREATEOBJECT(vsql,'create table err$_EXP_PROGRAM');
    
    /* support tables for exp_program */
    
  


   
  
       
    /* exp_Program Rating */
    vsql := 'CREATE TABLE '||vschema||'."EXP_PROGRAM_RATING"
       (	
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         AUTHORITY NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_PROGRAM_RATING   PRIMARY KEY (PROGRAM_ID, AUTHORITY) 
        ) nologging';
    CREATEOBJECT(vsql,'create table EXP_PROGRAM_RATING');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM_RATING'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_exp_program_rating','Y');
     
    /* exp_program generic */
    vsql := 'CREATE TABLE '||vschema||'."EXP_PROGRAM_GENERIC"
       (	
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_PROGRAM_GENERIC_PK   PRIMARY KEY (PROGRAM_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_PROGRAM_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_PROGRAM_GENERIC','Y');    
    
   /* exp_program audio component */
    vsql := 'CREATE TABLE '||vschema||'."EXP_PROGRAM_AUDIO_COMP"
       (	
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
		 TLA NVARCHAR2(600) NOT NULL,
         TYPE NVARCHAR2(200) NULL, 
         LANGUAGE NVARCHAR2(200) NULL, 
		 ATTRIBUTE_TYPE NVARCHAR2(200) NULL,
         VALUE NVARCHAR2(200) NULL, 
         CONSTRAINT EXP_PROGRAM_AUDIO_COMP_PK   PRIMARY KEY (PROGRAM_ID, TLA) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_PROGRAM_AUDIO_COMP');
    
    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM_AUDIO_COMP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_PROGRAM_AUDIO_COMP','Y');    
   
   /* exp_program subtitle component */
    vsql := 'CREATE TABLE '||vschema||'."EXP_PROGRAM_SUBTITLE_COMP"
       (	
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
		 TLA NVARCHAR2(600) NOT NULL,
         TYPE NVARCHAR2(200) NULL, 
         LANGUAGE NVARCHAR2(200) NULL, 
		 ATTRIBUTE_TYPE NVARCHAR2(200) NULL,
         VALUE NVARCHAR2(200) NULL, 
         CONSTRAINT EXP_PROGRAM_SUBTITLE_COMP_PK   PRIMARY KEY (PROGRAM_ID, TLA) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_PROGRAM_SUBTITLE_COMP');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM_SUBTITLE_COMP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_PROGRAM_SUBTITLE_COMP','Y');        
     
/*    vsql := 'CREATE INDEX "IDX_EXPPRG_PRGID" ON EXP_PROGRAM(DATABASE_KEY)';
    CREATEOBJECT(vsql,'Create Index IDX_EXPPRG_PRGID','Y');
*/
     vsql := 'CREATE TABLE '||vschema||'."EXP_SCHEDULE"
       (	
      "STATION_ID" NVARCHAR2 (100) NOT NULL,
      "PROGRAM_ID" NVARCHAR2(100) NOT NULL,
      "SCHEDULE_LANGUAGE" NVARCHAR2(100),
      "START_TIME" TIMESTAMP(0) with time zone,
      "END_TIME" timestamp(0) with time zone,
      "DURATION" NUMBER,
      "PART_NUMBER" NUMBER,
	  "SERIES_ID" NVARCHAR2(100),
      "NO_OF_PARTS" NUMBER,
      "CC" NVARCHAR2(1),
      "STEREO" NVARCHAR2(1),
      "REPEAT" NVARCHAR2(1),
      "LIVE_TAPE_DELAY" NVARCHAR2(100),
      "SUBTITLED" NVARCHAR2(1),
      "PREMIERE_FINALE" NVARCHAR2(100),
      "JOINED_IN_PROGRESS" NVARCHAR2(1),
      "CABLE_IN_CLASSROOM" NVARCHAR2(1),
      "TV_RATING" NVARCHAR2(1000),
      "SAP" NVARCHAR2(1),
      "BLACKOUT" NVARCHAR2(1),
      "SEX_RATING" NVARCHAR2(1),
      "VIOLENCE_RATING" NVARCHAR2(1),
      "LANGUAGE_RATING" NVARCHAR2(1),
      "DIALOG_RATING" NVARCHAR2(1),
      "FV_RATING" NVARCHAR2(1),
      "ENHANCED" NVARCHAR2(1),
      "THREE_D" NVARCHAR2(1),
      "LETTERBOX" NVARCHAR2(1),
      "HD_TV" NVARCHAR2(1),
      "DOLBY" NVARCHAR2(100),
      "DVS" NVARCHAR2(1),
      "USER_DATA1" NVARCHAR2(1000),
      "USER_DATA2" NVARCHAR2(1000),
      "USER_DATA3" NVARCHAR2(1000),
      "USER_DATA4" NVARCHAR2(1000),
      "USER_DATA5" NVARCHAR2(1000),
      "USER_DATA6" NVARCHAR2(1000),
      "USER_DATA7" NVARCHAR2(1000),
      "USER_DATA8" NVARCHAR2(1000),
      "USER_DATA9" NVARCHAR2(1000),
      "USER_DATA10" NVARCHAR2(1000),
      "USER_DATA11" NVARCHAR2(1000),
      "REQUEST_RECORD" VARCHAR2(1)
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_SCHEDULE');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_SCHEDULE'', ERR_LOG_TABLE_OWNER => '''||vschema||''',skip_unsupported=>TRUE);
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_SCHEDULE');
/*
    vsql := 'CREATE INDEX "IDX_EXPSCH_PK" ON EXP_SCHEDULE(STATION_NUM,DATABASE_KEY,AIR_DATE)';
    CREATEOBJECT(vsql,'Create Index IDX_EXPSCH_PK','Y');
*/
    /* exp schedule support tables */
    
    /* exp_schedule_event_id */
    vsql := 'CREATE TABLE '||vschema||'.EXP_SCHEDULE_EVENT_ID( 
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         IDENTIFIER_TYPE NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(200) NULL, 
         CONSTRAINT EXP_SCHEDULE_EVENT_IDS_PK   PRIMARY KEY (PROGRAM_ID, IDENTIFIER_TYPE) 
		) nologging';
    CREATEOBJECT(vsql,'create table EXP_SCHEDULE_EVENT_ID');
    
    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_SCHEDULE_EVENT_ID'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_EXP_SCHEDULE_EVENT_ID');
 
    /* exp_schedule_series_attribute */
    vsql := 'CREATE TABLE '||vschema||'.EXP_SCHEDULE_SERIES_ATTRIB(
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         ATTRIBUTE_TYPE NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_SCHEDULE_SERIES_ATTRIBS_PK   PRIMARY KEY (PROGRAM_ID, ATTRIBUTE_TYPE) 
		) nologging';
    CREATEOBJECT(vsql,'create table EXP_SCHEDULE_SERIES_ATTRIB');
    
    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_SCHEDULE_SERIES_ATTRIB'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_EXP_SCHEDULE_SERIES_ATTRIB');
    
    /* exp_schedule generic */
    vsql := 'CREATE TABLE '||vschema||'."EXP_SCHEDULE_GENERIC"
       (	
         PROGRAM_ID NVARCHAR2(100) NOT NULL, 
         ASSET NVARCHAR2(50) NULL, 
         NAME NVARCHAR2(200) NULL, 
         VALUE NVARCHAR2(2000) NULL, 
         CONSTRAINT EXP_SCHEDULE_GENERIC_PK   PRIMARY KEY (PROGRAM_ID, ASSET, NAME) 
       ) nologging';
    CREATEOBJECT(vsql,'create table EXP_SCHEDULE_GENERIC');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_SCHEDULE_GENERIC'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_SCHEDULE_GENERIC','Y');    

    
    
    creatcredittable();
    
 /* end export tables */
     
     /********** INSERT ROWS to PRODUCTION Tables ****************************/
   
	/* initial entity mapping fields - more may be added during epg ingest from more elaborate fields. */	
    INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleId','SCHEDULE','SCHEDULE_ID','Int',0,0);
    INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleStationId','SCHEDULE','STATION_ID','String',1,0);
    INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleProgramId','SCHEDULE','PROGRAM_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleLanguage','SCHEDULE','SCHEDULE_LANGUAGE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.startTime','SCHEDULE','START_TIME','DateTime',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.endTime','SCHEDULE','END_TIME','DateTime',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.duration','SCHEDULE','DURATION','Int',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.tvRating','SCHEDULE','TV_RATING','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.hd','SCHEDULE','HD_TV','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.partNumber','SCHEDULE','PART_NUMBER','Int',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.noOfParts','SCHEDULE','NO_OF_PARTS','Int',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.cc','SCHEDULE','CC','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stereo','SCHEDULE','STEREO','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.liveTapeDelay','SCHEDULE','LIVE_TAPE_DELAY','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.subtitled','SCHEDULE','SUBTITLED','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.premiereFinale','SCHEDULE','PREMIERE_FINALE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.joinedInProgress','SCHEDULE','JOINED_IN_PROGRESS','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.cableInClassroom','SCHEDULE','CABLE_IN_CLASSROOM','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.sap','SCHEDULE','SAP','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.sexRating','SCHEDULE','SEX_RATING','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.violenceRating','SCHEDULE','VIOLENCE_RATING','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.languageRating','SCHEDULE','LANGUAGE_RATING','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.dialogRating','SCHEDULE','DIALOG_RATING','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.fvRating','SCHEDULE','FV_RATING','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.enhanced','SCHEDULE','ENHANCED','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.threeD','SCHEDULE','THREE_D','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.letterBox','SCHEDULE','LETTERBOX','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.dvs','SCHEDULE','DVS','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.repeat','SCHEDULE','REPEAT','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.blackout','SCHEDULE','BLACKOUT','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.dolby','SCHEDULE','DOLBY','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.requestToRecord','SCHEDULE','REQUEST_RECORD','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programId','PROGRAM','PROGRAM_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.language','PROGRAM','LANGUAGE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.title','PROGRAM','TITLE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedTitle1','PROGRAM','REDUCED_TITLE1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedTitle2','PROGRAM','REDUCED_TITLE2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedTitle3','PROGRAM','REDUCED_TITLE3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedTitle4','PROGRAM','REDUCED_TITLE4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.altTitle','PROGRAM','ALT_TITLE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedDescription1','PROGRAM','REDUCED_DESCRIPTION1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedDescription2','PROGRAM','REDUCED_DESCRIPTION2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedDescription3','PROGRAM','REDUCED_DESCRIPTION3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.reducedDescription4','PROGRAM','REDUCED_DESCRIPTION4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.advisoryDescription1','PROGRAM','ADVISORY_DESC1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.advisoryDescription2','PROGRAM','ADVISORY_DESC2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.advisoryDescription3','PROGRAM','ADVISORY_DESC3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.advisoryDescription4','PROGRAM','ADVISORY_DESC4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.advisoryDescription5','PROGRAM','ADVISORY_DESC5','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.advisoryDescription6','PROGRAM','ADVISORY_DESC6','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.genereDescription1','PROGRAM','GENRE_DESC1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.genereDescription2','PROGRAM','GENRE_DESC2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.genereDescription3','PROGRAM','GENRE_DESC3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.genereDescription4','PROGRAM','GENRE_DESC4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.genereDescription5','PROGRAM','GENRE_DESC5','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.genereDescription6','PROGRAM','GENRE_DESC6','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.description1','PROGRAM','DESCRIPTION1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.description2','PROGRAM','DESCRIPTION2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.year','PROGRAM','YEAR_PROGRAM','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.mpaaRating','PROGRAM','MPAA_RATING','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.starRating','PROGRAM','STAR_RATING','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.runTime','PROGRAM','RUN_TIME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.colorCode','PROGRAM','COLOR_CODE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programLanguage','PROGRAM','PROGRAM_LANGUAGE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.countryOfOrigin','PROGRAM','COUNTRY_OF_ORIGIN','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.madeForTV','PROGRAM','MADE_FOR_TV','Boolean',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.sourceType','PROGRAM','SOURCE_TYPE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.showType','PROGRAM','SHOW_TYPE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.holiday','PROGRAM','HOLIDAY','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.syndicateEpisodeNum','PROGRAM','SYNDICATE_EPISODE_NUM','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.altSyndicateEpisodeNum','PROGRAM','ALT_SYNDICATE_EPI_NUM','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.episodeTitle','PROGRAM','EPISODE_TITLE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.netSynSource','PROGRAM','NET_SYN_SOURCE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.netSynType','PROGRAM','NET_SYN_TYPE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.orgStudio','PROGRAM','ORG_STUDIO','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.gameDateTime','PROGRAM','GAME_DATETIME','DateTime',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.gameTimezone','PROGRAM','GAME_TIMEZONE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.orgAirDate','PROGRAM','ORG_AIR_DATE','DateTime',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.uniqueId','PROGRAM','UNIQUE_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.seriesId','SCHEDULE','SERIES_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationId','STATION','STATION_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationName','STATION','STATION_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationTimezone','STATION','TIME_ZONE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.callSign','STATION','CALL_SIGN','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationCity','STATION','CITY','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationState','STATION','STATE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationZipcode','STATION','ZIPCODE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationCountry','STATION','COUNTRY','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.dmaName','STATION','DMA_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.dmaNumber','STATION','DMA_NUMBER','Int',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.fccChannelNumber','STATION','FCC_CHANNEL_NUM','Int',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.affiliation','STATION','AFFILIATION','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData1','PROGRAM','USER_DATA1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData2','PROGRAM','USER_DATA2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData3','PROGRAM','USER_DATA3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData4','PROGRAM','USER_DATA4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData5','PROGRAM','USER_DATA5','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData6','PROGRAM','USER_DATA6','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData7','PROGRAM','USER_DATA7','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData8','PROGRAM','USER_DATA8','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData9','PROGRAM','USER_DATA9','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData10','PROGRAM','USER_DATA10','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData11','PROGRAM','USER_DATA11','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData12','PROGRAM','USER_DATA12','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.programUserData13','PROGRAM','USER_DATA13','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData1','SCHEDULE','USER_DATA1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData2','SCHEDULE','USER_DATA2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData3','SCHEDULE','USER_DATA3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData4','SCHEDULE','USER_DATA4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData5','SCHEDULE','USER_DATA5','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData6','SCHEDULE','USER_DATA6','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData7','SCHEDULE','USER_DATA7','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData8','SCHEDULE','USER_DATA8','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData9','SCHEDULE','USER_DATA9','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData10','SCHEDULE','USER_DATA10','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.scheduleUserData11','SCHEDULE','USER_DATA11','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData1','STATION','USER_DATA1','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData2','STATION','USER_DATA2','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData3','STATION','USER_DATA3','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData4','STATION','USER_DATA4','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData5','STATION','USER_DATA5','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData6','STATION','USER_DATA6','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData7','STATION','USER_DATA7','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData8','STATION','USER_DATA8','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData9','STATION','USER_DATA9','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.stationUserData10','STATION','USER_DATA10','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendId','HEADEND','HEADEND_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendCommunityName','HEADEND','COMMUNITY_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendCountyName','HEADEND','COUNTY_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendStCountyCode','HEADEND','ST_COUNTY_CODE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendCountySize','HEADEND','COUNTY_SIZE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendState','HEADEND','STATE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendZipCode','HEADEND','ZIPCODE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendDMACode','HEADEND','DMA_CODE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendDMAName','HEADEND','DMA_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendMSOCode','HEADEND','MSO_CODE','Int',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendDMARank','HEADEND','DMA_RANK','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendName','HEADEND','HEADEND_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendLocation','HEADEND','HEADEND_LOCATION','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendMSOName','HEADEND','MSO_NAME','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.headendTimezone','HEADEND','TIME_ZONE','String',1,1);
	
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupId','LINEUP','LINEUP_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupStationId','LINEUP','STATION_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupHeadendId','LINEUP','HEADEND_ID','String',1,0);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupDevice','LINEUP','DEVICE','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupTMSChannel','LINEUP','TMS_CHANNEL','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupServiceTier','LINEUP','SERVICE_TIER','String',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupEffectiveDate','LINEUP','EFFECTIVE_DATE','DateTime',1,1);
	INSERTENTITYMAPPING('dynamicEntityMappingField.lineupExpirationDate','LINEUP','EXPIRATION_DATE','DateTime',1,1);	
	
	
      
    vsql := 'CREATE TABLE '||vschema||'.epgguidedata_log(logdatetime timestamp default localtimestamp,
              description clob)';
    CREATEOBJECT(vsql,'create table epgguidedata_log');
    
   
    /* only on the STATION and LINEUP tables? */    
    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STATION'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_STATION');

    vsql := 'BEGIN
      DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.LINEUP'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
    END;';
    CREATEOBJECT(vsql,'create table err$_LINEUP');

	vsql := 'create table '||vschema||'.temp_station_sch_range
					(
						STATION_ID NVARCHAR2(100) primary key,
						first_schedule_start_time timestamp(0) with time zone,
						last_schedule_end_time timestamp(0) with time zone
					)';
	CREATEOBJECT(vsql, 'create TABLE temp_station_sch_range');

    createupdatetrg('CONFIG_SETTING');
    createupdatetrg('HEADEND');
    createupdatetrg('HEADEND_GENERIC');
    createupdatetrg('LINEUP');
    createupdatetrg('LINEUP_GENERIC');    
    --createupdatetrg('SERIES');
    createupdatetrg('STATION');
    createupdatetrg('STATION_GENERIC');

    
    createupdatetrg('PROGRAM');
    createupdatetrg('PROGRAM_RATING');

    createupdatetrg('PROGRAM_GENERIC');
    createupdatetrg('PROGRAM_AUDIO_COMP');
    createupdatetrg('PROGRAM_SUBTITLE_COMP');
    
    createupdatetrg('SCHEDULE');
    createupdatetrg('SCHEDULE_EVENT_ID');    
    createupdatetrg('SCHEDULE_SERIES_ATTRIB');    
    createupdatetrg('SCHEDULE_GENERIC');
	
    createupdatetrg('ALARM_ACTIVITYLOG');
        
    createupdatetrg('NORM_RULE_CLAUSE');
    createupdatetrg('NORM_RULE');
    createupdatetrg('NORM_GROUP');
    createupdatetrg('NORM_PROVIDER_GROUP_MAP');
    createupdatetrg('NORM_PUBLISHER_GROUP_MAP');
    createupdatetrg('VALID_RULE_CLAUSE');
    createupdatetrg('VALID_RULE');
    createupdatetrg('VALID_GROUP');
    createupdatetrg('VALID_PUBLISHER_GROUP_MAP');
    createupdatetrg('VALID_PROVIDER_GROUP_MAP');
        
  --Version information will be updated by installer.
    vsql := 'INSERT INTO '||vschema||'.config_setting (key,value,CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE)
      VALUES(''epgmanagerversion'', ''&EPG_DB_VERSION'', ''System'', ''System'', current_timestamp, current_timestamp)';
    CREATEOBJECT(vsql,'INSERT INTO CONFIG_SETTING');
    
   vsql := 'INSERT INTO install_Table values(sysdate,''EPGManager Version &EPG_DB_VERSION install has completed SUCCESSFULLY.'')';
    CREATEOBJECT(vsql,'insert into install_table','Y');
    
   vsql := 'ALTER TABLESPACE epgmanager_data ADD DATAFILE SIZE 100M AUTOEXTEND ON NEXT 250M MAXSIZE unlimited';
		CREATEOBJECT(vsql,'Add Datafile for tablespace');
    
end createepgmanager;

  

PROCEDURE creatcredittable AS
    vsql varchar2(32767);
  BEGIN
  
      /* NEW CREDIT TABLE*/
    
       /* stg_program_credits */
    vsql := 'CREATE TABLE '||vschema||'.STG_PROGRAM_CREDITS
		(
			FILENAME VARCHAR2(4000 BYTE), 
			PROGRAM_ID NVARCHAR2(100) NOT NULL, 
			C_ID NVARCHAR2(50) NULL,
			SEQ NUMBER(6) DEFAULT 1 NOT NULL,
			ROLE NVARCHAR2(200) NOT NULL, 
			FIRST_NAME NVARCHAR2(200) NULL, 
			MIDDLE_NAME NVARCHAR2(200) NULL, 
			LAST_NAME NVARCHAR2(200) NULL, 
			CONSTRAINT U_STG_PROGRAM_CREDITS   UNIQUE (PROGRAM_ID, SEQ, ROLE) 
		)nologging'; 
	CREATEOBJECT(vsql,'create table STG_PROGRAM_CREDITS'); 

       
	vsql := 'BEGIN
			DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM_CREDITS'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
			END;';
	CREATEOBJECT(vsql,'create table err$_STG_PROGRAM_CREDITS','Y');
	
	/* stg_program cast */
    vsql := 'CREATE TABLE '||vschema||'."STG_PROGRAM_CAST"
		(	
			FILENAME VARCHAR2(4000 BYTE), 
			PROGRAM_ID NVARCHAR2(100) NOT NULL, 
			C_ID NVARCHAR2(50) NULL,
			SEQ NUMBER(6) DEFAULT 1 NOT NULL,
			ROLE NVARCHAR2(200) NOT NULL, 
			FIRST_NAME NVARCHAR2(200) NULL, 
			MIDDLE_NAME NVARCHAR2(200) NULL, 
			LAST_NAME NVARCHAR2(200) NULL, 
			CONSTRAINT U_STG_PROGRAM_CAST   UNIQUE (PROGRAM_ID, SEQ, ROLE) 
		) nologging';
	CREATEOBJECT(vsql,'create table STG_PROGRAM_CAST');

	vsql := 'BEGIN
			DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.STG_PROGRAM_CAST'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
			END;';
    CREATEOBJECT(vsql,'create table err$_STG_PROGRAM_CAST','Y');
	
	vsql := 'CREATE TABLE '||vschema||'.PROGRAM_CREDITS 
		(  
			PROGRAM_ID NVARCHAR2 (100) NOT NULL, 
			C_ID NVARCHAR2(50) NULL,
			SEQ NUMBER(6) DEFAULT 1 NOT NULL ,
			ROLE NVARCHAR2(200) NOT NULL, 
			FIRST_NAME NVARCHAR2(200) NULL, 
			MIDDLE_NAME NVARCHAR2(200) NULL, 
			LAST_NAME NVARCHAR2(200) NULL, 
			CREATED_BY NVARCHAR2 (50) NOT NULL, 
			UPDATED_BY NVARCHAR2 (50) NOT NULL,  
			CREATE_DATE TIMESTAMP NOT NULL, 
			UPDATE_DATE TIMESTAMP NOT NULL, 
			CONSTRAINT U_PROGRAM_CREDITS   UNIQUE (PROGRAM_ID, SEQ, ROLE),
			CONSTRAINT FK_PROGRAM_CREDIT_PROG_ID FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID) ON DELETE CASCADE
		)'; 
	CREATEOBJECT(vsql,'create table PROGRAM_CREDITS'); 
	
	vsql := 'CREATE TABLE '||vschema||'.PROGRAM_CAST
		(  
			PROGRAM_ID NVARCHAR2 (100) NOT NULL, 
			C_ID NVARCHAR2(50) NULL,
			SEQ NUMBER(6) DEFAULT 1 NOT NULL,
			ROLE NVARCHAR2(200) NOT NULL, 
			FIRST_NAME NVARCHAR2(200) NULL, 
			MIDDLE_NAME NVARCHAR2(200) NULL,
			LAST_NAME NVARCHAR2(200) NULL, 
			CREATED_BY NVARCHAR2 (50) NOT NULL, 
			UPDATED_BY NVARCHAR2 (50) NOT NULL,  
			CREATE_DATE TIMESTAMP NOT NULL, 
			UPDATE_DATE TIMESTAMP NOT NULL, 
			CONSTRAINT U_PROGRAM_CAST   UNIQUE (PROGRAM_ID, SEQ, ROLE),
			CONSTRAINT FK_PROGRAM_CAST_PROGRAM_ID FOREIGN KEY ( PROGRAM_ID) REFERENCES '||vschema||'.PROGRAM ( PROGRAM_ID) ON DELETE CASCADE
		)'; 
	CREATEOBJECT(vsql,'create table PROGRAM_CAST'); 	  
	
	createupdatetrg('PROGRAM_CAST');
	createupdatetrg('PROGRAM_CREDITS');	
	   
	/* exp_program_credits */   
	vsql := 'CREATE TABLE '||vschema||'.EXP_PROGRAM_CREDITS
		(  
			PROGRAM_ID NVARCHAR2(100) NOT NULL, 
			C_ID NVARCHAR2(50) NULL,
			SEQ NUMBER(6) NOT  NULL,
			ROLE NVARCHAR2(200) NOT NULL, 
			FIRST_NAME NVARCHAR2(200) NULL,
			MIDDLE_NAME NVARCHAR2(200) NULL, 
			LAST_NAME NVARCHAR2(200) NULL, 
			CONSTRAINT U_EXP_PROGRAM_CREDITS   UNIQUE (PROGRAM_ID,SEQ, ROLE) 
		) nologging'; 
    CREATEOBJECT(vsql,'create table EXP_PROGRAM_CREDITS'); 
       
    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM_CREDITS'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_PROGRAM_CREDITS','Y');

	/* exp_program_cast */	
	vsql := 'CREATE TABLE '||vschema||'."EXP_PROGRAM_CAST"
		(	
			PROGRAM_ID NVARCHAR2(100) NOT NULL,
			C_ID NVARCHAR2(50) NULL,
			SEQ NUMBER(6) NOT NULL,
			ROLE NVARCHAR2(200) NOT NULL,
			FIRST_NAME NVARCHAR2(200) NULL, 
			MIDDLE_NAME NVARCHAR2(200) NULL, 
			LAST_NAME NVARCHAR2(200) NULL, 
			CONSTRAINT U_EXP_PROGRAM_CAST   UNIQUE (PROGRAM_ID, SEQ, ROLE) 
		) nologging';
    CREATEOBJECT(vsql,'create table EXP_PROGRAM_CAST');

    vsql := 'BEGIN
              DBMS_ERRLOG.create_error_log (dml_table_name => '''||vschema||'.EXP_PROGRAM_CAST'', ERR_LOG_TABLE_OWNER => '''||vschema||''');
            END;';
    CREATEOBJECT(vsql,'create table err$_EXP_PROGRAM_CAST','Y');
  

END ;

PROCEDURE dropcreditsandcasttables(pattern varchar2) AS
	type vtablestobedropped is table of VARCHAR2(50) INDEX BY PLS_INTEGER;
	tabletobedrop vtablestobedropped;
  begin
    select table_name bulk collect into tabletobedrop from dba_tables where table_name like pattern and OWNER = upper(vschema);
    if tabletobedrop.count > 0 then
		FOR indx IN 1 .. tabletobedrop.COUNT
		LOOP
			execute immediate 'drop table '||vschema||'.'||tabletobedrop(indx)||'';
		END LOOP;
    end if;
    
    commit;
END dropcreditsandcasttables;

  PROCEDURE convertepgmanager AS
    vsql varchar2(32767);
    vepgdbver varchar2(100);
  BEGIN
    begin
      execute immediate 'SELECT value from '||vschema||'.config_setting where key=''epgmanagerversion'''  into vepgdbver;
    exception
      when others then
        vepgdbver := '&EPG_DB_VERSION';
    end;
    
    -- the algorithm here is to compare the version of the DB against the known versions ever created.
    -- then implement the changes needed in each block to move from what is known to what is current.
	
    if vepgdbver < '1.0.000.1' then
--    	-- perform steps need to go from 1.0.000.0 to 1.0.000.1

    	-- update entity mappings
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''Int'' where NAME=''dynamicEntityMappingField.headendMSOName''';
		CREATEOBJECT(vsql,'Update Entity Mapping 1','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.year''';
		CREATEOBJECT(vsql,'Update Entity Mapping 2','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.runTime''';
		CREATEOBJECT(vsql,'Update Entity Mapping 3','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.syndicateEpisodeNum''';
		CREATEOBJECT(vsql,'Update Entity Mapping 4','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.altSyndicateEpisodeNum''';
		CREATEOBJECT(vsql,'Update Entity Mapping 5','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.fccChannelNumber''';
		CREATEOBJECT(vsql,'Update Entity Mapping 6','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.dmaNumber''';
		CREATEOBJECT(vsql,'Update Entity Mapping 7','Y');
		
		-- ordering for rules and rule groups , do some back flips cause the columns MIGHT already exist..
		DECLARE
  		v_column_exists number := 0;  
		BEGIN
  			Select count(*) into v_column_exists from all_tab_columns where owner='EPGMANAGER' and table_name='RULE' and column_name = 'ORDERING';

  			if (v_column_exists = 0) then
      			execute immediate 'ALTER TABLE EPGMANAGER.RULE ADD (ORDERING NUMBER(19, 0) NOT NULL)';
  			end if;
  			
    		Select count(*) into v_column_exists from all_tab_columns where owner='EPGMANAGER' and table_name='NORM_RULE_GROUP' and column_name = 'ORDERING';
			
  			if (v_column_exists = 0) then
      			execute immediate 'ALTER TABLE EPGMANAGER.NORM_RULE_GROUP ADD (ORDERING NUMBER(19, 0) NOT NULL)';
  			end if;
  			
		end;
    
        -- set the version to now be this version.  This is in case there is an error in later updates blocks - we can rerun and only redo that part.
    	setInstalledDBVersion('1.0.000.1');
    end if;
    
    -- this is specifically NOT an else. We want to run the later checks AS WELL as the one above.
    if vepgdbver < '1.0.000.2' then
    	-- perform steps to update to 1.0.000.2
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.headendMSOName''';
		CREATEOBJECT(vsql,'Update Entity Mapping 8','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''Int'' where NAME=''dynamicEntityMappingField.headendMSOCode''';
		CREATEOBJECT(vsql,'Update Entity Mapping 9','Y');
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.runTime''';
		CREATEOBJECT(vsql,'Update Entity Mapping 2','Y');

		setInstalledDBVersion('1.0.000.2');
    end if;
	 
    if vepgdbver < '1.0.000.3' then
		vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set DATA_TYPE=''String'' where NAME=''dynamicEntityMappingField.premiereFinale''';
		CREATEOBJECT(vsql,'Update Entity Mapping 1','Y');

		setInstalledDBVersion('1.0.000.3');
	
      vsql := 'INSERT INTO install_Table values(sysdate,''EPGManager Version &EPG_DB_VERSION upgrade has completed SUCCESSFULLY.'')';
      CREATEOBJECT(vsql,'insert into install_table','Y');
   end if; 
       if vepgdbver < '1.1.000.1' then
       vsql := 'ALTER table '||vschema||'.ENTITY_MAPPING ADD ALLOW_AS_CRITERIA NUMBER(1,0) DEFAULT 1 NOT NULL';
       CREATEOBJECT(vsql,'add column ALLOW_AS_CRITERIA','Y');
       vsql := 'ALTER table '||vschema||'.ENTITY_MAPPING ADD  ALLOW_AS_PREDICATE NUMBER(1,0) DEFAULT 1 NOT NULL';
       CREATEOBJECT(vsql,'add column ALLOW_AS_PREDICATE','Y');
       vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=0,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''SCHEDULE'' AND COLUMN_NAME=''SCHEDULE_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
       vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''SCHEDULE'' AND COLUMN_NAME=''STATION_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
           vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''SCHEDULE'' AND COLUMN_NAME=''PROGRAM_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
       vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''PROGRAM'' AND COLUMN_NAME=''PROGRAM_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
       vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''PROGRAM'' AND COLUMN_NAME=''UNIQUE_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
           vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''SCHEDULE'' AND COLUMN_NAME=''SERIES_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
            vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''STATION'' AND COLUMN_NAME=''STATION_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
         vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''HEADEND'' AND COLUMN_NAME=''HEADEND_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
                 vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''LINEUP'' AND COLUMN_NAME=''LINEUP_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
         vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''LINEUP'' AND COLUMN_NAME=''STATION_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
                 vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''LINEUP'' AND COLUMN_NAME=''HEADEND_ID''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');
         vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set ALLOW_AS_CRITERIA=1,ALLOW_AS_PREDICATE=0 where TABLE_NAME=''SCHEDULE'' AND COLUMN_NAME=''END_TIME''';
                CREATEOBJECT(vsql,'Update Entity Mapping','Y');

                 vsql := 'ALTER TABLESPACE epgmanager_data ADD DATAFILE SIZE 100M AUTOEXTEND ON NEXT 250M MAXSIZE unlimited';
                CREATEOBJECT(vsql,'Add Datafile for tablespace','Y');

          vsql := 'CREATE TABLE '||vschema||'."TEMPORARY_STATION" AS SELECT * FROM ' ||vschema||'.STATION';
                 CREATEOBJECT(vsql,'Create table TEMPORARY_STATION','Y');
          vsql := 'UPDATE '||vschema||'.STATION set dma_number=null,fcc_channel_num = null';
                CREATEOBJECT(vsql,'Update STATION','Y');
          vsql := 'ALTER table '||vschema||'.STATION MODIFY (dma_number number(12),fcc_channel_num number(12))';
                CREATEOBJECT(vsql,'modify columns in station from string to number','Y');
           vsql :='update '||vschema||'.station set '||vschema||'.station.fcc_channel_num=(select to_number('||vschema||'.temporary_station.fcc_channel_num) from '||vschema||'.temporary_station where '||vschema||'.station.station_id='||vschema||'.temporary_station.station_id)';
                CREATEOBJECT(vsql,'Update station fcc_channel_num','Y');
          vsql := 'update '||vschema||'.station set  '||vschema||'.station.dma_number=(select to_number('||vschema||'.temporary_station.dma_number) from '||vschema||'.temporary_station where '||vschema||'.station.station_id='||vschema||'.temporary_station.station_id)';
                CREATEOBJECT(vsql,'Update STATION dma_number','Y');
           vsql := 'UPDATE '||vschema||'.ENTITY_MAPPING set data_type=''Int'' where table_name=''STATION'' and column_name in(''FCC_CHANNEL_NUM'',''DMA_NUMBER'')';
                 CREATEOBJECT(vsql,'Update Entity Mapping set data_type to int','Y');
      IF IS_TABLE_EXISTS(vschema,'ROVI_STG_STATION') THEN
            vsql := 'CREATE TABLE '||vschema||'."TEMP_ROVI_STG_STATION" AS SELECT * FROM ' ||vschema||'.ROVI_STG_STATION';
                 CREATEOBJECT(vsql,'Create table TEMP_ROVI_STG_STATION','Y');
            vsql := 'UPDATE '||vschema||'.ROVI_STG_STATION set dma_number=null,fcc_channel_num = null';
                CREATEOBJECT(vsql,'Update ROVI_STG_STATION','Y');
          vsql := 'ALTER table '||vschema||'.ROVI_STG_STATION MODIFY (dma_number number(12),fcc_channel_num number(12))';
                CREATEOBJECT(vsql,'modify columns in rovi_stg_station from string to number','Y');
           vsql :='update '||vschema||'.rovi_stg_station set '||vschema||'.rovi_stg_station.fcc_channel_num=(select to_number('||vschema||'.temp_rovi_stg_station.fcc_channel_num) from '||vschema||'.temp_rovi_stg_station where '||vschema||'.rovi_stg_station.station_id='||vschema||'.temp_rovi_stg_station.station_id)';
                CREATEOBJECT(vsql,'Update station fcc_channel_num','Y');
          vsql := 'update '||vschema||'.rovi_stg_station set  '||vschema||'.rovi_stg_station.dma_number=(select to_number('||vschema||'.temp_rovi_stg_station.dma_number) from '||vschema||'.temp_rovi_stg_station where '||vschema||'.rovi_stg_station.station_id='||vschema||'.temp_rovi_stg_station.station_id)';
                CREATEOBJECT(vsql,'Update ROVI_STG_STATION dma_number','Y');
     END IF;
        vsql := 'UPDATE '||vschema||'.STG_STATION set dma_number=null,fcc_channel_num = null';
                CREATEOBJECT(vsql,'Update STG_STATION','Y');
      IF IS_TABLE_EXISTS(vschema,'EXP_STATION') THEN
        vsql := 'UPDATE '||vschema||'.EXP_STATION set dma_number=null,fcc_channel_num = null';
                CREATEOBJECT(vsql,'Update EXP_STATION','Y');
      END IF;
        vsql := 'ALTER table '||vschema||'.STG_STATION MODIFY (dma_number number(12),fcc_channel_num number(12))';
                CREATEOBJECT(vsql,'modify columns in stg_station from string to number','Y');
       IF IS_TABLE_EXISTS(vschema,'EXP_STATION') THEN
          vsql := 'ALTER table '||vschema||'.EXP_STATION MODIFY (dma_number number(12),fcc_channel_num number(12))';
                CREATEOBJECT(vsql,'modify columns in exp_station from string to number','Y');
         END IF;
          vsql := 'drop table '||vschema||'.temporary_station';
                CREATEOBJECT(vsql,'Drop table temporary_station','Y');
       IF IS_TABLE_EXISTS(vschema,'TEMP_ROVI_STG_STATION') THEN
          vsql := 'drop table '||vschema||'.temp_rovi_stg_station';
                CREATEOBJECT(vsql,'Drop table temp_rovi_stg_station','Y');
       END IF;
     
       /* route status table */
	     vsql := 'DROP TABLE '||vschema||'.ROUTE_STATUS'; 
         CREATEOBJECT(vsql,'Drop table ROUTE_STATUS','Y');  

       /* route status table */
	     vsql := 'CREATE TABLE '||vschema||'."ROUTE_STATUS"; 
		 (  
		   ROUTE_ID NVARCHAR2(50) NOT NULL, 
		   STATE NVARCHAR2(50) NOT NULL, 
		   LAST_RUN_START_TIME TIMESTAMP NOT NULL, 
		   LAST_RUN_END_TIME TIMESTAMP,    
		   REMARKS NVARCHAR2(2000),
		   CREATED_BY NVARCHAR2 (50)  NOT NULL ,
		   CREATE_DATE TIMESTAMP  NOT NULL ,
		   UPDATED_BY NVARCHAR2 (50) NOT NULL,     
		   UPDATE_DATE TIMESTAMP NOT NULL, 
		   CONSTRAINT PK_ROUTE_STATUS_ROUTE_ID PRIMARY KEY (ROUTE_ID) 
		 )'; 
		 CREATEOBJECT(vsql,'create table ROUTE_STATUS','Y');  
       
	setInstalledDBVersion('1.1.000.1');

    vsql := 'INSERT INTO install_Table values(sysdate,''EPGManager Version &EPG_DB_VERSION upgrade has completed SUCCESSFULLY.'')';
    CREATEOBJECT(vsql,'insert into install_table','Y');
	
    end if;

    if vepgdbver < '1.1.000.2' then
	
    /* route status table */
	vsql := 'DROP TABLE '||vschema||'.ROUTE_STATUS';
    CREATEOBJECT(vsql,'Drop table ROUTE_STATUS','Y');  

    /* route status table */
	vsql := 'CREATE TABLE '||vschema||'."ROUTE_STATUS" 
		 (  
		   ROUTE_ID NVARCHAR2(50) NOT NULL, 
		   STATE NVARCHAR2(50) NOT NULL, 
		   LAST_RUN_START_TIME TIMESTAMP NOT NULL, 
		   LAST_RUN_END_TIME TIMESTAMP,    
		   REMARKS NVARCHAR2(2000),
		   CREATED_BY NVARCHAR2 (50)  NOT NULL ,
		   CREATE_DATE TIMESTAMP  NOT NULL ,
		   UPDATED_BY NVARCHAR2 (50) NOT NULL,     
		   UPDATE_DATE TIMESTAMP NOT NULL, 
		   CONSTRAINT PK_ROUTE_STATUS_ROUTE_ID PRIMARY KEY (ROUTE_ID) 
		 )'; 
	 CREATEOBJECT(vsql,'create table ROUTE_STATUS','Y');  

	setInstalledDBVersion('1.1.000.2');
	vsql := 'INSERT INTO install_Table values(sysdate,''EPGManager Version &EPG_DB_VERSION upgrade has completed SUCCESSFULLY.'')';
    CREATEOBJECT(vsql,'insert into install_table','Y');

    end if;
    
    IF vepgdbver < '1.1.000.3' then
	  IF IS_TABLE_EXISTS(vschema,'SCHEDULE_SERIES_ATTRIB') THEN
          vsql := 'ALTER table '||vschema||'.SCHEDULE_SERIES_ATTRIB MODIFY (value NVARCHAR2(2000))';
                CREATEOBJECT(vsql,'modify columns in schedule_series_attrib from nvarchar2(200) to nvarchar2(2000)','Y');
      END IF;
      
      IF IS_TABLE_EXISTS(vschema,'STG_SCHEDULE_SERIES_ATTRIB') THEN
          vsql := 'ALTER table '||vschema||'.STG_SCHEDULE_SERIES_ATTRIB MODIFY (value NVARCHAR2(2000))';
          CREATEOBJECT(vsql,'modify columns in stg_schedule_series_attrib from nvarchar2(200) to nvarchar2(2000)','Y');
      END IF;
      
      IF IS_TABLE_EXISTS(vschema,'EXP_SCHEDULE_SERIES_ATTRIB') THEN
          vsql := 'ALTER table '||vschema||'.EXP_SCHEDULE_SERIES_ATTRIB MODIFY (value NVARCHAR2(2000))';
                CREATEOBJECT(vsql,'modify columns in exp_schedule_series_attrib from nvarchar2(200) to nvarchar2(2000)','Y');
      END IF;
    END IF;
    
    
    IF vepgdbver < '1.1.000.4' then
      creatcredittable();
      
      	/* import data to new tables*/
      IF (IS_TABLE_EXISTS(vschema, 'PROGRAM_CREDITS_FIRSTNAME') AND IS_TABLE_EXISTS(vschema, 'PROGRAM_CREDITS_LASTNAME')) THEN
	      vsql := 'INSERT INTO '||vschema||'.PROGRAM_CREDITS( PROGRAM_ID, ROLE, FIRST_NAME, LAST_NAME, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE) 
			 select pf.program_id, pf.role, pf.value, pl.value, pf.created_by, pf.updated_by, pf.create_date, pf.update_date 
			 from '||vschema||'.PROGRAM_CREDITS_FIRSTNAME pf, '||vschema||'.PROGRAM_CREDITS_LASTNAME pl 
			 where pf.PROGRAM_ID = pl.PROGRAM_ID AND pf.ROLE = pl.ROLE';
          CREATEOBJECT(vsql,'import data to TABLE PROGRAM_CREDITS', 'Y');
      END IF;
		  
      IF (IS_TABLE_EXISTS(vschema, 'PROGRAM_CAST_FIRSTNAME') AND IS_TABLE_EXISTS(vschema, 'PROGRAM_CAST_LASTNAME')) THEN
	      vsql := 'INSERT INTO '||vschema||'.PROGRAM_CAST( PROGRAM_ID, ROLE, FIRST_NAME, LAST_NAME, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE) 
			select pf.program_id, pf.role, pf.value, pl.value, pf.created_by, pf.updated_by, pf.create_date, pf.update_date 
			from '||vschema||'.PROGRAM_CAST_FIRSTNAME pf, '||vschema||'.PROGRAM_CAST_LASTNAME pl 
			where pf.PROGRAM_ID = pl.PROGRAM_ID AND pf.ROLE = pl.ROLE';
          CREATEOBJECT(vsql,'import data to TABLE ROVI_STG_PROGRAM_CAST', 'Y');
      END IF;

	/* delete the old tables*/
      dropcreditsandcasttables('%PROGRAM_CREDITS_%');
      dropcreditsandcasttables('%PROGRAM_CAST_%');
      
      /*update lineup tabls*/
      IF IS_TABLE_EXISTS(vschema,'LINEUP') THEN
          vsql := 'ALTER table '||vschema||'.LINEUP MODIFY (LINEUP_ID NVARCHAR2(100))';
          CREATEOBJECT(vsql,'modify columns in lineup from nvarchar2(20) to nvarchar2(100)','Y');
      END IF;     
      
      IF IS_TABLE_EXISTS(vschema,'STG_LINEUP') THEN
          vsql := 'ALTER table '||vschema||'.STG_LINEUP MODIFY (LINEUP_ID NVARCHAR2(100))';
          CREATEOBJECT(vsql,'modify columns in stg_lineup from nvarchar2(20) to nvarchar2(100)','Y');
      END IF;    
      
      IF IS_TABLE_EXISTS(vschema,'EXP_LINEUP') THEN
          vsql := 'ALTER table '||vschema||'.EXP_LINEUP MODIFY (LINEUP_ID NVARCHAR2(100))';
          CREATEOBJECT(vsql,'modify columns in exp_lineup from nvarchar2(20) to nvarchar2(100)','Y');
      END IF; 
      
      setInstalledDBVersion('1.1.000.4');
	  vsql := 'INSERT INTO install_Table values(sysdate,''EPGManager Version &EPG_DB_VERSION upgrade has completed SUCCESSFULLY.'')';
      CREATEOBJECT(vsql,'insert into install_table','Y');
    END IF;		
	
	
  END convertepgmanager;

 
  procedure installepgmanager AS
    vcounttab number := 0;
  begin
	SELECT COUNT(1)
    INTO 	vcounttab
	FROM 	dba_tables
	WHERE	OWNER = upper(vschema);

    IF vcounttab = 0 THEN
        	BEGIN
	            configxmldb();       
	        	createepgmanager();
       		EXCEPTION
            WHEN OTHERS THEN
            --	rollback;
              raise;
	        END;
	 else
	 	BEGIN
	 		convertepgmanager();
	 	EXCEPTION
	 	WHEN OTHERS THEN
	 		raise;
	 	END;
     END IF;

  end installepgmanager;

END EPGMANAGERDB;
/

begin
	
 epgmanagerdb.installepgmanager();
 
end;
/

exit
