create or replace
PACKAGE EPGEXPORT AS
   procedure exportepgdata(publisher_id number default -1, lowerDate varchar2 default '', upperDate varchar2 default '', channelIds varchar2 default '');
   
   procedure createxml( v_filename varchar2 default 'GuideDataOut.xml', publisher_id number default -1, format varchar2 default 'ETV_20');
   
  --Threshold time for schedule downloads
  vcurrentdatetime timestamp with time zone;
  const_nls_charset_id constant varchar2(100) := 'UTF-8';
  g_WriteBuf nvarchar2(32767);
  g_WriteBufSize number := 0;
  g_TimeToFlush boolean := false;
  
  const_COMPONENT_NAME_INGEST constant nvarchar2(100) := 'INGEST';
  const_COMPONENT_NAME_MERGE constant nvarchar2(100) := 'MERGE';
  const_COMPONENT_NAME_EXPORT constant nvarchar2(100) := 'EXPORT_TO_STG';
  const_COMPONENT_NAME_PUBLISH constant nvarchar2(100) := 'EXPORT_TO_XML';
 
  const_ALARM_SEVERITY_WARNING constant nvarchar2(100) := 'WARNING';
  const_ALARM_SEVERITY_ERROR constant nvarchar2(100) := 'ERROR';
  const_ALARM_SEVERITY_CRITICAL constant nvarchar2(100) := 'CRITICAL';
  const_ALARM_SEVERITY_INFO constant nvarchar2(100) := 'INFORMATION';
 
  CURSOR getheadends is select * from exp_HEADEND order by HEADEND_ID;
  CURSOR getheadendgeneric(head_id varchar2) is select * from exp_headend_generic where exp_headend_generic.headend_id = head_id order by Name;
  CURSOR getprograms is	select * from exp_PROGRAM order by PROGRAM_ID;
  CURSOR getlineups is select * from exp_LINEUP order by LINEUP_ID,STATION_ID,HEADEND_ID;
  CURSOR getlineupgeneric(line_id varchar2, head_id varchar2, stat_id varchar2) is select * from exp_lineup_generic where exp_lineup_generic.lineup_id = line_id and exp_lineup_generic.headend_id = head_id and exp_lineup_generic.station_id = stat_id order by Name;  
  CURSOR getschedules is select * from exp_SCHEDULE  order by station_id, program_id, Start_time;
  CURSOR getschedule(prog_id varchar2) is select * from exp_SCHEDULE where exp_SCHEDULE.program_id = prog_id order by station_id, program_id, Start_time;  
  CURSOR getstations is select * from exp_STATION order by station_id;
  CURSOR getstationgeneric(stat_id varchar2) is select * from exp_station_generic where exp_station_generic.station_id = stat_id order by Name;  
  CURSOR getscheduleeventids(prog_id varchar2) is select * from exp_SCHEDULE_EVENT_ID where exp_SCHEDULE_EVENT_ID.program_id = prog_id order by Identifier_Type;
  CURSOR getschedulegeneric(prog_id varchar2) is select * from exp_SCHEDULE_GENERIC where exp_SCHEDULE_GENERIC.program_id = prog_id order by Name;
  CURSOR getscheduleseriesattribute(prog_id varchar2) is select * from exp_SCHEDULE_SERIES_ATTRIB where exp_SCHEDULE_SERIES_ATTRIB.program_id = prog_id order by Attribute_Type;
  CURSOR getprogramcast(prog_id varchar2) is select pc.first_name as FIRST_NAME, pc.last_name as LAST_NAME, pc.ROLE as ROLE from exp_PROGRAM_CAST pc WHERE pc.Program_ID = prog_id order by pc.seq;
  CURSOR getprogramcredits(prog_id varchar2) is select pc.first_name as FIRST_NAME, pc.last_name as LAST_NAME, pc.ROLE as ROLE from exp_PROGRAM_CREDITS pc WHERE pc.Program_ID = prog_id order by pc.seq;
  CURSOR getprogramrating(prog_id varchar2) is select * from exp_PROGRAM_RATING where exp_PROGRAM_RATING.program_id = prog_id order by Authority;
  CURSOR getprogramaudiotl(prog_id varchar2) is select unique Type, Language from exp_PROGRAM_AUDIO_COMP where exp_PROGRAM_AUDIO_COMP.program_id = prog_id order by Type,Language;
  CURSOR getprogramaudio(prog_id varchar2, t varchar2, l varchar2 ) is select * from exp_PROGRAM_AUDIO_COMP where exp_PROGRAM_AUDIO_COMP.program_id = prog_id and exp_PROGRAM_AUDIO_COMP.Type=t and exp_PROGRAM_AUDIO_COMP.Language=l order by Attribute_Type;
  CURSOR getprogramsubtitletl(prog_id varchar2) is select unique Type, Language from exp_PROGRAM_SUBTITLE_COMP where exp_PROGRAM_SUBTITLE_COMP.program_id = prog_id order by Type,Language;
  CURSOR getprogramsubtitle(prog_id varchar2, t varchar2, l varchar2 ) is select * from exp_PROGRAM_SUBTITLE_COMP where exp_PROGRAM_SUBTITLE_COMP.program_id = prog_id and exp_PROGRAM_subtitle_COMP.Type=t and exp_PROGRAM_SUBTITLE_COMP.Language=l order by Attribute_Type;
  CURSOR getprogramgeneric(prog_id varchar2) is select * from exp_PROGRAM_GENERIC where exp_PROGRAM_GENERIC.program_id = prog_id order by Name;

  END EPGEXPORT;
  /
 create or replace
PACKAGE BODY EPGEXPORT AS

FUNCTION IS_TABLE_EXISTS (IN_TABLE_NAME VARCHAR2) RETURN boolean is
counter number(5) := 0;
begin
  select count(1) into counter from user_tables where TABLE_NAME=UPPER(IN_TABLE_NAME);
  if counter > 0  then
    return true;
  else
  return false;
  end if;
end;
--Autonomous transaction procedure
  procedure autonomoustransaction(vsql varchar2,errmsg varchar2 default NULL, mask varchar2 default 'N') is
  PRAGMA AUTONOMOUS_TRANSACTION;
    verrmsg varchar2(32000);
  begin
    execute immediate vsql;
    commit;
  exception
    when others then
          verrmsg := errmsg ||' '||sqlerrm;
          select cast(verrmsg as varchar2(4000))
          into verrmsg
          from dual;
      if upper(mask) != 'Y' then
          execute immediate 'INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')';
          commit;
          raise;
      end if;
  end;
  
  
  procedure epgguidedatalogging(vsql varchar2) is
  begin
    autonomoustransaction('insert into epgguidedata_log(description) values ('''||vsql||''')');
  end;


  /* Procedure: CreateObject. Executes a dynamic SQL statement and logs error in install_table*/
  PROCEDURE CREATEOBJECT(vsql varchar2,errmsg varchar2, mask varchar2 default 'N') IS
    verrmsg varchar2(32000);
  BEGIN
--    dbms_output.put_line(vsql);
    execute immediate vsql;
  exception
    when others then
          verrmsg := errmsg ||' '||sqlerrm||' '||vsql;
          select cast(verrmsg as varchar2(4000))
          into verrmsg
          from dual;
      if upper(mask) != 'Y' then
          autonomoustransaction ('INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')');
          raise;
      end if;
  END CREATEOBJECT;
  
  /* Procedure: SetDbCreateFileDest. Sets the default directory for Oracle datafiles so the path
              need not be given when creating tablespaces. */
  PROCEDURE SETDBCREATEFILEDEST(vpath varchar2) IS
  BEGIN
    execute immediate 'alter system set db_create_file_dest='''||vpath||''' scope=both';
  END SETDBCREATEFILEDEST;
  
  -- Create Alarms
  procedure alarmactivitylogging(vdesc nvarchar2,
  				vcomponent nvarchar2 default const_COMPONENT_NAME_INGEST, 
  				vseverity nvarchar2 default const_ALARM_SEVERITY_INFO) is
    vsql varchar2(32767);
  begin
      vsql := 'insert into alarm_activitylog (ID,SEVERITY,COMPONENT,DESCRIPTION,CREATED_BY,UPDATED_BY)
                select alarm_activitylog_seq.nextval,'''||vseverity||''','''||vcomponent||''',
                      substr('''||vdesc||''',1,2000),''SYSTEM'',''SYSTEM''
                      from dual';
      --dbms_output.put_line(vsql);
     autonomoustransaction(vsql);
  end alarmactivitylogging;
  
   /* Procedure: Populate variables*/
  PROCEDURE init is
  BEGIN
    begin
      select current_timestamp + numtodsinterval(nvl(value,0),'SECOND')
      into vcurrentdatetime
      from config_setting, dual
      where key = 'EPG_Schedule_Delay_Secs';
    exception
      when others then
        select current_timestamp
        into vcurrentdatetime
        from dual;
    end;
   
  END init;
  
    procedure delete_leftover_files(v_filename varchar2) is

	  type vtabdeletefilename is table of varchar2(4000) INDEX BY PLS_INTEGER;

	  vdeletefilenames vtabdeletefilename;

			begin

			  select any_path

			  bulk collect into vdeletefilenames

			  from resource_view

			  where any_path like '/public/%'

			  and any_path != '/public/'||v_filename;



			  if vdeletefilenames.count > 0 then

			     FOR indx IN 1 .. vdeletefilenames.COUNT

			     LOOP

			      dbms_xdb.deleteresource(vdeletefilenames(indx),dbms_xdb.delete_recursive_force);

			     END LOOP;

			  end if;

			end;

	-- support methods for createXML
  procedure fflush(f utl_file.file_type) is
  BEGIN
	  -- write out what you have.
	  if g_WriteBufSize > 0 then
	  	utl_file.put_line_nchar(f,g_WriteBuf);
	  	g_WriteBufSize := 0;
	  	g_WriteBuf := '';
	  end if;
  END fflush;		
			
  procedure fWrite(f utl_file.file_type, buf nvarchar2 ) is
  	bufSize number := LENGTHB(buf);
  BEGIN
	 if g_WriteBufSize + bufSize > 28000 then
	  	-- time to flush
	  	g_TimeToFlush := true;
	 end if;
	 
	 if g_TimeToFlush and SUBSTR(g_WriteBuf,-1,1) = '>' then
	 	-- flush
	  	utl_file.put_line_nchar(f,g_WriteBuf);
	  	g_WriteBufSize := bufSize;
	  	g_WriteBuf := buf;
	 	g_TimeToFlush := false;
	 
	 else
	 	-- append to string
	  	g_WriteBufSize := g_WriteBufSize + bufSize;
	  	g_WriteBuf := g_WriteBuf||buf; 
	 end if;
  END fWrite;
  
  procedure fWriteBTag(f utl_file.file_type, buf nvarchar2) is
  BEGIN
	  -- begin an XML tag - i.e. make the <tag> open
	  fWrite(f,'<'||buf||'>');
  END fWriteBTag;
  
  procedure fWriteETag(f utl_file.file_type, buf nvarchar2) is
  BEGIN
	  -- end an XML tag - i.e. make the </tag> end
	  fWrite(f,'</'||buf||'>');
  END fWriteETag;
  
 procedure fWriteTag(f utl_file.file_type,
  						tag varchar2,
  						value nvarchar2,
  						attribTag1 nvarchar2 default '',
  						attribValue1 nvarchar2 default '',
  						attribTag2 nvarchar2 default '',
  						attribValue2 nvarchar2 default ''
  						) is
  begin
	  if value is not null then
	  	-- put in <ta
	  	fWrite(f,'<'||tag);
	  	
    	if attribTag1 is not null then
    		-- put in  attribTag1="attribValue1"
    		fWrite(f,' '||attribTag1||'="'||htf.escape_sc(attribValue1)||'"');
    	end if;
    	if attribTag2 is not null then
    		-- put in  attribTag2="attribValue2"
    		fWrite(f,' '||attribTag2||'="'||htf.escape_sc(attribValue2)||'"');
    	end if;
    	fWrite(f,'>'||htf.escape_sc(value)||'</'||tag||'>');

     end if;
     
  end fWriteTag;
  		
 
  procedure exportepg(publisherName nvarchar2, lowerDate varchar2, upperDate varchar2, channelIds varchar2) is
    vsql varchar2(32767);
    vcurrentdatetimeforlog timestamp with time zone;
    minutesittook varchar2(10);
    secondsittook varchar2(10);
    hoursittook varchar2(10);
    filterByChannels boolean := channelIds is not null;
    channelFilter varchar2(32767);
    filterByDate boolean := lowerDate is not null;
    dateFilter varchar2(200);
  begin

	  channelFilter := '('||channelIds||')';
	  dateFilter := ' s.START_TIME >= '''||lowerDate||''' and s.START_TIME <= '''||upperDate||'''';
	  	  
   --Logging start of load to export tables
     vcurrentdatetimeforlog := current_timestamp;

	 epgguidedatalogging('Start of EPG Export. StartTime = '||current_timestamp);
	 
     alarmactivitylogging('Starting EPG export for publisher '||publisherName, const_COMPONENT_NAME_EXPORT);


	 /* wipe export tables and load from production based on desired filters of publisher
	  * 
	  */
	 
    vsql := 'truncate table exp_headend';
    CREATEOBJECT(vsql,'truncate table EXP_HEADEND');

    vsql := 'truncate table err$_exp_headend';
    CREATEOBJECT(vsql,'truncate table err$_exp_headend');

    epgguidedatalogging('Start of Export Headend load. ');

    vsql := 'INSERT INTO exp_headend (
    HEADEND_ID,
    COMMUNITY_NAME,
    COUNTY_NAME,
    COUNTY_SIZE,
    ST_COUNTY_CODE,
    STATE,
    ZIPCODE,
    DMA_CODE,
    DMA_NAME,
    MSO_CODE,
    DMA_RANK,
    HEADEND_NAME,
    HEADEND_LOCATION,
    MSO_NAME,
    TIME_ZONE)
    SELECT 
    unique h.HEADEND_ID,
    h.COMMUNITY_NAME,
    h.COUNTY_NAME,
    h.COUNTY_SIZE,
    h.ST_COUNTY_CODE,
    h.STATE,
    h.ZIPCODE,
    h.DMA_CODE,
    h.DMA_NAME,
    h.MSO_CODE,
    h.DMA_RANK,
    h.HEADEND_NAME,
    h.HEADEND_LOCATION,
    h.MSO_NAME,
    h.TIME_ZONE FROM headend h';
	if filterByChannels then
		vsql := vsql||', lineup l where h.HEADEND_ID = l.HEADEND_ID and l.STATION_ID in '||channelFilter;
	end if;
    vsql:=vsql||' log errors into err$_exp_headend reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO EXP_HEADEND');
    commit;

    vsql := 'truncate table exp_headend_generic';
    CREATEOBJECT(vsql,'truncate table EXP_HEADEND_GENERIC');

    vsql := 'truncate table err$_exp_headend_generic';
    CREATEOBJECT(vsql,'truncate table err$_exp_headend_generic');

    epgguidedatalogging('Start of Export Headend_Generic load. ');

    vsql := 'INSERT INTO exp_headend_generic (
    HEADEND_ID,
	ASSET,
	NAME,
	VALUE)
    SELECT 
    hg.HEADEND_ID,
	hg.ASSET,
	hg.NAME,
	hg.VALUE
    FROM headend_generic hg';
	if filterByChannels then
		vsql := vsql||', lineup l where hg.HEADEND_ID = l.HEADEND_ID and l.STATION_ID in '||channelFilter;
	end if;
    vsql:=vsql||' log errors into err$_exp_headend_generic reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO EXP_HEADEND_GENERIC');
    commit;
    
    vsql := 'truncate table exp_station';
    CREATEOBJECT(vsql,'truncate table EXP_STATION');

    vsql := 'truncate table err$_exp_station';
    CREATEOBJECT(vsql,'truncate table err$_exp_station');
    
     /* Station */
     epgguidedatalogging('Start of export Station');

    vsql := 'insert into exp_station (
	 station_id,
	 station_name, 
	 time_zone, 
	 call_sign, 
	 city, 
	 state, 
	 zipcode, 
	 country, 
	 dma_name, 
	 dma_number, 
	 fcc_channel_num, 
	 affiliation,
     USER_DATA1,
     USER_DATA2,
     USER_DATA3,
     USER_DATA4,
     USER_DATA5,
     USER_DATA6,
     USER_DATA7,
     USER_DATA8,
     USER_DATA9,
     USER_DATA10 )
    	select 
		station_id,
		station_name, 
		time_zone, 
		call_sign, 
		city, 
		state,
		zipcode,
		country, 
		dma_name, 
		dma_number, 
		fcc_channel_num,
		affiliation,
     	USER_DATA1,
     	USER_DATA2,
     	USER_DATA3,
     	USER_DATA4,
     	USER_DATA5,
     	USER_DATA6,
     	USER_DATA7,
     	USER_DATA8,
     	USER_DATA9,
     	USER_DATA10
    	from STATION';
	if filterByChannels then
		vsql := vsql||' where station_id in '||channelFilter;
	end if;
	vsql := vsql||'  log errors into err$_exp_station reject limit unlimited';
    CREATEOBJECT(vsql,'EXPORT FROM STATION');

    commit;
    
        vsql := 'truncate table exp_station_generic';
    CREATEOBJECT(vsql,'truncate table exp_station_generic');

    vsql := 'truncate table err$_exp_station_generic';
    CREATEOBJECT(vsql,'truncate table err$_exp_station_generic');

    epgguidedatalogging('Start of Export Station_Generic load. ');

    vsql := 'INSERT INTO exp_station_generic (
    STATION_ID,
	ASSET,
	NAME,
	VALUE)
    SELECT 
    STATION_ID,
	ASSET,
	NAME,
	VALUE
    FROM station_generic ';
	if filterByChannels then
		vsql := vsql||' where STATION_ID in '||channelFilter;
	end if;
    vsql:=vsql||' log errors into err$_exp_station_generic reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO EXP_STATION_GENERIC');
    commit;
    
    vsql := 'truncate table exp_lineup';
    CREATEOBJECT(vsql,'truncate table exp_lineup');

    vsql := 'truncate table err$_exp_lineup';
    CREATEOBJECT(vsql,'truncate table err$_exp_lineup');

/* Lineup */
     epgguidedatalogging('Start of Export Lineup');

    vsql := 'insert into exp_lineup (
	lineup_id,
	device, 
	tms_channel, 
	service_tier, 
	effective_date, 
	expiration_date, 
	headend_id, 
	station_id
	)
	select
	lineup_id,
	device,
 	tms_channel, 
	service_tier, 
	effective_date, 
	expiration_date,
 	headend_id, 
	station_id
	from LINEUP';
	if filterByChannels then
		vsql := vsql||' where station_id in '||channelFilter;
	end if;
	vsql := vsql||' log errors into err$_exp_lineup reject limit unlimited';
    CREATEOBJECT(vsql,'EXPORT FROM LINEUP');

    commit;
    
    vsql := 'truncate table exp_lineup_generic';
    CREATEOBJECT(vsql,'truncate table exp_lineup_generic');

    vsql := 'truncate table err$_exp_lineup_generic';
    CREATEOBJECT(vsql,'truncate table err$_exp_lineup_generic');

    epgguidedatalogging('Start of Export Lineup_Generic load. ');

    vsql := 'INSERT INTO exp_lineup_generic (
	LINEUP_ID,
	HEADEND_ID,
    STATION_ID,
	ASSET,
	NAME,
	VALUE)
    SELECT 
	LINEUP_ID,
	HEADEND_ID,
    STATION_ID,
	ASSET,
	NAME,
	VALUE
    FROM lineup_generic ';
	if filterByChannels then
		vsql := vsql||' where STATION_ID in '||channelFilter;
	end if;
    vsql:=vsql||' log errors into err$_exp_lineup_generic reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO EXP_LINEUP_GENERIC');
    commit;
    
    
    vsql := 'truncate table exp_program';
    CREATEOBJECT(vsql,'truncate table exp_program');

    vsql := 'truncate table err$_exp_program';
    CREATEOBJECT(vsql,'truncate table err$_exp_program');
    
     /*Program */
     epgguidedatalogging('Start of export Program');

    vsql := 'insert into EXP_PROGRAM (
    PROGRAM_ID,
    LANGUAGE,
    TITLE,
    REDUCED_TITLE1,
    REDUCED_TITLE2,
    REDUCED_TITLE3,
    REDUCED_TITLE4,
    ALT_TITLE,
    REDUCED_DESCRIPTION1,
    REDUCED_DESCRIPTION2,
    REDUCED_DESCRIPTION3,
    REDUCED_DESCRIPTION4,
    ADVISORY_DESC1,
    ADVISORY_DESC2,
    ADVISORY_DESC3,
    ADVISORY_DESC4,
    ADVISORY_DESC5,
    ADVISORY_DESC6,
    GENRE_DESC1,
    GENRE_DESC2,
    GENRE_DESC3,
    GENRE_DESC4,
    GENRE_DESC5,
    GENRE_DESC6,
    DESCRIPTION1,
    DESCRIPTION2,
    YEAR_PROGRAM,
    MPAA_RATING,
    STAR_RATING,
    RUN_TIME,
    COLOR_CODE,
    PROGRAM_LANGUAGE,
    COUNTRY_OF_ORIGIN,
    MADE_FOR_TV,
    SOURCE_TYPE,
    SHOW_TYPE,
    HOLIDAY,
    SYNDICATE_EPISODE_NUM,
    ALT_SYNDICATE_EPI_NUM,
    EPISODE_TITLE,
    NET_SYN_SOURCE,
    NET_SYN_TYPE,
    ORG_STUDIO,
    GAME_DATETIME,
    GAME_TIMEZONE,
    ORG_AIR_DATE,
    UNIQUE_ID,
    USER_DATA1,
    USER_DATA2,
    USER_DATA3,
    USER_DATA4,
    USER_DATA5,
    USER_DATA6,
    USER_DATA7,
    USER_DATA8,
    USER_DATA9,
    USER_DATA10,
    USER_DATA11,
    USER_DATA12,
    USER_DATA13
    )
    SELECT  
    unique p.program_id,
    p.LANGUAGE,
    p.TITLE,
    p.REDUCED_TITLE1,
    p.REDUCED_TITLE2,
    p.REDUCED_TITLE3,
    p.REDUCED_TITLE4,
    p.ALT_TITLE,
    p.REDUCED_DESCRIPTION1,
    p.REDUCED_DESCRIPTION2,
    p.REDUCED_DESCRIPTION3,
    p.REDUCED_DESCRIPTION4,
    p.ADVISORY_DESC1,
    p.ADVISORY_DESC2,
    p.ADVISORY_DESC3,
    p.ADVISORY_DESC4,
    p.ADVISORY_DESC5,
    p.ADVISORY_DESC6,
    p.GENRE_DESC1,
    p.GENRE_DESC2,
    p.GENRE_DESC3,
    p.GENRE_DESC4,
    p.GENRE_DESC5,
    p.GENRE_DESC6,
    p.DESCRIPTION1,
    p.DESCRIPTION2,
    p.YEAR_PROGRAM,
    p.MPAA_RATING,
    p.STAR_RATING,
    p.RUN_TIME,
    p.COLOR_CODE,
    p.PROGRAM_LANGUAGE,
    p.COUNTRY_OF_ORIGIN,
    p.MADE_FOR_TV,
    p.SOURCE_TYPE,
    p.SHOW_TYPE,
    p.HOLIDAY,
    p.SYNDICATE_EPISODE_NUM,
    p.ALT_SYNDICATE_EPI_NUM,
    p.EPISODE_TITLE,
    p.NET_SYN_SOURCE,
    p.NET_SYN_TYPE,
    p.ORG_STUDIO,
    p.GAME_DATETIME,
    p.GAME_TIMEZONE,
    p.ORG_AIR_DATE,
    p.UNIQUE_ID,
    p.USER_DATA1,
    p.USER_DATA2,
    p.USER_DATA3,
    p.USER_DATA4,
    p.USER_DATA5,
    p.USER_DATA6,
    p.USER_DATA7,
    p.USER_DATA8,
    p.USER_DATA9,
    p.USER_DATA10,
    p.USER_DATA11,
    p.USER_DATA12,
    p.USER_DATA13
    from PROGRAM p';
    
    if filterByChannels OR filterByDate then
    	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
    end if;
    
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    
    vsql := vsql||' log errors into err$_exp_program reject limit unlimited';    
    
    CREATEOBJECT(vsql,'EXPORT FROM PROGRAM');

    commit;
   
    vsql := 'truncate table exp_program_rating';
    CREATEOBJECT(vsql,'truncate table exp_program_rating');

    vsql := 'truncate table err$_exp_program_rating';
    CREATEOBJECT(vsql,'truncate table err$_exp_program_rating');
   
 	epgguidedatalogging('Start of export Program_rating');    
    
    -- merge into program_rating
	vsql:='insert into exp_program_rating (
    	program_id,
    	authority,
    	value
    )
    select 
    	unique p.program_id,
    	p.authority,
    	p.value
	from PROGRAM_RATING p';
	
    if filterByChannels OR filterByDate then
    	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
    end if;
    
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    vsql := vsql||' log errors into err$_exp_program_rating reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM PROGRAM_Rating');
	
    vsql := 'truncate table exp_program_audio_comp';
    CREATEOBJECT(vsql,'truncate table exp_program_audio_comp');

    vsql := 'truncate table err$_exp_program_audio_comp';
    CREATEOBJECT(vsql,'truncate table err$_exp_program_audio_comp');
   
  	epgguidedatalogging('Start of export Program_Audio_Comp');    
    
    vsql:= 'insert into exp_program_audio_comp (
    	program_id,
		tla,
    	type,
    	language,
		attribute_type,
    	value
    )
    select 
    	unique p.program_id,
		p.tla,
    	p.type,
    	p.language,
		p.attribute_type,
    	p.value
    from PROGRAM_AUDIO_COMP p';
	
    if filterByChannels OR filterByDate then
    	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
    end if;
    
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    vsql := vsql||' log errors into err$_exp_program_audio_comp reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM PROGRAM_AUDIO_COMP');
    
    vsql := 'truncate table exp_program_subtitle_comp';
    CREATEOBJECT(vsql,'truncate table exp_program_subtitle_comp');

    vsql := 'truncate table err$_exp_program_subtitle_comp';
    CREATEOBJECT(vsql,'truncate table err$_exp_program_subtitle_comp');
	
  	epgguidedatalogging('Start of export Program_Subtitle_Comp');    
    
    vsql:= 'insert into exp_program_subtitle_comp (
    	program_id,
		tla,
    	type,
    	language,
		attribute_type,
    	value
    )
    select 
    	unique p.program_id,
		p.tla,
    	p.type,
    	p.language,
		p.attribute_type,
    	p.value
    from PROGRAM_SUBTITLE_COMP p';
	
    if filterByChannels OR filterByDate then
    	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
    end if;
    
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    vsql := vsql||' log errors into err$_exp_program_subtitle_comp reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM PROGRAM_SUBTITLE_COMP');    
   
    IF IS_TABLE_EXISTS('exp_program_cast') THEN
      vsql := 'truncate table exp_program_cast';
      CREATEOBJECT(vsql,'truncate table exp_program_cast');
    END IF;

    IF IS_TABLE_EXISTS('exp_program_cast') THEN
      vsql := 'truncate table err$_exp_program_cast';
      CREATEOBJECT(vsql,'truncate table err$_exp_program_cast');
    END IF;
    
    IF (IS_TABLE_EXISTS('EXP_PROGRAM_CAST') AND IS_TABLE_EXISTS('PROGRAM_CAST') AND IS_TABLE_EXISTS('err$_exp_program_cast')) THEN
       epgguidedatalogging('Start of export Program_Cast');    
       
       vsql :='insert into EXP_PROGRAM_CAST
       (
       	program_id,
       	c_id,
       	seq,
       	role,
       	first_name,
       	middle_name,
       	last_name
       )
       select     
       	unique p.program_id,
       	p.c_id,
       	p.seq,
       	p.role,
       	p.first_name,
       	p.middle_name,
       	p.last_name
       FROM PROGRAM_CAST p';
       
       if filterByChannels OR filterByDate then
       	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
       end if;
       
       if filterByChannels then
       	vsql := vsql||' and s.station_id in '||channelFilter;
       end if;
       
       if filterByDate then
       	vsql := vsql||' and '||dateFilter;
       end if;
       vsql := vsql||' log errors into err$_exp_program_cast reject limit unlimited';
       
       CREATEOBJECT(vsql,'EXPORT FROM PROGRAM_CAST'); 
    END IF;
    
    IF IS_TABLE_EXISTS('exp_program_credits') THEN
      vsql := 'truncate table exp_program_credits';
      CREATEOBJECT(vsql,'truncate table exp_program_credits');
    END IF;

    IF IS_TABLE_EXISTS('err$_exp_program_credits') THEN  
      vsql := 'truncate table err$_exp_program_credits';
      CREATEOBJECT(vsql,'truncate table err$_exp_program_credits');
    END IF;
    
    IF (IS_TABLE_EXISTS('EXP_PROGRAM_CREDITS') AND IS_TABLE_EXISTS('PROGRAM_CREDITS') AND IS_TABLE_EXISTS('err$_exp_program_credits')) THEN
       epgguidedatalogging('Start of export Program_Credits');    
       
       vsql :='insert into EXP_PROGRAM_CREDITS
       (
       	program_id,
       	c_id,
       	seq,
       	role,
       	first_name,
       	middle_name,
       	last_name
       )
       select     
       	unique p.program_id,
       	p.c_id,
       	p.seq,
       	p.role,
       	p.first_name,
       	p.middle_name,
       	p.last_name
       FROM PROGRAM_CREDITS p';
       
       if filterByChannels OR filterByDate then
       	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
       end if;
       
       if filterByChannels then
       	vsql := vsql||' and s.station_id in '||channelFilter;
       end if;
       
       if filterByDate then
       	vsql := vsql||' and '||dateFilter;
       end if;
       vsql := vsql||' log errors into err$_exp_program_credits reject limit unlimited';
       
       CREATEOBJECT(vsql,'EXPORT FROM PROGRAM_CREDITS');  
    END IF;
    
    vsql := 'truncate table exp_program_generic';
    CREATEOBJECT(vsql,'truncate table exp_program_generic');

    vsql := 'truncate table err$_exp_program_generic';
    CREATEOBJECT(vsql,'truncate table err$_exp_program_generic');    
    
 	epgguidedatalogging('Start of export Program_Generic ');    
    
 	vsql := 'insert into EXP_PROGRAM_GENERIC
	(
    	program_id,
    	asset,
    	name,
    	value
    )
    select    
    	unique p.program_id,
    	p.asset,
    	p.name,
    	p.value
	FROM PROGRAM_GENERIC p';
	
    if filterByChannels OR filterByDate then
    	vsql:=vsql||', SCHEDULE s where p.program_id = s.program_id ';
    end if;
    
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    vsql := vsql||' log errors into err$_exp_program_generic reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM PROGRAM_GENERIC');     
    
    vsql := 'truncate table exp_schedule';
    CREATEOBJECT(vsql,'truncate table exp_schedule');

    vsql := 'truncate table err$_exp_schedule';
    CREATEOBJECT(vsql,'truncate table err$_exp_schedule');
        
    --SCHEDULE MERGE
   epgguidedatalogging('Start of export Schedule');
    
	vsql :='insert into EXP_SCHEDULE(
          SCHEDULE_LANGUAGE,
          STATION_ID,
          PROGRAM_ID,
          START_TIME,
          END_TIME,
          DURATION,
          TV_RATING,
          HD_TV,
          PART_NUMBER,
          NO_OF_PARTS,
          CC,
          STEREO,
          LIVE_TAPE_DELAY,
          SUBTITLED,
          PREMIERE_FINALE,
          JOINED_IN_PROGRESS,
          CABLE_IN_CLASSROOM,
          SAP,
          SEX_RATING,
          VIOLENCE_RATING,
          LANGUAGE_RATING,
          DIALOG_RATING,
          FV_RATING,
          ENHANCED,
          THREE_D,
          LETTERBOX,
          DVS,
          series_id,
          REQUEST_RECORD,
          REPEAT,
          BLACKOUT,
          DOLBY,
          USER_DATA1,
          USER_DATA2,
          USER_DATA3,
          USER_DATA4,
          USER_DATA5,
          USER_DATA6,
          USER_DATA7,
          USER_DATA8,
          USER_DATA9,
          USER_DATA10,
          USER_DATA11
			)
		  select 
          SCHEDULE_LANGUAGE,
          STATION_ID,
          PROGRAM_ID,
          start_time,
          end_time,
          DURATION,
          TV_RATING,
          HD_TV,
          PART_NUMBER,
          NO_OF_PARTS,
          CC,
          STEREO,
          LIVE_TAPE_DELAY,
          SUBTITLED,
          PREMIERE_FINALE,
          JOINED_IN_PROGRESS,
          CABLE_IN_CLASSROOM,
          SAP,
          SEX_RATING,
          VIOLENCE_RATING,
          LANGUAGE_RATING,
          DIALOG_RATING,
          FV_RATING,
          ENHANCED,
          THREE_D,
          LETTERBOX,
          DVS,
          series_id,
          REQUEST_RECORD,
          REPEAT,
          BLACKOUT,
          DOLBY,
          USER_DATA1,
          USER_DATA2,
          USER_DATA3,
          USER_DATA4,
          USER_DATA5,
          USER_DATA6,
          USER_DATA7,
          USER_DATA8,
          USER_DATA9,
          USER_DATA10,
          USER_DATA11
	FROM SCHEDULE';
	
	if filterByChannels OR filterByDate then
		vsql := vsql||' s where ';
	end if;
    if filterByChannels then
    	vsql := vsql||' s.station_id in '||channelFilter;
    	if filterByDate then
    		vsql := vsql||' and ';
    	end if;
    end if;
    if filterByDate then
    	vsql := vsql||dateFilter;
    end if;
    
    vsql := vsql||' log errors into err$_exp_schedule reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM SCHEDULE');     
          
    vsql := 'truncate table exp_schedule_event_id';
    CREATEOBJECT(vsql,'truncate table exp_schedule_event_id');

    vsql := 'truncate table err$_exp_schedule_event_id';
    CREATEOBJECT(vsql,'truncate table err$_exp_schedule_event_id');
    

	epgguidedatalogging('Start of export Schedule_Event_Id');    
    
     vsql :='insert into exp_schedule_event_id (
    	program_id,
    	identifier_type,
    	value
    )
    select     
    	p.program_id,
    	p.identifier_type,
    	p.value
 	FROM SCHEDULE_EVENT_ID p';
	
	if filterByChannels OR filterByDate then
		vsql := vsql||', SCHEDULE s where p.program_id = s.program_id ';
	end if;
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    
    vsql := vsql||' log errors into err$_exp_schedule_EVENT_ID reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM SCHEDULE_EVENT_ID');     

    vsql := 'truncate table exp_schedule_series_attrib';
    CREATEOBJECT(vsql,'truncate table exp_schedule_series_attrib');

    vsql := 'truncate table err$_exp_schedule_series_attri';
    CREATEOBJECT(vsql,'truncate table err$_exp_schedule_series_attri');
    
	epgguidedatalogging('Start of export Schedule_Series_Attrib');    
    
	vsql := 'insert into exp_schedule_series_attrib (
    	program_id,
    	attribute_type,
    	value
    )
    select     
    	p.program_id,
    	p.attribute_type,
    	p.value
	FROM SCHEDULE_SERIES_ATTRIB p';
	
	if filterByChannels OR filterByDate then
		vsql := vsql||', SCHEDULE s where p.program_id = s.program_id ';
	end if;
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    
    vsql := vsql||' log errors into err$_exp_schedule_series_attri reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM SCHEDULE_SERIES_ATTRIB');     

    vsql := 'truncate table exp_schedule_generic';
    CREATEOBJECT(vsql,'truncate table exp_schedule_generic');

    vsql := 'truncate table err$_exp_schedule_generic';
    CREATEOBJECT(vsql,'truncate table err$_exp_schedule_generic');
    
 	epgguidedatalogging('Start of export Schedule_Generic ');    
    
 	vsql := 'insert into EXP_Schedule_GENERIC
	(
    	program_id,
    	asset,
    	name,
    	value
    )
    select    
    	p.program_id,
    	p.asset,
    	p.name,
    	p.value
	FROM Schedule_GENERIC p';
	
	if filterByChannels OR filterByDate then
		vsql := vsql||', SCHEDULE s where p.program_id = s.program_id ';
	end if;
    if filterByChannels then
    	vsql := vsql||' and s.station_id in '||channelFilter;
    end if;
    if filterByDate then
    	vsql := vsql||' and '||dateFilter;
    end if;
    
    vsql := vsql||' log errors into err$_exp_Schedule_generic reject limit unlimited';
    
    CREATEOBJECT(vsql,'EXPORT FROM Schedule_GENERIC');     
   
   SELECT floor(Extract(Minute from (current_timestamp - vcurrentdatetimeforlog))) into minutesittook from dual;
   SELECT floor(Extract(Second from (current_timestamp - vcurrentdatetimeforlog))) into secondsittook from dual;    
   SELECT floor(Extract(Hour from (current_timestamp - vcurrentdatetimeforlog))) into hoursittook from dual;
   
   epgguidedatalogging('End of Export Load . It took '||hoursittook||' hours and '||LPAD(minutesittook,2,'0')||' minutes and '||LPAD(secondsittook,2,'0')||' seconds');
   
   alarmactivitylogging('Finished EPG export for publisher '||publisherName||' (took '||hoursittook||':'||LPAD(minutesittook,2,'0')||':'||LPAD(secondsittook,2,'0')||')', const_COMPONENT_NAME_EXPORT);
       
 end exportepg;
  
function XSDDTValue(datetime TIMESTAMP with TIME ZONE) return nvarchar2 as
begin
	return TO_CHAR(datetime, 'YYYY-MM-DD"T"HH24:MI:SS');
end;

function XSDDurValue(dur NUMBER) return nvarchar2 as
begin
	return to_char(floor(dur/(60*60)),'FM9900')||to_char( mod( dur , (60*60) )/60,'FM00')||to_char(mod( dur, 60),'FM00');
end;

procedure fWriteScheduleTag(f utl_file.file_type, sr getSchedules%ROWTYPE) is
begin
	
		fWriteBTag(f,'row');
    
    	fWriteTag(f, 'STATION_NUM',sr.STATION_ID);
    	fWriteTag(f, 'DATABASE_KEY',sr.PROGRAM_ID);
    	fWriteTag(f, 'LANGUAGE', sr.SCHEDULE_LANGUAGE);
    	fWriteTag(f, 'AIR_DATE', XSDDTValue(sr.START_TIME));
    	fWriteTag(f, 'DURATION', XSDDurValue(sr.DURATION));
    
    -- Schedule.Event_ID  
   		for sei IN getscheduleeventids(sr.PROGRAM_ID) LOOP
    		fWriteTag(f, 'EVENT_ID', sei.value, 'IdentifierType', sei.Identifier_Type);
    	END LOOP;
    
    	fWriteTag(f, 'PART_NUM', sr.PART_NUMBER);
    	fWriteTag(f, 'NUM_OF_PARTS', sr.NO_OF_PARTS);
    
    -- SERIES_LINK - build up manually
    	if sr.Series_ID is not null then
    		fWriteBTag(f,'SERIES_LINK');
    		fWriteTag(f, 'SeriesIdentifier', sr.Series_ID);
	  		for sar IN getscheduleseriesattribute(sr.PROGRAM_ID) LOOP
  					fWriteTag(f, 'SeriesAttribute', sar.value, 'AttributeType', sar.Attribute_Type);
  			END LOOP;
  			fWriteETag(f,'SERIES_LINK');
  		end if;
    
    	fWriteTag(f, 'CC', sr.CC);
    	fWriteTag(f, 'STEREO', sr.STEREO);
    	fWriteTag(f, 'REPEAT', sr.REPEAT);
    	fWriteTag(f, 'LIVE_TAPE_DELAY', sr.LIVE_TAPE_DELAY);
    	fWriteTag(f, 'SUBTITLED', sr.SUBTITLED);
    	fWriteTag(f, 'PREMIERE_FINALE', sr.PREMIERE_FINALE);
    	fWriteTag(f, 'JOINED_IN_PROGRESS', sr.JOINED_IN_PROGRESS);
    	fWriteTag(f, 'CABLE_IN_THE_CLASSROOM', sr.CABLE_IN_CLASSROOM);
    	fWriteTag(f, 'TV_RATING', sr.TV_RATING);
    	fWriteTag(f, 'SAP', sr.SAP);
    	fWriteTag(f, 'BLACKOUT', sr.BLACKOUT);
    	fWriteTag(f, 'SEX_RATING', sr.SEX_RATING);
    	fWriteTag(f, 'VIOLENCE_RATING', sr.VIOLENCE_RATING);
    	fWriteTag(f, 'LANGUAGE_RATING', sr.LANGUAGE_RATING);
    	fWriteTag(f, 'DIALOG_RATING', sr.DIALOG_RATING);
    	fWriteTag(f, 'FV_RATING', sr.FV_RATING);
    	fWriteTag(f, 'ENHANCED', sr.ENHANCED);
    	fWriteTag(f, 'THREE_D', sr.THREE_D);
    	fWriteTag(f, 'LETTERBOX', sr.LETTERBOX);
    	fWriteTag(f, 'HDTV', sr.HD_TV);
    	fWriteTag(f, 'DOLBY', sr.DOLBY);
    	fWriteTag(f, 'DVS', sr.DVS);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA1);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA2);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA3);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA4);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA5);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA6);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA7);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA8);
   		fWriteTag(f, 'USER_DATA', sr.USER_DATA9);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA10);
    	fWriteTag(f, 'USER_DATA', sr.USER_DATA11);
    
    -- SCHEDULE GENERIC
    
    FOR sg in getschedulegeneric(sr.program_id) LOOP
    		fWriteTag(f, 'SCHEDULE_GENERIC', sg.value, 'Name', sg.name, 'Asset', sg.asset);
    END LOOP;
    
    	fWriteTag(f, 'REQUEST_RECORD', sr.REQUEST_RECORD);
        
    -- close row
    
    fWriteETag(f,'row');
	
end;
  
procedure createxml( v_filename varchar2 default 'GuideDataOut.xml', publisher_id number default -1, format varchar2 default 'ETV_20') is
  
  f utl_file.file_type;

  rc number;
  
  publisherName nvarchar2(50);
  
  vcurrentdatetimeforlog timestamp with time zone;
  minutesittook varchar2(10);
  secondsittook varchar2(10);
  hoursittook varchar2(10);
  
  -- create base ETV format?
  createETV_18 boolean := format = 'ETV_18';
  createETV_MIX boolean := format = 'ETV_MIX';

 begin
	 
	 -- TODO - resolve schedule.program_id vs schedule.schedule_id for link tables
	  	 
	 publisherName := 'unknown';
	 begin
	 	select name into publisherName from publisher where id = publisher_id;
	 exception
	 	when NO_DATA_FOUND then
	 		publisherName :='unknown';
	 end;
	 
    vcurrentdatetimeforlog := current_timestamp;

    epgguidedatalogging('createXML: v_filename:'||v_filename||' publisher_id:'||to_char(publisher_id)||' format:'||format);
      
    alarmactivitylogging('Starting XML generation for publisher '||publisherName , const_COMPONENT_NAME_PUBLISH,const_ALARM_SEVERITY_INFO);


	f := utl_file.fopen_nchar('EPGPUBLISH',v_filename,'W',32767);

	-- initialize as an xml file
	fWrite(f,'<?xml version="1.0" encoding="UTF-8"?>');
	fWrite(f,'<GuideData xmlns:fn="http://www.w3.org/2005/02/xpath-functions" xmlns:xdt="http://www.w3.org/2005/xpath-datatypes" xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="ETV_EPG');

	if createETV_18 then
		fWrite(f,'.xsd" Spec_Version="1.8"');
	elsif createETV_MIX then
		fWrite(f,'_MIX.xsd" Spec_Version="3.0"');
	else
		fWrite(f,'.xsd" Spec_Version="2.0"');
	end if;

	fWrite(f,'><SlicerGuideData>');
    
	--- Headends
   	epgguidedatalogging('Start of Headend write.');

	fWriteBTag(f,'HEADEND');

	rc := 0;
   
   	for hr in getheadends LOOP

		fWriteBTag(f,'row');
   
    		fWriteTag(f,'HE_HEADEND_ID',hr.HEADEND_ID);
    		fWriteTag(f,'HE_COMMUNITY_NAME', hr.COMMUNITY_NAME);
    		fWriteTag(f,'HE_COUNTY_NAME',hr.COUNTY_NAME);
    		fWriteTag(f,'HE_COUNTY_SIZE', hr.COUNTY_SIZE);
    		fWriteTag(f,'HE_ST_COUNTY_CODE', hr.ST_COUNTY_CODE);
    		fWriteTag(f,'HE_STATE_SERVED', hr.STATE);
    		fWriteTag(f,'HE_ZIP_CODE', hr.ZIPCODE);
    		fWriteTag(f,'HE_DMA_CODE', hr.DMA_CODE);
    		fWriteTag(f,'HE_DMA_NAME', hr.DMA_NAME);
    		fWriteTag(f,'HE_MSO_CODE', hr.MSO_CODE);
    		fWriteTag(f,'HE_DMA_RANK', hr.DMA_RANK);
    		fWriteTag(f,'HE_HEADEND_NAME', hr.HEADEND_NAME);
    		fWriteTag(f,'HE_HEADEND_LOCATION', hr.HEADEND_LOCATION);
    		fWriteTag(f,'HE_MSO_NAME', hr.MSO_NAME);
    		fWriteTag(f,'HE_TIME_ZONE_CODE', hr.TIME_ZONE);
    
    		if (not createETV_18) then
	    		-- add headend expansion fields 
    			FOR hg in getheadendgeneric(hr.headend_id) LOOP
    				fWriteTag(f, 'HE_GENERIC', hg.value, 'Name', hg.name, 'Asset', hg.asset);
    			END LOOP;

	   	end if;

		fWRiteETag(f,'row');
 	    	
	    rc:=getheadends%ROWCOUNT;
    
  	end loop;

	fWriteETag(f,'HEADEND');
  
  	-- log no headends.
  	if  rc = 0 then
  		-- no headends??
    		alarmactivitylogging('All Headends filtered out of published data for publisher '||publisherName , const_COMPONENT_NAME_PUBLISH,const_ALARM_SEVERITY_WARNING);
    	end if;
  
  -- Lineup
  epgguidedatalogging('Start of Lineup write.');

  fWriteBTag(f,'LINEUP');
  
  rc := 0;
   
  FOR lr IN getlineups LOOP

  	fWriteBTag(f,'row');
  
   	fWriteTag(f,'CL_ROW_ID',lr.lineup_id);
    fWriteTag(f,'CL_STATION_NUM',lr.station_id);
    fWriteTag(f,'CL_HEADEND_ID',lr.headend_id);
    if (not createETV_18) then
    -- create lr.device as whatever it is
    	fWriteTag(f,'CL_DEVICE',lr.device);
    else
    -- create lr.device as either an X or blank as per ETV_EPG.XSD ver 1.8
    	if ( lr.device = 'X') then
    		fWriteTag(f, 'CL_DEVICE','X');
    	elsif lr.device is not null then
    		fWriteTag(f, 'CL_DEVICE',' ');
    	end if;
    end if;
    fWriteTag(f,'CL_TMS_CHAN',lr.tms_channel);
    fWriteTag(f,'CL_SERVICE_TIER',lr.service_tier);
    fWriteTag(f,'CL_EFFECTIVE_DATE',XSDDTValue(lr.effective_date));
    fWriteTag(f,'CL_EXPIRATION_DATE',XSDDTValue(lr.expiration_date));
    
    if (not createETV_18) then
	    -- add lineup expansion fields 
    	FOR lg in getlineupgeneric(lr.lineup_id, lr.headend_id, lr.station_id) LOOP
    		fWriteTag(f, 'CL_GENERIC', lg.value, 'Name', lg.name, 'Asset', lg.asset);
    	END LOOP;
    end if;
    
     rc:=getlineups%rowcount;
     
     fWriteETag(f,'row');

  END LOOP;
  
  fWriteETag(f,'LINEUP');
  
   -- log no lineup.
  	if  rc = 0 then
  		-- no lineup??
    	alarmactivitylogging('All lineups filtered out of published data for publisher  '||publisherName , const_COMPONENT_NAME_PUBLISH,const_ALARM_SEVERITY_WARNING);
    end if;

    if  not createETV_MIX then
  
	  	-- schedule   
	  	epgguidedatalogging('Start of Schedule write.');
	
	  	fWriteBTag(f,'SCHEDULE');
	  	rc := 0;
	   
	  	FOR sr IN getschedules LOOP
	
  			fWriteScheduleTag(f,sr);
	    
		    rc := getschedules%rowcount;
    		
	   	END LOOP;
	   
   		-- close SCHEDULE
   		fWriteETag(f,'SCHEDULE');
   
    	-- log no schedule.
  		if rc = 0 then
	  		-- no schedule??
    		alarmactivitylogging('All schedules filtered out of published data for publisher  '||publisherName , const_COMPONENT_NAME_PUBLISH,const_ALARM_SEVERITY_WARNING);
    	end if;
    	
    end if;


  -- station  
  epgguidedatalogging('Start of Station write.');
  
  fWriteBTag(f,'STATION');
  
   rc := 0;
   
  FOR sr IN getstations LOOP
    
  	-- start row
  	fWriteBTag(f,'row');
    
    fWriteTag(f,'STATION_NUM',sr.station_id);
    fWriteTag(f,'STATION_TIME_ZONE',sr.time_zone);
    fWriteTag(f,'STATION_NAME',sr.station_name);
    fWriteTag(f,'STATION_CALL_SIGN',sr.call_sign);
    fWriteTag(f,'STATION_AFFIL',sr.affiliation);
    fWriteTag(f,'STATION_CITY',sr.city);
    fWriteTag(f,'STATION_STATE',sr.state);
    fWriteTag(f,'STATION_ZIP_CODE',sr.zipcode);
    fWriteTag(f,'STATION_COUNTRY',sr.country);
    fWriteTag(f,'DMA_NAME',sr.DMA_NAME);
    fWriteTag(f,'DMA_NUM',sr.DMA_NUMBER);
    fWriteTag(f,'FCC_CHANNEL_NUM',sr.FCC_CHANNEL_NUM);
    fWriteTag(f,'USER_DATA',sr.USER_DATA1);
    fWriteTag(f,'USER_DATA',sr.USER_DATA2);
    fWriteTag(f,'USER_DATA',sr.USER_DATA3);
    fWriteTag(f,'USER_DATA',sr.USER_DATA4);
    fWriteTag(f,'USER_DATA',sr.USER_DATA5);
    fWriteTag(f,'USER_DATA',sr.USER_DATA6);
    fWriteTag(f,'USER_DATA',sr.USER_DATA7);
    fWriteTag(f,'USER_DATA',sr.USER_DATA8);
    fWriteTag(f,'USER_DATA',sr.USER_DATA9);
    fWriteTag(f,'USER_DATA',sr.USER_DATA10);
    
    if (not createETV_18) then
	    -- add station expansion fields 
    	FOR sg in getstationgeneric(sr.station_id) LOOP
    		fWriteTag(f, 'STATION_GENERIC', sg.value, 'Name', sg.name, 'Asset', sg.asset);
    	END LOOP;
    end if;
    
    -- close the row
    fWriteETag(f,'row');
    
    rc := getstations%rowcount;
   END LOOP;
   
   -- close the station
   fWriteETag(f,'STATION');
  
      -- log no station.
  	if  rc = 0 then
  		-- no station??
    	alarmactivitylogging('All channels filtered out of published data for publisher  '||publisherName , const_COMPONENT_NAME_PUBLISH,const_ALARM_SEVERITY_WARNING);
    end if;

   
  -- Programs    
  epgguidedatalogging('Start of Program write.');
    
  fWriteBTag(f,'PROGRAM');
  
   rc := 0;
   
  FOR pr IN getprograms LOOP
    
  	-- open the row
  	fWriteBTag(f,'row');
  
    fWriteTag(f,'DATABASE_KEY',pr.PROGRAM_ID);
    fWriteTag(f,'LANGUAGE',pr.LANGUAGE);
    fWriteTag(f,'TITLE',pr.TITLE);
    fWriteTag(f,'REDUCED_TITLE', pr.REDUCED_TITLE1);
    fWriteTag(f,'REDUCED_TITLE', pr.REDUCED_TITLE2);
    fWriteTag(f,'REDUCED_TITLE', pr.REDUCED_TITLE3);
    fWriteTag(f,'REDUCED_TITLE', pr.REDUCED_TITLE4);
    fWriteTag(f,'ALT_TITLE', pr.ALT_TITLE);
    fWriteTag(f,'REDUCED_DESC', pr.REDUCED_DESCRIPTION1);
    fWriteTag(f,'REDUCED_DESC', pr.REDUCED_DESCRIPTION2);
    fWriteTag(f,'REDUCED_DESC', pr.REDUCED_DESCRIPTION3);
    fWriteTag(f,'REDUCED_DESC', pr.REDUCED_DESCRIPTION4);
    fWriteTag(f,'ADVISORY_DESC', pr.ADVISORY_DESC1);
    fWriteTag(f,'ADVISORY_DESC', pr.ADVISORY_DESC2);
    fWriteTag(f,'ADVISORY_DESC', pr.ADVISORY_DESC3);
    fWriteTag(f,'ADVISORY_DESC', pr.ADVISORY_DESC4);
    fWriteTag(f,'ADVISORY_DESC', pr.ADVISORY_DESC5);
    fWriteTag(f,'ADVISORY_DESC', pr.ADVISORY_DESC6);
    
    -- CAST
    for pc in getprogramcast(pr.PROGRAM_ID) LOOP
    	-- create CAST_NAME_ROLE
    	fWriteBTag(f,'CAST_NAME_ROLE');
  		fWriteTag(f, 'CAST_FIRST_NAME', pc.FIRST_NAME);
  		fWriteTag(f, 'CAST_LAST_NAME', pc.LAST_NAME);
  		fWriteTag(f, 'CAST_ROLE_DESC', pc.ROLE);
  		fWriteETag(f,'CAST_NAME_ROLE');
  	END LOOP;
 
	-- CREDITS
    for pc in getprogramcredits(pr.PROGRAM_ID) LOOP
    	fWriteBTag(f,'CREDITS_NAME_ROLE');
  		fWriteTag(f, 'CREDITS_FIRST_NAME', pc.FIRST_NAME);
  		fWriteTag(f, 'CREDITS_LAST_NAME', pc.LAST_NAME);
  		fWriteTag(f, 'CREDITS_ROLE_DESC', pc.ROLE);
  		fWriteETag(f,'CREDITS_NAME_ROLE');
  	END LOOP;
	    
    fWriteTag(f,'GENRE_DESC', pr.GENRE_DESC1);
    fWriteTag(f,'GENRE_DESC', pr.GENRE_DESC2);
    fWriteTag(f,'GENRE_DESC', pr.GENRE_DESC3);
    fWriteTag(f,'GENRE_DESC', pr.GENRE_DESC4);
    fWriteTag(f,'GENRE_DESC', pr.GENRE_DESC5);
    fWriteTag(f,'GENRE_DESC', pr.GENRE_DESC6);
    fWriteTag(f,'DESC', pr.DESCRIPTION1);
    fWriteTag(f,'DESC', pr.DESCRIPTION2);
    fWriteTag(f,'YEAR', pr.YEAR_PROGRAM);
    fWriteTag(f,'MPAA_RATING', pr.MPAA_RATING);
    fWriteTag(f,'STAR_RATING', pr.STAR_RATING);
    
    -- Program Rating
    for prt in getprogramrating(pr.PROGRAM_ID) LOOP
    	fWriteTag(f,'PROGRAM_RATING', prt.value, 'Authority', prt.Authority);
    END LOOP;
        
    fWriteTag(f,'RUN_TIME', pr.RUN_TIME);
    fWriteTag(f,'COLOR_CODE', pr.COLOR_CODE);
    fWriteTag(f,'PROGRAM_LANGUAGE', pr.PROGRAM_LANGUAGE);
    
    -- Audio Component
    for pa in getprogramaudiotl(pr.PROGRAM_ID) LOOP

    	fWrite(f,'<AUDIO_COMPONENT Type="');
    	fWrite(f,pa.Type);
    	fWrite(f,'" Language="');
    	fWrite(f,pa.Language);
    	fWrite(f,'">');
   	
    	for ca in getprogramaudio(pr.PROGRAM_ID, pa.Type, pa.Language) LOOP
  			fWriteTag(f, 'ComponentAttribute', ca.value, 'AttributeType', ca.attribute_type);
  		END LOOP;
  		
  		fWrite(f,'</AUDIO_COMPONENT>');
  		
    END LOOP;
  
     -- Subtitle Component
    for ps in getprogramsubtitletl(pr.PROGRAM_ID) LOOP

       	fWrite(f,'<SUBTITLE_COMPONENT Type="');
    	fWrite(f,ps.Type);
    	fWrite(f,'" Language="');
    	fWrite(f,ps.Language);
    	fWrite(f,'">');
   	
    	for ca in getprogramsubtitle(pr.PROGRAM_ID, ps.Type, ps.Language) LOOP
  			fWriteTag(f, 'ComponentAttribute', ca.value, 'AttributeType', ca.attribute_type);
  		END LOOP;
 		
  		fWrite(f,'</SUBTITLE_COMPONENT>');

      END LOOP;
    
    fWriteTag(f,'ORG_COUNTRY', pr.COUNTRY_OF_ORIGIN);
    fWriteTag(f,'MADE_FOR_TV', pr.MADE_FOR_TV);
    fWriteTag(f,'SOURCE_TYPE', pr.SOURCE_TYPE);
    fWriteTag(f,'SHOW_TYPE', pr.SHOW_TYPE);
    fWriteTag(f,'HOLIDAY', pr.HOLIDAY);
    fWriteTag(f,'SYN_EPI_NUM', pr.SYNDICATE_EPISODE_NUM);
    fWriteTag(f,'ALT_SYN_EPI_NUM', pr.ALT_SYNDICATE_EPI_NUM);
    fWriteTag(f,'EPI_TITLE', pr.EPISODE_TITLE);
    fWriteTag(f,'NET_SYN_SOURCE', pr.NET_SYN_SOURCE);
    fWriteTag(f,'NET_SYN_TYPE', pr.NET_SYN_TYPE);
    fWriteTag(f,'ORG_STUDIO', pr.ORG_STUDIO);
    fWriteTag(f,'GAME_DATE', XSDDTValue(pr.GAME_DATETIME));
    fWriteTag(f,'GAME_TIME_ZONE', pr.GAME_TIMEZONE);
    fWriteTag(f,'ORG_AIR_DATE', XSDDTValue(pr.ORG_AIR_DATE));
    fWriteTag(f,'UNIQUE_ID', pr.UNIQUE_ID);
    fWriteTag(f,'USER_DATA', pr.USER_DATA1);
    fWriteTag(f,'USER_DATA', pr.USER_DATA2);
    fWriteTag(f,'USER_DATA', pr.USER_DATA3);
    fWriteTag(f,'USER_DATA', pr.USER_DATA4);
    fWriteTag(f,'USER_DATA', pr.USER_DATA5);
    fWriteTag(f,'USER_DATA', pr.USER_DATA6);
    fWriteTag(f,'USER_DATA', pr.USER_DATA7);
    fWriteTag(f,'USER_DATA', pr.USER_DATA8);
    fWriteTag(f,'USER_DATA', pr.USER_DATA9);
    fWriteTag(f,'USER_DATA', pr.USER_DATA10);
    fWriteTag(f,'USER_DATA', pr.USER_DATA11);
    fWriteTag(f,'USER_DATA', pr.USER_DATA12);
    fWriteTag(f,'USER_DATA', pr.USER_DATA13);
   
    -- PROGARM GENERIC    
    FOR sg in getprogramgeneric(pr.program_id) LOOP
    	fWriteTag(f, 'PROGRAM_GENERIC', sg.value, 'Name', sg.name, 'Asset', sg.asset);
    END LOOP;
    
    rc := getprograms%rowcount;
    
    if createETV_MIX then
    	-- include schedule tags here.
    	fWriteBTag(f,'SCHEDULE');
    	
    	for sr in getschedule(pr.program_id) LOOP
    		fWriteScheduleTag(f,sr);
    	end loop;
    	
    	fWriteETag(f,'SCHEDULE');
    	
    end if;
      
    -- close row
    fWriteETag(f,'row');
    
   END LOOP;
   
   -- close program
   fWriteETag(f,'PROGRAM');
   
   -- close Clicer guide data
   fWriteETag(f,'SlicerGuideData');
   -- close GuideData
   fWriteETag(f,'GuideData');
   
	fflush(f);
   
   -- close file
   utl_file.fclose(f);
   
    -- log no program.
  	if  rc = 0 then
  		-- no program??
    	alarmactivitylogging('All programs filtered out of published data for publisher  '||publisherName , const_COMPONENT_NAME_PUBLISH,const_ALARM_SEVERITY_WARNING);
    end if;

    
   SELECT floor(Extract(Minute from (current_timestamp - vcurrentdatetimeforlog))) into minutesittook from dual;
   SELECT floor(Extract(Second from (current_timestamp - vcurrentdatetimeforlog))) into secondsittook from dual;
   SELECT floor(Extract(Hour from (current_timestamp - vcurrentdatetimeforlog))) into hoursittook from dual;
   
   epgguidedatalogging('End of XML write. It took '||hoursittook||' hours and '||LPAD(minutesittook,2,'0')||' minutes and '||LPAD(secondsittook,2,'0')||' seconds');
   
   alarmactivitylogging('Finished XML generation for publisher '||publisherName||' (took '||hoursittook||':'||LPAD(minutesittook,2,'0')||':'||LPAD(secondsittook,2,'0')||')', const_COMPONENT_NAME_PUBLISH);

 end createxml;



 
 procedure exportepgdata(publisher_id number default -1, lowerDate varchar2 default '', upperDate varchar2 default '', channelIds varchar2 default '') is
 	publisherName nvarchar2(50);
  begin
	  -- Date value formats are '26-MAR-12 12.30.00.00000000 AM UTC' 
	publisherName := 'unknown';
    init();
    begin
    	select Name into publisherName from Publisher where ID = publisher_id;
    exception
    	WHEN NO_DATA_FOUND then
    		publisherName := 'unknown';
    end;
    
    -- log params
    epgguidedatalogging('exportepgdata:  publisher_id:'||to_char(publisher_id)||' lowerDate: '||lowerDate||' upperDate: '||upperDate);
     
    -- check for NO production data.
    DECLARE
    	total integer := 0;
    	num integer := 0;
    BEGIN
	    select count(*) into num from HEADEND where rownum = 1;
	    total := total + num;
	    select count(*) into num from LINEUP where rownum = 1;
	    total := total + num;
	    select count(*) into num from SCHEDULE where rownum = 1;
	    total := total + num;
	    select count(*) into num from STATION where rownum = 1;
	    total := total + num;
	    select count(*) into num from PROGRAM where rownum = 1;
	    total := total + num;
	    
	    if total = 0 then
	    	-- no production data.
     		alarmactivitylogging('Export aborted for publisher '||publisherName||'. No production data found.', const_COMPONENT_NAME_EXPORT, const_ALARM_SEVERITY_ERROR);
   			epgguidedatalogging('Export aborted for publisher '||publisherName||'. No production data found.');
     		
	    	raise_application_error(-20130,' Export failed - no production data');
	   	end if;
	    
	END;
    
    exportepg(publisherName, lowerDate, upperDate, channelIds);
    commit;
  exception
	WHEN xmldom.INDEX_SIZE_ERR THEN
	  	raise_application_error(-20120, 'Index Size error');
	WHEN xmldom.DOMSTRING_SIZE_ERR THEN
  		raise_application_error(-20120, 'String Size error');
	WHEN xmldom.HIERARCHY_REQUEST_ERR THEN
	  	raise_application_error(-20120, 'Hierarchy request error');
	WHEN xmldom.WRONG_DOCUMENT_ERR THEN
  		raise_application_error(-20120, 'Wrong doc error');
	WHEN xmldom.INVALID_CHARACTER_ERR THEN
  		raise_application_error(-20120, 'Invalid Char error');
	WHEN xmldom.NO_DATA_ALLOWED_ERR THEN
  		raise_application_error(-20120, 'No data allowed error');
	WHEN xmldom.NO_MODIFICATION_ALLOWED_ERR THEN
  		raise_application_error(-20120, 'No mod allowed error');
	WHEN xmldom.NOT_FOUND_ERR THEN
  		raise_application_error(-20120, 'Not found error');
	WHEN xmldom.NOT_SUPPORTED_ERR THEN
  		raise_application_error(-20120, 'Not supported error');
	WHEN xmldom.INUSE_ATTRIBUTE_ERR THEN
  		raise_application_error(-20120, 'In use attr error');
     when others then
      -- Error Code 11 = EPG Download Abnormal Error
      -- commit;
      alarmactivitylogging('Abnormal Error during EPG export for '||publisherName, const_COMPONENT_NAME_EXPORT, const_ALARM_SEVERITY_ERROR);

      raise;
  end exportepgdata;
END EPGEXPORT;
/
exit