CREATE OR REPLACE PACKAGE EPGRULES AS

	function processProviderValidRules( provider_id number ) return boolean;
	procedure processProviderNormRules(provider_id number);
	procedure processPublisherNormRules(publisher_id number);
	
	procedure gr_processProviderValidRules( provider_id number);
	procedure createSampleRules;
	procedure dropRules;
	
	-- join operators constants
	const_Rule_Clause_JP_NULL constant number := 1;	
	const_Rule_Clause_JP_AND constant number := 2;
	const_Rule_Clause_JP_OR constant number := 3;
	
	-- compare operator constants
	const_Compare_Equals 		constant number := 1;
	const_Compare_NotEquals 	constant number := 2;
	const_Compare_GreaterThan 	constant number := 3;
	const_Compare_GreaterThanEqTo constant number := 4;
	const_Compare_LessThan 		constant number := 5;
	const_Compare_LessThanEqTo  constant number := 6;
	const_Compare_Contains 		constant number := 7;
	const_Compare_In			constant number :=8;
	const_Compare_StartsWith	constant number :=9;
	const_Compare_EndsWith		constant number :=10;
	const_Compare_IsEmpty		constant number :=11;
	const_Compare_Like			constant number := 12;
	const_Compare_NotEmpty		constant number := 13;	
	const_Compare_Max			constant number := 14;

	-- Alarm table constants
  const_ALARM_TYPE_ALARM constant number := 1;
  const_ALARM_TYPE_ACTIVITY  constant number := 2;
  const_ALARM_TYPE_LOG constant number := 3;
  const_ALARM_TYPE_VALID_FAILED constant number := 4;

  const_ALARM_SEVERITY_WARNING constant number := 1;
  const_ALARM_SEVERITY_MINOR constant number := 2;
  const_ALARM_SEVERITY_MAJOR constant number := 3;
  const_ALARM_SEVERITY_CRITICAL constant number := 4;
  const_ALARM_SEVERITY_INFO constant number := 5;
	
	
	
END EPGRULES;
/
create or replace
PACKAGE BODY EPGRULES AS


  /* Procedure: CreateObject. Executes a dynamic SQL statement and logs error in install_table*/
  PROCEDURE CREATEOBJECT(vsql varchar2,errmsg varchar2, mask varchar2 default 'N') IS
    verrmsg varchar2(32000);
  BEGIN
    execute immediate vsql;
  exception
    when others then
          verrmsg := errmsg ||' '||sqlerrm||' '||vsql;
          select cast(verrmsg as varchar2(4000))
          into verrmsg
          from dual;
      if upper(mask) != 'Y' then
          execute immediate 'INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')';
          commit;
          raise;
      end if;
  END CREATEOBJECT;
  
  	procedure dropRules is
  		vsql nvarchar2(1000);
  	begin
	  execute immediate 'truncate table Norm_Rule_Clause';
	  execute immediate 'truncate table Norm_Rule';
	  execute immediate 'truncate table Norm_Rule_Group_Map';
	  execute immediate 'truncate table Norm_group';
	  execute immediate 'truncate table Norm_Provider_Group_Map';
	  execute immediate 'truncate table Norm_Publisher_Group_Map';
	  execute immediate 'truncate table Valid_Rule_Clause';
	  execute immediate 'truncate table Valid_Rule';
	  execute immediate 'truncate table Valid_Rule_Group_Map';
	  execute immediate 'truncate table Valid_group';
	  execute immediate 'truncate table Valid_Provider_Group_Map';
	  execute immediate 'truncate table Valid_Publisher_Group_Map';
	  	
	  commit;
	  
	end;
  
  	procedure createSampleRules is
  		vsql nvarchar2(1000);
  	begin
	  	
	  	--populate some sample rules into the validation and normalization rules tables
	  	-- this creates validation rule id 100 for provider id 100
	  	-- in group 100
	  	
	  	
	  	-- create validation rule group
	  	vsql := 'INSERT INTO VALID_GROUP (id,name,CREATED_BY, UPDATED_BY) VALUES(100,''ChrisGroup'', ''System'', ''System'' )';
	    CREATEOBJECT(vsql,'Sample Rules Entry');
	  		  	
	  	-- map fictional provider 100 to new group 
	  	vsql := 'INSERT INTO VALID_PROVIDER_GROUP_MAP ( id, provider_id, valid_group_id, enabled, CREATED_BY, UPDATED_BY) VALUES( 100, 100, 100, ''Y'', ''System'', ''System'')';
		CREATEOBJECT(vsql,'Sample Rules Entry');
	  		  	
		-- creat a new rule
		vsql:= 'INSERT INTO VALID_RULE (id, name, errorlevel, msg, CREATED_BY, UPDATED_BY) VALUES ( 100, ''ChrisProvider'', 2, ''This is a test validation rule'', ''System'', ''System'')';
		CREATEOBJECT(vsql,'Sample Rules Entry');
		
	    -- map new rule to groupo
	    vsql := 'INSERT INTO VALID_RULE_GROUP_MAP (id, VALID_RULE_ID, VALID_GROUP_ID, ENABLED, CREATED_BY, UPDATED_BY) values (100, 100, 100, ''Y'', ''System'', ''System'')';
		CREATEOBJECT(vsql,'Sample Rules Entry');
	    
	    
	    -- create clause for validation rule
	    -- em 8 is schedule.Num_Of_Parts - an int field.
	    vsql := 'INSERT INTO VALID_RULE_CLAUSE 
				(id, 
					VALID_RULE_ID, 
					entity_mapping_id, 
					Rule_Compare_Operator_id, 
					Rule_Clause_join_Prefix_ID, 
					Condition_Value_Type, 
					Condition_Value, 
					Layer_Number, 
					Ordering, 
					CREATED_BY, 
					UPDATED_BY)
			values
				( 100, 
					100,
					8, 
					'||const_Compare_Equals||',
					'||const_Rule_Clause_JP_NULL||', 
					0,
					''512'',
					0,
					0, 
					''SYSTEM'', 
					''SYSTEM'')';
		CREATEOBJECT(vsql,'Sample Rules Entry');

		-- em.id 18 is violence rating
			    vsql := 'INSERT INTO VALID_RULE_CLAUSE 
				(id, 
					VALID_RULE_ID, 
					entity_mapping_id, 
					Rule_Compare_Operator_id, 
					Rule_Clause_join_Prefix_ID, 
					Condition_Value_Type, 
					Condition_Value, 
					Layer_Number, 
					Ordering, 
					CREATED_BY, 
					UPDATED_BY)
			values
				( 101, 
					100,
					18, 
					'||const_Compare_NotEquals||',
					'||const_Rule_Clause_JP_AND||', 
					0,
					''Y'',
					0,
					1, 
					''SYSTEM'', 
					''SYSTEM'')';
		CREATEOBJECT(vsql,'Sample Rules Entry');
		
	end;
	
	function noCriticalsInValidationResults return boolean as
		c number;
		result boolean := true;
	begin
		select count(*) into c from Valid_Rule_Result where ErrorLevel = const_ALARM_SEVERITY_CRITICAL;
		
		if c > 0 then
			result := false;
		end if;
		
		return result;
	end;
  
  	-- return the string version of the comparison enumeration
  	function RuleCompareOperatorString(Compare_ID number) return nvarchar2 as
  		compare nvarchar2(50);
  	begin
	  	return case(Compare_ID)
	  		WHEN const_Compare_Equals THEN ' = '
	  		WHEN const_Compare_NotEquals THEN ' != '
	  		WHEN const_Compare_GreaterThan THEN ' > '	
	  		WHEN const_Compare_GreaterThanEqTo THEN ' >= ' 
	  		WHEN const_Compare_LessThan THEN ' < '		
	  		WHEN const_Compare_LessThanEqTo THEN ' <= ' 
	  		WHEN const_Compare_Contains THEN ' like '		
	  		WHEN const_Compare_In THEN ' in '		
	  		WHEN const_Compare_StartsWith THEN ' like '	
	  		WHEN const_Compare_EndsWith THEN ' like '
	  		WHEN const_Compare_IsEmpty THEN	' is null '
	  		WHEN const_Compare_Like THEN ' like '
	  		WHEN const_Compare_NotEmpty then ' is not null '
	  		ELSE '-BAD-OPR-'
	  		end;
	end;
	
	function RuleInValueString(value nvarchar2) return nvarchar2 as
		valueset nvarchar2(1000);
		maxpos number;
		pos number;
		ch nvarchar2(1);
	begin
		-- create a set structure as ('value1','value2','value3') from the provided input of value1,value2,value3
		SELECT LENGTH(value) into maxpos from DUAL;
		
		valueset := '(''';
		FOR pos in 1..maxpos LOOP
		
			ch := SUBSTR(value,pos,1);
			if ch = ',' then
				valueset := valueset || ''',''';
			else
				valueset := valueset || ch;
			end if;
		END LOOP;
		
		valueset := valueset || ''')';

		return valueset;
		
	end;
	
	-- build up the right side of the comparison operation based on the compare operator and THIS as the value to compare against.
	-- this assumes the value is a constant that needs to be quoted.
	function RuleCompareValueString(Compare_ID number, value nvarchar2) return nvarchar2 as
	begin
		-- how and if we modify the compare value is based on what kind of compare operator we're using
		return case(Compare_ID)
	  		WHEN const_Compare_Equals THEN ''''||value||''''
	  		WHEN const_Compare_NotEquals THEN ''''||value||''''
	  		WHEN const_Compare_GreaterThan THEN ''''||value||''''	
	  		WHEN const_Compare_GreaterThanEqTo THEN ''''||value||'''' 
	  		WHEN const_Compare_LessThan THEN ''''||value||''''		
	  		WHEN const_Compare_LessThanEqTo THEN ''''||value||'''' 
	  		WHEN const_Compare_Contains THEN '''%'||value||'%'''	
	  		WHEN const_Compare_In THEN 	RuleInValueString(value)	
	  		WHEN const_Compare_StartsWith THEN ''''||value||'%'''	
	  		WHEN const_Compare_EndsWith THEN '''%'||value||''''
	  		WHEN const_Compare_IsEmpty THEN	''
	  		WHEN const_Compare_Like THEN ''''||value||''''
	  		WHEN const_Compare_NotEmpty THEN ''
	  		ELSE '-BAD-OPR-FOR-VALUE-'
	  		end;
	end;
	
	-- build up the right side of the comparison operation based on the compare operator and THIS as the value to compare against.
	-- this assumes the value is a constant that needs to be quoted.
	function RuleCompareValueDate(Compare_ID number, datevalue nvarchar2) return nvarchar2 as
	begin
		-- how and if we modify the compare value is based on what kind of compare operator we're using
		return case(Compare_ID)
	  		WHEN const_Compare_Equals THEN 'to_timestamp_tz('''||datevalue||''',''YYYY/MM/DD HH24:MI:SS TZH:TZM'')'
	  		WHEN const_Compare_NotEquals THEN 'to_timestamp_tz('''||datevalue||''',''YYYY/MM/DD HH24:MI:SS TZH:TZM'')'
	  		WHEN const_Compare_GreaterThan THEN 'to_timestamp_tz('''||datevalue||''',''YYYY/MM/DD HH24:MI:SS TZH:TZM'')'	
	  		WHEN const_Compare_GreaterThanEqTo THEN 'to_timestamp_tz('''||datevalue||''',''YYYY/MM/DD HH24:MI:SS TZH:TZM'')'
	  		WHEN const_Compare_LessThan THEN 'to_timestamp_tz('''||datevalue||''',''YYYY/MM/DD HH24:MI:SS TZH:TZM'')'		
	  		WHEN const_Compare_LessThanEqTo THEN 'to_timestamp_tz('''||datevalue||''',''YYYY/MM/DD HH24:MI:SS TZH:TZM'')' 
	  		WHEN const_Compare_Contains THEN '-BAD-OPR-FOR-DATE-'	
	  		WHEN const_Compare_In THEN 	'-BAD-OPR-FOR-DATE-'	
	  		WHEN const_Compare_StartsWith THEN '-BAD-OPR-FOR-DATE-'	
	  		WHEN const_Compare_EndsWith THEN '-BAD-OPR-FOR-DATE-'
	  		WHEN const_Compare_IsEmpty THEN	''
	  		WHEN const_Compare_Like THEN '-BAD-OPR-FOR-DATE-'
	  		WHEN const_Compare_NotEmpty THEN ''
	  		ELSE '-BAD-OPR-FOR-DATE-'
	  		end;
	end;  

	
  	-- build up the condition that this rule checks against.
  	function buildValidInsertWhenClause( rule_id number) return nvarchar2 as
  		clause nvarchar2(1000);
  		depth number := 0;
  		tableName varchar(50);
  		columnName varchar(50);
	   	CURSOR getRuleClauses(rule_id number) is select 
                                Rule_Clause_Join_Prefix_ID,
                                Layer_Number,
                                Ordering,
                                Rule_Compare_Operator_ID,
                                Condition_Value_Type,
                                Condition_Value,
                                Table_Name,
                                Column_Name,
                                Alt_Column_Name1,
                                Alt_Column_Value1,
                                Alt_Column_Name2,
                                Alt_Column_Value2,
                                Data_Type
                        from 	Valid_Rule_Clause vrc,
  								Entity_Mapping em
  						where  vrc.Valid_Rule_ID = rule_id  and
  								em.ID = vrc.Entity_Mapping_ID
  						order by vrc.Ordering;  	
  	begin
	  	
	  	clause := '( ';
	  	for vr in getRuleClauses(rule_id) LOOP
	  	
	  		-- include this clause in the check.
	  		
	  		-- joiner?  AND/OR
	  		clause := clause|| case (vr.Rule_Clause_Join_Prefix_ID)
	  			  			   WHEN const_Rule_Clause_JP_AND THEN ' AND '
	  			  			   WHEN const_Rule_Clause_JP_OR THEN ' OR '
	  			  			   ELSE ''
	  			  			   END;
	  		
	  		-- depth change? brackets
	  		if (vr.Layer_Number != depth) then
	  		
	  			if (vr.Layer_Number > depth) then
	  				clause := clause||' ( ';
	  			else
	  				clause := clause||' ) ';
	  			end if;
	  			
	  			depth := vr.Layer_Number;
	  			
	  		end if;
	  		
	  		-- any additional alt columns to initially include?
	  			  		
	  		if vr.Alt_Column_Name1 is not null then
	  			clause := clause||vr.Alt_Column_Name1||RuleCompareOperatorString(const_Compare_Equals)||''''||vr.Alt_Column_Value1||''' AND ';
        	end if;
	  		if vr.Alt_Column_Name2 is not null then
	  			clause := clause||vr.Alt_Column_Name2||RuleCompareOperatorString(const_Compare_Equals)||''''||vr.Alt_Column_Value2||''' AND ';
       		end if;
       		
	  		-- field compare  RuleCompareOperatorString adds a space before and after
  			clause := clause||vr.Column_Name||RuleCompareOperatorString(vr.Rule_Compare_Operator_ID);
	  		
	  		-- based on what to compare to...
	  		if (vr.Condition_Value_Type = 0) then
	  			-- static value compare of string	  		
	  			if vr.Data_type = 'String' then
	  				clause := clause||RuleCompareValueString(vr.Rule_Compare_Operator_ID,vr.Condition_value);
	  			elsif vr.Data_type = 'DateTime' then
	  				clause := clause||RuleCompareValueDate(vr.Rule_Compare_Operator_ID, vr.Condition_value);
	  			elsif vr.Data_type = 'Int' then
	  				clause := clause||RuleCompareValueString(vr.Rule_Compare_Operator_ID,vr.Condition_value);
	  			end if;
	  		else
	  			-- field value compare
	  			select Table_Name, Column_Name into tableName, columnName from Entity_Mapping WHERE Entity_Mapping.ID = vr.Condition_Value; 
	  			clause := clause||columnName;
			end if;
				
	  	END LOOP;
	  	
	  	clause := clause||' ) ';
	  	
	  	return clause;
	  	
	end;
	
	-- include a column in the list of column's being selected
	function addColumnNamesToSelect( currentColumns varchar2, newColumn nvarchar2) return varchar2 as
		columnNames varchar(500);
	begin
		columnNames := currentColumns;
		
		if newColumn is not null then
		
			if columnNames is null then
		  		columnNames := newColumn;
	  		else
		  		columnNames := columnNames||','||newColumn;
	  		end if;
	  		
		end if;
		
	  	return columnNames;
	 end;
	  		
	
  	-- build up the condition that this rule checks against.
  	function buildValidInsertSelectClause( rule_id number) return nvarchar2 as
  		clause nvarchar2(1000);
  		depth number := 0;
  		tableNames varchar(500);
  		columnNames varchar(500);
  		columnName varchar(100);
	   	CURSOR getRuleClauses(rule_id number) is select 
                                Rule_Clause_Join_Prefix_ID,
                                Layer_Number,
                                Ordering,
                                Rule_Compare_Operator_ID,
                                Condition_Value_Type,
                                Condition_Value,
                                Table_Name,
                                Column_Name,
                                Alt_Column_Name1,
                                Alt_Column_Value1,
                                Alt_Column_Name2,
                                Alt_Column_Value2
                        from 	Valid_Rule_Clause vrc,
  								Entity_Mapping em
  						where  vrc.Valid_Rule_ID = rule_id  and
  								em.ID = vrc.Entity_Mapping_ID
  						order by vrc.Ordering;  	
  	begin
	  	
	  	for vr in getRuleClauses(rule_id) LOOP
			
	  		-- tables for from
	  		if (tableNames is null) then
	  			tableNames := 'stg_'||vr.Table_Name;
	  		elsif (INSTR(tableNames, vr.Table_Name) = 0 ) then
	  			tableNames := tableNames||', stg_'||vr.Table_Name;
	  		end if;
	  		
	  		-- columns for select
	  		columnNames := addColumnNamesToSelect(columnNames, vr.Column_Name);
	  		columnNames := addColumnNamesToSelect(columnNames, vr.Alt_Column_Name1);
	  		columnNames := addColumnNamesToSelect(columnNames, vr.Alt_Column_Name2);
	  		if vr.Condition_Value_Type = 1 then
	  			-- include the right side of the comparison in the colum list
				select Column_Name into columnName from Entity_Mapping WHERE Entity_Mapping.ID = vr.Condition_Value;	  			
	  			columnNames:=AddColumnNamesToSelect(columnNames,columnName);
	  		end if;
  						
	  	END LOOP;
	  	
	  	-- limit the number of rows returned here to just be 2 - so we only do a single insert into the valid_rule_results table (vs one for every row found...)
	  	clause := 'select '||columnNames||' from '||tableNames||' where '||buildValidInsertWhenClause(rule_id)||' and rownum < 2';
	  	
	  	return clause;
	  	
	end;
	
	

	-- do this validation rule from this name'ed provider.
	procedure executeValidRule(valid_rule_id number, Name varchar2) as
	-- CURSORS
		failMessage nvarchar2(2000);
		errLevel number;
		vsql nvarchar2(2000);
	begin
		-- get the result of the validation failure
		select Msg,ErrorLevel into failMessage, errLevel 
		from Valid_Rule where ID=valid_rule_id;
		
		-- build up the sql
		-- insert when CALL_SIGN like '%ATT2%' 
		--   THEN INTO 
    	--   VALID_RULE_RESULT 
    	--		(Name, Msg, errorLevel) 
    	--		values ( 'chris', 'test message', 4)
		--	select CALL_SIGN from stg_Station where CALL_SIGN like '%ATT2%'and rownum < 2;
  	
		vsql := 'INSERT WHEN '||buildValidInsertWhenClause(valid_rule_id)||' THEN INTO VALID_RULE_RESULT ( Name, Msg, ErrorLevel) VALUES ('''||Name||''','''||failMessage||''','||errLevel||') '||buildValidInsertSelectClause(valid_rule_id);
		
		dbms_output.put_line(vsql);
		
		CREATEOBJECT(vsql,'Execute Validation Rule id '||valid_rule_id);
		
	end executeValidRule;
		

	function processProviderValidRules( provider_id number ) return boolean is
	-- CURSORS
	
	-- get the list of validation rule groups this provider is in
	CURSOR getProvValidGroups(prov_id number) is select * from VALID_PROVIDER_GROUP_MAP where Provider_ID = prov_id and Enabled='Y';
	-- get the list of rules in a validation rule group 
	CURSOR getValidGroupRules( group_id number) is select * from Valid_Rule_Group_Map where Valid_Group_ID = group_id and Enabled='Y';
	
	-- variables
	result boolean := true;	
	provider_name nvarchar2(50);
	vsql nvarchar2(2000);
	Name nvarchar2(50);
	c number;
	begin
		dbms_output.put_line('Perform provider valid Rules for '||provider_id);
		
		--select provider_name into Name from Provider where ID = provider_id;
		name :='ChrisProvider';
		
		-- truncate the validation results table.
		vsql := 'truncate table VALID_RULE_RESULT';
    	CREATEOBJECT(vsql,'truncate table VALID_RULE_RESULT');
		
		-- for all the validation rule groups this provider is in
		for vg in getProvValidGroups(provider_id) LOOP
		
			-- for all the validation rules that are in this validation rule group
			for vr in getValidGroupRules(vg.Valid_Group_ID) LOOP
			
				executeValidRule(vr.Valid_Rule_ID, Name);
			
			END LOOP;
		
		END LOOP;
		
		-- more results to alarm activity log
		vsql := 'INSERT INTO ALARM_ACTIVITYLOG (ID,SEVERITY,COMPONENT,DESCRIPTION,CREATED_BY,UPDATED_BY) 
				SELECT alarm_activitylog_seq.nextval,Errorlevel,''RULES'', ''Provider '||name||' Validation Rule Failed: ''||Msg, ''SYSTEM'',''SYSTEM'' from VALID_RULE_RESULT';
				
		CREATEOBJECT(vsql,'Insert Validation Results');
		
		-- anything with a critial?
		result := noCriticalsInValidationResults();
			
		return result;
	end;
	
	
	--production method
	procedure processProviderNormRules(provider_id number) as
	begin
		
		-- TODO
		dbms_output.put_line('Perform provider norm Rules for '||provider_id);
		
	end processProviderNormRules;
	
	-- production method
	procedure processPublisherNormRules(publisher_id number) as
	begin
		
		-- TODO
		dbms_output.put_line('Peform publisher norm Rules for '||publisher_id);
		
	end processPublisherNormRules;

	
	procedure gr_processProviderValidRules( provider_id number) as
	 result boolean := false;
	begin
		-- just for debugging the fuction as a procedure
		result := processProviderValidRules(provider_id);
		if result = true then
			dbms_output.put_line('processProviderValidRules for '||provider_id||' is true');
		else
			dbms_output.put_line('processProviderValidRules for '||provider_id||' is false');		
		end if;
		
	end gr_processProviderValidRules;

END EPGRULES;
/
exit;
