
CREATE OR REPLACE PACKAGE PROCESSTSTV AS
  procedure processepgdata(v_filename varchar2);
  vfileid number;
  vfilename varchar2(1000);
  vsql varchar2(32000);

  --Threshold time for schedule downloads
  vcurrentdatetime timestamp with time zone;
  vepgrecordingspan timestamp with time zone;
  procedure rightsprocess(vfilename varchar2 default '');
  procedure rulesprocess(vrights_rules varchar2 default 'RULES', vfilename varchar2 default '');
  procedure rightsedit(in_right_id number, p_countrecording OUT number);
  procedure removermsfromright(rm_comma_separated_values VARCHAR2, in_right_id NUMBER,  record_count out number);
  procedure removermlineup (in_rm_id number, in_station_id varchar2);
  procedure deleteright(in_right_id number);
  procedure processrecordables(in_contract_id number);
  PROCEDURE RESOURCELOCK(vresourcetype varchar2,vepgresourcetimeout varchar2 default null);
  PROCEDURE RECORDABLESRECRESOURCELOCK;
  procedure processrightdeletion(in_right_id number);
  procedure processrmsdeletionfromright(in_right_id number);
  procedure deleterecording(in_recording_id number);
  procedure runschemastatistics;
  procedure insertRecQueue;
  FUNCTION rowtocol( p_slct IN VARCHAR2, p_dlmtr IN VARCHAR2 DEFAULT ',' ) RETURN VARCHAR2;

  const_nls_charset_id constant varchar2(100) := 'UTF-8';
  const_ALARM_TYPE_ALARM constant number := 1;
  const_ALARM_TYPE_ACTIVITY  constant number := 2;
  const_ALARM_TYPE_LOG constant number := 3;

  const_ALARM_SEVERITY_WARNING constant number := 1;
  const_ALARM_SEVERITY_MINOR constant number := 2;
  const_ALARM_SEVERITY_MAJOR constant number := 3;
  const_ALARM_SEVERITY_CRITICAL constant number := 4;
  const_ALARM_SEVERITY_INFO constant number := 5;

  /* Recording Queue Table constants */
  const_recordingstatuspending constant number := 200;
  const_recording_UPDATE_REQUEST constant number := 1;
  const_recording_DELETE_REQUEST constant number := 2;
  const_recording_STOP_REQUEST constant number := 3;

  /* Recording table constants  */
  const_recording_DELETED constant number := 1;
  const_recording_SUCCESS constant number := 2;
  const_recording_SCHEDULED constant number := 3;
  const_recording_CURRENT constant number := 4;
  const_recording_FAILED constant number := 5;
  const_recording_PRTL_SUCCESS constant number := 6;


  /** Recording status_code constants */
  const_rec_status_code_DELETED constant number := 300;

      /** Event name keys**/
  const_EventNameKey_RECORDINGS constant varchar2(50) :='RECORDINGS';

      /** Recording Request Types sync with RCSConstants , do not change these values **/
  const_RecordingReqType_UPDATE constant number := 1;
  const_RecordingReqType_DELETE constant number :=2;
  const_RecordingReqType_STOP constant number :=3;

    /** Right, Rule state constants **/
  const_STATE_ACTIVE  constant number := 1;
  const_STATE_INACTIVE constant number := 0;
  const_STATE_DELETED constant number := 2;

  /* EPG Resource Timeout for acquiring Recordables and Recordings operations (seconds) */
  const_EPGRESOURCETIMEOUT constant number := 300;
END PROCESSTSTV;
/


CREATE OR REPLACE PACKAGE BODY     PROCESSTSTV AS

  --Autonomous transaction procedure
  procedure autonomoustransaction(vsql varchar2,errmsg varchar2 default NULL, mask varchar2 default 'N') is
  PRAGMA AUTONOMOUS_TRANSACTION;
    verrmsg varchar2(32000);
  begin
    execute immediate vsql;
    commit;
  exception
    when others then
          verrmsg := errmsg ||' '||sqlerrm;
          select cast(verrmsg as varchar2(4000))
          into verrmsg
          from dual;
      if upper(mask) != 'Y' then
          execute immediate 'INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')';
          commit;
          raise;
      end if;
  end;



   /* Procedure: CreateObject. Executes a dynamic SQL statement and logs error in install_table*/
  PROCEDURE CREATEOBJECT(vsql varchar2,errmsg varchar2, mask varchar2 default 'N') IS
    verrmsg varchar2(32000);
  BEGIN
--    dbms_output.put_line(vsql);
    execute immediate vsql;
  exception
    when others then
          verrmsg := errmsg ||' '||sqlerrm;
          select cast(verrmsg as varchar2(4000))
          into verrmsg
          from dual;
      if upper(mask) != 'Y' then
          autonomoustransaction ('INSERT INTO install_table values('''||sysdate||''','''||replace(verrmsg,'''')||''')');
          raise;
      end if;
 
  END CREATEOBJECT;


function getpropertyvalue(property_key varchar2) return nvarchar2 as
  property_value nvarchar2(2000);
  language_id backend_application_resources.language%type;
  begin
     select value into language_id from config_setting where key='I18N_LANGUAGE_ID';

    if language_id is null then
      language_id :='en';
    end if;

    select value into property_value from backend_application_resources where key = property_key and language = language_id;

   if (property_value is null) then
      return '';
    else
      return property_value;
    end if;
end;


/* Procedure: SetDbCreateFileDest. Sets the default directory for Oracle datafiles so the path
              need not be given when creating tablespaces. */
  PROCEDURE SETDBCREATEFILEDEST(vpath varchar2) IS
  BEGIN
    execute immediate 'alter system set db_create_file_dest='''||vpath||''' scope=both';
  END SETDBCREATEFILEDEST;

/* Procedure: CreateSeq. Creates a sequence on a primary key of a table.
              Sequence name is <tablename>SEQ.*/
  PROCEDURE CREATESEQ(vtabname varchar2, vstartwith number default 1) IS
    vcolname user_cons_columns.column_name%type;
    vsql varchar2(32000);
  BEGIN
    select dcc.column_name
    into vcolname
    from user_constraints dc, user_cons_columns dcc
    where dc.table_name = dcc.table_name
    and dc.table_name=upper(vtabname)
    and dc.constraint_type='P'
    and dc.constraint_name = dcc.constraint_name
    and rownum < 2;

		-- Create Sequence for Table
      vsql :=  'CREATE SEQUENCE rcs.'||vtabname||'_SEQ INCREMENT BY 1 START WITH '||vstartwith||' NOCACHE';
      CREATEOBJECT(vsql,'Create Sequence'||vtabname||'_SEQ','Y');

    -- Create before insert trigger to populate the PK column
      vsql := 'CREATE OR REPLACE TRIGGER rcs.'||vtabname||'TRG BEFORE INSERT ON rcs.'||vtabname||' FOR EACH ROW
            DECLARE
              vseq number;
            BEGIN
              if (:new.'||vcolname||' is null) then
                select rcs.'||vtabname||'_SEQ.nextval into vseq from dual;
                :new.'||vcolname||' := vseq;
              end if;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'TRG');
   END CREATESEQ;

/* Procedure: CreateUpdTimeTRG. Updates the UpdateTime column with the current date and time when any
              update statement is issued against the table.*/
  PROCEDURE CREATEUPDATETRG(vtabname varchar2) IS
    vcount number := 0;
    vsql varchar2(32000);
  BEGIN
    select count(*)
    into vcount
    from user_tables dt, user_tab_columns dtc
    where dt.table_name = dtc.table_name
    and dt.table_name=upper(vtabname)
    and dtc.column_name = 'UPDATE_DATE';

	  if ( vcount > 0 ) then
      vsql := 'CREATE OR REPLACE TRIGGER rcs.'||vtabname||'UPD BEFORE UPDATE ON rcs.'||vtabname||' FOR EACH ROW
            DECLARE
              vsysdate timestamp;
            BEGIN
              select systimestamp into vsysdate from dual;
              :new.update_date := vsysdate;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'UPD');
    end if;

    select count(*)
    into vcount
    from user_tables dt, user_tab_columns dtc
    where dt.table_name = dtc.table_name
    and dt.table_name=upper(vtabname)
    and dtc.column_name = 'CREATE_DATE';

	  if ( vcount > 0 ) then
      vsql := 'CREATE OR REPLACE TRIGGER rcs.'||vtabname||'CRT BEFORE INSERT ON rcs.'||vtabname||' FOR EACH ROW
            DECLARE
              vsysdate timestamp;
            BEGIN
              select systimestamp into vsysdate from dual;
              :new.create_date := vsysdate;
              :new.update_date := vsysdate;
            END;';
      CREATEOBJECT(vsql,'Create Trigger '||vtabname||'UPD');
    end if;

   END CREATEUPDATETRG;

  --rowtocol function to pivot rows to a single column list with delimiter
   FUNCTION rowtocol( p_slct IN VARCHAR2,p_dlmtr IN VARCHAR2 DEFAULT ',' ) RETURN VARCHAR2
        AS
       TYPE c_refcur IS REF CURSOR;
       lc_str VARCHAR2(32767);
       lc_colval VARCHAR2(4000);
       c_dummy c_refcur;
       l number;
   BEGIN

     OPEN c_dummy FOR p_slct;
     LOOP
       FETCH c_dummy INTO lc_colval;
       EXIT WHEN c_dummy%NOTFOUND;
       lc_str := lc_str || p_dlmtr || lc_colval;
     END LOOP;
     CLOSE c_dummy;

     RETURN SUBSTR(lc_str,length(p_dlmtr)+1);

     EXCEPTION
       WHEN OTHERS THEN
         lc_str := SQLERRM;
           IF c_dummy%ISOPEN THEN
             CLOSE c_dummy;
           END IF;
         RETURN lc_str;
  END rowtocol;


  procedure epgguidedatalogging(vsql varchar2) is
  begin
    autonomoustransaction('insert into epgguidedata_log(description) values ('''||vsql||''')');
  end;

  --Create Alarms
  procedure alarmactivitylogging(vtype number default const_ALARM_TYPE_ALARM,vseverity number default  const_ALARM_SEVERITY_CRITICAL,vdesc varchar2,verrorcode number default null, vevent_name varchar2 default 'EPG_DOWNLOAD') is
    vsql varchar2(32767);
  begin
      vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY)
                select alarm_activitylog_seq.nextval,'||vtype||','||vseverity||','''||getpropertyvalue('rcs')||''',
                      substr('''||vdesc||''',1,2000),'''||vevent_name||''','''||nvl(verrorcode,'')||''','''||getpropertyvalue('system')||''','''||getpropertyvalue('system')||'''
                      from dual';
      --dbms_output.put_line(vsql);
      autonomoustransaction(vsql);
  end alarmactivitylogging;

  /*Procedure to serialize Recordables and Recordings operations in the db.
    Arguments:
      1. For locking Recordables operations, pass the following as argument: RECORDABLES_RESOURCE_LOCK
      2. For locking Recordings operations, pass the following as argument: RECORDINGS_RESOURCE_LOCK
  */
  PROCEDURE RESOURCELOCK(vresourcetype varchar2,vepgresourcetimeout varchar2 default null) IS
    vkeyvalue varchar2(1000);
    vresourcetimeout VARCHAR2(100);
  begin
    if vepgresourcetimeout is null then
      execute immediate 'select value from config_setting where key=:1' into vresourcetimeout using 'RESOURCE_TIMEOUT_SECS';
    else
      vresourcetimeout := vepgresourcetimeout ;
    end if;

    execute immediate 'select value from config_setting where key=:1 for update wait '||vresourcetimeout into vkeyvalue using vresourcetype;
  exception
    when others then
        /* Error Code 15 = Resource Locking Failure */
        alarmactivitylogging(const_ALARM_TYPE_ALARM,const_ALARM_SEVERITY_CRITICAL,replace(replace(getpropertyvalue('resource_lock_failed'), '{0}',vresourcetype), '{1}', regexp_replace(sqlerrm,'[[:punct:]]',' ')),15, 'RESOURCE_LOCK');
        raise;
  end;

  PROCEDURE RECORDABLESRECRESOURCELOCK IS
  begin
      RESOURCELOCK('RECORDABLES_RESOURCE_LOCK');
      RESOURCELOCK('RECORDINGS_RESOURCE_LOCK');
  end RECORDABLESRECRESOURCELOCK;


   /* Procedure: Populate vmaxscheduledatetime variable*/
  PROCEDURE init is
  BEGIN
    begin
      select current_timestamp + numtodsinterval(nvl(value,0),'SECOND')
      into vcurrentdatetime
      from config_setting, dual
      where key = 'EPG_Schedule_Delay_Secs';
    exception
      when others then
        select current_timestamp
        into vcurrentdatetime
        from dual;
    end;

    begin
      select current_timestamp + numtodsinterval(nvl(value,0),'SECOND')
      into vepgrecordingspan
      from config_setting, dual
      where key = 'EPG_AUTOCREATE_RECORDING_SPAN';
    exception
      when others then
        select current_timestamp + 1
        into vepgrecordingspan
        from dual;
    end;

  END init;


  procedure loadtstv(v_filename varchar2) is
    vsql varchar2(32767);

  begin

    --Logging start of load
    epgguidedatalogging('Start of EPG File = '||v_filename||'. StartTime = '||current_timestamp);
    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(getpropertyvalue('epg_file_loading_started'), '{0}',v_filename), '{1}', current_timestamp));

    vsql := 'truncate table stg_headend';
    CREATEOBJECT(vsql,'truncate table STG_HEADEND');

    vsql := 'truncate table err$_stg_headend';
    CREATEOBJECT(vsql,'truncate table err$_stg_headend');

    epgguidedatalogging('Start of Staging Headend Load. StartTime = '||current_timestamp);

    vsql := 'INSERT /*+ append */ INTO stg_headend (FILENAME,
    HEADEND_ID,
    COMMUNITY_NAME,
    COUNTY_NAME,
    COUNTY_SIZE,
    ST_COUNTY_CODE,
    STATE,
    ZIPCODE,
    DMA_CODE,
    DMA_NAME,
    MSO_CODE,
    DMA_RANK,
    HEADEND_NAME,
    HEADEND_LOCATION,
    MSO_NAME,
    TIME_ZONE)
    SELECT '''||v_filename||''',
    HE_HEADEND_ID,
    HE_COMMUNITY_NAME,
    HE_COUNTY_NAME,
    HE_COUNTY_SIZE,
    HE_ST_COUNTY_CODE,
    HE_STATE_SERVED,
    HE_ZIP_CODE,
    HE_DMA_CODE,
    HE_DMA_NAME,
    HE_MSO_CODE,
    HE_DMA_RANK,
    HE_HEADEND_NAME,
    HE_HEADEND_LOCATION,
    HE_MSO_NAME,
    HE_TIME_ZONE_CODE FROM stg_headend_view
    MINUS
    SELECT     '''||v_filename||''',
    HEADEND_ID,
    COMMUNITY_NAME,
    COUNTY_NAME,
    COUNTY_SIZE,
    ST_COUNTY_CODE,
    STATE,
    ZIPCODE,
    DMA_CODE,
    DMA_NAME,
    MSO_CODE,
    DMA_RANK,
    HEADEND_NAME,
    HEADEND_LOCATION,
    MSO_NAME,
    TIME_ZONE
    FROM HEADEND
    log errors into err$_stg_headend reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO STG_HEADEND');

    epgguidedatalogging('Start of Staging Station Load. StartTime = '||current_timestamp);
    commit;

    vsql := 'truncate table STG_STATION';
    CREATEOBJECT(vsql,'truncate table STG_STATION');

    vsql := 'truncate table err$_stg_station';
    CREATEOBJECT(vsql,'truncate err$_stg_station');

    --To resolve a bug with character mismatch with station_id, we are creating a temp table and then populating values into stg_station
    vsql := 'drop table temp_station';
    CREATEOBJECT(vsql,'drop table temp_station','Y');

    vsql := 'CREATE TABLE TEMP_STATION
    (FILENAME,
    STATION_ID,
    TIME_ZONE,
    STATION_NAME,
    CALL_SIGN,
    AFFILIATION,
    CITY,
    STATE,
    ZIPCODE,
    COUNTRY,
    DMA_NAME,
    DMA_NUMBER,
    FCC_CHANNEL_NUM,
    USER_DATA1,
    USER_DATA2,
    USER_DATA3,
    USER_DATA4,
    USER_DATA5,
    USER_DATA6,
    USER_DATA7,
    USER_DATA8,
    USER_DATA9,
    USER_DATA10) nologging
    AS
    SELECT '''||v_filename||''',
    STATION_ID,
    STATION_TIME_ZONE,
    STATION_NAME,
    STATION_CALL_SIGN,
    STATION_AFFIL,
    STATION_CITY,
    STATION_STATE,
    STATION_ZIP_CODE,
    STATION_COUNTRY,
    DMA_NAME,
    DMA_NUM,
    FCC_CHANNEL_NUM,
    USER_DATA1,
    USER_DATA2,
    USER_DATA3,
    USER_DATA4,
    USER_DATA5,
    USER_DATA6,
    USER_DATA7,
    USER_DATA8,
    USER_DATA9,
    USER_DATA10
    FROM STG_STATION_VIEW';
    CREATEOBJECT(vsql,'create table temp_station');


    vsql := 'INSERT /*+ append */ INTO STG_STATION
    (FILENAME,
    STATION_ID,
    TIME_ZONE,
    STATION_NAME,
    CALL_SIGN,
    AFFILIATION,
    CITY,
    STATE,
    ZIPCODE,
    COUNTRY,
    DMA_NAME,
    DMA_NUMBER,
    FCC_CHANNEL_NUM,
    USER_DATA1,
    USER_DATA2,
    USER_DATA3,
    USER_DATA4,
    USER_DATA5,
    USER_DATA6,
    USER_DATA7,
    USER_DATA8,
    USER_DATA9,
    USER_DATA10)
    SELECT FILENAME,
    STATION_ID,
    TIME_ZONE,
    STATION_NAME,
    CALL_SIGN,
    AFFILIATION,
    CITY,
    STATE,
    ZIPCODE,
    COUNTRY,
    DMA_NAME,
    DMA_NUMBER,
    FCC_CHANNEL_NUM,
    USER_DATA1,
    USER_DATA2,
    USER_DATA3,
    USER_DATA4,
    USER_DATA5,
    USER_DATA6,
    USER_DATA7,
    USER_DATA8,
    USER_DATA9,
    USER_DATA10
    FROM TEMP_STATION
    log errors into err$_stg_station reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO stg_station');

    epgguidedatalogging('Start of Staging Lineup Load. StartTime = '||current_timestamp);

    commit;


    vsql := 'truncate table STG_LINEUP';
    CREATEOBJECT(vsql,'truncate table STG_LINEUP');

    vsql := 'truncate table err$_stg_lineup';
    CREATEOBJECT(vsql,'truncate table err$_stg_lineup');

    --To resolve a bug with character mismatch with station_id, we are creating a temp table and then populating values into stg_lineup
    vsql := 'drop table temp_lineup';
    CREATEOBJECT(vsql,'drop table temp_lineup','Y');

    vsql := 'CREATE TABLE TEMP_LINEUP (FILENAME,
    CL_ROW_ID,
    CL_STATION_NUM,
    CL_HEADEND_ID,
    CL_DEVICE,
    CL_TMS_CHAN,
    CL_SERVICE_TIER,
    CL_EFFECTIVE_DATE,
    CL_EXPIRATION_DATE) nologging
    AS
    SELECT '''||v_filename||''',
    CL_ROW_ID,
    CL_STATION_NUM,
    CL_HEADEND_ID,
    CL_DEVICE,
    CL_TMS_CHAN,
    CL_SERVICE_TIER,
    case when cl_effective_date is not null then to_timestamp_tz(CL_EFFECTIVE_DATE||'' UTC'',''YYYY-MM-DD"T"HH24:MI:SS.FF TZR'') else null end as CL_EFFECTIVE_DATE,
    case when cl_expiration_date is not null then to_timestamp_tz(CL_EXPIRATION_DATE ||'' UTC'',''YYYY-MM-DD"T"HH24:MI:SS.FF TZR'') else null end as CL_EXPIRATION_DATE
    FROM STG_LINEUP_VIEW
    MINUS
    SELECT '''||v_filename||''',
    LINEUP_ID,
    STATION_ID,
    HEADEND_ID,
    DEVICE,
    TMS_CHANNEL,
    SERVICE_TIER,
    EFFECTIVE_DATE,
    EXPIRATION_DATE FROM LINEUP';
    CREATEOBJECT(vsql,'CREATE TABLE TEMP_LINEUP');

    vsql := 'INSERT /*+ append */ INTO STG_LINEUP (FILENAME,
    ROW_ID,
    STATION_NUM,
    HEADEND_ID,
    DEVICE,
    TMS_CHANNEL,
    SERVICE_TIER,
    EFFECTIVE_DATE,
    EXPIRATION_DATE)
    SELECT FILENAME,
    CL_ROW_ID,
    CL_STATION_NUM,
    CL_HEADEND_ID,
    CL_DEVICE,
    CL_TMS_CHAN,
    CL_SERVICE_TIER,
    CL_EFFECTIVE_DATE,
    CL_EXPIRATION_DATE FROM TEMP_LINEUP
    log errors into err$_stg_lineup reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO STG_LINEUP');

    epgguidedatalogging('Start of Staging Program Load. StartTime = '||current_timestamp);

    commit;

    vsql := 'truncate table stg_program';
    CREATEOBJECT(vsql,'truncate table stg_program');

    vsql := 'truncate table err$_stg_program';
    CREATEOBJECT(vsql,'truncate err$_stg_program');

    vsql := 'DROP INDEX IDX_STGPRG_PRGID';
    CREATEOBJECT(vsql,'drop index IDX_STGPRG_PRGID','Y');

    vsql := 'insert /*+ append */ into stg_program(FILENAME,
        program_id,
        LANGUAGE,
        TITLE,
        REDUCED_TITLE1,
        REDUCED_TITLE2,
        REDUCED_TITLE3,
        REDUCED_TITLE4,
        ALT_TITLE,
        REDUCED_DESCRIPTION1,
        REDUCED_DESCRIPTION2,
        REDUCED_DESCRIPTION3,
        REDUCED_DESCRIPTION4,
        ADVISORY_DESC1,
        ADVISORY_DESC2,
        ADVISORY_DESC3,
        ADVISORY_DESC4,
        ADVISORY_DESC5,
        ADVISORY_DESC6,
        GENRE_DESC1,
        GENRE_DESC2,
        GENRE_DESC3,
        GENRE_DESC4,
        GENRE_DESC5,
        GENRE_DESC6,
        DESCRIPTION1,
        DESCRIPTION2,
        YEAR_PROGRAM,
        MPAA_RATING,
        STAR_RATING,
        RUN_TIME,
        COLOR_CODE,
        PROGRAM_LANGUAGE,
        COUNTRY_OF_ORIGIN,
        MADE_FOR_TV,
        SOURCE_TYPE,
        SHOW_TYPE,
        HOLIDAY,
        SYNDICATE_EPISODE_NUM,
        ALT_SYNDICATE_EPI_NUM,
        EPISODE_TITLE,
        NET_SYN_SOURCE,
        NET_SYN_TYPE,
        ORG_STUDIO,
        GAME_DATETIME,
        GAME_TIMEZONE,
        ORG_AIR_DATE,
        UNIQUE_ID,
--        AUDIO_COMPONENT,
--        SUBTITLE_COMPONENT,
        USER_DATA1 ,
        USER_DATA2 ,
        USER_DATA3 ,
        USER_DATA4 ,
        USER_DATA5 ,
        USER_DATA6 ,
        USER_DATA7 ,
        USER_DATA8 ,
        USER_DATA9 ,
        USER_DATA10 ,
        USER_DATA11 ,
        USER_DATA12 ,
        USER_DATA13 ,
--        PROGRAM_RATING,
        PROGRAM_GENERIC,
        CAST_NAME_ROLE,
        CREDITS_NAME_ROLE
        )
        select '''||v_filename||''',
        DATABASE_KEY,
        LANGUAGE,
        TITLE,
        REDUCED_TITLE1,
        REDUCED_TITLE2,
        REDUCED_TITLE3,
        REDUCED_TITLE4,
        ALT_TITLE,
        REDUCED_DESC1,
        REDUCED_DESC2,
        REDUCED_DESC3,
        REDUCED_DESC4,
        ADVISORY_DESC1,
        ADVISORY_DESC2,
        ADVISORY_DESC3,
        ADVISORY_DESC4,
        ADVISORY_DESC5,
        ADVISORY_DESC6,
        GENRE_DESC1,
        GENRE_DESC2,
        GENRE_DESC3,
        GENRE_DESC4,
        GENRE_DESC5,
        GENRE_DESC6,
        DESCRIPTION1,
        DESCRIPTION2,
        YEAR_PROGRAM,
        MPAA_RATING,
        STAR_RATING,
        RUN_TIME,
        COLOR_CODE,
        PROGRAM_LANGUAGE,
        ORG_COUNTRY,
        MADE_FOR_TV,
        SOURCE_TYPE,
        SHOW_TYPE,
        HOLIDAY,
        SYN_EPI_NUM,
        ALT_SYN_EPI_NUM,
        EPI_TITLE,
        NET_SYN_SOURCE,
        NET_SYN_TYPE,
        ORG_STUDIO,
        case when GAME_DATE is not null then to_timestamp_tz(GAME_DATE||'' UTC'',''YYYY-MM-DD"T"HH24:MI:SS.FF TZR'') else null end as GAME_DATE,
        GAME_TIMEZONE,
        case when ORG_AIR_DATE is not null then to_timestamp_tz(ORG_AIR_DATE ||'' UTC'',''YYYY-MM-DD"T"HH24:MI:SS.FF TZR'') else null end as ORG_AIR_DATE,
        UNIQUE_ID,
--        sys_XMLGen(AUDIO_COMPONENT),
--        sys_XMLGen(SUBTITLE_COMPONENT),
        USER_DATA1 ,
        USER_DATA2 ,
        USER_DATA3 ,
        USER_DATA4 ,
        USER_DATA5 ,
        USER_DATA6 ,
        USER_DATA7 ,
        USER_DATA8 ,
        USER_DATA9 ,
        USER_DATA10,
        USER_DATA11,
        USER_DATA12,
        USER_DATA13,
--        sys_XMLGen(PROGRAM_RATING),
        sys_XMLGen(PROGRAM_GENERIC),
        sys_XMLGen(CAST_NAME_ROLE),
        sys_XMLGen(CREDITS_NAME_ROLE)
      from stg_program_view
    log errors into err$_stg_program reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO stg_program');

    commit;

    vsql := 'CREATE INDEX IDX_STGPRG_PRGID ON STG_PROGRAM(program_id) nologging';
    CREATEOBJECT(vsql,'create index stg_program','Y');
/*    epgguidedatalogging('Delete Duplicate Program Load. StartTime = '||current_timestamp);

    vsql := 'delete from stg_program
                where database_key not in (
                  select database_key from
                    (SELECT DATABASE_KEY,
                      LANGUAGE,
                      TITLE,
                      REDUCED_TITLE1,
                      REDUCED_TITLE2,
                      REDUCED_TITLE3,
                      REDUCED_TITLE4,
                      ALT_TITLE,
                      REDUCED_DESCRIPTION1,
                      REDUCED_DESCRIPTION2,
                      REDUCED_DESCRIPTION3,
                      REDUCED_DESCRIPTION4,
                      ADVISORY_DESC1,
                      ADVISORY_DESC2,
                      ADVISORY_DESC3,
                      ADVISORY_DESC4,
                      ADVISORY_DESC5,
                      ADVISORY_DESC6,
                      GENRE_DESC1,
                      GENRE_DESC2,
                      GENRE_DESC3,
                      GENRE_DESC4,
                      GENRE_DESC5,
                      GENRE_DESC6,
                      DESCRIPTION1,
                      DESCRIPTION2,
                      YEAR_PROGRAM,
                      MPAA_RATING,
                      STAR_RATING,
                      RUN_TIME,
                      COLOR_CODE,
                      PROGRAM_LANGUAGE,
                      COUNTRY_OF_ORIGIN,
                      MADE_FOR_TV,
                      SOURCE_TYPE,
                      SHOW_TYPE,
                      HOLIDAY,
                      SYNDICATE_EPISODE_NUM,
                      ALT_SYNDICATE_EPI_NUM,
                      EPISODE_TITLE,
                      NET_SYN_SOURCE,
                      NET_SYN_TYPE,
                      ORG_STUDIO,
                      GAME_DATETIME,
                      GAME_TIMEZONE,
                      ORG_AIR_DATE,
                      UNIQUE_ID
                    from stg_program
                    MINUS
                    SELECT    PROGRAM_ID,
                      LANGUAGE,
                      TITLE,
                      REDUCED_TITLE1,
                      REDUCED_TITLE2,
                      REDUCED_TITLE3,
                      REDUCED_TITLE4,
                      ALT_TITLE,
                      REDUCED_DESCRIPTION1,
                      REDUCED_DESCRIPTION2,
                      REDUCED_DESCRIPTION3,
                      REDUCED_DESCRIPTION4,
                      ADVISORY_DESC1,
                      ADVISORY_DESC2,
                      ADVISORY_DESC3,
                      ADVISORY_DESC4,
                      ADVISORY_DESC5,
                      ADVISORY_DESC6,
                      GENRE_DESC1,
                      GENRE_DESC2,
                      GENRE_DESC3,
                      GENRE_DESC4,
                      GENRE_DESC5,
                      GENRE_DESC6,
                      DESCRIPTION1,
                      DESCRIPTION2,
                      YEAR_PROGRAM,
                      MPAA_RATING,
                      STAR_RATING,
                      RUN_TIME,
                      COLOR_CODE,
                      PROGRAM_LANGUAGE,
                      COUNTRY_OF_ORIGIN,
                      MADE_FOR_TV,
                      SOURCE_TYPE,
                      SHOW_TYPE,
                      HOLIDAY,
                      SYNDICATE_EPISODE_NUM,
                      ALT_SYNDICATE_EPI_NUM,
                      EPISODE_TITLE,
                      NET_SYN_SOURCE,
                      NET_SYN_TYPE,
                      ORG_STUDIO,
                      GAME_DATETIME,
                      GAME_TIMEZONE,
                      ORG_AIR_DATE,
                      UNIQUE_ID
                      FROM PROGRAM
                    )
                )';
    CREATEOBJECT(vsql,'DELETE DUPLICATES FROM stg_program');
*/
    epgguidedatalogging('Start of Staging CastCredits Load. StartTime = '||current_timestamp);

    commit;

    vsql := 'truncate table stg_castcredits';
    CREATEOBJECT(vsql,'truncate table stg_castcredits');

    vsql := 'truncate table err$_stg_castcredits';
    CREATEOBJECT(vsql,'truncate err$_stg_castcredits');

    vsql := 'insert /*+ append */ into stg_castcredits(FILENAME,
    DATABASE_KEY,
    TYPE,
    FIRST_NAME,
    LAST_NAME,
    ROLE_DESC)
    select pv.FILENAME,
    pv.program_id,
    cast(''CAST'' as nvarchar2(100)),
    xmldata.cast_first_name,
    xmldata.cast_last_name,
    xmldata.CAST_ROLE_DESC
    from stg_program pv,
      XMLTABLE(''/CAST_NAME_ROLE/CAST_NAME_ROLE'' passing pv.CAST_NAME_ROLE columns
      CAST_FIRST_NAME NVARCHAR2(100) path ''CAST_FIRST_NAME'',
      CAST_LAST_NAME NVARCHAR2(100) path ''CAST_LAST_NAME'',
      CAST_ROLE_DESC NVARCHAR2(100) path ''CAST_ROLE_DESC''
      ) xmldata
    WHERE pv.FILENAME = '''||v_filename||'''
    and pv.cast_name_role is not null
    log errors into err$_stg_castcredits reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO CAST');

    commit;

    vsql := 'insert /*+ append */ into stg_castcredits(FILENAME,
    DATABASE_KEY,
    TYPE,
    FIRST_NAME,
    LAST_NAME,
    ROLE_DESC)
    select pv.FILENAME,
    pv.program_id,
    cast(''CREDITS'' as nvarchar2(100)),
    xmldata.CREDITS_FIRST_NAME,
    xmldata.CREDITS_LAST_NAME,
    xmldata.CREDITS_ROLE_DESC
    from stg_program pv,
    XMLTABLE(''/CREDITS_NAME_ROLE/CREDITS_NAME_ROLE'' passing pv.CREDITS_NAME_ROLE columns
        CREDITS_FIRST_NAME NVARCHAR2(100) path ''CREDITS_FIRST_NAME'',
        CREDITS_LAST_NAME NVARCHAR2(100) path ''CREDITS_LAST_NAME'',
        CREDITS_ROLE_DESC NVARCHAR2(100) path ''CREDITS_ROLE_DESC''
        ) xmldata
    WHERE pv.FILENAME = '''||v_filename||'''
    and pv.credits_name_role is not null
    log errors into err$_stg_castcredits reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO CREDITS');

    epgguidedatalogging('Start of Staging Schedule Load. StartTime = '||current_timestamp);

    commit;

    vsql := 'truncate table stg_schedule';
    CREATEOBJECT(vsql,'truncate table stg_schedule');

    vsql := 'truncate table err$_stg_schedule';
    CREATEOBJECT(vsql,'truncate err$_stg_schedule');

    vsql := 'DROP INDEX IDX_STGSCH_PK';
    CREATEOBJECT(vsql,'Drop Index IDX_STGSCH_PK','Y');

    vsql := 'DROP INDEX index_sid_stg_schedule';
    CREATEOBJECT(vsql,'Drop Index index_sid_stg_schedule','Y');

    commit;

    vsql := 'insert /*+ append  */ into stg_schedule(FILENAME,
    STATION_NUM,
    DATABASE_KEY,
    SCHEDULE_LANGUAGE,
    start_time,
    DURATION,
    PART_NUMBER,
    NO_OF_PARTS,
    CC,
    STEREO,
    REPEAT,
    LIVE_TAPE_DELAY,
    SUBTITLED,
    PREMIERE_FINALE,
    JOINED_IN_PROGRESS,
    CABLE_IN_CLASSROOM,
    TV_RATING,
    SAP,
    BLACKOUT,
    SEX_RATING,
    VIOLENCE_RATING,
    LANGUAGE_RATING,
    DIALOG_RATING,
    FV_RATING,
    ENHANCED,
    THREE_D,
    LETTERBOX,
    HD_TV,
    DOLBY,
    DVS,
    REQUEST_RECORD,
--    EVENT_ID,
     USER_DATA1,
     USER_DATA2,
     USER_DATA3,
     USER_DATA4,
     USER_DATA5,
     USER_DATA6,
     USER_DATA7,
     USER_DATA8,
     USER_DATA9,
     USER_DATA10,
     USER_DATA11,
     SCHEDULE_GENERIC,
    SERIES_LINK)
    SELECT '''||v_filename||''',
    STATION_NUM,
    DATABASE_KEY,
    SCHEDULE_LANGUAGE,
 	 to_timestamp_tz(AIR_DATE||'' UTC'',''YYYY-MM-DD"T"HH24:MI:SS.FF TZR'')  as AIR_DATE,
     /** The Duration Format is HHMMSS  054530 **/
    (floor(to_number(DURATION)/10000)) *60 *60 + floor (mod(to_number(DURATION),10000) /100)*60 + mod(to_number(DURATION),100) as DURATION,
   PART_NUM,
    NUM_OF_PARTS,
    CC,
    STEREO,
    REPEAT,
    LIVE_TAPE_DELAY,
    SUBTITLED,
    PREMIERE_FINALE,
    JOINED_IN_PROGRESS,
    CABLE_IN_THE_CLASSROOM,
    TV_RATING,
    SAP,
    BLACKOUT,
    SEX_RATING,
    VIOLENCE_RATING,
    LANGUAGE_RATING,
    DIALOG_RATING,
    FV_RATING,
    ENHANCED,
    THREE_D,
    LETTERBOX,
    HDTV,
    DOLBY,
    DVS,
    REQUEST_RECORD,
--    sys_XMLGen(EVENT_ID),
     USER_DATA1,
     USER_DATA2,
     USER_DATA3,
     USER_DATA4,
     USER_DATA5,
     USER_DATA6,
     USER_DATA7,
     USER_DATA8,
     USER_DATA9,
     USER_DATA10,
     USER_DATA11,
    sys_XMLGen(SCHEDULE_GENERIC),
    sys_XMLGen(SERIES_LINK)
    from stg_schedule_view
    log errors into err$_stg_schedule reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO stg_schedule');

    commit;
    vsql := 'CREATE INDEX IDX_STGSCH_PK ON STG_SCHEDULE(STATION_NUM,DATABASE_KEY,start_time)  nologging';
    CREATEOBJECT(vsql,'Create Index IDX_STGSCH_PK','Y');

    commit;
    vsql :=' create index index_sid_stg_schedule on stg_schedule(database_key)   nologging';
    CREATEOBJECT(vsql,'create index on database_key on the table STG_SCHEDULE');

    epgguidedatalogging('Start of Staging Series Load. StartTime = '||current_timestamp);

    commit;

    vsql := 'truncate table stg_series';
    CREATEOBJECT(vsql,'truncate table stg_schedule');

    vsql := 'truncate table err$_stg_series';
    CREATEOBJECT(vsql,'truncate err$_stg_series');

    vsql := 'insert /*+ append */ into stg_series(FILENAME,
    PROGRAM_ID,
    SERIES_ID)
    select sv.FILENAME,
    sv.DATABASE_KEY,
    xmldata.SERIESIDENTIFIER
    from stg_schedule sv,
    XMLTABLE(''/SERIES_LINK/SERIES_LINK'' passing sv.SERIES_LINK columns
        SERIESIDENTIFIER NVARCHAR2(200) path ''SeriesIdentifier''
        ) xmldata
    WHERE sv.FILENAME = '''||v_filename||'''
    log errors into err$_stg_series reject limit unlimited';
    CREATEOBJECT(vsql,'LOAD INTO SERIES');

     epgguidedatalogging('End of Staging Load. StartTime = '||current_timestamp);

    commit;

    --Logging End of loading EPG File to Staging tables
    epgguidedatalogging('End of Loading EPG File = '||v_filename||'. EndTime = '||current_timestamp);
    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(getpropertyvalue('epg_file_loading_ended'), '{0}',v_filename), '{1}', current_timestamp));


/*
    exception
    when others then
      autonomoustransaction('update epgxmlfiles set processendtime = systimestamp where fileid = '||vfileid);
      raise; */
  end loadtstv;

  procedure mergetstv is
    vsql varchar2(32767);
    vsqlwhere varchar2(32767);
    vschsqlwhere varchar2(32767);
    vmaxschedulestarttime schedule.start_time%type;
    vminstgschdate timestamp(0) with time zone;
    vmaxstgschdate timestamp(0) with time zone;
    vcnt_affectedrecording number := 0;
    is_epg_sch_conflict_completed varchar2(1) := 'Y';
    sch_prg_md_chng_update_p_v backend_application_resources.value%type := getpropertyvalue('schedule_program_metadata_changed_to_update');
    sch_prg_md_chng_delete_p_v backend_application_resources.value%type := getpropertyvalue('schedule_program_metadata_changed_to_delete');
    sch_prg_md_chng_too_late_p_v backend_application_resources.value%type := getpropertyvalue('schedule_program_metadata_changed_too_late');
    rm_name_p_v  backend_application_resources.value%type := getpropertyvalue('rm_name');
    id_p_v  backend_application_resources.value%type := getpropertyvalue('id');
    schedule_start_time_p_v  backend_application_resources.value%type := getpropertyvalue('schedule_start_time');
    program_title_p_v  backend_application_resources.value%type := getpropertyvalue('program_title');
    station_call_sign_p_v  backend_application_resources.value%type := getpropertyvalue('station_call_sign');
    rcs_p_v backend_application_resources.value%type := getpropertyvalue('rcs');
    system_p_v backend_application_resources.value%type := getpropertyvalue('system');
    prg_md_chng_p_v backend_application_resources.value%type := getpropertyvalue('program_metadata_changed');
  begin

    init();

     epgguidedatalogging('Start of Merge Headend. StartTime = '||current_timestamp);
    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(getpropertyvalue('epg_file_merging_started'), '{0}',vfilename), '{1}', current_timestamp));

    vsql := 'merge into headend hprod
    using stg_headend hstg
    on (hstg.headend_id = hprod.headend_id)
    when matched then
    update set hprod.community_name = hstg.community_name, hprod.county_name = hstg.county_name, hprod.county_size = hstg.county_size, hprod.st_county_code = hstg.st_county_code, hprod.state = hstg.state , hprod.zipcode = hstg.zipcode, hprod.dma_code = hstg.dma_code, hprod.dma_name = hstg.dma_name , hprod.mso_code = hstg.mso_code, hprod.dma_rank = hstg.dma_rank , hprod.headend_name = hstg.headend_name , hprod.headend_location = hstg.headend_location, hprod.mso_name = hstg.mso_name, hprod.time_zone = hstg.time_zone, hprod.created_by = ''SYSTEM'', hprod.updated_by = ''SYSTEM''
    when not matched then
    insert (hprod.headend_id,hprod.community_name, hprod.county_name, hprod.county_size, hprod.st_county_code, hprod.state, hprod.zipcode, hprod.dma_code, hprod.dma_name, hprod.mso_code, hprod.dma_rank, hprod.headend_name, hprod.headend_location, hprod.mso_name, hprod.time_zone, hprod.created_by,hprod.updated_by)
    values (hstg.headend_id,hstg.community_name, hstg.county_name, hstg.county_size, hstg.st_county_code, hstg.state, hstg.zipcode, hstg.dma_code, hstg.dma_name, hstg.mso_code, hstg.dma_rank, hstg.headend_name, hstg.headend_location, hstg.mso_name, hstg.time_zone,''SYSTEM'',''SYSTEM'')';
    CREATEOBJECT(vsql,'MERGE INTO HEADEND');

	 vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||getpropertyvalue('rcs')||''',
                  replace(replace(replace('''||getpropertyvalue('defaulting_headend_timezone_to_UTC')||''', ''{0}'', he.headend_name), ''{1}'', he.headend_location), ''{2}'',he.headend_id),

                  ''EPG_DOWNLOAD'',23,'''||getpropertyvalue('system')||''','''||getpropertyvalue('system')||'''
                  from (select he.headend_id, he.headend_location, he.headend_name from headend he where not exists (
            (SELECT 1 from V$TIMEZONE_NAMES vt where vt.TZNAME not in (''GMT+0'',''GMT-0'',''ROC'')  and vt.tzname = he.time_zone))) he';

     vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;

     CREATEOBJECT(vsql,'Creating alarms for headends that have invalid time zones ');


     update headend set time_zone = 'UTC' where headend_id in (select he.headend_id from headend he where not exists (
            (SELECT 1 from V$TIMEZONE_NAMES vt where vt.TZNAME not in ('GMT+0','GMT-0','ROC')  and vt.tzname = he.time_zone)));

    commit;

     epgguidedatalogging('Start of Merge Station. StartTime = '||current_timestamp);

    vsql := 'merge into station sprod
    using stg_station sstg
    on (sstg.station_id = sprod.station_id)
    when matched then
    update set
    sprod.station_name = sstg.station_name,
    sprod.time_zone = sstg.time_zone,
    sprod.call_sign = sstg.call_sign,
    sprod.city = sstg.city,
    sprod.state = sstg.state,
    sprod.zipcode = sstg.zipcode,
    sprod.country = sstg.country,
    sprod.dma_name = sstg.dma_name,
    sprod.dma_number = sstg.dma_number,
    sprod.fcc_channel_num = sstg.fcc_channel_num,
    sprod.affiliation = sstg.affiliation ,
    sprod.USER_DATA1 = sstg.USER_DATA1,
    sprod.USER_DATA2 = sstg.USER_DATA2,
    sprod.USER_DATA3 = sstg.USER_DATA3,
    sprod.USER_DATA4 = sstg.USER_DATA4,
    sprod.USER_DATA5 = sstg.USER_DATA5,
    sprod.USER_DATA6 = sstg.USER_DATA6,
    sprod.USER_DATA7 = sstg.USER_DATA7,
    sprod.USER_DATA8 = sstg.USER_DATA8,
    sprod.USER_DATA9 = sstg.USER_DATA9,
    sprod.USER_DATA10 = sstg.USER_DATA10,
    sprod.created_by = ''SYSTEM'', sprod.updated_by = ''SYSTEM''
    when not matched then
    insert (sprod.station_id,sprod.station_name, sprod.time_zone, sprod.call_sign, sprod.city, sprod.state, sprod.zipcode, sprod.country, sprod.dma_name, sprod.dma_number, sprod.fcc_channel_num, sprod.affiliation,
     sprod.USER_DATA1,
     sprod.USER_DATA2,
     sprod.USER_DATA3,
     sprod.USER_DATA4,
     sprod.USER_DATA5,
     sprod.USER_DATA6,
     sprod.USER_DATA7,
     sprod.USER_DATA8,
     sprod.USER_DATA9,
     sprod.USER_DATA10,
     sprod.created_by, sprod.updated_by)
    values (sstg.station_id,sstg.station_name, sstg.time_zone, sstg.call_sign, sstg.city, sstg.state,sstg.zipcode,sstg.country, sstg.dma_name, sstg.dma_number, sstg.fcc_channel_num,sstg.affiliation,
     sstg.USER_DATA1,
     sstg.USER_DATA2,
     sstg.USER_DATA3,
     sstg.USER_DATA4,
     sstg.USER_DATA5,
     sstg.USER_DATA6,
     sstg.USER_DATA7,
     sstg.USER_DATA8,
     sstg.USER_DATA9,
     sstg.USER_DATA10,
    ''SYSTEM'', ''SYSTEM'')
    log errors into err$_station reject limit unlimited';
    CREATEOBJECT(vsql,'MERGE INTO STATION');

    commit;

     epgguidedatalogging('Start of Merge Lineup. StartTime = '||current_timestamp);

    vsql := 'merge into lineup lprod
    using stg_lineup lstg
    on (lstg.row_id = lprod.lineup_id and lprod.headend_id = lstg.headend_id and lprod.station_id = lstg.station_num)
    when matched then
    update set lprod.device = lstg.device, lprod.tms_channel = lstg.tms_channel, lprod.service_tier = lstg.service_tier, lprod.effective_date = lstg.effective_date, lprod.expiration_date = lstg.expiration_date , lprod.created_by = ''SYSTEM'' , lprod.updated_by = ''SYSTEM''
    when not matched then
    insert (lprod.lineup_id,lprod.device, lprod.tms_channel, lprod.service_tier, lprod.effective_date, lprod.expiration_date, lprod.headend_id, lprod.station_id, lprod.created_by, lprod.updated_by)
    values (lstg.row_id,lstg.device, lstg.tms_channel, lstg.service_tier, lstg.effective_date, lstg.expiration_date, lstg.headend_id, lstg.station_num, ''SYSTEM'', ''SYSTEM'')
    log errors into err$_lineup reject limit unlimited';
    CREATEOBJECT(vsql,'MERGE INTO LINEUP');

    commit;

    /* Program Metadata has changed? If so, raise Alarm. Error Code 13 = Program Change */
    vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||rcs_p_v||''','''||prg_md_chng_p_v||'''||p.program_id,''EPG_DOWNLOAD'',13,'''||system_p_v||''','''||system_p_v||''' from program p, stg_program pstg where p.program_id = pstg.program_id and exists (select 1 from schedule s, recording r where s.schedule_id=r.schedule_id and s.start_time > to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'') and s.program_id = pstg.program_id ) ';

    --Get list of program table columns that constitute as a metadata from Config_settings table.
/*    select '('||rowtocol ('select ''p.''||col||'' != pstg.''||col from
    (select regexp_substr(value, ''[^\,]+'', 1, ao.rn) as col from config_setting cs
    join
    (select rownum rn from all_objects where rownum < 100) ao
    on
    (rn <= length(regexp_replace(value, ''[^\,]'',''''))+1 and cs.key=''PROGRAM.METADATA'')) tabcol',' OR ') || ')'
    into vsqlwhere
    from dual;
*/
    vsqlwhere := ' and ('||rowtocol ('select ''nvl(p.''||col||'','''' '''') != nvl(pstg.''||col||'','''' '''')'' from (select regexp_substr(value, ''[^\,]+'', 1, ao.rn) as col from config_setting cs join (select rownum rn from all_objects where rownum < 100) ao on (rn <= length(regexp_replace(value, ''[^\,]'',''''))+1 and cs.key=''PROGRAM.METADATA'')) tabcol',' OR ') || ')';

    if vsqlwhere = ' and ()'
      then vsqlwhere := null;
    end if;

    epgguidedatalogging('Start of Program Metadata Check and store in temp_recording table . StartTime = '||current_timestamp);


    vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql || vsqlwhere;
--    dbms_output.put_line(vsql);
    CREATEOBJECT(vsql,'PROGRAM METADATA CHECK');

    commit;

    --Insert into Recording_queue here
    vsql := 'drop table temp_recording';
    CREATEOBJECT(vsql,'DROP TABLE TEMP_RECORDING','Y');

     vsql := 'truncate  table temp_station_sch_range';
    CREATEOBJECT(vsql,'DROP TABLE temp_station_sch_range','Y');


    vsql := 'create table temp_recording nologging as
    select distinct r.recording_id, s.schedule_id,s.station_id,cast(''PROGRAM.METADATA'' as varchar2(4000)) as description,
    s.start_time as schedule_start_time, s.end_time as schedule_end_time, s.duration as schedule_duration,
    s.hd_tv as hd_tv, s.cc as cc,s.repeat as repeat,
    p.program_id, pstg.episode_title as episode_title,
    pstg.year_program as year_program, cast(null as nvarchar2(100)) as tv_rating,
    case when mcmr.name != p.mpaa_rating then mcmr.name else null end as mpaa_rating,
    pstg.title as program_title, pstg.description1 as program_description, rm.name as rm_name , rm.rm_id,
    case
    when  r.adjusted_start_time < current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0), ''SECOND'') then cast(0 as number(1))
    else cast(1 as number(1))
    end as update_allowed
    from stg_program pstg, program p, schedule s, recording r, rtc_pool rp, rm,
    (select * from metadata_config where type=''MPAARating'' ) mcmr
    where pstg.program_id = p.program_id
    and r.schedule_id = s.schedule_id
    and r.pool_id = rp.pool_id
    and rm.rm_id = rp.rm_id
    and s.program_id = p.program_id
    and r.status != '||const_recording_DELETED||'
    and mcmr.name(+) = upper(pstg.mpaa_rating)
    and s.start_time > to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')
    '||vsqlwhere;
    CREATEOBJECT(vsql,'CREATE TABLE TEMP_RECORDING');

    --Construct dynamic where clause for entity mapping entries with entity_mapping.type = 1


    vsqlwhere := ' and ('||rowtocol ('SELECT distinct ''(ENTITY_MAPPING.MAPPING_ID=''||em.mapping_id||''
            AND  nvl(''||decode(em.column_name,''PROGRAM_GENERIC'',em.table_name||''.''||''PROGRAM_GENERIC.getStringVal()''
            ,''SCHEDULE_GENERIC'',em.table_name||''.''||''SCHEDULE_GENERIC.getStringVal()''
            ,''SERIES_LINK'',em.table_name||''.''||''SERIES_LINK.getStringVal()''
            ,''to_nchar(''||em.table_name||''.''||em.column_name||'')'')||'',''''X'''')

            !=  nvl(''||decode(em.column_name,''PROGRAM_GENERIC'',
            ''STG_''||em.table_name||''.''||''PROGRAM_GENERIC.getStringVal()'',
            ''SCHEDULE_GENERIC'',''STG_''||em.table_name||''.''||''SCHEDULE_GENERIC.getStringVal()'',
            ''SERIES_LINK'',''STG_''||em.table_name||''.''||''SERIES_LINK.getStringVal()'',
            ''to_nchar(STG_''||em.table_name||''.''||em.column_name||'')'')||'',''''X''''))'' AS dynclause
    from schedule s, recording r , rec_trial_use_fields rtuf, entity_mapping em
    where s.schedule_id = r.schedule_id
    and r.recording_id = rtuf.recording_id
    and rtuf.mapping_id = em.mapping_id
    and em.type in (1,3)
    and s.start_time > to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')',' OR ') || ')';

    if vsqlwhere = ' and ()'
      then vsqlwhere := null;
    end if;
  --  dbms_output.put_line('em: '||vsqlwhere);
    --Create Recording queue entries if trial use fields change in subsequent EPG
    vsql := 'insert into temp_recording (recording_id,schedule_id,station_id,description,
    schedule_start_time,schedule_end_time,schedule_duration,
    hd_tv,cc,repeat,
    program_id,episode_title,year_program,tv_rating,mpaa_rating,program_title,program_description, rm_name, rm_id, update_allowed)
    SELECT  distinct r.recording_id, schedule.schedule_id,schedule.station_id,cast(''SCHEDULE.METADATA'' as varchar2(4000)) as description,
    schedule.start_time as schedule_start_time, schedule.end_time as schedule_end_time, schedule.duration as schedule_duration,
    schedule.hd_tv as hd_tv, schedule.cc as cc,schedule.repeat as repeat,
    program.program_id, stg_program.episode_title as episode_title,
    stg_program.year_program as year_program, stg_schedule.tv_rating as tv_rating,
    program.mpaa_rating as mpaa_rating,
    stg_program.title as program_title, stg_program.description1 as program_description, rm.name as rm_name, rm.rm_id,
    case
    when  r.adjusted_start_time < current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0), ''SECOND'') then cast(0 as number(1))
    else cast(1 as number(1))
    end as update_allowed
    FROM schedule schedule join stg_schedule stg_schedule
      on (stg_schedule.station_num = schedule.station_id
              and stg_schedule.database_key = schedule.program_id
              and stg_schedule.start_time = schedule.start_time
              and schedule.start_time > to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')
          )
    join program program on ( schedule.program_id = program.program_id)
    join stg_program stg_program on ( program.program_id = stg_program.program_id)
    join station on ( schedule.station_id = station.station_id)
    left outer join stg_station on ( station.station_id = stg_station.station_id)
    join lineup on (lineup.station_id = schedule.station_id)
    left outer join stg_lineup on ( lineup.lineup_id = stg_lineup.row_id)
    join headend on (lineup.headend_id = headend.headend_id)
    left outer join stg_headend on (headend.headend_id = stg_headend.headend_id)
    join recording r on (schedule.schedule_id = r.schedule_id  and r.status != '||const_recording_DELETED||' )
    join rtc_pool rp on (rp.pool_id = r.pool_id)
    join rm on (rp.rm_id = rm.rm_id )
    join rec_trial_use_fields rtuf on ( r.recording_id = rtuf.recording_id)
    join entity_mapping on ( rtuf.mapping_id = entity_mapping.mapping_id and entity_mapping.type in (1,3))'
    ||vsqlwhere;
    CREATEOBJECT(vsql,'INSERT INTO TEMP_RECORDING');

    epgguidedatalogging('Start of Program merge. StartTime = '||current_timestamp);


    merge into program pprod
    using stg_program pstg
    on (pstg.program_id = pprod.program_id)
    when matched then
    update set pprod.LANGUAGE = pstg.language,
    pprod.TITLE = pstg.title,
    pprod.REDUCED_TITLE1 = pstg.REDUCED_TITLE1 ,
    pprod.REDUCED_TITLE2 = pstg.REDUCED_TITLE2,
    pprod.REDUCED_TITLE3 = pstg.REDUCED_TITLE3,
    pprod.REDUCED_TITLE4 = pstg.REDUCED_TITLE4,
    pprod.ALT_TITLE = pstg.ALT_TITLE,
    pprod.REDUCED_DESCRIPTION1 = pstg.REDUCED_DESCRIPTION1,
    pprod.REDUCED_DESCRIPTION2 = pstg.REDUCED_DESCRIPTION2,
    pprod.REDUCED_DESCRIPTION3 = pstg.REDUCED_DESCRIPTION3,
    pprod.REDUCED_DESCRIPTION4 = pstg.REDUCED_DESCRIPTION4,
    pprod.ADVISORY_DESC1 = pstg.ADVISORY_DESC1,
    pprod.ADVISORY_DESC2 = pstg.ADVISORY_DESC2,
    pprod.ADVISORY_DESC3 = pstg.ADVISORY_DESC3,
    pprod.ADVISORY_DESC4 = pstg.ADVISORY_DESC4,
    pprod.ADVISORY_DESC5 = pstg.ADVISORY_DESC5,
    pprod.ADVISORY_DESC6 = pstg.ADVISORY_DESC6,
    pprod.GENRE_DESC1 = pstg. GENRE_DESC1,
    pprod.GENRE_DESC2 = pstg. GENRE_DESC2,
    pprod.GENRE_DESC3 = pstg. GENRE_DESC3,
    pprod.GENRE_DESC4 = pstg. GENRE_DESC4,
    pprod.GENRE_DESC5 = pstg. GENRE_DESC5,
    pprod.GENRE_DESC6 = pstg. GENRE_DESC6,
    pprod.DESCRIPTION1 = pstg.DESCRIPTION1,
    pprod.DESCRIPTION2 = pstg.DESCRIPTION2,
    pprod.YEAR_PROGRAM = pstg.YEAR_PROGRAM,
    pprod.MPAA_RATING = pstg.MPAA_RATING,
    pprod.STAR_RATING = pstg.STAR_RATING,
    pprod.RUN_TIME = pstg.RUN_TIME,
    pprod.COLOR_CODE = pstg.COLOR_CODE,
    pprod.PROGRAM_LANGUAGE = pstg.PROGRAM_LANGUAGE,
    pprod.COUNTRY_OF_ORIGIN = pstg.COUNTRY_OF_ORIGIN,
    pprod.MADE_FOR_TV = pstg.MADE_FOR_TV,
    pprod.SOURCE_TYPE = pstg.SOURCE_TYPE,
    pprod.SHOW_TYPE = pstg.SHOW_TYPE,
    pprod.HOLIDAY = pstg.HOLIDAY,
    pprod.SYNDICATE_EPISODE_NUM = pstg.SYNDICATE_EPISODE_NUM,
    pprod.ALT_SYNDICATE_EPI_NUM = pstg.ALT_SYNDICATE_EPI_NUM,
    pprod.EPISODE_TITLE = pstg.EPISODE_TITLE,
    pprod.NET_SYN_SOURCE = pstg.NET_SYN_SOURCE,
    pprod.NET_SYN_TYPE = pstg.NET_SYN_TYPE,
    pprod.ORG_STUDIO = pstg.ORG_STUDIO,
    pprod.GAME_DATETIME = pstg.GAME_DATETIME,
    pprod.GAME_TIMEZONE = pstg.GAME_TIMEZONE,
    pprod.ORG_AIR_DATE = pstg.ORG_AIR_DATE,
    pprod.UNIQUE_ID = pstg.UNIQUE_ID,
    pprod.USER_DATA1 = pstg.user_data1,
    pprod.USER_DATA2 = pstg.user_data2,
    pprod.USER_DATA3 = pstg.user_data3,
    pprod.USER_DATA4 = pstg.user_data4,
    pprod.USER_DATA5 = pstg.user_data5,
    pprod.USER_DATA6 = pstg.user_data6,
    pprod.USER_DATA7 = pstg.user_data7,
    pprod.USER_DATA8 = pstg.user_data8,
    pprod.USER_DATA9 = pstg.user_data9,
    pprod.USER_DATA10 = pstg.user_data10,
    pprod.USER_DATA11 = pstg.user_data11,
    pprod.USER_DATA12 = pstg.user_data12,
    pprod.USER_DATA13 = pstg.user_data13,
    pprod.PROGRAM_GENERIC = pstg.PROGRAM_GENERIC,
    pprod.CREATED_BY = 'SYSTEM',
    pprod.UPDATED_BY = 'SYSTEM'
    when not matched then
    insert (pprod.PROGRAM_ID,
    pprod.LANGUAGE,
    pprod.TITLE,
    pprod.REDUCED_TITLE1,
    pprod.REDUCED_TITLE2,
    pprod.REDUCED_TITLE3,
    pprod.REDUCED_TITLE4,
    pprod.ALT_TITLE,
    pprod.REDUCED_DESCRIPTION1,
    pprod.REDUCED_DESCRIPTION2,
    pprod.REDUCED_DESCRIPTION3,
    pprod.REDUCED_DESCRIPTION4,
    pprod.ADVISORY_DESC1,
    pprod.ADVISORY_DESC2,
    pprod.ADVISORY_DESC3,
    pprod.ADVISORY_DESC4,
    pprod.ADVISORY_DESC5,
    pprod.ADVISORY_DESC6,
    pprod.GENRE_DESC1,
    pprod.GENRE_DESC2,
    pprod.GENRE_DESC3,
    pprod.GENRE_DESC4,
    pprod.GENRE_DESC5,
    pprod.GENRE_DESC6,
    pprod.DESCRIPTION1,
    pprod.DESCRIPTION2,
    pprod.YEAR_PROGRAM,
    pprod.MPAA_RATING,
    pprod.STAR_RATING,
    pprod.RUN_TIME,
    pprod.COLOR_CODE,
    pprod.PROGRAM_LANGUAGE,
    pprod.COUNTRY_OF_ORIGIN,
    pprod.MADE_FOR_TV,
    pprod.SOURCE_TYPE,
    pprod.SHOW_TYPE,
    pprod.HOLIDAY,
    pprod.SYNDICATE_EPISODE_NUM,
    pprod.ALT_SYNDICATE_EPI_NUM,
    pprod.EPISODE_TITLE,
    pprod.NET_SYN_SOURCE,
    pprod.NET_SYN_TYPE,
    pprod.ORG_STUDIO,
    pprod.GAME_DATETIME,
    pprod.GAME_TIMEZONE,
    pprod.ORG_AIR_DATE,
    pprod.UNIQUE_ID,
    pprod.USER_DATA1,
    pprod.USER_DATA2,
    pprod.USER_DATA3,
    pprod.USER_DATA4,
    pprod.USER_DATA5,
    pprod.USER_DATA6,
    pprod.USER_DATA7,
    pprod.USER_DATA8,
    pprod.USER_DATA9,
    pprod.USER_DATA10,
    pprod.USER_DATA11,
    pprod.USER_DATA12,
    pprod.USER_DATA13,
    pprod.PROGRAM_GENERIC,
    pprod.CREATED_BY,
    pprod.UPDATED_BY)
    values (pstg.program_id,
    pstg.LANGUAGE,
    pstg.TITLE,
    pstg.REDUCED_TITLE1,
    pstg.REDUCED_TITLE2,
    pstg.REDUCED_TITLE3,
    pstg.REDUCED_TITLE4,
    pstg.ALT_TITLE,
    pstg.REDUCED_DESCRIPTION1,
    pstg.REDUCED_DESCRIPTION2,
    pstg.REDUCED_DESCRIPTION3,
    pstg.REDUCED_DESCRIPTION4,
    pstg.ADVISORY_DESC1,
    pstg.ADVISORY_DESC2,
    pstg.ADVISORY_DESC3,
    pstg.ADVISORY_DESC4,
    pstg.ADVISORY_DESC5,
    pstg.ADVISORY_DESC6,
    pstg.GENRE_DESC1,
    pstg.GENRE_DESC2,
    pstg.GENRE_DESC3,
    pstg.GENRE_DESC4,
    pstg.GENRE_DESC5,
    pstg.GENRE_DESC6,
    pstg.DESCRIPTION1,
    pstg.DESCRIPTION2,
    pstg.YEAR_PROGRAM,
    pstg.MPAA_RATING,
    pstg.STAR_RATING,
    pstg.RUN_TIME,
    pstg.COLOR_CODE,
    pstg.PROGRAM_LANGUAGE,
    pstg.COUNTRY_OF_ORIGIN,
    pstg.MADE_FOR_TV,
    pstg.SOURCE_TYPE,
    pstg.SHOW_TYPE,
    pstg.HOLIDAY,
    pstg.SYNDICATE_EPISODE_NUM,
    pstg.ALT_SYNDICATE_EPI_NUM,
    pstg.EPISODE_TITLE,
    pstg.NET_SYN_SOURCE,
    pstg.NET_SYN_TYPE,
    pstg.ORG_STUDIO,
    pstg.GAME_DATETIME,
    pstg.GAME_TIMEZONE,
    pstg.ORG_AIR_DATE,
    pstg.UNIQUE_ID,
    pstg.USER_DATA1,
    pstg.USER_DATA2,
    pstg.USER_DATA3,
    pstg.USER_DATA4,
    pstg.USER_DATA5,
    pstg.USER_DATA6,
    pstg.USER_DATA7,
    pstg.USER_DATA8,
    pstg.USER_DATA9,
    pstg.USER_DATA10,
    pstg.USER_DATA11,
    pstg.USER_DATA12,
    pstg.USER_DATA13,
    pstg.PROGRAM_GENERIC,
    'SYSTEM',
    'SYSTEM');


    -- Adding entity mapping entries for the program_generic values

    insert into entity_mapping(mapping_id, name, table_name, column_name, type, generic_name, asset_type)
    select entity_mapping_seq.nextval as mapping_id,'PROGRAM.'||spgv.trialname as name, 'PROGRAM' as table_name,'PROGRAM_GENERIC' as column_name, cast(3 as number) as type , SPGV.trialname, spgv.assettype
    FROM entity_mapping em right outer join
        (select distinct trialname, assettype  from stg_program_generic_view) spgv on (em.generic_name = spgv.trialname and em.asset_type = spgv.assettype
        and upper(em.table_name)='PROGRAM' and upper(em.column_name) = 'PROGRAM_GENERIC')
        where em.generic_name is null and em.asset_type is null;

    epgguidedatalogging('Start of cast credits merge. StartTime = '||current_timestamp);

    delete from cast_credits
    where program_id in (select database_key from stg_castcredits);

    insert into cast_credits (CAST_CREDITS_ID,
      PROGRAM_ID,
      FIRST_NAME,
      LAST_NAME,
      DESCRIPTION,
      TYPE,
      CREATED_BY,
      UPDATED_BY)
    select cast_credits_seq.nextval,
      database_key,
      first_name,
      last_name,
      role_desc,
      type,
      'SYSTEM',
      'SYSTEM'
    from stg_castcredits;

      commit;

 	 epgguidedatalogging('Start of Series merge. StartTime = '||current_timestamp);

    --MERGE SERIES
    merge into series s
    using (select distinct ss.series_id as series_id,max(p.title) over (partition by ss.series_id) as title
            From stg_series ss,program p
            where p.program_id = ss.program_id) sstg
    on (sstg.series_id = s.series_id)
    when matched then
    update set s.title = sstg.title
    when not matched then
    insert (s.series_id,s.title,s.created_by,s.updated_by)
    values (sstg.series_id,sstg.title,'SYSTEM','SYSTEM');

    insert into series_program_map (SERIES_ID,PROGRAM_ID,CREATED_BY,UPDATED_BY)
    select distinct ss.series_id as series_id,ss.program_id,'SYSTEM','SYSTEM'
            From stg_series ss where not exists
              (select 1 from series_program_map spm where spm.series_id = ss.series_id
                   and spm.program_id = ss.program_id);

    -- Delete sereies_program_map entries, if coming schedules do not have series information or different series information
	delete from series_program_map spm where exists
    (select database_key from stg_schedule ss where spm.program_id = ss.database_key and ss.start_time > nvl (vcurrentdatetime, current_timestamp)
          and  (series_link is null or
                  extract (series_link, '/SERIES_LINK/SERIES_LINK/SeriesIdentifier/text()').getStringVal() != spm.series_id
              )
    );

   epgguidedatalogging('End of Series merge. StartTime = '||current_timestamp);
   commit;

  select  value into is_epg_sch_conflict_completed from config_setting where key ='Is_EPG_Schedule_Conflict_Completed';

  if (is_epg_sch_conflict_completed = 'Y') then
    -- truncate the previously deleted recordings trap messages from the schedule conflicts
    autonomoustransaction(' truncate table temp_deleted_recordings ');
  end if;

 --SCHEDULE MERGE

    --Get max start time from schedule
    select nvl(max(start_time),vcurrentdatetime)
    into vmaxschedulestarttime
    from schedule;

    --Get Schedule Metadata columns
/*    select '('||rowtocol ('select ''s.''||col||'' != sstg.''||col from
    (select regexp_substr(value, ''[^\,]+'', 1, ao.rn) as col from config_setting cs
    join
    (select rownum rn from all_objects where rownum < 100) ao
    on
    (rn <= length(regexp_replace(value, ''[^\,]'',''''))+1 and cs.key=''SCHEDULE.METADATA'')) tabcol',' OR ') || ')'
    into vschsqlwhere
    from dual;
*/
    vschsqlwhere := ' ('||rowtocol ('select ''nvl(to_nchar(s.''||col||''),'''' '''') != nvl(to_nchar(sstg.''||col||''),'''' '''')'' from (select regexp_substr(value, ''[^\,]+'', 1, ao.rn) as col from config_setting cs join (select rownum rn from all_objects where rownum < 100) ao on (rn <= length(regexp_replace(value, ''[^\,]'',''''))+1 and cs.key=''SCHEDULE.METADATA'')) tabcol',' OR ') || ')';

    if vschsqlwhere = ' ()'
      then vschsqlwhere := null;
    end if;

    select greatest(min(start_time),vcurrentdatetime) as minstgschdate, greatest(max(start_time + numtodsinterval(duration,'SECOND')),vcurrentdatetime)  as maxstgschdate
    into vminstgschdate, vmaxstgschdate
    from stg_schedule;

    insert into temp_station_sch_range(station_id, first_schedule_start_time, last_schedule_end_time)
          select sst.station_id,  min(start_time) as minstgschdate, max(start_time + numtodsinterval(duration,'SECOND'))  as maxstgschdate
          from stg_schedule ssch right join stg_station sst on (sst.station_id = ssch.station_num)
    group by sst.station_id;

    epgguidedatalogging('Start of Schedule merge. StartTime = '||current_timestamp);

     -- Get Schedule MetaData changed against the values for the key SCHEDULE.METADATA  in the config_setting table
    vsql := 'select r.recording_id,s.schedule_id,s.station_id,''SCHEDULE.METADATA'',
    s.start_time,sstg.start_time + numtodsinterval(sstg.duration,''SECOND''),sstg.duration,
    sstg.hd_tv,sstg.cc,sstg.repeat,
    p.program_id,p.episode_title,p.year_program,
    case when mctv.name != s.tv_rating then mctv.name else null end as tv_rating,
    null as mpaa_rating,p.title,p.description1 ,rm.name as rm_name, rm.rm_id,
        case
    when  r.adjusted_start_time < current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0), ''SECOND'') then cast(0 as number(1))
    else cast(1 as number(1))
    end as update_allowed
    from recording r, rtc_pool rp, rm , schedule s, stg_schedule sstg, program p, (select * from metadata_config where type=''TVRating'' ) mctv
    where
    r.schedule_id = s.schedule_id
    and r.pool_id = rp.pool_id
    and rp.rm_id = rm.rm_id
    and p.program_id = s.program_id
    and sstg.station_num = s.station_id
    and sstg.database_key = s.program_id
    and sstg.start_time = s.start_time
    and mctv.name(+) = upper(sstg.tv_rating)
    and r.status != '||const_recording_DELETED||'
    and s.start_time > nvl(to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR''),current_timestamp) ';

    if vschsqlwhere is not null then
        vsql := vsql || ' and ( ' || vschsqlwhere ||' )';
    end if;

    --Get all schedule recordings that have changed
    execute immediate 'insert into temp_recording (recording_id,schedule_id,station_id,description,
              schedule_start_time,schedule_end_time,schedule_duration,
              hd_tv,cc,repeat,
              program_id,episode_title,year_program,tv_rating,mpaa_rating,program_title,program_description, rm_name, rm_id, update_allowed)
              select r.recording_id,s.schedule_id,s.station_id,''SCHEDULE.DELETED '',
              s.start_time,s.end_time,s.duration,
              s.hd_tv,s.cc,s.repeat,
              p.program_id,p.episode_title,p.year_program, null as tv_rating,
              null as mpaa_rating,p.title,p.description1, rm.name as rm_name, rm.rm_id, null as update_allowed
              from recording r, rtc_pool rp, rm , schedule s, program p
              where
              r.schedule_id = s.schedule_id
              and r.pool_id = rp.pool_id
              and rp.rm_id = rm.rm_id
              and s.program_id = p.program_id
              and s.start_time >= to_timestamp_tz('''||to_char(vminstgschdate,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')
              and s.start_time < to_timestamp_tz('''||to_char(vmaxstgschdate,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')
              and not exists ( select 1 from stg_schedule sstg
                              where sstg.station_num = s.station_id
                              and sstg.database_key = s.program_id
                              and sstg.start_time = s.start_time
                            )
                and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,to_timestamp_tz('''||to_char(vminstgschdate,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR''))
                and s.start_time <  nvl(tssr.last_schedule_end_time,to_timestamp_tz('''||to_char(vmaxstgschdate,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR''))
                 )
              UNION '||vsql;

          -- do the proceedings

          execute immediate 'select count(1) from temp_recording' into vcnt_affectedrecording;

          if vcnt_affectedrecording > 0 then
              --Create Alerts
              /* Schedule Metadata has changed? If so, raise Alarm. Error Code 12 = Schedule Change */
              vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||rcs_p_v||''',
                          '''||sch_prg_md_chng_update_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.program_title||'','||station_call_sign_p_v||'= ''||tr.station_call_sign,
                          ''EPG_DOWNLOAD'',12,'''||system_p_v||''','''||system_p_v||'''
                          from (select distinct prg.title as program_title, st.call_sign as station_call_sign, tr.schedule_id, tr.schedule_start_time at time zone to_char (current_timestamp, ''TZR'') as schedule_start_time,
                          description, tr.rm_name, tr.rm_id  from temp_recording tr , station st, program prg    where tr.update_allowed =1 and  tr.station_id = st.station_id  and tr.program_id =prg.program_id and tr.description not like ''%SCHEDULE.DELETED%'' ) tr';

              vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
              CREATEOBJECT(vsql,'SCHEDULE ALARMS');

              vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||rcs_p_v||''',
                          '''||sch_prg_md_chng_too_late_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.program_title||'','||station_call_sign_p_v||'= ''||tr.station_call_sign,
                          ''EPG_DOWNLOAD'',12,'''||system_p_v||''','''||system_p_v||'''
                          from (select distinct prg.title as program_title, st.call_sign as station_call_sign, tr.schedule_id, tr.schedule_start_time at time zone to_char (current_timestamp, ''TZR'') as schedule_start_time,
                          description, tr.rm_name, tr.rm_id  from temp_recording tr , recording rec,  station st, program prg    where tr.recording_id = rec.recording_id and rec.status != '||const_recording_DELETED||' and tr.update_allowed = 0 and  tr.station_id = st.station_id  and tr.program_id =prg.program_id and tr.description not like ''%SCHEDULE.DELETED%'' ) tr';

              vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
              CREATEOBJECT(vsql,'SCHEDULE ALARMS');

               vsql := 'insert into recording_log(RECORDING_LOG_ID,
                                      RECORDING_ID,
                                      LOG_MESSAGE,
                                      CREATED_BY,
                                      UPDATED_BY,
                                      CREATE_DATE,
                                      UPDATE_DATE)
                select RECORDING_LOG_SEQ.nextval,tr.recording_id,''Schedule or Program metadata has changed. But it is too late to submit the update request to the RM  for Recording'',''SYSTEM'',''SYSTEM'',current_timestamp,current_timestamp
                from temp_recording tr  where tr.update_allowed =0 ';

               CREATEOBJECT(vsql,'Writing Recording logs');

              vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||rcs_p_v||''',
                          '''||sch_prg_md_chng_delete_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.program_title||'','||station_call_sign_p_v||'= ''||tr.station_call_sign,
                          ''EPG_DOWNLOAD'',24,'''||system_p_v||''','''||system_p_v||'''
                          from (select distinct prg.title as program_title, st.call_sign as station_call_sign, tr.schedule_id, tr.schedule_start_time at time zone to_char (current_timestamp, ''TZR'') as schedule_start_time,
                          description, tr.rm_name, tr.rm_id  from temp_recording tr , station st, program prg, recording rec    where rec.recording_id = tr.recording_id and rec.status != '||const_recording_DELETED||' and tr.station_id = st.station_id  and tr.program_id =prg.program_id and  tr.description  like ''%SCHEDULE.DELETED%'' ) tr';

              vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
              CREATEOBJECT(vsql,'Deleted Recordings Alarms');


              -- insert the deleted recordings for the logging traps

              vsql := ' insert into temp_deleted_recordings (rm_id, rm_name, program_title, station_call_sign, schedule_start_time)
                        select distinct   tr.rm_id, tr.rm_name, prg.title, st.call_sign, tr.schedule_start_time at time zone to_char (current_timestamp, ''TZR'') as schedule_start_time
                        from temp_recording tr, station st, program prg , recording rec
                        where tr.recording_id = rec.recording_id and rec.status != '||const_recording_DELETED||' and tr.station_id = st.station_id  and tr.program_id =prg.program_id and tr.description like ''%SCHEDULE.DELETED%''
                      ';
               CREATEOBJECT(vsql,'Deleted Recordings');

              --Update Recording Metadata.
              vsql := 'update recording r
                        set r.program_title = nvl((select distinct tr.program_title from temp_recording tr where tr.description=''PROGRAM.METADATA'' and tr.recording_id = r.recording_id),r.program_title),
                          r.program_description = nvl((select distinct tr.program_description from temp_recording tr where tr.description=''PROGRAM.METADATA'' and tr.recording_id = r.recording_id),r.program_description),
                          r.mpaa_rating = nvl((select distinct tr.mpaa_rating from temp_recording tr where tr.description=''PROGRAM.METADATA'' and tr.recording_id = r.recording_id),r.mpaa_rating),
                          r.tv_rating = nvl((select distinct tr.tv_rating from temp_recording tr where tr.description=''SCHEDULE.METADATA'' and tr.recording_id = r.recording_id),r.tv_rating)
                       where r.recording_id in (select tr.recording_id from temp_recording tr where tr.update_allowed= 1 and  description like ''%METADATA%'')
                      ';
              CREATEOBJECT(vsql,'Update Recording Metadata');
			   -- Create Recording
            
                      
              vsql := 'INSERT INTO recording_queue (
                                      RECORDING_ID,
                                      SCHEDULE_ID,
                                      POOL_ID,
                                      RM_ID,
                                      SCHEDULE_START_TIME,
                                      SCHEDULE_END_TIME,
                                      SCHEDULE_DURATION,
                                      STATION_ID,
                                      PROGRAM_ID,
                                      HD_TV,
                                      CC,
                                      REPEAT,
                                      RM_NAME,
                                      RM_IP_ADDRESS,
                                      RM_PORT,
                                      EPISODE_TITLE,
                                      STATION_OFFSET,
                                      HEADEND_NAME,
                                      CONTRACT_ID,
                                      YEAR_PROGRAM,
                                      ASSET_ID,
                                      MOVIE_ASSET_ID,
                                      TITLE_ASSET_ID,
                                      POSTER_ASSET_ID,
                                      POSTER_IMAGE_NAME,
                                      POSTER_DESCRIPTION,
                                      PROGRAM_TITLE,
                                      PROGRAM_DESCRIPTION,
                                      OFFSET_START_TIME,
                                      OFFSET_END_TIME,
                                      RECORDING_MODE,
                                      PRIORITY,
                                      PROVIDER_QA_CONTACT,
                                      CATALOG_ENTRY,
                                      BILLING_ID,
                                      MPAA_RATING,
                                      TV_RATING,
                                      PROVIDER_ID,
                                      PROVIDER_NAME,
                                      PRODUCT,
                                      CATEGORY,
                                      AUDIO_TYPE,
                                      max_copies,
                            		  preview_period,
                            		  REDUNDANCY_COUNT,
                                      LICENCE_START_DATE,
                                      LICENSE_END_DATE,
                                      REQUEST_TYPE,
                                      VOD_START_TIME,
                                      VOD_DURATION,
                                      VOD_RESTRICTIONS,
                                      RTI_START_TIME,
                                      RTI_DURATION,
                                      RTI_RESTRICTIONS,
                                      QC_START_TIME,
                                      QC_DURATION,
                                      QC_RESTRICTIONS,
                                      ADVISORIES,
                                      CREATED_BY,
                                      UPDATED_BY)
                select rec_queue.*
                	  FROM
                	  (SELECT
                      tr.recording_id,
                      tr.schedule_id,
                      r.pool_id,
                      rp.rm_id,
                      tr.schedule_start_time,
                      tr.schedule_end_time,
                      tr.schedule_duration,
                      tr.station_id,
                      tr.program_id,
                      tr.hd_tv,
                      tr.cc,
                      tr.repeat,
                      rm.name,
                      rm.ip_address,
                      rm.port,
                      tr.episode_title,
                      rl.offset,
                      h.headend_location,
                      rrmouter.contract_id,
                      tr.year_program,
                      r.asset_id,
                      r.MOVIE_ASSET_ID,
                      r.TITLE_ASSET_ID,
                      r.POSTER_ASSET_ID,
                      r.POSTER_IMAGE_NAME,
                      r.POSTER_DESCRIPTION,
                      tr.program_TITLE,
                      tr.program_DESCRIPTION,
                      r.OFFSET_START_TIME,
                      r.OFFSET_END_TIME,
                      r.RECORDING_MODE,
                      r.PRIORITY,
                      r.PROVIDER_QA_CONTACT,
                      r.CATALOG_ENTRY,
                      r.BILLING_ID,
                      case when tr.mpaa_rating is null then r.mpaa_rating else tr.mpaa_rating end as MPAA_RATING,
                      case when tr.tv_rating is null then r.TV_RATING else tr.tv_rating end as TV_RATING,
                      r.PROVIDER_ID,
                      r.PROVIDER_NAME,
                      r.PRODUCT,
                      r.CATEGORY,
                      r.AUDIO_TYPE,
					            r.max_copies,
                      r.preview_period,
                      r.REDUNDANCY_COUNT,
                      r.LICENCE_START_DATE,
                      r.LICENSE_END_DATE,
                      case when tr.description like ''%METADATA%'' then '''||const_recording_UPDATE_REQUEST||''' when tr.description like ''%DELETED%'' then '''||const_recording_DELETE_REQUEST||''' else '''||const_recording_UPDATE_REQUEST||''' end,
                      pwr_vod.start_time as vod_start_time,
                      pwr_vod.duration as vod_duration,
                      pwr_vod.restrictions as vod_restrictions,
                      pwr_rti.start_time as rti_start_time,
                      pwr_rti.duration as rti_duration,
                      pwr_rti.restrictions as rti_restrictions,
                      pwr_qc.start_time as qc_start_time,
                      pwr_qc.duration as qc_duration,
                      pwr_qc.restrictions as qc_restrictions,
                      rtrim(case when p.advisory_desc1 is not null then p.advisory_desc1||''|'' end ||
                      case when p.advisory_desc2 is not null then p.advisory_desc2||''|'' end ||
                      case when p.advisory_desc3 is not null then p.advisory_desc3||''|'' end ||
                      case when p.advisory_desc4 is not null then p.advisory_desc4||''|'' end ||
                      case when p.advisory_desc5 is not null then p.advisory_desc5||''|'' end ||
                      case when p.advisory_desc6 is not null then p.advisory_desc6||''|'' end, ''|'') as advisories ,
                      ''SYSTEM'' as created_by,
                      ''SYSTEM'' as updated_by
              FROM (select nvl(str.recording_id,ptr.recording_id) as recording_id,
                    nvl(str.station_id,ptr.station_id) as station_id,
                    nvl(str.schedule_id,ptr.schedule_id) as schedule_id,
                    nvl(str.description,ptr.description) as description ,
                    nvl(str.schedule_start_time,ptr.schedule_start_time) as schedule_start_time,
                    nvl(str.schedule_end_time,ptr.schedule_end_time) as schedule_end_time,
                    nvl(str.schedule_duration,ptr.schedule_duration) as schedule_duration,
                    nvl(str.program_id,ptr.program_id) as program_id,
                    nvl(str.hd_tv,ptr.hd_tv) as hd_tv,
                    nvl(str.cc,ptr.cc) as cc,
                    nvl(str.repeat,ptr.repeat) as repeat,
                    nvl(ptr.episode_title,str.episode_title) as episode_title,
                    nvl(ptr.year_program,str.year_program) as year_program,
                    nvl(str.tv_rating,ptr.tv_rating) as tv_rating,
                    nvl(ptr.mpaa_rating,str.mpaa_rating) as mpaa_rating,
                    nvl(ptr.program_title,str.program_title) as program_title,
                    nvl(ptr.program_description,str.program_description) as program_description,
                    nvl(str.duprow,1) as duprow,
                    str.update_allowed
                    from
                    ((select temp_recording.*,row_number() over (partition by recording_id order by description desc) as duprow
                      from TEMP_RECORDING
                      where description like ''%SCHEDULE%'') str
                        full outer join
                      (select * from temp_recording where description = ''PROGRAM.METADATA'') ptr
                      on (str.recording_id = ptr.recording_id))
                    ) TR
              join recording r on (tr.recording_id = r.recording_id and r.status != '||const_recording_DELETED||' and tr.duprow=1 and (tr.update_allowed =1 or tr.update_allowed is null)  )
              join program p on (p.program_id = tr.program_id)
              join rtc_pool rp on (r.pool_id = rp.pool_id)
              join rm on (rm.rm_id = rp.rm_id)
              join rm_lineup rl on (rl.rm_id = rm.rm_id and rl.station_id = tr.station_id and rl.headend_id = rm.headend_id)
              join headend h on (h.headend_id = rm.headend_id)
              join (select row_number() over (partition by rrm.schedule_id, rrm.rm_id order by r.create_date ) as rownumber,
                  rrm.schedule_id, rrm.rm_id, r.contract_id
                  from recordable_rights_map rrm , right r where r.right_id = rrm.right_id and r.state=1) rrmouter
                on (rrmouter.schedule_id = tr.schedule_id
                    and rrmouter.rownumber = 1)
              left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_vod on (pwr_vod.recording_id = tr.recording_id and pwr_vod.type=''VOD'')
              left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_rti on (pwr_rti.recording_id = tr.recording_id and pwr_rti.type=''RTI'')
              left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_qc on (pwr_qc.recording_id = tr.recording_id and pwr_qc.type=''QC'')
            order by r.adjusted_start_time, r.PRIORITY) rec_queue ';
              CREATEOBJECT(vsql,'RECORDING SCHEDULE ALARMS');
              -- new HA change
                insertRecQueue;

--              --Delete Trial Use Fields
               delete from rec_trial_use_fields rtuf
               where rtuf.recording_id in (
               select r.recording_id from schedule s,recording r
               where
               r.schedule_id = s.schedule_id
               and s.start_time >= vminstgschdate
               and s.start_time < vmaxstgschdate
               and not exists (select 1 from stg_schedule stg
                           where stg.start_time = s.start_time
                           and stg.station_num = s.station_id
                           and stg.database_key = s.program_id
                        )
                 and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                 where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                 and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                  )
                ) ;

               --DELETE CONFLICTING SCHEDULE RECORDS
               delete from RECORDING_REC_RULES_MAP rrrm
               where rrrm.recording_id in (
               select r.recording_id from schedule s,recording r
               where  r.schedule_id = s.schedule_id
               and s.start_time >= vminstgschdate
               and s.start_time < vmaxstgschdate
                 and not exists (select 1 from stg_schedule stg
                           where stg.start_time = s.start_time
                           and stg.station_num = s.station_id
                           and stg.database_key = s.program_id
                        )
                 and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                 where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                 and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                  )

                );

               delete from PLAYBACK_WINDOW_RESTRICTIONS pwr
               where pwr.recording_id in (
               select r.recording_id from schedule s,recording r
               where r.schedule_id = s.schedule_id
               and s.start_time >= vminstgschdate
               and s.start_time < vmaxstgschdate
                 and not exists (select 1 from stg_schedule stg
                           where stg.start_time = s.start_time
                           and stg.station_num = s.station_id
                           and stg.database_key = s.program_id
                        )
               and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                 where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                 and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                  )
                 );


               delete from RECORDING_LOG rl
               where rl.recording_id in (
               select r.recording_id from schedule s,recording r
               where r.schedule_id = s.schedule_id
               and s.start_time >= vminstgschdate
               and s.start_time < vmaxstgschdate
               and not exists (select 1 from stg_schedule stg
                           where stg.start_time = s.start_time
                           and stg.station_num = s.station_id
                           and stg.database_key = s.program_id
                        )
                and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                 where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                 and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                  )
                 );

               delete from recording r
               where r.schedule_id in
               ( select s.schedule_id from schedule s
                   where
                   s.start_time >= vminstgschdate
                   and s.start_time < vmaxstgschdate
                   and not exists (select 1 from stg_schedule stg
                                   where stg.start_time = s.start_time and
                                   stg.station_num = s.station_id and
                                   stg.database_key = s.program_id
                                  )
                  and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                               where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                               and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                              )
                      );


           end if;

           delete from recordable_rights_map rrm
           where rrm.schedule_id in ( select s.schedule_id from schedule s
           where
           s.start_time >= vminstgschdate
           and s.start_time < vmaxstgschdate
           and not exists (select 1 from stg_schedule stg
                           where stg.start_time = s.start_time and
                           stg.station_num = s.station_id and
                           stg.database_key = s.program_id
                          )
          and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                       where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                       and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                      )
           );

         delete from schedule s
           where
           s.start_time >= vminstgschdate
           and s.start_time < vmaxstgschdate
           and not exists (select 1 from stg_schedule stg
                           where stg.start_time = s.start_time and
                           stg.station_num = s.station_id and
                           stg.database_key = s.program_id
                          )
          and exists (select 1 from stg_station sst join temp_station_sch_range tssr on (sst.station_id = tssr.station_id)
                       where sst.station_id = s.station_id  and s.start_time >= nvl(tssr.first_schedule_start_time,vminstgschdate)
                       and s.start_time <  nvl(tssr.last_schedule_end_time,vmaxstgschdate)
                    );










         if (is_epg_sch_conflict_completed = 'N') then

            merge into temp_schedule_conflicts tsc
            using ( SELECT  distinct s.schedule_id, s.start_time,  p.title ,  st.call_sign ,
                          CASE WHEN upper(s.request_record) ='Y' and upper(s.request_record) != upper(NVL(stg.request_record,'X')) THEN CAST (1 AS NUMBER(1)) ELSE CAST (0 AS NUMBER(1)) END AS rrflag,
                          CASE when NVL (s.duration, 1) != nvl (stg.duration, 1)  THEN CAST (1 AS NUMBER(1)) ELSE CAST (0 AS NUMBER(1)) END AS duration_change
                    FROM SCHEDULE s JOIN stg_schedule stg ON (stg.start_time     = s.start_time AND stg.station_num  = s.station_id AND stg.database_key = s.program_id)
                    JOIN program p ON ( p.program_id = s.program_id)
                    JOIN station st ON (st.station_id = s.station_id)
                    WHERE  s.start_time > CURRENT_TIMESTAMP
                      AND ( upper(NVL(stg.hd_tv,'1'))!= upper(NVL(s.hd_tv,'1')) OR
                            upper(NVL(stg.repeat,'1')) != upper(NVL(s.repeat,'1')) OR
                            upper(NVL(stg.live_tape_delay,'1')) != upper(NVL(s.live_tape_delay,'1')) OR
                            upper(s.request_record) ='Y' and upper(s.request_record) != upper(NVL(stg.request_record,'X')) OR
                            NVL(s.series_link.getStringVal(),'X') != NVL(stg.series_link.getStringVal(),'X')  OR
                            NVL (s.duration, 1) != nvl (stg.duration, 1)
                  )) conflicts
              on (  tsc.schedule_id = conflicts.schedule_id)
              when matched then

              update set tsc.schedule_start_time = conflicts.start_time,
                  tsc.program_title = conflicts.title,
                  tsc.station_call_sign = conflicts.call_sign,
                  tsc.rr_flag_change = conflicts.rrflag,
                  tsc.duration_change = conflicts.duration_change

              when not matched then

              insert (schedule_id, schedule_start_time, program_title, station_call_sign, rr_flag_change, duration_change )
              values (conflicts.schedule_id,
                      conflicts.start_time,
                      conflicts.title,
                      conflicts.call_sign,
                      conflicts.rrflag,
                      conflicts.duration_change
                      );

          else
             -- Delete the previous entries
              delete from temp_schedule_conflicts;

                -- insert conflicted schedules with rights
            INSERT INTO temp_schedule_conflicts(schedule_id, schedule_start_time, program_title, station_call_sign, rr_flag_change, duration_change )
            SELECT  distinct s.schedule_id, s.start_time,  p.title ,  st.call_sign ,
                  CASE WHEN upper(s.request_record) ='Y' and upper(s.request_record) != upper(NVL(stg.request_record,'X')) THEN CAST (1 AS NUMBER(1)) ELSE CAST (0 AS NUMBER(1)) END AS rrflag,
                  CASE when NVL (s.duration, 1) != nvl (stg.duration, 1)  THEN CAST (1 AS NUMBER(1)) ELSE CAST (0 AS NUMBER(1)) END AS duration_change
            FROM SCHEDULE s JOIN stg_schedule stg ON (stg.start_time     = s.start_time AND stg.station_num  = s.station_id AND stg.database_key = s.program_id)
            JOIN program p ON ( p.program_id = s.program_id)
            JOIN station st ON (st.station_id = s.station_id)
            WHERE  s.start_time > CURRENT_TIMESTAMP
              AND ( upper(NVL(stg.hd_tv,'1'))!= upper(NVL(s.hd_tv,'1')) OR
                    upper(NVL(stg.repeat,'1')) != upper(NVL(s.repeat,'1')) OR
                    upper(NVL(stg.live_tape_delay,'1')) != upper(NVL(s.live_tape_delay,'1')) OR
                    upper(s.request_record) ='Y' and upper(s.request_record) != upper(NVL(stg.request_record,'X')) OR
                    NVL(s.series_link.getStringVal(),'X') != NVL(stg.series_link.getStringVal(),'X')  OR
                    NVL (s.duration, 1) != nvl (stg.duration, 1)
                  );


          end if;




         --Merge the schedule records
          merge into schedule sprod
          using (select * from stg_schedule where start_time > nvl(vcurrentdatetime,current_timestamp)) sstg
          on (sstg.database_key = sprod.program_id and sstg.station_num = sprod.station_id and sstg.start_time = sprod.start_time)
          when matched then
          update set sprod.schedule_language = sstg.schedule_language,
          sprod.end_time = sstg.end_time,
          sprod.duration = sstg.duration,
          sprod.tv_rating = sstg.tv_rating,
          sprod.hd_tv = sstg.hd_tv,
          sprod.part_number = sstg.part_number,
          sprod.no_of_parts = sstg.no_of_parts,
          sprod.cc = sstg.cc,
          sprod.stereo = sstg.stereo,
          sprod.live_tape_delay = sstg.live_tape_delay,
          sprod.subtitled = sstg.subtitled,
          sprod.premiere_finale = sstg.premiere_finale,
          sprod.joined_in_progress = sstg.joined_in_progress,
          sprod.cable_in_classroom = sstg.cable_in_classroom,
          sprod.sap = sstg.sap,
          sprod.sex_rating = sstg.sex_rating,
          sprod.violence_rating = sstg.violence_rating,
          sprod.language_rating = sstg.language_rating,
          sprod.dialog_rating = sstg.dialog_rating,
          sprod.fv_rating = sstg.fv_rating,
          sprod.enhanced = sstg.enhanced,
          sprod.three_d = sstg.three_d,
          sprod.letterbox = sstg.letterbox,
          sprod.dvs = sstg.dvs,
          sprod.series_link = sstg.series_link,
          sprod.request_record = sstg.request_record,
          sprod.repeat = sstg.repeat,
          sprod.blackout = sstg.blackout,
          sprod.dolby = sstg.dolby,
          sprod.USER_DATA1 = sstg.user_data1,
          sprod.USER_DATA2 = sstg.user_data2,
          sprod.USER_DATA3 = sstg.user_data3,
          sprod.USER_DATA4 = sstg.user_data4,
          sprod.USER_DATA5 = sstg.user_data5,
          sprod.USER_DATA6 = sstg.user_data6,
          sprod.USER_DATA7 = sstg.user_data7,
          sprod.USER_DATA8 = sstg.user_data8,
          sprod.USER_DATA9 = sstg.user_data9,
          sprod.USER_DATA10 = sstg.user_data10,
          sprod.USER_DATA11 = sstg.user_data11,
          sprod.SCHEDULE_GENERIC = sstg.SCHEDULE_GENERIC,
          sprod.created_by = 'SYSTEM',
          sprod.updated_by = 'SYSTEM'
          when not matched then
          insert (sprod.schedule_id,
          sprod.SCHEDULE_LANGUAGE,
          sprod.STATION_ID,
          sprod.PROGRAM_ID,
          sprod.START_TIME,
          sprod.END_TIME,
          sprod.DURATION,
          sprod.TV_RATING,
          sprod.HD_TV,
          sprod.PART_NUMBER,
          sprod.NO_OF_PARTS,
          sprod.CC,
          sprod.STEREO,
          sprod.LIVE_TAPE_DELAY,
          sprod.SUBTITLED,
          sprod.PREMIERE_FINALE,
          sprod.JOINED_IN_PROGRESS,
          sprod.CABLE_IN_CLASSROOM,
          sprod.SAP,
          sprod.SEX_RATING,
          sprod.VIOLENCE_RATING,
          sprod.LANGUAGE_RATING,
          sprod.DIALOG_RATING,
          sprod.FV_RATING,
          sprod.ENHANCED,
          sprod.THREE_D,
          sprod.LETTERBOX,
          sprod.DVS,
          sprod.series_link,
          sprod.REQUEST_RECORD,
          sprod.REPEAT,
          sprod.BLACKOUT,
          sprod.DOLBY,
          sprod.USER_DATA1,
          sprod.USER_DATA2,
          sprod.USER_DATA3,
          sprod.USER_DATA4,
          sprod.USER_DATA5,
          sprod.USER_DATA6,
          sprod.USER_DATA7,
          sprod.USER_DATA8,
          sprod.USER_DATA9,
          sprod.USER_DATA10,
          sprod.USER_DATA11,
          sprod.SCHEDULE_GENERIC,
          sprod.CREATED_BY,
          sprod.UPDATED_BY)
          values (SCHEDULE_SEQ.nextval,
          sstg.SCHEDULE_LANGUAGE,
          sstg.STATION_NUM,
          sstg.DATABASE_KEY,
          sstg.start_time,
          sstg.end_time,
          sstg.DURATION,
          sstg.TV_RATING,
          sstg.HD_TV,
          sstg.PART_NUMBER,
          sstg.NO_OF_PARTS,
          sstg.CC,
          sstg.STEREO,
          sstg.LIVE_TAPE_DELAY,
          sstg.SUBTITLED,
          sstg.PREMIERE_FINALE,
          sstg.JOINED_IN_PROGRESS,
          sstg.CABLE_IN_CLASSROOM,
          sstg.SAP,
          sstg.SEX_RATING,
          sstg.VIOLENCE_RATING,
          sstg.LANGUAGE_RATING,
          sstg.DIALOG_RATING,
          sstg.FV_RATING,
          sstg.ENHANCED,
          sstg.THREE_D,
          sstg.LETTERBOX,
          sstg.DVS,
          sstg.series_link,
          sstg.REQUEST_RECORD,
          sstg.REPEAT,
          sstg.BLACKOUT,
          sstg.DOLBY,
          sstg.USER_DATA1,
          sstg.USER_DATA2,
          sstg.USER_DATA3,
          sstg.USER_DATA4,
          sstg.USER_DATA5,
          sstg.USER_DATA6,
          sstg.USER_DATA7,
          sstg.USER_DATA8,
          sstg.USER_DATA9,
          sstg.USER_DATA10,
          sstg.USER_DATA11,
          sstg.SCHEDULE_GENERIC,
          'SYSTEM',
          'SYSTEM');

          commit;

         -- Adding entity mapping entries for the schedule_generic values
        insert into entity_mapping(mapping_id, name, table_name, column_name, type, generic_name, asset_type)
        select entity_mapping_seq.nextval as mapping_id,'SCHEDULE.'||ssgv.trialname as name, 'SCHEDULE' as table_name,'SCHEDULE_GENERIC' as column_name, cast (3 as number) as type , ssgv.trialname, ssgv.assettype
        FROM entity_mapping em right outer join
            (select distinct trialname, assettype  from stg_SCHEDULE_generic_view) ssgv on (em.generic_name = ssgv.trialname and em.asset_type = ssgv.assettype
            and upper(em.table_name)='SCHEDULE' and upper(em.column_name) = 'SCHEDULE_GENERIC')
            where em.generic_name is null and em.asset_type is null;

    epgguidedatalogging('End of Schedule merge. StartTime = '||current_timestamp);

    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(getpropertyvalue('epg_file_merging_ended'), '{0}',vfilename), '{1}', current_timestamp));

    --
    update config_setting set value ='N'  where key='Is_EPG_Schedule_Conflict_Completed';

    commit;




  end mergetstv;

  procedure rightsprocess (vfilename VARCHAR2 default '') is
    vscheduleid schedule.schedule_id%type;
    vepg_download_msg varchar2(1000);
    vevent_name varchar2(1000);
    vsql varchar2(32000);
    temp_table_name varchar2(30) := 'temp_rec_right_map';
    temp_count number := 0;
    vminstgschdate timestamp(0) with time zone;
    vmaxstgschdate timestamp(0) with time zone;
    rec_count number := 0;
    is_epg_sch_conflict_completed VARCHAR2(1) :='Y';
    sch_prg_md_chng_update_p_v backend_application_resources.value%type := getpropertyvalue('schedule_program_metadata_changed_to_update');
    sch_prg_md_chng_delete_p_v backend_application_resources.value%type := getpropertyvalue('schedule_program_metadata_changed_to_delete');
    sch_durtn_chng_too_late_p_v backend_application_resources.value%type := getpropertyvalue('schedule_duration_changed_too_late');
    sch_durtn_chng_p_v backend_application_resources.value%type := getpropertyvalue('schedule_duration_changed');

    rm_name_p_v  backend_application_resources.value%type := getpropertyvalue('rm_name');
    id_p_v  backend_application_resources.value%type := getpropertyvalue('id');
    schedule_start_time_p_v  backend_application_resources.value%type := getpropertyvalue('schedule_start_time');
    program_title_p_v  backend_application_resources.value%type := getpropertyvalue('program_title');
    station_call_sign_p_v  backend_application_resources.value%type := getpropertyvalue('station_call_sign');
    rcs_p_v backend_application_resources.value%type := getpropertyvalue('rcs');
    system_p_v backend_application_resources.value%type := getpropertyvalue('system');
   begin
    init();


    if (vfilename is not null) then
      --To serialize Recordables creation, acquire exclusive lock
      RESOURCELOCK('RECORDABLES_RESOURCE_LOCK',const_EPGRESOURCETIMEOUT);
      RESOURCELOCK('RECORDINGS_RESOURCE_LOCK',const_EPGRESOURCETIMEOUT);
      vepg_download_msg := replace(getpropertyvalue('epg_file'), '{0}',vfilename);
      vevent_name := 'EPG_DOWNLOAD';
    else
      --To serialize Recordables creation, acquire exclusive lock
      RESOURCELOCK('RECORDABLES_RESOURCE_LOCK');
      vepg_download_msg := getpropertyvalue('empty');
      vevent_name := 'RIGHTS';
    end if;

    epgguidedatalogging('Start of Rights processing. StartTime = '||current_timestamp);
    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(getpropertyvalue('rights_process_started'), '{0}',vepg_download_msg), '{1}', current_timestamp), null, vevent_name);

    merge into recordable_rights_map rrm
    using (select  distinct sschedule_id, rright_id, rm_id from
                            (select s.schedule_id as sschedule_id,r.right_id as rright_id, rrm.rm_id as rm_id , s.station_id as sstation_id,s.program_id as sprogram_id,s.start_time as sstart_time, s.end_time as send_time,
                            upper(s.hd_tv) as shd_tv, upper(s.repeat) as srepeat, case when upper(s.live_tape_delay) = 'DELAY' then cast('TAPE' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,
                            nvl(r.start_date,c.effective_date) as rstart_date, nvl(r.end_date,c.expiry_date) as rend_date,
                            r.start_time as rstart_time, r.end_time as rend_time,
                            upper(trim(TO_CHAR(s.start_time at TIME zone TO_CHAR(he.time_zone),'DAY')))                    AS sdayofweek,
                            upper(NVL(r.day_of_week,trim(TO_CHAR(s.start_time at TIME zone TO_CHAR(he.time_zone),'DAY')))) AS rdayofweek,
                            upper(r.repeat) as rrepeat, upper(r.hd) as rhd_tv, upper(r.live) as rlive,
                            nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                            case
                                when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                                when rserm.series_id  is not null then rserm.program_id
                            end as rsermprogram_id,
                            nvl(rstam.station_id,s.station_id) as rstamstation_id,
                            rl.station_id as rlstation_id,
                            he.time_zone as htime_zone
                            from schedule s join contract c on (s.start_time >= nvl(vcurrentdatetime,current_timestamp) and s.start_time >= c.effective_date and s.end_time <= c.expiry_date and c.status =1 /*Active Contracts*/ )
                            join right r on (r.contract_id = c.contract_id and r.state = 1 /* Active Right */)
                            join right_rm_map rrm on (r.right_id = rrm.right_id)
                            join rm_lineup rl on (rl.rm_id = rrm.rm_id and rl.station_id = s.station_id)
                            join rm on (rm.rm_id = rrm.rm_id)
                            join headend he on (rm.headend_id = he.headend_id)
                            left outer join right_program_map rpm on (r.right_id = rpm.right_id )
                            left outer join (select rserm_inner.series_id as series_id, rserm_inner.right_id as right_id, spm.program_id as program_id
                                from right_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                                on (r.right_id = rserm.right_id)
                            left outer join right_station_map rstam on (r.right_id = rstam.right_id)
                            )
                          where sstation_id = rstamstation_id
                          and sstation_id = rlstation_id
                          and sprogram_id = rpmprogram_id
                          and sprogram_id = rsermprogram_id
                          and sstart_time >= rstart_date
                          and send_time <= rend_date
                          and  trim(rdayofweek) like '%'||trim(sdayofweek)||'%'
                          and (srepeat = rrepeat or rrepeat is null)
                          and (shd_tv = rhd_tv or rhd_tv is null)
                          and (slive = rlive or rlive is null)
                          AND NVL(rstart_time,to_dsinterval('0 '||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), 'hh24:mi:ss'))) <= to_dsinterval('0 '||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), 'hh24:mi:ss'))
                          AND NVL(rend_time + numtodsinterval(TO_CHAR(sstart_time at TIME zone TO_CHAR(htime_zone),'J'),'DAY'),

                                          to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), 'J hh24:mi:ss')))

                                            >= to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), 'J hh24:mi:ss'))






                       ) sr
      on (sr.sschedule_id = rrm.schedule_id and sr.rright_id = rrm.right_id and sr.rm_id = rrm.rm_id)
      when not matched then
      insert (schedule_id,right_id, rm_id , created_by,updated_by,create_date,update_date)
      values (sr.sschedule_id,sr.rright_id, sr.rm_id, 'SYSTEM','SYSTEM',sysdate,sysdate);

      update schedule s
      set s.recordable = 1
      where s.start_time >= nvl(vcurrentdatetime,current_timestamp)
      and s.recordable = 0
      and exists (select 1 from recordable_rights_map rrm where rrm.schedule_id = s.schedule_id);

       select  value into is_epg_sch_conflict_completed from config_setting where key ='Is_EPG_Schedule_Conflict_Completed';

      -- Evaluate all the new schedule conflicts with any new recordings
       if (vfilename is not null or is_epg_sch_conflict_completed = 'N' ) then
           select greatest(min(start_time),vcurrentdatetime) as minstgschdate, greatest(max(start_time + numtodsinterval(duration,'SECOND')),vcurrentdatetime)  as maxstgschdate
            into vminstgschdate, vmaxstgschdate
            from stg_schedule;


        -- delete the previous entries
         vsql := ' delete from temp_del_conflicted_recables ';
        autonomoustransaction(vsql);

         -- insert the invalid recordables into the table
        vsql := ' insert into temp_del_conflicted_recables(schedule_id)
         ( select  distinct rrm.schedule_id from recordable_rights_map rrm join schedule sc on (rrm.schedule_id = sc.schedule_id and sc.start_time >= nvl(to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR''),current_timestamp))
           minus
          select  distinct sschedule_id from
                             (select s.schedule_id as sschedule_id,r.right_id as rright_id, rrm.rm_id as rm_id , s.station_id as sstation_id,s.program_id as sprogram_id,s.start_time as sstart_time, s.end_time as send_time,
                            upper(s.hd_tv) as shd_tv, upper(s.repeat) as srepeat, case when upper(s.live_tape_delay) = ''DELAY'' then cast(''TAPE'' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,
                            nvl(r.start_date,c.effective_date) as rstart_date, nvl(r.end_date,c.expiry_date) as rend_date,
                            r.start_time as rstart_time, r.end_time as rend_time,
                            upper(trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY''))) as sdayofweek,
                            upper(nvl(r.day_of_week,trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY'')))) as rdayofweek,
                            upper(r.repeat) as rrepeat, upper(r.hd) as rhd_tv, upper(r.live) as rlive,
                            nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                            case
                                when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                                when rserm.series_id  is not null then rserm.program_id
                            end as rsermprogram_id,
                            nvl(rstam.station_id,s.station_id) as rstamstation_id,
                            rl.station_id as rlstation_id,
                            he.time_zone as htime_zone
                            from schedule s join contract c on (s.start_time >= nvl(to_timestamp_tz('''||to_char(vcurrentdatetime,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR''),current_timestamp) and s.start_time >= c.effective_date and s.end_time <= c.expiry_date and c.status =1 /*Active Contracts*/ )
                            join right r on (r.contract_id = c.contract_id and r.state = 1 /* Active Right */)
                            join right_rm_map rrm on (r.right_id = rrm.right_id)
                            join rm_lineup rl on (rl.rm_id = rrm.rm_id and rl.station_id = s.station_id)
                            join rm on (rm.rm_id = rrm.rm_id)
                            join headend he on (rm.headend_id = he.headend_id)
                            left outer join right_program_map rpm on (r.right_id = rpm.right_id )
                            left outer join (select rserm_inner.series_id as series_id, rserm_inner.right_id as right_id, spm.program_id as program_id
                                from right_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                                on (r.right_id = rserm.right_id)
                            left outer join right_station_map rstam on (r.right_id = rstam.right_id )
                            )
                          where sstation_id = rstamstation_id
                          and sstation_id = rlstation_id
                          and sprogram_id = rpmprogram_id
                          and sprogram_id = rsermprogram_id
                          and sstart_time >= rstart_date
                          and send_time <= rend_date
                          and  trim(rdayofweek) like ''%''||trim(sdayofweek)||''%''
                          and (srepeat = rrepeat or rrepeat is null)
                          and (shd_tv = rhd_tv or rhd_tv is null)
                          and (slive = rlive or rlive is null)
                          AND NVL(rstart_time,to_dsinterval(''0 ''||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), ''hh24:mi:ss''))) <= to_dsinterval(''0 ''||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), ''hh24:mi:ss''))
                          AND NVL(rend_time + numtodsinterval(TO_CHAR(sstart_time at TIME zone TO_CHAR(htime_zone),''J''),''DAY''),
                          to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), ''J hh24:mi:ss'')))
                           >= to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), ''J hh24:mi:ss''))

        )';
        autonomoustransaction(vsql);


       select count(1) into rec_count
       from
            ( select rec.* from recording rec where  rec.status not in (const_recording_FAILED, const_recording_SUCCESS ,const_recording_DELETED, const_recording_PRTL_SUCCESS) and  rec.schedule_id in (select schedule_id from temp_del_conflicted_recables)
                 union all
                select rec.* from recording rec where  rec.status not in (const_recording_FAILED, const_recording_SUCCESS, const_recording_DELETED, const_recording_PRTL_SUCCESS) and  rec.schedule_id in ( select schedule_id from temp_schedule_conflicts tsc where tsc.rr_flag_change = 1) and rec.approval_type ='AUTO'
                UNION all
                select rec.* from recording rec where  rec.status not in (const_recording_FAILED, const_recording_SUCCESS, const_recording_DELETED,const_recording_PRTL_SUCCESS) and  rec.schedule_id in ( select schedule_id from temp_schedule_conflicts tsc where tsc.duration_change = 1)
              ) ;


       if ( rec_count > 0 ) then

                select count(1) into temp_count from user_tables where upper(table_name)=upper(temp_table_name);

                    if (temp_count = 1) then
                       vsql := 'drop table   ' ||temp_table_name||'  purge';
                      autonomoustransaction(vsql);
                    end if;

                -- Creating  a temp table
                vsql := ' create table   ' ||temp_table_name||'    nologging as
                      select distinct rec.recording_id,
                            rec.schedule_id,
                            rec.pool_id,
                            rp.rm_id,
                            sch.start_time as SCHEDULE_START_TIME ,
                            sch.end_time as SCHEDULE_END_TIME,
                            sch.duration as SCHEDULE_DURATION ,
                            sch.station_id,
                            sch.program_id,
                            sch.hd_tv,
                            sch.cc,
                            sch.repeat,
                            rm.name as RM_NAME,
                            rm.ip_address as RM_IP_ADDRESS,
                            rm.port as RM_PORT,
                            prg.episode_title,
                            rl.offset as STATION_OFFSET,
                            h.headend_location,
                            contract.contract_id,
                            prg.year_program,
                            rec.asset_id,
                            rec.title_asset_id,
                            rec.movie_asset_id,
                            rec.POSTER_ASSET_ID,
                            rec.POSTER_IMAGE_NAME,
                            rec.POSTER_DESCRIPTION,
                            prg.title as PROGRAM_TITLE,
                            rec.program_description,
                            rec.offset_start_time,
                            rec.offset_end_time,
                            rec.recording_mode,
                            rec.priority,
                            cast(0 as number) as state,
                            rec.provider_qa_contact,
                            rec.catalog_entry,
                            rec.billing_id,
                            rec.mpaa_rating,
                            rec.tv_rating,
                            rec.provider_id,
                            rec.provider_name,
                            rec.product,
                            rec.category,
                            rec.audio_type,
                            rec.max_copies,
                            rec.preview_period,
                            rec.REDUNDANCY_COUNT,
                            rec.licence_start_date,
                            rec.license_end_date,
                            rec.request_type,
                            pwr_vod.start_time as vod_start_time,
                            pwr_vod.duration as vod_duration,
                            pwr_vod.restrictions as vod_restrictions,
                            pwr_rti.start_time as rti_start_time ,
                            pwr_rti.duration as rti_duration,
                            pwr_rti.restrictions as rti_restrictions,
                            pwr_qc.start_time as qc_start_time ,
                            pwr_qc.duration as qc_duration,
                            pwr_qc.restrictions as qc_restrictions,
                            ''SYSTEM'' as created_by,
                            ''SYSTEM'' as updated_by,
                            current_timestamp as create_date,
                            current_timestamp as update_date,
                             rtrim(case when prg.advisory_desc1 is not null then prg.advisory_desc1||''|'' end ||
                                   case when prg.advisory_desc2 is not null then prg.advisory_desc2||''|'' end ||
                                   case when prg.advisory_desc3 is not null then prg.advisory_desc3||''|'' end ||
                                   case when prg.advisory_desc4 is not null then prg.advisory_desc4||''|'' end ||
                                   case when prg.advisory_desc5 is not null then prg.advisory_desc5||''|'' end ||
                                   case when prg.advisory_desc6 is not null then prg.advisory_desc6||''|'' end, ''|'') as advisories


                            from

                            (
                                with to_be_deleted_recordings as
                                       (select rec.* from recording rec where  rec.status not in ('||const_recording_FAILED||','||const_recording_SUCCESS||','||const_recording_DELETED||', '||const_recording_PRTL_SUCCESS||') and  rec.schedule_id in (select schedule_id from temp_del_conflicted_recables)
                                         union
                                        select rec.* from recording rec where  rec.status not in ('||const_recording_FAILED||','||const_recording_SUCCESS||','||const_recording_DELETED||', '||const_recording_PRTL_SUCCESS||' ) and  rec.schedule_id in ( select schedule_id from temp_schedule_conflicts tsc where tsc.rr_flag_change = 1) and rec.approval_type =''AUTO''
                                        )
                                      select  dr.*, cast('||const_recording_DELETE_REQUEST||' as number) as request_type from to_be_deleted_recordings dr

                                        union

                                        (
                                          select rec.*, cast('||const_recording_UPDATE_REQUEST||' as number) as request_type  from recording rec where  rec.status not in ('||const_recording_FAILED||','||const_recording_SUCCESS||','||const_recording_DELETED||', '||const_recording_PRTL_SUCCESS||' ) and  rec.schedule_id in ( select schedule_id from temp_schedule_conflicts tsc where tsc.duration_change = 1)
                                           minus
                                          select dr.*, cast('||const_recording_UPDATE_REQUEST||' as number) as request_type  from to_be_deleted_recordings dr
                                        )
                             ) rec

                            join( select sc.* , sc1.right_id from  ( select * from schedule where schedule.end_time > current_timestamp) sc
                                    join  (select schedule_id , right_id, rm_id from ( select row_number() over (partition by rrm1.schedule_id, rrm1.rm_id order by r.create_date ) as rownumber,
                                    rrm1.schedule_id,r.right_id, rrm1.rm_id
                                    from recordable_rights_map rrm1 , right r where
                                    r.right_id = rrm1.right_id and r.state=1) rrmouter where rrmouter.rownumber =1) sc1
                                    on (sc.schedule_id = sc1.schedule_id)) sch on  (sch.schedule_id = rec.schedule_id)
                            join right on (right.right_id = sch.right_id and right.state = 1)
                            join contract on (right.contract_id = contract.contract_id)
                            join program prg on (sch.program_id = prg.program_id)
                            join rtc_pool rp on (rec.pool_id = rp.pool_id)
                            join rm on (rm.rm_id = rp.rm_id)
                            join rm_lineup rl on (rl.rm_id = rm.rm_id and rl.station_id = sch.station_id and rl.headend_id = rm.headend_id)
                            join headend h on (h.headend_id = rm.headend_id)
                            left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_vod on (pwr_vod.recording_id = rec.recording_id and pwr_vod.type=''VOD'')
                            left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_rti on (pwr_rti.recording_id = rec.recording_id and pwr_rti.type=''RTI'')
                            left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_qc on (pwr_qc.recording_id = rec.recording_id and pwr_qc.type=''QC'')
                            where rec.status not in ('||const_recording_FAILED||','||const_recording_SUCCESS||','||const_recording_DELETED||', '||const_recording_PRTL_SUCCESS||' )
                            and sch.END_TIME + numtodsinterval(nvl(rec.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(rl.OFFSET,0),''SECOND'') >= current_timestamp
                            ';

                   autonomoustransaction(vsql);

                    -- insert the alarm activity logs for the deleted conflicted recordings
                         vsql :=' insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY, CREATE_DATE, UPDATE_DATE)

                          select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''' as component,
                           '''||sch_prg_md_chng_delete_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||start_time||'','||program_title_p_v||' = ''||program_title||'','||station_call_sign_p_v||'= ''||station_call_sign as description,
                         ''EPG_DOWNLOAD'',
                            24,
                          '''||system_p_v||''' as created_by,'''||system_p_v||''' as updated_by, current_timestamp, current_timestamp
                               from  (
                                        select prg.title as program_title, s.start_time at time zone to_char (current_timestamp, ''TZR'') as start_time,
                                        st.call_sign as station_call_sign , trrm.request_type , rm.name as rm_name, rm.rm_id from  ' ||temp_table_name||'   trrm
                                        join schedule s on (trrm.schedule_id = s.schedule_id)
                                        join rcs.program prg on (prg.program_id = s.program_id)
                                        join station st on (st.station_id = s.station_id)
                                        join rtc_pool rp on (trrm.pool_id = rp.pool_id)
                                        join rm on (rm.rm_id = rp.rm_id)
                                        where  trrm.request_type = '||const_recording_DELETE_REQUEST||'
                                      )';
                       CREATEOBJECT(vsql,'inserting the alarm activity log  for updated recordings', 'Y');

                   vsql := ' insert into temp_deleted_recordings (rm_id, rm_name, program_title, station_call_sign, schedule_start_time)
                          select distinct  rm.rm_id , rm.name as rm_name, prg.title as program_title, st.call_sign  as station_call_sign, s.start_time at time zone to_char (current_timestamp, ''TZR'') as start_time
                                        from  ' ||temp_table_name||'   trrm
                                        join schedule s on (trrm.schedule_id = s.schedule_id)
                                        join rcs.program prg on (prg.program_id = s.program_id)
                                        join station st on (st.station_id = s.station_id)
                                        join rtc_pool rp on (trrm.pool_id = rp.pool_id)
                                        join rm on (rm.rm_id = rp.rm_id)
                                         where  trrm.request_type = '||const_recording_DELETE_REQUEST||'
                      ';
               CREATEOBJECT(vsql,'Deleted Recordings');



         -- Insert Recording_queue entries for all future and current recordings
           -- new change for HA
         
   
                   
           vsql := ' insert into recording_queue(
                    RECORDING_ID,
                    SCHEDULE_ID,
                    POOL_ID,
                    RM_ID,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    STATION_ID,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    REPEAT,
                    RM_NAME,
                    RM_IP_ADDRESS,
                    RM_PORT,
                    EPISODE_TITLE,
                    STATION_OFFSET,
                    HEADEND_NAME,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    ASSET_ID,
                    TITLE_ASSET_ID,
                    MOVIE_ASSET_ID,
                    POSTER_ASSET_ID,
                    POSTER_IMAGE_NAME,
                    POSTER_DESCRIPTION,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    STATE,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REQUEST_TYPE,
                    VOD_START_TIME,
                    VOD_DURATION,
                    VOD_RESTRICTIONS,
                    RTI_START_TIME,
                    RTI_DURATION,
                    RTI_RESTRICTIONS,
                    QC_START_TIME,
                    QC_DURATION,
                    QC_RESTRICTIONS,
                    CREATED_BY,
                    UPDATED_BY,
                    CREATE_DATE,
                    UPDATE_DATE,
                    advisories) 
                    select rec_queue.* FROM
                    (SELECT distinct
                    trrm.RECORDING_ID,
                    trrm.SCHEDULE_ID,
                    trrm.POOL_ID,
                    trrm.RM_ID,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    STATION_ID,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    REPEAT,
                    RM_NAME,
                    RM_IP_ADDRESS,
                    RM_PORT,
                    EPISODE_TITLE,
                    STATION_OFFSET,
                    HEADEND_LOCATION,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    trrm.ASSET_ID,
                    trrm.TITLE_ASSET_ID,
                    trrm.MOVIE_ASSET_ID,
                    trrm.POSTER_ASSET_ID,
                    trrm.POSTER_IMAGE_NAME,
                    trrm.POSTER_DESCRIPTION,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    STATE,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REQUEST_TYPE,
                    VOD_START_TIME,
                    VOD_DURATION,
                    VOD_RESTRICTIONS,
                    RTI_START_TIME,
                    RTI_DURATION,
                    RTI_RESTRICTIONS,
                    QC_START_TIME,
                    QC_DURATION,
                    QC_RESTRICTIONS,
                    trrm.CREATED_BY,
                    trrm.UPDATED_BY,
                    trrm.CREATE_DATE,
                    trrm.UPDATE_DATE,
                    advisories

                    from   ' ||temp_table_name||'  trrm
                   	join rtc_pool rp on (trrm.pool_id = rp.pool_id) where
                  	trrm.rm_id = rp.rm_id
                    where  trrm.request_type = '||const_recording_DELETE_REQUEST||'
                    order by (trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'')), trrm.PRIORITY) rec_queue';

                   CREATEOBJECT(vsql,'inserting Recording_queue records for future deleted recordings', 'Y');                   
             
              -- new HA change
                insertRecQueue;
                
                   -- Mark the recordings are as deleted status
                   vsql := 'update recording rec set status ='||const_recording_DELETED||' , status_code='||const_rec_status_code_DELETED||'   where exists (select 1 from '|| temp_table_name ||'  trrm where rec.recording_id = trrm.recording_id and trrm.request_type ='||const_recording_DELETE_REQUEST|| ') ';
                  CREATEOBJECT(vsql,'Updating the recordings status to be deleted', 'Y');


                  -- create alarms for non-updatable recordings because of pre_processing_interval or stop_block_out properties of the RM
                  vsql :=' insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY, CREATE_DATE, UPDATE_DATE)

                          select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''' as component,
                            '''||sch_durtn_chng_too_late_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||start_time||'','||program_title_p_v||' = ''||program_title||'','||station_call_sign_p_v||'= ''||station_call_sign  as description,

                          ''EPG_DOWNLOAD'',
                            12,
                          '''||system_p_v||''' as created_by,'''||system_p_v||''' as updated_by, current_timestamp, current_timestamp
                               from  (
                                        select prg.title as program_title, s.start_time at time zone to_char (current_timestamp, ''TZR'') as start_time,
                                        st.call_sign as station_call_sign , trrm.request_type , rm.name as rm_name, rm.rm_id from  ' ||temp_table_name||'   trrm
                                        join (select rec.recording_id,rec.pool_id,
                                                case
                                                    when current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval, 0), ''SECOND'') <= rec.adjusted_start_time then ''UPDATABLE''
                                                    WHEN nvl(rm.stop_black_out, 0) = 0 then ''NONUPDATABLE''
                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') > rec.adjusted_end_time THEN ''NONUPDATABLE''

                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') < rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') >
                                                            trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') THEN ''NONUPDATABLE''
                                                 ELSE ''UPDATABLE''

                                                 END AS IS_UPDATABLE
                                                from
                                                ' ||temp_table_name||'   trrm, recording rec, rm
                                                where
                                                trrm.recording_id = rec.recording_id and
                                                trrm.rm_id = rm.rm_id and

                                                trrm.request_type ='||const_recording_UPDATE_REQUEST||'
                                             ) nur on (nur.IS_UPDATABLE = ''NONUPDATABLE'' and nur.recording_id = trrm.recording_id)

                                        join schedule s on (trrm.schedule_id = s.schedule_id)
                                        join rcs.program prg on (prg.program_id = s.program_id)
                                        join station st on (st.station_id = s.station_id)
                                        join rtc_pool rp on (trrm.pool_id = rp.pool_id)
                                        join rm on (rm.rm_id = rp.rm_id)
                                        where  trrm.request_type = '||const_recording_UPDATE_REQUEST||'
                                      )';
                       CREATEOBJECT(vsql,'inserting the alarm activity log  for non-updated recordings', 'Y');

              -- recording logs

                vsql := 'insert into recording_log(RECORDING_LOG_ID,
                                      RECORDING_ID,
                                      LOG_MESSAGE,
                                      CREATED_BY,
                                      UPDATED_BY,
                                      CREATE_DATE,
                                      UPDATE_DATE)
                select RECORDING_LOG_SEQ.nextval,tr.recording_id,''Schedule duration has changed. But it is too late to submit the duration update request to the RM  for Recording'',''SYSTEM'',''SYSTEM'',current_timestamp,current_timestamp
                from (select rec.recording_id,rec.pool_id,
                                                case
                                                    when current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval, 0), ''SECOND'') <= rec.adjusted_start_time then ''UPDATABLE''
                                                    WHEN nvl(rm.stop_black_out, 0) = 0 then ''NONUPDATABLE''
                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') > rec.adjusted_end_time THEN ''NONUPDATABLE''

                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') < rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') >
                                                            trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') THEN ''NONUPDATABLE''
                                                 ELSE ''UPDATABLE''

                                                 END AS IS_UPDATABLE
                                                from
                                                ' ||temp_table_name||'   trrm, recording rec, rm
                                                where
                                                trrm.recording_id = rec.recording_id and
                                                trrm.rm_id = rm.rm_id and

                                                trrm.request_type ='||const_recording_UPDATE_REQUEST||'
                                             ) tr  where tr.IS_UPDATABLE =''NONUPDATABLE'' ';

               CREATEOBJECT(vsql,'Writing Recording logs');

              vsql :=' insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY, CREATE_DATE, UPDATE_DATE)

                          select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''' as component,
                                '''||sch_durtn_chng_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||start_time||'','||program_title_p_v||' = ''||program_title||'','||station_call_sign_p_v||'= ''||station_call_sign  as description,

                           ''EPG_DOWNLOAD'',
                            12,
                          '''||system_p_v||''' as created_by,'''||system_p_v||''' as updated_by, current_timestamp, current_timestamp
                               from  (
                                        select prg.title as program_title, s.start_time at time zone to_char (current_timestamp, ''TZR'') as start_time,
                                        st.call_sign as station_call_sign , trrm.request_type , rm.name as rm_name, rm.rm_id from  ' ||temp_table_name||'   trrm
                                        join (select rec.recording_id,rec.pool_id,
                                                case
                                                    when current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval, 0), ''SECOND'') <= rec.adjusted_start_time then ''UPDATABLE''
                                                    WHEN nvl(rm.stop_black_out, 0) = 0 then ''NONUPDATABLE''
                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') > rec.adjusted_end_time THEN ''NONUPDATABLE''

                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') < rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') >
                                                            trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') THEN ''NONUPDATABLE''
                                                 ELSE ''UPDATABLE''

                                                 END AS IS_UPDATABLE
                                                from
                                                ' ||temp_table_name||'   trrm, recording rec, rm
                                                where
                                                trrm.recording_id = rec.recording_id and
                                                trrm.rm_id = rm.rm_id and

                                                trrm.request_type ='||const_recording_UPDATE_REQUEST||'
                                             ) nur on (nur.IS_UPDATABLE = ''UPDATABLE'' and nur.recording_id = trrm.recording_id)
                                        join schedule s on (trrm.schedule_id = s.schedule_id)
                                        join rcs.program prg on (prg.program_id = s.program_id)
                                        join station st on (st.station_id = s.station_id)
                                        join rtc_pool rp on (trrm.pool_id = rp.pool_id)
                                        join rm on (rm.rm_id = rp.rm_id)
                                        where  trrm.request_type = '||const_recording_UPDATE_REQUEST||'
                                      )';
                CREATEOBJECT(vsql,'inserting the alarm activity log  for updated recordings');


          -- update the recordings of the schedules that have changes in the duration field.

          vsql := ' update recording rec set adjusted_end_time = (select trrm.schedule_end_time + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.station_offset,0),''SECOND'') as adjusted_end_time
                                                                      from  ' ||temp_table_name||'   trrm
                                                                      join (select rec.recording_id,rec.pool_id,
                                                                              case
                                                                                  when current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval, 0), ''SECOND'') <= rec.adjusted_start_time then ''UPDATABLE''
                                                                                  WHEN nvl(rm.stop_black_out, 0) = 0 then ''NONUPDATABLE''
                                                                                  when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > rec.adjusted_end_time
                                                                                          and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') > rec.adjusted_end_time THEN ''NONUPDATABLE''

                                                                                  when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') < rec.adjusted_end_time
                                                                                          and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') >
                                                                                          trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') THEN ''NONUPDATABLE''
                                                                               ELSE ''UPDATABLE''

                                                                               END AS IS_UPDATABLE
                                                                              from
                                                                              ' ||temp_table_name||'   trrm, recording rec, rm
                                                                              where
                                                                              trrm.recording_id = rec.recording_id and
                                                                              trrm.rm_id = rm.rm_id and

                                                                              trrm.request_type ='||const_recording_UPDATE_REQUEST||'
                                                                           ) nur on (nur.IS_UPDATABLE = ''UPDATABLE'' and nur.recording_id = trrm.recording_id)
                                                                            where trrm.recording_id = rec.recording_id and trrm.request_type ='||const_recording_UPDATE_REQUEST||' )
                    where rec.recording_id in (select nur.recording_id from (select rec.recording_id,rec.pool_id,
                                                                              case
                                                                                  when current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval, 0), ''SECOND'') <= rec.adjusted_start_time then ''UPDATABLE''
                                                                                  WHEN nvl(rm.stop_black_out, 0) = 0 then ''NONUPDATABLE''
                                                                                  when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > rec.adjusted_end_time
                                                                                          and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') > rec.adjusted_end_time THEN ''NONUPDATABLE''

                                                                                  when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') < rec.adjusted_end_time
                                                                                          and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') >
                                                                                          trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') THEN ''NONUPDATABLE''
                                                                               ELSE ''UPDATABLE''

                                                                               END AS IS_UPDATABLE
                                                                              from
                                                                              ' ||temp_table_name||'   trrm, recording rec, rm
                                                                              where
                                                                              trrm.recording_id = rec.recording_id and
                                                                              trrm.rm_id = rm.rm_id and

                                                                              trrm.request_type ='||const_recording_UPDATE_REQUEST||'
                                                                           ) nur     where nur.IS_UPDATABLE =''UPDATABLE'' )
                        and  schedule_id in (select schedule_id from temp_schedule_conflicts tsc where tsc.duration_change = 1)
                     ';

        CREATEOBJECT(vsql,'Updating the recordings adjusted_end_time with duration changes ');

              
              
              
               vsql := ' insert into recording_queue(
                    RECORDING_ID,
                    SCHEDULE_ID,
                    POOL_ID,
                    RM_ID,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    STATION_ID,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    REPEAT,
                    RM_NAME,
                    RM_IP_ADDRESS,
                    RM_PORT,
                    EPISODE_TITLE,
                    STATION_OFFSET,
                    HEADEND_NAME,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    ASSET_ID,
                    TITLE_ASSET_ID,
                    MOVIE_ASSET_ID,
                    POSTER_ASSET_ID,
                    POSTER_IMAGE_NAME,
                    POSTER_DESCRIPTION,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    STATE,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REQUEST_TYPE,
                    VOD_START_TIME,
                    VOD_DURATION,
                    VOD_RESTRICTIONS,
                    RTI_START_TIME,
                    RTI_DURATION,
                    RTI_RESTRICTIONS,
                    QC_START_TIME,
                    QC_DURATION,
                    QC_RESTRICTIONS,
                    CREATED_BY,
                    UPDATED_BY,
                    CREATE_DATE,
                    UPDATE_DATE,
                    advisories)
                    select rec_queue.* FROM
                    (SELECT distinct
                    trrm.RECORDING_ID,
                    trrm.SCHEDULE_ID,
                    trrm.POOL_ID,
                    trrm.RM_ID,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    STATION_ID,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    REPEAT,
                    RM_NAME,
                    RM_IP_ADDRESS,
                    RM_PORT,
                    EPISODE_TITLE,
                    STATION_OFFSET,
                    HEADEND_LOCATION,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    trrm.ASSET_ID,
                    trrm.TITLE_ASSET_ID,
                    trrm.MOVIE_ASSET_ID,
                    trrm.POSTER_ASSET_ID,
                    trrm.POSTER_IMAGE_NAME,
                    trrm.POSTER_DESCRIPTION,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    STATE,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REQUEST_TYPE,
                    VOD_START_TIME,
                    VOD_DURATION,
                    VOD_RESTRICTIONS,
                    RTI_START_TIME,
                    RTI_DURATION,
                    RTI_RESTRICTIONS,
                    QC_START_TIME,
                    QC_DURATION,
                    QC_RESTRICTIONS,
                    trrm.CREATED_BY,
                    trrm.UPDATED_BY,
                    trrm.CREATE_DATE,
                    trrm.UPDATE_DATE,
                    advisories

                    from   ' ||temp_table_name||'  trrm
                    join (select rec.recording_id,rec.pool_id,
                                                case
                                                    when current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval, 0), ''SECOND'') <= rec.adjusted_start_time then ''UPDATABLE''
                                                    WHEN nvl(rm.stop_black_out, 0) = 0 then ''NONUPDATABLE''
                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') > rec.adjusted_end_time THEN ''NONUPDATABLE''

                                                    when trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') < rec.adjusted_end_time
                                                            and current_timestamp + numtodsinterval(nvl(rm.stop_black_out,0), ''SECOND'') >
                                                            trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') THEN ''NONUPDATABLE''
                                                 ELSE ''UPDATABLE''

                                                 END AS IS_UPDATABLE
                                                from
                                                ' ||temp_table_name||'   trrm, recording rec, rm
                                                where
                                                trrm.recording_id = rec.recording_id and
                                                trrm.rm_id = rm.rm_id and

                                                trrm.request_type = 1
                        ) nur on (nur.IS_UPDATABLE = ''UPDATABLE'' and nur.recording_id = trrm.recording_id)
                   	join rtc_pool rp on (trrm.pool_id = rp.pool_id) where
                  	trrm.rm_id = rp.rm_id
                    and  trrm.request_type = '||const_recording_UPDATE_REQUEST||'
                    order by (trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'')), trrm.PRIORITY) rec_queue';

                  CREATEOBJECT(vsql,'creating the recording queue entry  with duration changes ');                                  
              -- new HA change
                insertRecQueue;
            
             end if; -- row_count > 0

                   -- delete all the invalid recordables

             delete from recordable_rights_map where (schedule_id, right_id, rm_id ) in (
                     select   rrm.schedule_id, rrm.right_id, rrm.rm_id from recordable_rights_map rrm join schedule sc on (rrm.schedule_id = sc.schedule_id and sc.start_time >= nvl(vcurrentdatetime,current_timestamp))
                      minus
                     select  sschedule_id, rright_id, rm_id from
                          (select s.schedule_id as sschedule_id,r.right_id as rright_id, rrm.rm_id as rm_id , s.station_id as sstation_id,s.program_id as sprogram_id,s.start_time as sstart_time, s.end_time as send_time,
                            upper(s.hd_tv) as shd_tv, upper(s.repeat) as srepeat, case when upper(s.live_tape_delay) = 'DELAY' then cast('TAPE' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,
                            nvl(r.start_date,c.effective_date) as rstart_date, nvl(r.end_date,c.expiry_date) as rend_date,
                            r.start_time as rstart_time, r.end_time as rend_time,
                            upper(trim(to_char(s.start_time at time zone to_char(he.time_zone),'DAY'))) as sdayofweek,
                            upper(nvl(r.day_of_week,trim(to_char(s.start_time at time zone to_char(he.time_zone),'DAY')))) as rdayofweek,
                            upper(r.repeat) as rrepeat, upper(r.hd) as rhd_tv, upper(r.live) as rlive,
                            nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                            case
                                when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                                when rserm.series_id  is not null then rserm.program_id
                            end as rsermprogram_id,
                            nvl(rstam.station_id,s.station_id) as rstamstation_id,
                            rl.station_id as rlstation_id,
                            he.time_zone as htime_zone
                            from schedule s join contract c on (s.start_time >= nvl(vcurrentdatetime,current_timestamp) and s.start_time >= c.effective_date and s.end_time <= c.expiry_date and c.status =1 /*Active Contracts*/ )
                            join right r on (r.contract_id = c.contract_id and r.state = 1 /* Active Right */)
                            join right_rm_map rrm on (r.right_id = rrm.right_id)
                            join rm_lineup rl on (rl.rm_id = rrm.rm_id and rl.station_id = s.station_id)
                            join rm on (rm.rm_id = rrm.rm_id)
                            join headend he on (rm.headend_id = he.headend_id)
                            left outer join right_program_map rpm on (r.right_id = rpm.right_id )
                            left outer join (select rserm_inner.series_id as series_id, rserm_inner.right_id as right_id, spm.program_id as program_id
                                from right_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                                on (r.right_id = rserm.right_id)
                            left outer join right_station_map rstam on (r.right_id = rstam.right_id )
                            )
                          where sstation_id = rstamstation_id
                          and sstation_id = rlstation_id
                          and sprogram_id = rpmprogram_id
                          and sprogram_id = rsermprogram_id
                          and sstart_time >= rstart_date
                          and send_time <= rend_date
                          and  trim(rdayofweek) like '%'||trim(sdayofweek)||'%'
                          and (srepeat = rrepeat or rrepeat is null)
                          and (shd_tv = rhd_tv or rhd_tv is null)
                          and (slive = rlive or rlive is null)
                          AND NVL(rstart_time,to_dsinterval('0 '||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), 'hh24:mi:ss'))) <= to_dsinterval('0 '||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), 'hh24:mi:ss'))
                          AND NVL(rend_time + numtodsinterval(TO_CHAR(sstart_time at TIME zone TO_CHAR(htime_zone),'J'),'DAY'),

                                          to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), 'J hh24:mi:ss')))

                                            >= to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), 'J hh24:mi:ss'))



             );

            -- update the schedules recordable status
            update schedule set recordable = default
            where start_time > current_timestamp and recordable = 1
                    and schedule_id in ( select schedule_id from temp_del_conflicted_recables);

          update config_setting set value ='Y'  where key='Is_EPG_Schedule_Conflict_Completed';

         end if;  -- vfilename is not null

     -- Delete all the recordings if they are not under the covered rights


    epgguidedatalogging('End of Rights processing. StartTime = '||current_timestamp);
    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(getpropertyvalue('rights_process_ended'), '{0}',vepg_download_msg), '{1}', current_timestamp), null, vevent_name);


  end rightsprocess;

   procedure rightsedit(in_right_id number, p_countrecording OUT number) is

  temp_recordable_right_map varchar2 (30) := 'temp_recordable_right_map';
  temp_deleted_recordings varchar2 (30) := 'temp_deleted_recordings';
  temp_deleted_recordables varchar2 (30) := 'temp_deleted_recordables';
  p_deleted_recordables number := 0;
  p_deleted_recordings number :=0;
  right_state right.state%type;
  vsql varchar2(32000);
  temp_count number ;

   begin

    RECORDABLESRECRESOURCELOCK();
     -- Drop temp tables
          temp_count := 0;
          select count(1) into temp_count from user_tables where upper(table_name)=upper(temp_recordable_right_map);
           if (temp_count = 1) then
             vsql := 'drop table   ' ||temp_recordable_right_map||'  purge';
            autonomoustransaction(vsql);
          end if;

      -- Create a temp Recordable and Right table with new Recordable set for the Right

      vsql := ' create table ' || temp_recordable_right_map || ' (schedule_id number(19,0), right_id number(19,0), rm_id number(19,0)) ';
      autonomoustransaction(vsql);
      -- create index on temp table

        vsql := ' CREATE UNIQUE INDEX t_sch_right_uni_index  ON ' || temp_recordable_right_map || '  (schedule_id,right_id, rm_id) ';

         autonomoustransaction(vsql);

      vsql := ' insert into ' || temp_recordable_right_map || ' (schedule_id, right_id, rm_id)
               select distinct sschedule_id as schedule_id, right_id, rm_id from
              (select s.schedule_id as sschedule_id,r.right_id , rrm.rm_id , s.station_id as sstation_id,s.program_id as sprogram_id,s.start_time as sstart_time, s.end_time as send_time,
                            upper(s.hd_tv) as shd_tv, upper(s.repeat) as srepeat, case when upper(s.live_tape_delay) = ''DELAY'' then cast(''TAPE'' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,
                            nvl(r.start_date,c.effective_date) as rstart_date, nvl(r.end_date,c.expiry_date) as rend_date,
                            r.start_time as rstart_time, r.end_time as rend_time,
                            upper(trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY''))) as sdayofweek,
                            upper(nvl(r.day_of_week,trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY'')))) as rdayofweek,
                            upper(r.repeat) as rrepeat, upper(r.hd) as rhd_tv, upper(r.live) as rlive,
                            nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                            case
                                when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                                when rserm.series_id  is not null then rserm.program_id
                            end as rsermprogram_id,
                            nvl(rstam.station_id,s.station_id) as rstamstation_id,
                            rl.station_id as rlstation_id,
                            he.time_zone as htime_zone
                            from schedule s join contract c on (s.start_time >= c.effective_date and s.end_time <= c.expiry_date and c.status =1 /*Active Contracts*/ )
                            join right r on (r.right_id = '||in_right_id || ' and r.contract_id = c.contract_id and r.state = 1 /* Active Right */)
                            join right_rm_map rrm on (r.right_id = rrm.right_id)
                            join rm_lineup rl on (rl.rm_id = rrm.rm_id and rl.station_id = s.station_id)
                            join rm on (rm.rm_id = rrm.rm_id)
                            join headend he on (rm.headend_id = he.headend_id)
                            left outer join right_program_map rpm on (r.right_id = rpm.right_id )
                            left outer join (select rserm_inner.series_id as series_id, rserm_inner.right_id as right_id, spm.program_id as program_id
                                from right_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                                on (r.right_id = rserm.right_id)
                            left outer join right_station_map rstam on (r.right_id = rstam.right_id )

                            )
                          where sstation_id = rstamstation_id
                          and sstation_id = rlstation_id
                          and sprogram_id = rpmprogram_id
                          and sprogram_id = rsermprogram_id
                          and sstart_time >= rstart_date
                          and send_time <= rend_date
                          and  trim(rdayofweek) like ''%''||trim(sdayofweek)||''%''
                          and (srepeat = rrepeat or rrepeat is null)
                          and (shd_tv = rhd_tv or rhd_tv is null)
                          and (slive = rlive or rlive is null)
                          AND NVL(rstart_time,to_dsinterval(''0 ''||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), ''hh24:mi:ss''))) <= to_dsinterval(''0 ''||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), ''hh24:mi:ss''))
                          AND NVL(rend_time + numtodsinterval(TO_CHAR(sstart_time at TIME zone TO_CHAR(htime_zone),''J''),''DAY''),
                          to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), ''J hh24:mi:ss'')))
                           >= to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), ''J hh24:mi:ss''))

                       ';

        execute immediate (vsql);


      -- Get any exclusive current/future Recording's recordables are missed from the Right update
      vsql := '  select count(1) as recording_count
                from (
                        select distinct rec.schedule_id, rm.rm_id from recording rec join rtc_pool rp on (rp.pool_id = rec.pool_id) join rm on (rp.rm_id = rm.rm_id)

                            where  rec.status != '||const_recording_DELETED ||'
                                  and rec.adjusted_end_time >= current_timestamp and
                                  exists (
                                            select 1 from (select schedule_id, right_id, rm_id,  count(1) over ( partition by schedule_id, rm_id) as
                                            counter from recordable_rights_map )  temp where temp.counter =1
                                            and temp.right_id= '||in_right_id || ' and temp.schedule_id = rec.schedule_id and rm.rm_id = temp.rm_id )

                minus

                select distinct schedule_id, rm_id from '||temp_recordable_right_map || ' )

              ';

      execute immediate vsql into p_countrecording;


      -- If there are no missed recordables for any current/future recordings, delete all the invalid recorables for this right
      if (p_countrecording = 0 ) then

          select state into right_state from right where right_id = in_right_id;

          if (right_state = const_STATE_INACTIVE) then

              -- delete the  temp_recordable_right_map and rerun the right with inactive mode
              vsql := ' delete from ' || temp_recordable_right_map;
              execute immediate vsql;

              -- Rerunning with Inactive mode
                        vsql := ' insert into ' || temp_recordable_right_map || ' (schedule_id, right_id, rm_id)
                   select distinct sschedule_id as schedule_id, right_id, rm_id from
                    (select s.schedule_id as sschedule_id,r.right_id , rrm.rm_id , s.station_id as sstation_id,s.program_id as sprogram_id,s.start_time as sstart_time, s.end_time as send_time,
                            upper(s.hd_tv) as shd_tv, upper(s.repeat) as srepeat, case when upper(s.live_tape_delay) = ''DELAY'' then cast(''TAPE'' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,

                            nvl(r.start_date,c.effective_date) as rstart_date, nvl(r.end_date,c.expiry_date) as rend_date,
                            r.start_time as rstart_time, r.end_time as rend_time,
                            upper(trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY''))) as sdayofweek,
                             upper(nvl(r.day_of_week,trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY'')))) as rdayofweek,
                            upper(r.repeat) as rrepeat, upper(r.hd) as rhd_tv, upper(r.live) as rlive,
                            nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                            case
                                when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                                when rserm.series_id  is not null then rserm.program_id
                            end as rsermprogram_id,
                            nvl(rstam.station_id,s.station_id) as rstamstation_id,
                            rl.station_id as rlstation_id,
                            he.time_zone as htime_zone
                            from schedule s join contract c on (s.start_time >= c.effective_date and s.end_time <= c.expiry_date and c.status =1 /*Active Contracts*/ )
                            join right r on (r.right_id = '||in_right_id || ' and r.contract_id = c.contract_id  /* InActive Right */)
                            join right_rm_map rrm on (r.right_id = rrm.right_id)
                            join rm_lineup rl on (rl.rm_id = rrm.rm_id and rl.station_id = s.station_id)
                            join rm on (rm.rm_id = rrm.rm_id)
                            join headend he on (rm.headend_id = he.headend_id)
                            left outer join right_program_map rpm on (r.right_id = rpm.right_id )
                            left outer join (select rserm_inner.series_id as series_id, rserm_inner.right_id as right_id, spm.program_id as program_id
                                from right_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                                on (r.right_id = rserm.right_id)
                            left outer join right_station_map rstam on (r.right_id = rstam.right_id )
                            )
                          where sstation_id = rstamstation_id
                          and sstation_id = rlstation_id
                          and sprogram_id = rpmprogram_id
                          and sprogram_id = rsermprogram_id
                          and sstart_time >= rstart_date
                          and send_time <= rend_date
                          and  trim(rdayofweek) like ''%''||trim(sdayofweek)||''%''
                          and (srepeat = rrepeat or rrepeat is null)
                          and (shd_tv = rhd_tv or rhd_tv is null)
                          and (slive = rlive or rlive is null)
                          AND NVL(rstart_time,to_dsinterval(''0 ''||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), ''hh24:mi:ss''))) <= to_dsinterval(''0 ''||TO_CHAR (sstart_time at TIME zone TO_CHAR(htime_zone), ''hh24:mi:ss''))
                          AND NVL(rend_time + numtodsinterval(TO_CHAR(sstart_time at TIME zone TO_CHAR(htime_zone),''J''),''DAY''),
                          to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), ''J hh24:mi:ss'')))
                           >= to_dsinterval(TO_CHAR (send_time at TIME zone TO_CHAR(htime_zone), ''J hh24:mi:ss''))
                   ';

            execute immediate (vsql);

          end if;

          -- Check for the recording history for the existing Rights compliance

                  vsql := ' select count(1) from ( '|| '
                        select distinct rec.recording_id, rsch.schedule_id, rsch.right_id from
                          (
                            select temp.schedule_id, temp.right_id, temp.rm_id from (select schedule_id,
                            right_id, rm_id , count(1) over ( partition by schedule_id, rm_id) as
                            counter from recordable_rights_map )  temp where temp.counter =1 and temp.right_id='||in_right_id ||'
                              minus
                            select distinct schedule_id, right_id, rm_id from '|| temp_recordable_right_map || '
                          ) rsch join right_rm_map rrm on (rrm.right_id = rsch.right_id and rrm.rm_id =  rsch.rm_id)  join  rtc_pool rp on (rp.rm_id = rsch.rm_id) join recording rec on ( rec.schedule_id = rsch.schedule_id and rec.pool_id = rp.pool_id and rec.status != '||const_recording_DELETED ||' )
                      )';
          execute immediate vsql into p_countrecording;

         if (p_countrecording = 0 ) then

                  temp_count := 0;
                  select count(1) into temp_count from user_tables where  upper(table_name)=upper(temp_deleted_recordables);
                   if (temp_count = 1) then
                     vsql := 'drop table   ' ||temp_deleted_recordables||'  purge';
                    autonomoustransaction(vsql);
                  end if;
                    vsql := ' create table   ' || temp_deleted_recordables || ' (schedule_id number(19,0), right_id number(19,0), rm_id number(19,0))';
                    autonomoustransaction(vsql);

                    vsql := ' CREATE UNIQUE INDEX t_del_recordables_index ON ' || temp_deleted_recordables || '  (schedule_id, right_id, rm_id) ';
                    autonomoustransaction(vsql);

              -- If right is updated as Inactive, the non-exclusive recordables for the past recordings and future recordables associated with this Right need to be deleted

                if (right_state = const_STATE_INACTIVE) then
                     vsql := ' insert into  ' || temp_deleted_recordables || '(schedule_id, right_id, rm_id)
                                select  distinct rrmap.schedule_id , rrmap.right_id , rrmap.rm_id
                                 from recordable_rights_map rrmap, schedule sch where sch.start_time > current_timestamp
                                        and  rrmap.schedule_id = sch.schedule_id and rrmap.right_id='||in_right_id;
                  execute immediate vsql;

                elsif (right_state = const_STATE_ACTIVE)  then

                    -- All excluded future recordables for this right
                    vsql := ' insert into  ' || temp_deleted_recordables || ' (schedule_id, right_id, rm_id)

                             select distinct rsch.schedule_id, rsch.right_id, rsch.rm_id from (
                                select  distinct rrmap.schedule_id , rrmap.right_id, rrmap.rm_id
                                 from recordable_rights_map rrmap, schedule sch where sch.start_time > current_timestamp and  rrmap.schedule_id = sch.schedule_id and rrmap.right_id='||in_right_id ||'
                               minus
                              select distinct schedule_id, right_id, rm_id from '|| temp_recordable_right_map || ' )rsch ';
                  execute immediate vsql;
                end  if;

                  -- If deleted future recordables greater than 0 then do the following

                  vsql := ' select count(1) as recordable_count from ' || temp_deleted_recordables;

                  execute immediate vsql into p_deleted_recordables;


                  if (p_deleted_recordables > 0 ) then

                                -- Delete all the future non-recordables mapppings for the right

                                vsql := '  delete from recordable_rights_map rrm_base where right_id = ' ||in_right_id || '  and  (schedule_id, right_id, rm_id)
                                           in ( select schedule_id , right_id, rm_id from  ' || temp_deleted_recordables || ' )

                                        ';
                                   execute immediate vsql;

                                   -- Change schedule recorable status to default for all non-recordables
                                   update schedule set recordable = default where recordable = 1
                                     and not exists (select 1 from recordable_rights_map  rrm where rrm.schedule_id = schedule.schedule_id);
                 end if; --  if (p_deleted_recordables > 0 ) then

                 -- Delete non-matched past recordings recorables
                vsql := ' delete from recordable_rights_map rrmap where right_id ='||in_right_id||' and  not exists (select 1 from '|| temp_recordable_right_map || ' trrmap where trrmap.right_id = rrmap.right_id and trrmap.schedule_id = rrmap.schedule_id and trrmap.rm_id = rrmap.rm_id) ';
                execute immediate vsql;

                if (right_state = const_STATE_INACTIVE) then
                  vsql := ' delete from recordable_rights_map rrmap where right_id ='||in_right_id||' and  not exists (select 1 from '|| temp_recordable_right_map || ' trrmap join right_rm_map rrm1 on (trrmap.right_id= rrm1.right_id and trrmap.rm_id = rrm1.rm_id) join rtc_pool rp on (rp.rm_id = rrm1.rm_id) join recording rec on (trrmap.schedule_id = rec.schedule_id and rec.pool_id = rp.pool_id and rec.status != '||const_recording_DELETED || ' )  where trrmap.right_id = rrmap.right_id and trrmap.schedule_id = rrmap.schedule_id and trrmap.rm_id = rrmap.rm_id) ';
                  execute immediate vsql;
                end if;


            end if;  --   if (p_countrecording = 0 ) then
         end if;

   end rightsedit;

  procedure dynexprclause(vrights_rules varchar2 default 'RIGHTS',vdynexpr out varchar2)is
    cursor cright is
      select case when r.rowcnt = 1 and r.rowcnt != r.cnt then ' AND (('||r.dynclause||')'  else
              case when r.rowcnt = 1 and r.cnt = 1 then ' AND ('||r.dynclause||' )' else
                case when r.rowcnt > 1 and r.rowcnt < r.cnt then ' OR ('||r.dynclause||')' else
                  case when r.rowcnt = r.cnt then ' OR ('||r.dynclause||'))' end
                end
              end
            end as stringconcat
      from
      (select 'RIGHT.RIGHT_ID='||right_id||case when dynexpr is not null then ' AND '||dynexpr end as dynclause ,
        count(*) over () as cnt,
        row_number() over (order by right_id) as rowcnt
        from right
        where state =1 and auto_approve=1
      ) r;

    cursor crule is
      select case when r.rowcnt = 1 and r.rowcnt != r.cnt then '  (('||r.dynclause||')'  else
              case when r.rowcnt = 1 and r.cnt = 1 then '  ('||r.dynclause||' )' else
                case when r.rowcnt > 1 and r.rowcnt < r.cnt then ' OR ('||r.dynclause||')' else
                  case when r.rowcnt = r.cnt then ' OR ('||r.dynclause||'))' end
                end
              end
            end as stringconcat
      from
      (select 'RECORDING_RULES.RULE_ID='||rule_id||case when dynexpr is not null then ' AND '||dynexpr end as dynclause ,
        count(*) over () as cnt,
        row_number() over (order by rule_id) as rowcnt
        from recording_rules
        where state =1 and auto_approve=1
      ) r;

  begin
    if vrights_rules = 'RIGHTS' then
      for recright in cright loop
        vdynexpr := vdynexpr||recright.stringconcat;
      end loop;
    elsif vrights_rules = 'RULES' then
      for recrule in crule loop
        vdynexpr := vdynexpr||recrule.stringconcat;
      end loop;
    end if;
    --dbms_output.put_line(vdynexpr);
  end;

  procedure rulesprocess(vrights_rules varchar2 default 'RULES', vfilename varchar2 default '') is
    vcntrecordinglog number;
    vsql varchar2(32767);
    vepg_download_msg varchar2(1000);
    vevent_name varchar2(1000);
    vrights_rules_msg varchar2(50);
    vdynamicexpr varchar2(32000) := null;
    vdynamicexproutput varchar2(32000) := null;
    asset_generation_type varchar2(200) := null;
    rm_name_p_v  backend_application_resources.value%type := getpropertyvalue('rm_name');
    id_p_v  backend_application_resources.value%type := getpropertyvalue('id');
    schedule_start_time_p_v  backend_application_resources.value%type := getpropertyvalue('schedule_start_time');
    program_title_p_v  backend_application_resources.value%type := getpropertyvalue('program_title');
    station_call_sign_p_v  backend_application_resources.value%type := getpropertyvalue('station_call_sign');
    rcs_p_v backend_application_resources.value%type := getpropertyvalue('rcs');
    system_p_v backend_application_resources.value%type := getpropertyvalue('system');
    rights_offset_err_p_v backend_application_resources.value%type := getpropertyvalue('rights_offset_error');
    rights_offset_war_p_v backend_application_resources.value%type := getpropertyvalue('rights_offset_warning');
    right_title_p_v backend_application_resources.value%type := getpropertyvalue('right_title');
    approve_aa_rights_p_v backend_application_resources.value%type := getpropertyvalue('approve_succeeded_aa_rights');
    approve_epg_rights_p_v backend_application_resources.value%type := getpropertyvalue('approve_succeeded_epg_rights');
    approve_epg_rules_p_v backend_application_resources.value%type := getpropertyvalue('approve_succeeded_epg_rules');
    approve_aa_rules_p_v backend_application_resources.value%type := getpropertyvalue('approve_succeeded_aa_rules');

  begin

     init();

    if (vfilename is not null) then
      --To serialize Recordings creation, acquire exclusive lock
      RESOURCELOCK('RECORDINGS_RESOURCE_LOCK',const_EPGRESOURCETIMEOUT);
      vepg_download_msg := replace(getpropertyvalue('epg_file'), '{0}',vfilename);
      vevent_name := 'EPG_DOWNLOAD';
      if vrights_rules = 'RULES' then
       		vrights_rules_msg := getpropertyvalue('auto_approve_rules');
      	elsif vrights_rules = 'RIGHTS' then
      		vrights_rules_msg := getpropertyvalue('auto_approve_rights');
      end if;
    elsif vrights_rules = 'RULES' then
      --To serialize Recordings creation, acquire exclusive lock
      RESOURCELOCK('RECORDINGS_RESOURCE_LOCK');
      vrights_rules_msg := getpropertyvalue('auto_approve_rules');
      vepg_download_msg := getpropertyvalue('empty');
      vevent_name := 'RECORDINGS_RULES';
    elsif vrights_rules = 'RIGHTS' then
      --To serialize Recordings creation, acquire exclusive lock
      RESOURCELOCK('RECORDINGS_RESOURCE_LOCK');
      vrights_rules_msg := getpropertyvalue('auto_approve_rights');
      vepg_download_msg := getpropertyvalue('empty');
      vevent_name := 'RIGHTS';
    end if;


    if vrights_rules = 'RULES' then
        epgguidedatalogging('Start of '||vrights_rules_msg||' processing. StartTime = '||current_timestamp);
        alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(replace(getpropertyvalue('auto_approve_rules_rights_process_started'), '{0}',vrights_rules_msg), '{1}', vepg_download_msg), '{2}',current_timestamp), null, vevent_name);

        vsql := ' delete from temp_rec_rules ';
        CREATEOBJECT(vsql,'Delete Table temp_rec_rules');


        --Initialize, Just in Case!
        vdynamicexproutput := null;
        dynexprclause('RULES',vdynamicexpr);

        vsql := 'insert into temp_rec_rules (RECORDING_ID,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REPEAT,
                    CREATED_BY,
                    UPDATED_BY,
                    CREATE_DATE,
                    UPDATE_DATE,
                    SCHEDULE_ID,
                    HEADEND_ID,
                    RM_ID,
                    POOL_ID,
                    STATION_ID,
                    RIGHT_ID,
                    RULE_ID,
                    RULE_CREATE_DATE,
                    EPISODE_TITLE,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    RL_STATION_OFFSET,
                    ADVISORIES,
                    REQUEST_RECORD,
                    start_offset_adjustment_flag,
                    end_offset_adjustment_flag,
                    POSTER_IMAGE_NAME,
        						POSTER_DESCRIPTION
                    )
        select trr.* from (
            select distinct cast(1 as number) as RECORDING_ID,
            p.title as PROGRAM_TITLE,
            p.description1 as PROGRAM_DESCRIPTION,
            case
            when  sch_rec.offset_start_time is null then sch_rec.offset_start_time
            when  sch_rec.offset_start_time is not null and rrmouter.right_start_offset is null then sch_rec.offset_start_time
            when  sch_rec.offset_start_time is not null and rrmouter.right_start_offset = 0  then rrmouter.right_start_offset
            when  sch_rec.offset_start_time is not null and rrmouter.right_start_offset is not null

                  and ( (rrmouter.right_start_offset < 0  and sch_rec.offset_start_time >=rrmouter.right_start_offset )
                        or
                        (rrmouter.right_start_offset >= 0  and sch_rec.offset_start_time <=rrmouter.right_start_offset and sch_rec.offset_start_time >=0 )
                      ) then sch_rec.offset_start_time
            when  sch_rec.offset_start_time is not null and rrmouter.right_start_offset is not null
                  and ( (rrmouter.right_start_offset < 0  and sch_rec.offset_start_time < rrmouter.right_start_offset )
                        or
                        (rrmouter.right_start_offset >= 0  and (sch_rec.offset_start_time > rrmouter.right_start_offset or  sch_rec.offset_start_time < 0 ))
                      ) then null

            end as OFFSET_START_TIME,
            case
            when  sch_rec.offset_end_time is null then sch_rec.offset_end_time
            when  sch_rec.offset_end_time is not null and rrmouter.right_end_offset is null  then sch_rec.offset_end_time
            when  sch_rec.offset_end_time is not null and rrmouter.right_end_offset = 0  then rrmouter.right_end_offset
            when  sch_rec.offset_end_time is not null and rrmouter.right_end_offset is not null
                  and ( (rrmouter.right_end_offset < 0  and sch_rec.offset_end_time >=rrmouter.right_end_offset and sch_rec.offset_end_time <=0 )
                        or
                        (rrmouter.right_end_offset > 0  and sch_rec.offset_end_time <= rrmouter.right_end_offset )

                      ) then sch_rec.offset_end_time

           when  sch_rec.offset_end_time is not null and rrmouter.right_end_offset is not null
                  and ( (rrmouter.right_end_offset < 0  and (sch_rec.offset_end_time < rrmouter.right_end_offset or sch_rec.offset_end_time > 0 ))
                        or
                        (rrmouter.right_end_offset >= 0  and sch_rec.offset_end_time > rrmouter.right_end_offset )
                      ) then null

            end as OFFSET_end_TIME,

            rrmouter.recording_mode as RECORDING_MODE,
            sch_rec.priority as PRIORITY,
            sch_rec.provider_qa_contact as PROVIDER_QA_CONTACT,
            sch_rec.catalog_entry as CATALOG_ENTRY,
            sch_rec.billing_id as BILLING_ID,
            case when mcmpaa.name is null then sch_rec.mpaa_rating else mcmpaa.name end as MPAA_RATING,
            sch_rec.tv_rating as TV_RATING,
            sch_rec.provider_id as PROVIDER_ID,
            sch_rec.provider_name as PROVIDER_NAME,
            sch_rec.product as PRODUCT,
            sch_rec.category as CATEGORY,
            sch_rec.audio_type as AUDIO_TYPE,
            sch_rec.max_copies as MAX_COPIES,
            sch_rec.preview_period as PREVIEW_PERIOD,
            sch_rec.REDUNDANCY_COUNT as REDUNDANCY_COUNT,
            sch_rec.sstart_time + numtodsinterval(rrmouter.license_relative_start,''SECOND'') + numtodsinterval(nvl(sch_rec.offset_start_time,0),''SECOND'') as LICENCE_START_DATE,
            sch_rec.sstart_time + numtodsinterval(rrmouter.license_relative_start,''SECOND'') + numtodsinterval(nvl(sch_rec.offset_start_time,0),''SECOND'') + numtodsinterval(rrmouter.license_relative_duration,''SECOND'') + numtodsinterval(nvl(sch_rec.offset_end_time,0),''SECOND'') as LICENSE_END_DATE,
            sch_rec.REPEAT as REPEAT,
            ''SYSTEM'' as CREATED_BY,
            ''SYSTEM'' as UPDATED_BY,
            sysdate as CREATE_DATE,
            sysdate as UPDATE_DATE,
            sch_rec.schedule_id as SCHEDULE_ID,
            rm.headend_id as HEADEND_ID,
            rm.rm_id as RM_ID,
            rtc_pool.pool_id as POOL_ID,
            sch_rec.sstation_id as STATION_ID,
            rrmouter.right_id as RIGHT_ID,
            sch_rec.rule_id as RULE_ID,
            sch_rec.RULE_CREATE_DATE as RULE_CREATE_DATE,
            p.EPISODE_TITLE as EPISODE_TITLE,
            sch_rec.sprogram_id as PROGRAM_ID,
            sch_rec.shd_tv as HD_TV,
            sch_rec.sCC as CC,
            sch_rec.sstart_time as SCHEDULE_START_TIME,
            sch_rec.sEND_TIME as SCHEDULE_END_TIME,
            sch_rec.sDURATION as SCHEDULE_DURATION,
            rrmouter.contract_id as contract_id,
            p.year_program as YEAR_PROGRAM,
            rl.offset as RL_STATION_OFFSET,
            rtrim(case when p.advisory_desc1 is not null then p.advisory_desc1||''|'' end ||
            case when p.advisory_desc2 is not null then p.advisory_desc2||''|'' end ||
            case when p.advisory_desc3 is not null then p.advisory_desc3||''|'' end ||
            case when p.advisory_desc4 is not null then p.advisory_desc4||''|'' end ||
            case when p.advisory_desc5 is not null then p.advisory_desc5||''|'' end ||
            case when p.advisory_desc6 is not null then p.advisory_desc6||''|'' end, ''|'') as advisories,
            request_record,
            case
            when  sch_rec.offset_start_time is not null and rrmouter.right_start_offset = 0  then cast(1 as number)

            when  sch_rec.offset_start_time is not null and rrmouter.right_start_offset is not null
                  and ( (rrmouter.right_start_offset < 0  and sch_rec.offset_start_time < rrmouter.right_start_offset )
                        or
                        (rrmouter.right_start_offset >= 0  and (sch_rec.offset_start_time > rrmouter.right_start_offset or  sch_rec.offset_start_time < 0 ))
                      ) then  cast(1 as number)
            end as start_offset_adjustment_flag,
            case
            when  sch_rec.offset_end_time is not null and rrmouter.right_end_offset = 0  then cast(1 as number)
            when  sch_rec.offset_end_time is not null and rrmouter.right_end_offset is not null
                  and ( (rrmouter.right_end_offset < 0  and (sch_rec.offset_end_time < rrmouter.right_end_offset or sch_rec.offset_end_time > 0 ))
                        or
                        (rrmouter.right_end_offset >= 0  and sch_rec.offset_end_time > rrmouter.right_end_offset )
                      ) then  cast(1 as number)
            end as end_offset_adjustment_flag,
            sch_rec.POSTER_IMAGE_NAME,
            sch_rec.POSTER_DESCRIPTION
            from (  select
                    s.program_id as sprogram_id,
                    s.station_id as sstation_id,
                    rr.offset_start_time as OFFSET_START_TIME,
                    rr.offset_end_time as OFFSET_END_TIME,
                    rr.priority as PRIORITY,
                    rr.provider_qa_contact as PROVIDER_QA_CONTACT,
                    rr.catalog_entry as CATALOG_ENTRY,
                    rr.billing_id as BILLING_ID,
                    rr.mpaa_rating as MPAA_RATING,
                    case when mctv.name is null then rr.tv_rating else mctv.name end as TV_RATING,
                    rr.provider_id as PROVIDER_ID,
                    rr.provider_name as PROVIDER_NAME,
                    rr.product as PRODUCT,
                    rr.category as CATEGORY,
                    rr.audio_type as AUDIO_TYPE,
                    rr.max_copies as MAX_COPIES,
                    rr.preview_period as PREVIEW_PERIOD,
                    rr.REDUNDANCY_COUNT as REDUNDANCY_COUNT,
                    rr.POSTER_IMAGE_NAME,
                    rr.POSTER_DESCRIPTION,
                    s.schedule_id as SCHEDULE_ID,
                    s.start_time as sstart_time,
                    s.end_time as send_time,
                    s.duration as sduration,
                    s.repeat as REPEAT,
                    rr.rule_id as RULE_ID,
                    rr.create_date as RULE_CREATE_DATE,
                    upper(s.hd_tv) as shd_tv,
                    upper(s.repeat) as srepeat,
                    case when upper(s.live_tape_delay) = ''DELAY'' then cast(''TAPE'' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,
                    upper(trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY''))) as sdayofweek,
                    upper(nvl(trim(rr.day_of_week),trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY'')))) as rdayofweek,
                    upper(rr.repeat) as rrepeat,
                    upper(rr.hd) as rhd_tv,
                    upper(rr.live) as rlive,
                    nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                    case
                         when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                         when rserm.series_id  is not null then rserm.program_id
                    end as rsermprogram_id,
                    nvl(rstam.station_id,s.station_id) as rstamstation_id,
                    s.cc as sCC,
                    s.request_record as request_record,
                    s.USER_DATA1 as user_data1,
                    s.USER_DATA2 as user_data2,
                    s.USER_DATA3 as user_data3,
                    s.USER_DATA4 as user_data4,
                    s.USER_DATA5 as user_data5,
                    s.USER_DATA6 as user_data6,
                    s.USER_DATA7 as user_data7,
                    s.USER_DATA8 as user_data8,
                    s.USER_DATA9 as user_data9,
                    s.USER_DATA10 as user_data10,
                    s.USER_DATA11 as user_data11,
                    rm.rm_id as rm_id
                    from
                    schedule s
                    join contract c on (c.status = 1 and s.start_time >= current_timestamp and s.start_time >= c.effective_date and s.end_time <= least(c.expiry_date,to_timestamp_tz('''||to_char(vepgrecordingspan,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')) and s.recordable = 1)
                    join recording_rules rr
                      on (((s.start_time >= rr.start_date at time zone to_char(s.start_time,''TZR'') and s.end_time <= rr.end_date at time zone to_char(s.end_time,''TZR'')) OR (rr.start_date is null and rr.end_date is null)) and rr.state = 1 and rr.auto_approve = 1 )
                    join rec_rules_rm_map rrrm on (rrrm.rule_id = rr.rule_id)
                    join rm on (rrrm.rm_id = rm.rm_id)
                    join headend he on (rm.headend_id = he.headend_id)
                    left outer join rec_rules_program_map rpm on (rr.rule_id = rpm.rule_id )
                    left outer join (select rserm_inner.series_id as series_id, rserm_inner.rule_id as rule_id, spm.program_id as program_id
                        from rec_rules_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                        on (rr.rule_id = rserm.rule_id)
                    left outer join rec_rules_station_map rstam on (rr.rule_id = rstam.rule_id)
                    left outer join metadata_config mctv on (mctv.type=''TVRating'' and mctv.name=upper(s.tv_rating))
                     where
                           NVL(rr.start_time,to_dsinterval(''0 ''||TO_CHAR (s.start_time at TIME zone TO_CHAR(he.time_zone), ''hh24:mi:ss''))) <= to_dsinterval(''0 ''||TO_CHAR (s.start_time at TIME zone TO_CHAR(he.time_zone), ''hh24:mi:ss''))
                              AND NVL(rr.end_time + numtodsinterval(TO_CHAR(s.start_time at TIME zone TO_CHAR(he.time_zone),''J''),''DAY''),
                              to_dsinterval(TO_CHAR (s.end_time at TIME zone TO_CHAR(he.time_zone), ''J hh24:mi:ss'')))
                               >= to_dsinterval(TO_CHAR (s.end_time at TIME zone TO_CHAR(he.time_zone), ''J hh24:mi:ss''))

            ) sch_rec
            join program p
              on ( sch_rec.sprogram_id = p.program_id)
            join (select row_number() over (partition by rrm.schedule_id, rrm.rm_id order by r.create_date ) as rownumber,
                rrm.schedule_id, rrm.rm_id, r.right_id, r.recording_mode, r.license_relative_start,r.license_relative_duration ,
                r.contract_id,
                r.offset_start_time as right_start_offset,
                r.offset_end_time as right_end_offset
                from recordable_rights_map rrm , right r where r.right_id = rrm.right_id and r.state=1) rrmouter
              on (rrmouter.schedule_id = sch_rec.schedule_id and rrmouter.rm_id = sch_rec.rm_id
                  and rrmouter.rownumber = 1)
            join right_rm_map rrm on (rrm.right_id = rrmouter.right_id and rrm.rm_id = rrmouter.rm_id )
            join rm on (rm.rm_id = rrm.rm_id)
            join rec_rules_rm_map rrrm on (rrm.rm_id = rrrm.rm_id and sch_rec.rule_id = rrrm.rule_id)
            join rtc_pool on (rm.rm_id = rtc_pool.rm_id)
            join rm_lineup rl on (rl.rm_id = rm.rm_id and rl.station_id = sch_rec.sstation_id and rl.headend_id = rm.headend_id)
            left outer join metadata_config mcmpaa on (mcmpaa.type=''MPAARating'' and mcmpaa.name=upper(p.mpaa_rating))
            where
            sch_rec.sstation_id = sch_rec.rstamstation_id
            and sch_rec.sprogram_id = sch_rec.rpmprogram_id
            and sch_rec.sprogram_id = sch_rec.rsermprogram_id
            and  sch_rec.rdayofweek like ''%''||sch_rec.sdayofweek||''%''
            and (sch_rec.srepeat = sch_rec.rrepeat or sch_rec.rrepeat is null)
            and (sch_rec.shd_tv = sch_rec.rhd_tv or sch_rec.rhd_tv is null)
            and (sch_rec.slive = sch_rec.rlive or sch_rec.rlive is null)
            and sch_rec.sstart_time + numtodsinterval(nvl(sch_rec.OFFSET_START_TIME,0), ''SECOND'') + numtodsinterval(nvl(rl.offset,0), ''SECOND'') >= current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0), ''SECOND'')
            and not exists (select 1 from recording r
                            where
                            r.schedule_id = sch_rec.schedule_id and r.status in (2,3,4,5,6)
                            and rtc_pool.pool_id = r.pool_id)
        ) trr

        join recording_rules on (recording_rules.rule_id = trr.rule_id)
        join schedule on (trr.schedule_id = schedule.schedule_id)
        join program on (trr.program_id = program.program_id)
        join station on (trr.station_id = station.station_id)
        join headend on (trr.headend_id = headend.headend_id)
        join rm_lineup on (trr.rm_id = rm_lineup.rm_id and  rm_lineup.headend_id = trr.headend_id and rm_lineup.station_id = trr.station_id)
        join lineup on  ( lineup.headend_id = rm_lineup.headend_id and lineup.station_id = rm_lineup.station_id and rm_lineup.lineup_id = lineup.lineup_id )
        left join (select ser.series_id, ser.title, spm.program_id as program_id from series ser join series_program_map spm on (ser.series_id = spm.series_id))  series on  (trr.program_id = series.program_id) ';

        if length(ltrim(rtrim(vdynamicexpr))) > 0 then
          vsql := vsql || ' where '||vdynamicexpr;
        end if;


        CREATEOBJECT(vsql,'Insert into Table temp_rec_rules');

        vsql := 'delete from temp_rec_rules trr
                      where rowid in (select row_identifier
                                          from (select rowid as row_identifier,dense_rank() over (partition by schedule_id,pool_id order by rule_create_date) as rownumber
                                                from temp_rec_rules)
                                          where rownumber > 1)';
        CREATEOBJECT(vsql,'Delete temp_rec_rules with multiple recording_rules for recordables');

    elsif vrights_rules = 'RIGHTS' then

        epgguidedatalogging('Start of '||vrights_rules_msg||' processing. StartTime = '||current_timestamp);
        alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(replace(getpropertyvalue('auto_approve_rules_rights_process_started'), '{0}',vrights_rules_msg), '{1}', vepg_download_msg), '{2}',current_timestamp), null, vevent_name);

        vsql := 'delete from temp_rec_rules';
        CREATEOBJECT(vsql,'Delete Table temp_rec_rules');

        --Initialize, Just in Case!
        vdynamicexproutput := null;
        dynexprclause('RIGHTS',vdynamicexproutput);

        --dbms_output.put_line('ROUTPUT '||vdynamicexproutput);

        select replace(replace(vdynamicexproutput,'RIGHT.','sch_rec.'),'SCHEDULE.','sch_rec.')
        into vdynamicexpr
        from dual;

        --dbms_output.put_line('RFINAL '||vdynamicexpr);

        vsql := 'insert into temp_rec_rules(RECORDING_ID,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    SCHEDULE_ID,
                    HEADEND_ID,
                    RM_ID,
                    POOL_ID,
                    STATION_ID,
                    RIGHT_ID,
                    RIGHT_CREATE_DATE,
                    EPISODE_TITLE,
                    REPEAT,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    CREATED_BY,
                    UPDATED_BY,
                    CREATE_DATE,
                    UPDATE_DATE,
                    RL_STATION_OFFSET,
                    ADVISORIES,
                    REQUEST_RECORD,
                    RIGHT_ID_USED_FOR_APPROVAL

                    )
        select distinct cast(1 as number) as RECORDING_ID,
        p.title as PROGRAM_TITLE,
        p.description1 as PROGRAM_DESCRIPTION,
        rrmouter.offset_start_time as OFFSET_START_TIME,
        rrmouter.offset_end_time as OFFSET_END_TIME,
        rrmouter.recording_mode as RECORDING_MODE,
        rrmouter.priority as PRIORITY,
        rrmouter.provider_qa_contact as PROVIDER_QA_CONTACT,
        rrmouter.catalog_entry as CATALOG_ENTRY,
        rrmouter.billing_id as BILLING_ID,
    	case when mcmr.name is null then rrmouter.mpaa_rating else mcmr.name end as MPAA_RATING,
        case when mctv.name is null then rrmouter.tv_rating else mctv.name end as TV_RATING,
        rrmouter.provider_id as PROVIDER_ID,
        rrmouter.provider_name as PROVIDER_NAME,
        rrmouter.product as PRODUCT,
        rrmouter.category as CATEGORY,
        rrmouter.audio_type as AUDIO_TYPE,
        rrmouter.max_copies as MAX_COPIES,
        rrmouter.preview_period as PREVIEW_PERIOD,
        cast(1 as number) as REDUNDANCY_COUNT,
        sch_rec.sstart_time + numtodsinterval(rrmouter.license_relative_start,''SECOND'') + numtodsinterval(nvl(rrmouter.offset_start_time,0),''SECOND'') as LICENCE_START_DATE,
        sch_rec.sstart_time + numtodsinterval(rrmouter.license_relative_start,''SECOND'')  + numtodsinterval(nvl(rrmouter.offset_start_time,0),''SECOND'')+ numtodsinterval(rrmouter.license_relative_duration,''SECOND'') + numtodsinterval(nvl(rrmouter.offset_end_time,0),''SECOND'') as LICENSE_END_DATE,
        sch_rec.schedule_id as SCHEDULE_ID,
        rm.headend_id as HEADEND_ID,
        rm.rm_id as RM_ID,
        rtc_pool.pool_id as POOL_ID,
        sch_rec.sstation_id as STATION_ID,
        rrmouter.right_id as RIGHT_ID,
        rrmouter.right_create_date,
        p.EPISODE_TITLE as EPISODE_TITLE,
        sch_rec.srepeat as REPEAT,
        sch_rec.sprogram_id as PROGRAM_ID,
        sch_rec.shd_tv as HD_TV,
        sch_rec.sCC as CC,
        sch_rec.sstart_time as SCHEDULE_START_TIME,
        sch_rec.sEND_TIME as SCHEDULE_END_TIME,
        sch_rec.sDURATION as SCHEDULE_DURATION,
        rrmouter.contract_id as contract_id,
        p.year_program as YEAR_PROGRAM,
        ''SYSTEM'' as CREATED_BY,
        ''SYSTEM'' as UPDATED_BY,
        sysdate as CREATE_DATE,
        sysdate as UPDATE_DATE,
        rl.offset as RL_STATION_OFFSET,
        rtrim(case when p.advisory_desc1 is not null then p.advisory_desc1||''|'' end ||
        case when p.advisory_desc2 is not null then p.advisory_desc2||''|'' end ||
        case when p.advisory_desc3 is not null then p.advisory_desc3||''|'' end ||
        case when p.advisory_desc4 is not null then p.advisory_desc4||''|'' end ||
        case when p.advisory_desc5 is not null then p.advisory_desc5||''|'' end ||
        case when p.advisory_desc6 is not null then p.advisory_desc6||''|'' end, ''|'') as advisories,
        request_record,
        sch_rec.right_id as right_id_used_for_approval
        from (  select
                r.right_id as right_id,
                s.program_id as sprogram_id,
                s.station_id as sstation_id,
                s.schedule_id as SCHEDULE_ID,
                s.start_time as sstart_time,
                s.end_time as sEND_TIME,
                s.duration as sDURATION,
                upper(s.hd_tv) as shd_tv,
                upper(s.repeat) as srepeat,
                case when upper(s.live_tape_delay) = ''DELAY'' then cast(''TAPE'' as nvarchar2(100)) else upper(s.live_tape_delay) end as slive,
                upper(trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY''))) as sdayofweek,
                upper(nvl(trim(r.day_of_week),trim(to_char(s.start_time at time zone to_char(he.time_zone),''DAY'')))) as rdayofweek,
                upper(r.repeat) as rrepeat,
                upper(r.hd) as rhd_tv,
                upper(r.live) as rlive,
                nvl(rpm.program_id,s.program_id) as rpmprogram_id,
                case
                    when rserm.series_id  is null then nvl(rserm.program_id,s.program_id)
                    when rserm.series_id  is not null then rserm.program_id
                end as rsermprogram_id,
                nvl(rstam.station_id,s.station_id) as rstamstation_id,
                s.cc as sCC,
                s.tv_rating stv_rating,
                s.request_record as request_record,
                s.USER_DATA1 as user_data1,
                s.USER_DATA2 as user_data2,
                s.USER_DATA3 as user_data3,
                s.USER_DATA4 as user_data4,
                s.USER_DATA5 as user_data5,
                s.USER_DATA6 as user_data6,
                s.USER_DATA7 as user_data7,
                s.USER_DATA8 as user_data8,
                s.USER_DATA9 as user_data9,
                s.USER_DATA10 as user_data10,
                s.USER_DATA11 as user_data11,
                rm.rm_id as rm_id
                from
                schedule s
                join contract c on (c.status = 1 and s.start_time >= current_timestamp and s.start_time >= c.effective_date and s.end_time <= least(c.expiry_date,to_timestamp_tz('''||to_char(vepgrecordingspan,'DD-MON-YYYY HH24:MI:SS TZR')||''',''DD-MON-YYYY HH24:MI:SS TZR'')) and s.recordable = 1)
                join right r
                  on (((s.start_time >= r.start_date  at time zone to_char(s.start_time,''TZR'')  and s.end_time <= r.end_date  at time zone to_char(s.end_time,''TZR'') ) OR (r.start_date is null and r.end_date is null)) and r.state = 1 and r.auto_approve = 1 )
                join right_rm_map rrm1 on (r.right_id = rrm1.right_id)
                join rm on (rm.rm_id = rrm1.rm_id)
                join headend he on (rm.headend_id = he.headend_id)
                join recordable_rights_map rrm on (r.right_id = rrm.right_id and rrm.schedule_id = s.schedule_id and rrm.rm_id = rrm1.rm_id)
                left outer join right_program_map rpm on (r.right_id= rpm.right_id)
                left outer join (select rserm_inner.series_id as series_id, rserm_inner.right_id as right_id, spm.program_id as program_id
                    from right_series_map rserm_inner left outer join series_program_map spm on (spm.series_id = rserm_inner.series_id)) rserm
                    on (r.right_id = rserm.right_id)
                left outer join right_station_map rstam on (r.right_id = rstam.right_id)
                where
                        NVL(r.start_time,to_dsinterval(''0 ''||TO_CHAR (s.start_time at TIME zone TO_CHAR(he.time_zone), ''hh24:mi:ss''))) <= to_dsinterval(''0 ''||TO_CHAR (s.start_time at TIME zone TO_CHAR(he.time_zone), ''hh24:mi:ss''))
                          AND NVL(r.end_time + numtodsinterval(TO_CHAR(s.start_time at TIME zone TO_CHAR(he.time_zone),''J''),''DAY''),
                          to_dsinterval(TO_CHAR (s.end_time at TIME zone TO_CHAR(he.time_zone), ''J hh24:mi:ss'')))
                           >= to_dsinterval(TO_CHAR (s.end_time at TIME zone TO_CHAR(he.time_zone), ''J hh24:mi:ss''))

        ) sch_rec
        join program p
          on ( sch_rec.sprogram_id = p.program_id)
         join (select row_number() over (partition by rrm.schedule_id, rrm.rm_id order by r.create_date ) as rownumber,
            rrm.schedule_id,rrm.rm_id, r.right_id,
                r.offset_start_time as OFFSET_START_TIME,
                r.offset_end_time as OFFSET_END_TIME,
                r.priority as PRIORITY,
                r.provider_qa_contact as PROVIDER_QA_CONTACT,
                r.catalog_entry as CATALOG_ENTRY,
                r.billing_id as BILLING_ID,
                r.mpaa_rating as MPAA_RATING,
                r.tv_rating as TV_RATING,
                r.provider_id as PROVIDER_ID,
                r.provider_name as PROVIDER_NAME,
                r.product as PRODUCT,
                r.category as CATEGORY,
                r.audio_type as AUDIO_TYPE,
                r.max_copies as MAX_COPIES,
                r.preview_period as PREVIEW_PERIOD,
                r.create_date as right_create_date,
                r.recording_mode as RECORDING_MODE,
                r.license_relative_start as LICENSE_RELATIVE_START,
                r.license_relative_duration as LICENSE_RELATIVE_DURATION,
                r.contract_id as CONTRACT_ID

            from recordable_rights_map rrm , right r where r.right_id = rrm.right_id and r.state=1) rrmouter
          on (rrmouter.schedule_id = sch_rec.schedule_id  and rrmouter.rm_id = sch_rec.rm_id
              and rrmouter.rownumber = 1)
        join right_rm_map rrm on (rrm.right_id = rrmouter.right_id and rrm.rm_id = rrmouter.rm_id)
        join rm on (rm.rm_id = rrm.rm_id)
        join rtc_pool on (rm.rm_id = rtc_pool.rm_id)
        join rm_lineup rl on (rl.rm_id = rm.rm_id and rl.station_id = sch_rec.sstation_id and rl.headend_id = rm.headend_id)
        left outer join metadata_config mctv on (mctv.type=''TVRating'' and mctv.name=upper(sch_rec.stv_rating))
        left outer join metadata_config mcmr on (mcmr.type=''MPAARating'' and mcmr.name=upper(p.mpaa_rating))
        where
        sch_rec.sstation_id = sch_rec.rstamstation_id
        and sch_rec.sprogram_id = sch_rec.rpmprogram_id
        and sch_rec.sprogram_id = sch_rec.rsermprogram_id
        and  sch_rec.rdayofweek like ''%''||sch_rec.sdayofweek||''%''
        and (sch_rec.srepeat = sch_rec.rrepeat or sch_rec.rrepeat is null)
        and (sch_rec.shd_tv = sch_rec.rhd_tv or sch_rec.rhd_tv is null)
        and (sch_rec.slive = sch_rec.rlive or sch_rec.rlive is null)
        and sch_rec.sstart_time + numtodsinterval(nvl(rrmouter.OFFSET_START_TIME,0), ''SECOND'') + numtodsinterval(nvl(rl.offset,0), ''SECOND'') >= current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0), ''SECOND'')
        and not exists (select 1 from recording r
                      where
                      r.schedule_id = sch_rec.schedule_id and r.status in (2,3,4,5,6)
                      and rtc_pool.pool_id = r.pool_id)'
        ||
        vdynamicexpr;
        --dbms_output.put_line('SQL '||vsql);
        CREATEOBJECT(vsql,'Insert into Table temp_rec_rules');

        vsql := 'delete from temp_rec_rules trr
                      where rowid in (select rowidentifier
                                          from (select rowid as rowidentifier,dense_rank() over (partition by schedule_id,pool_id order by right_create_date, right_id_used_for_approval ) as rownumber
                                                from temp_rec_rules)
                                          where rownumber > 1)';
        CREATEOBJECT(vsql,'Delete temp_rec_rules with multiple rights for recordables');
    end if;



    -- Raise Alarms for the schedules those cannot create any recordings because of the offsets
    vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||rcs_p_v||''',

				    '''||rights_offset_err_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign||'','||right_title_p_v||'= ''||tr.right_title,

                '''||vevent_name||''' , 14,'''||system_p_v||''','''||system_p_v||'''
                      from (
                              select distinct rm.rm_id, rm.name as rm_name, station.call_sign, SCHEDULE_START_TIME at time zone to_char (current_timestamp, ''TZR'') as SCHEDULE_START_TIME, PROGRAM_TITLE, right.name as right_title
                              from
                             temp_rec_rules trr join right on (trr.right_id =  right.right_id)
                             join rm on (rm.rm_id = trr.rm_id)
                            join contract on (right.contract_id  = contract.contract_id)
                            join station on (station.station_id = trr.station_id)
                            where

                              (
                                   ( nvl(right.start_date,contract.effective_date) >= trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')
                                      and
                                    nvl(right.start_date,contract.effective_date) >= trr.schedule_end_time + numtodsinterval(nvl(trr.offset_end_time,0),''SECOND'')
                                   )
                                   OR
                                   (
                                      nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')
                                        and
                                      nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')
                                   )
                                   OR
                                   (
                                    trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') >= trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')
                                   )
                                )

                  ) tr';

    vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
    CREATEOBJECT(vsql,'Offset ALARMS');

      vsql := '
                delete from temp_rec_rules trr where  (trr.schedule_id,trr.pool_id) in
                  (
                  select trr.schedule_id, trr.pool_id

                    from
                  temp_rec_rules trr join right on (trr.right_id =  right.right_id)
                  join contract on (right.contract_id  = contract.contract_id)
                  where

                    (
                         ( nvl(right.start_date,contract.effective_date) >= trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')
                            and
                          nvl(right.start_date,contract.effective_date) >= trr.schedule_end_time + numtodsinterval(nvl(trr.offset_end_time,0),''SECOND'')
                         )
                         OR
                         (
                            nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')
                              and
                            nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')
                         )
                          OR
                         (
                          trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') >= trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')
                         )
                      )
                )
            ';

      CREATEOBJECT(vsql,'deleting the recordings those offsets are beyond the Rights');


            vsql := '
               merge into temp_rec_rules trr using
                ( select trr.schedule_id, trr.pool_id,
                  case
                    when nvl(right.start_date,contract.effective_date) > trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') then
                        -1 * (
                        to_number(extract(second from (trr.schedule_start_time - nvl(right.start_date,contract.effective_date) ))) +
                        to_number(extract(minute from (trr.schedule_start_time - nvl(right.start_date,contract.effective_date) ))) * 60 +
                        to_number(extract(hour from (trr.schedule_start_time - nvl(right.start_date,contract.effective_date) )))   * 60 * 60 +
                        to_number(extract(day from (trr.schedule_start_time - nvl(right.start_date,contract.effective_date) )))  * 60 * 60* 24)
                   else   trr.offset_start_time
                   end as new_offset_start_time,
                   case
                    when nvl(right.start_date,contract.effective_date) > trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') then
                        cast(1 as number)
                    else cast(0 as number)
                    end as new_start_offset_adj_flag,

                   case
                    when nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'') then

                        to_number(extract(second from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - trr.schedule_end_time  ))) +
                        to_number(extract(minute from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - trr.schedule_end_time  ))) * 60 +
                        to_number(extract(hour from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - trr.schedule_end_time  )))   * 60 * 60 +
                        to_number(extract(day from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - trr.schedule_end_time  )))  * 60 * 60* 24
                  else trr.offset_END_time
                  end as new_offset_end_time,
                  case
                    when nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'') then
                      cast(1 as number)
                      else cast(0 as number)
                  end as new_end_offset_adj_flag

                from
              temp_rec_rules trr join right on (trr.right_id =  right.right_id)
              join contract on (right.contract_id  = contract.contract_id)
              where
                  (
                    nvl(right.start_date,contract.effective_date) > trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')
                      OR
                    nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < trr.schedule_end_time + numtodsinterval(nvl(trr.offset_end_time,0),''SECOND'')
                   )
                ) temp  on (temp.schedule_id = trr.schedule_id and temp.pool_id = trr.pool_id)

                when matched then
                update set trr.offset_start_time = temp.new_offset_start_time , trr.offset_end_time = temp.new_offset_end_time, trr.start_offset_adjustment_flag = temp.new_start_offset_adj_flag,
                trr.end_offset_adjustment_flag = temp.new_end_offset_adj_flag
          ';

        CREATEOBJECT(vsql,'Updating the recordings those offsets are beyond the Rights');


    vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_WARNING||','''||rcs_p_v||''',
				'''||rights_offset_war_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign||'','||right_title_p_v||'= ''||tr.right_title,
                 '''||vevent_name||''' , null,'''||system_p_v||''','''||system_p_v||'''
                      from (
                              select distinct rm.rm_id, rm.name as rm_name, station.call_sign, SCHEDULE_START_TIME  at time zone to_char (current_timestamp, ''TZR'') as SCHEDULE_START_TIME ,PROGRAM_TITLE, right.name as right_title
                              from
                             temp_rec_rules trr join right on (trr.right_id =  right.right_id)
                            join rm on (rm.rm_id = trr.rm_id)
                            join contract on (right.contract_id  = contract.contract_id)
                            join station on (station.station_id = trr.station_id)
                            where
                              trr.start_offset_adjustment_flag=1 or trr.end_offset_adjustment_flag=1
      ) tr';

    vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
    CREATEOBJECT(vsql,'Offset Warnings');


--        --Create    Alerts for Recordings with Offset Adjusted Start Time , Offset Adjusted End Time are beyond Right/Contract limitations
--    /*Error Code 14 = Recording Offset Error*/
--    vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||',''RCS'',
--                ''Rights Offset error! Recording adjusted start/end time is beyond the Right/Contract limits and cannot create recording for schedule start time = ''||tr.schedule_start_time||''. Program Title = ''||tr.PROGRAM_TITLE||''. Station Id= ''||tr.station_id,
--                '''||vevent_name||''' , 14,''SYSTEM'',''SYSTEM''
--                from (select distinct SCHEDULE_ID,STATION_ID,PROGRAM_ID, SCHEDULE_START_TIME,PROGRAM_TITLE
--                from
--               temp_rec_rules trr join right on (trr.right_id =  right.right_id)
--              join contract on (right.contract_id  = contract.contract_id)
--
--            where  (
--                    (trr.offset_start_time + trr.rl_station_offset < 0)
--                     or
--                    (trr.offset_end_time + trr.rl_station_offset > 0)
--                  )
--                  and  (nvl(right.start_date,contract.effective_date) > trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                        OR
--                        nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                       )
--
--                       and not exists (select 1 from recordable_rights_map rrm join right right1 on (rrm.right_id = right1.right_id)
--                                                join contract contract1 on (contract1.contract_id = right1.contract_id)
--                                         where rrm.schedule_id  = trr.schedule_id and rrm.rm_id = trr.rm_id and trr.right_id != rrm.right_id
--                                          and
--                                         ( nvl(right1.start_date,contract1.effective_date) <= trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                                          and
--                                          nvl(RIGHT1.END_DATE,CONTRACT1.EXPIRY_DATE) >= trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                                        )
--
--                                    )
--                          ) tr';
--
--    vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
--    CREATEOBJECT(vsql,'SCHEDULE ALARMS', 'Y');
--
--  vsql  := ' delete from temp_rec_rules trr where  (trr.schedule_id,trr.pool_id) in
--
--            (
--              select trr.schedule_id, trr.pool_id
--              from
--               temp_rec_rules trr join right on (trr.right_id =  right.right_id)
--              join contract on (right.contract_id  = contract.contract_id)
--
--            where  (
--                    (trr.offset_start_time + trr.rl_station_offset < 0)
--                     or
--                    (trr.offset_end_time + trr.rl_station_offset > 0)
--                  )
--                  and  (nvl(right.start_date,contract.effective_date) > trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                        OR
--                        nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                       )
--
--                  and not exists
--                    (select 1 from recordable_rights_map rrm join right right1 on (rrm.right_id = right1.right_id)
--                                join contract contract1 on (contract1.contract_id = right1.contract_id)
--                         where rrm.schedule_id  = trr.schedule_id and rrm.rm_id = trr.rm_id and trr.right_id != rrm.right_id
--                          and
--                         ( nvl(right1.start_date,contract1.effective_date) <= trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                          and
--                          nvl(RIGHT1.END_DATE,CONTRACT1.EXPIRY_DATE) >= trr.schedule_END_time + numtodsinterval(nvl(trr.offset_END_time,0),''SECOND'')  + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'')
--                        )
--
--                    )
--            ) ';
--
--    CREATEOBJECT(vsql,'Deleting the recordings that adjusted start/end time is beyond the Right/Contract limitations', 'Y');
/*
    vsql := 'update temp_rec_rules trr
    set recording_id = (select r.recording_id from recording r where r.schedule_id = trr.schedule_id and r.headend_id = trr.headend_id)';
    CREATEOBJECT(vsql,'Assign Recording Sequence for New RM addition to Rights');

    vsql := 'update temp_rec_rules rr
    set recording_id = recording_seq.nextval
    where rowid  ';
    CREATEOBJECT(vsql,'Assign Recording Sequence');
*/
/*
    vsql := 'create index trr_RECORDING_ID on temp_rec_rules(recording_id)';
    CREATEOBJECT(vsql,'Creating index on temp_rec_rules(recording_id)','Y');
*/
    --Delete soft delete recording but preserve recording_log
    vsql := 'delete from temp_recording_log ';
    CREATEOBJECT(vsql,'Delete table temp_recording_log');

    vsql := 'insert into temp_recording_log (RECORDING_LOG_ID,
              RECORDING_ID,
              POOL_ID,
              LOG_MESSAGE,
              SUB_COMPONENT_STATUS,
              SUB_COMPONENT_TYPE,
              SUB_COMPONENT_ID,
              CREATED_BY,
              UPDATED_BY,
              CREATE_DATE,
              UPDATE_DATE,
              OLDRECORDINGID,
              SCHEDULE_ID,
              HEADEND_ID)
              select rl.recording_log_id,trr.recording_id,r.pool_id,rl.log_message,rl.SUB_COMPONENT_STATUS,rl.SUB_COMPONENT_TYPE,rl.SUB_COMPONENT_ID,rl.created_by,rl.updated_by,rl.create_date,rl.update_date,r.recording_id as oldrecordingid,trr.schedule_id as schedule_id,trr.headend_id as headend_id
                from temp_rec_rules trr join recording r on (r.schedule_id = trr.schedule_id and trr.pool_id = r.pool_id and r.status = '||const_recording_DELETED||')
                left outer join recording_log rl on (r.recording_id = rl.recording_id)';
    CREATEOBJECT(vsql,'insert into table temp_recording_log');

    execute immediate 'select count(1) from temp_recording_log' into vcntrecordinglog;

    --DELETE SOFT DELETE RECORDINGS
    if vcntrecordinglog > 0 then

      vsql := 'delete from REC_TRIAL_USE_FIELDS rtuf
      where exists (select 1 from temp_recording_log trl where trl.oldrecordingid = rtuf.recording_id)';
      CREATEOBJECT(vsql,'delete REC_TRIAL_USE_FIELDS');

      vsql := 'delete from RECORDING_REC_RULES_MAP rrrm
      where exists (select 1 from temp_recording_log trl where trl.oldrecordingid = rrrm.recording_id)';
      CREATEOBJECT(vsql,'delete RECORDING_REC_RULES_MAP');

      vsql := 'delete from PLAYBACK_WINDOW_RESTRICTIONS pwr
      where exists (select 1 from temp_recording_log trl where trl.oldrecordingid = pwr.recording_id)';
      CREATEOBJECT(vsql,'delete PLAYBACK_WINDOW_RESTRICTIONS');

      vsql := 'delete from RECORDING_LOG rl
      where exists (select 1 from temp_recording_log trl where trl.oldrecordingid = rl.recording_id)';
      CREATEOBJECT(vsql,'delete RECORDING_LOG');


      vsql := 'delete from recording r
      where exists (select 1 from temp_recording_log trl where trl.oldrecordingid = r.recording_id)';
      CREATEOBJECT(vsql,'delete recording');

    end if;

    select  value into asset_generation_type from config_setting where key ='MOVIE_ASSET_ID_GEN_LOGIC_KEY';


    /* Create Recordings where schedule_start_time + right_offset_start_time < schedule_ends_time + right_offset_end_time */
    vsql := 'insert into recording (
    RECORDING_ID,
    PROGRAM_TITLE,
    PROGRAM_DESCRIPTION,
    OFFSET_START_TIME,
    OFFSET_END_TIME,
    RECORDING_MODE,
    PRIORITY,
    PROVIDER_QA_CONTACT,
    CATALOG_ENTRY,
    BILLING_ID,
    MPAA_RATING,
    TV_RATING,
    PROVIDER_ID,
    PROVIDER_NAME,
    PRODUCT,
    CATEGORY,
    STATUS,
    AUDIO_TYPE,
    MAX_COPIES,
    PREVIEW_PERIOD,
    REDUNDANCY_COUNT,
    LICENCE_START_DATE,
    LICENSE_END_DATE,
    CREATED_BY,
    UPDATED_BY,
    CREATE_DATE,
    UPDATE_DATE,
    SCHEDULE_ID,
    POOL_ID,
    ASSET_ID,
    TITLE_ASSET_ID,
    MOVIE_ASSET_ID,
    POSTER_ASSET_ID,
    POSTER_IMAGE_NAME,
    POSTER_DESCRIPTION,
    STATUS_CODE,
    ADJUSTED_START_TIME,
    ADJUSTED_END_TIME
    )
    SELECT
    recording_seq.nextval,
    rec.PROGRAM_TITLE,
    rec.PROGRAM_DESCRIPTION,
    rec.OFFSET_START_TIME,
    rec.OFFSET_END_TIME,
    rec.RECORDING_MODE,
    rec.PRIORITY,
    rec.PROVIDER_QA_CONTACT,
    rec.CATALOG_ENTRY,
    rec.BILLING_ID,
    rec.MPAA_RATING,
    rec.TV_RATING,
    rec.PROVIDER_ID,
    rec.PROVIDER_NAME,
    rec.PRODUCT,
    rec.CATEGORY,
    '||const_recording_SCHEDULED||',
    rec.AUDIO_TYPE,
    rec.MAX_COPIES,
    rec.PREVIEW_PERIOD,
    rec.REDUNDANCY_COUNT,
    rec.LICENCE_START_DATE,
    rec.LICENSE_END_DATE,
    rec.CREATED_BY,
    rec.UPDATED_BY,
    rec.CREATE_DATE,
    rec.UPDATE_DATE,
    rec.SCHEDULE_ID,
    rec.POOL_ID,
    ''BSPK''||substr(lpad(rec.pool_id, 20, 0), -4)|| substr(lpad(recording_seq.currval, 20, 0), -12) as asset_id,
    ''BSTL''||substr(lpad(rec.pool_id, 20, 0), -4)|| substr(lpad(recording_seq.currval, 20, 0), -12) as title_asset_id,
    case when '''||upper(asset_generation_type)||'''=''SCHEDULE_ID'' then ''BSMV''||substr(lpad(rec.SCHEDULE_ID, 20, 0), -16)
      else
    ''BSMV''||substr(lpad(rec.pool_id, 20, 0), -4)|| substr(lpad(recording_seq.currval, 20, 0), -12)
    end
    as movie_asset_id,
    case when rec.attach_poster is not null and  rec.attach_poster = 1 then ''BSPO''||substr(lpad(rec.pool_id, 20, 0), -4)|| substr(lpad(recording_seq.currval, 20, 0), -12)
    	else
    	null
    end
    as POSTER_ASSET_ID,
    rec.POSTER_IMAGE_NAME,
    rec.POSTER_DESCRIPTION,
    rec.STATUS_CODE,
    rec.ADJUSTED_START_TIME,
    rec.ADJUSTED_END_TIME
    FROM
    (SELECT DISTINCT
      trr.PROGRAM_TITLE,
      trr.PROGRAM_DESCRIPTION,
      trr.OFFSET_START_TIME,
      trr.OFFSET_END_TIME,
      trr.RECORDING_MODE,
      trr.PRIORITY,
      trr.PROVIDER_QA_CONTACT,
      trr.CATALOG_ENTRY,
      trr.BILLING_ID,
      trr.MPAA_RATING,
      trr.TV_RATING,
      trr.PROVIDER_ID,
      trr.PROVIDER_NAME,
      trr.PRODUCT,
      trr.CATEGORY,
      trr.AUDIO_TYPE,
      trr.MAX_COPIES,
      trr.PREVIEW_PERIOD,
      trr.REDUNDANCY_COUNT,
      trr.LICENCE_START_DATE,
      trr.LICENSE_END_DATE,
      trr.CREATED_BY,
      trr.UPDATED_BY,
      trr.CREATE_DATE,
      trr.UPDATE_DATE,
      trr.SCHEDULE_ID,
      trr.pool_id,
      trr.POSTER_IMAGE_NAME,
      trr.POSTER_DESCRIPTION,
      rr.attach_poster as attach_poster,
      '||const_recordingstatuspending||' as status_code,
      trr.schedule_start_time + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'') as adjusted_start_time,
      trr.schedule_end_time + numtodsinterval(nvl(trr.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'') as adjusted_end_time
       FROM TEMP_REC_RULES trr left join recording_rules rr on (trr.rule_id = rr.rule_id)
      WHERE trr.SCHEDULE_START_TIME + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') < trr.schedule_end_time + numtodsinterval(nvl(trr.offset_end_time,0),''SECOND'')  and
        trr.SCHEDULE_START_TIME + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') + numtodsinterval(nvl(trr.rl_station_offset,0),''SECOND'') >= current_timestamp
        and not exists (select 1 from recording r where r.schedule_id = trr.schedule_id and r.pool_id = trr.pool_id)
      ) rec';
    CREATEOBJECT(vsql,'Insert into recording from temp_rec_rules');

    vsql := 'update temp_rec_rules trr
    set trr.recording_id = (select r.recording_id from recording r where r.schedule_id = trr.schedule_id and r.pool_id = trr.pool_id)';
    CREATEOBJECT(vsql,'Assign Recording Sequence for New RM addition to Rights');

    vsql := 'update temp_recording_log trl
    set trl.recording_id = (select r.recording_id from recording r where r.schedule_id = trl.schedule_id and r.pool_id = trl.pool_id)';
    CREATEOBJECT(vsql,'Assign Recording Sequence for soft delete recording logs');

    --Create    Alerts for Recordings with Offset Adjusted Start Time >= Offset Adjusted End Time
    /*Error Code 14 = Recording Offset Error*/
--    vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||',''RCS'',
--                ''RIGHTS OFFSET ERROR! CANNOT CREATE RECORDING FOR SCHEDULE_START_TIME = ''||tr.schedule_start_time||''. PROGRAM TITLE = ''||tr.PROGRAM_TITLE||''. STATION ID= ''||tr.station_id,
--                '''||vevent_name||''',14,''SYSTEM'',''SYSTEM''
--                from (select distinct SCHEDULE_ID,STATION_ID,PROGRAM_ID, SCHEDULE_START_TIME,PROGRAM_TITLE from TEMP_REC_RULES trr WHERE trr.SCHEDULE_START_TIME + numtodsinterval(nvl(trr.offset_start_time,0),''SECOND'') >= trr.schedule_end_time + numtodsinterval(nvl(trr.offset_end_time,0),''SECOND'') ) tr';
--
--    vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
--    CREATEOBJECT(vsql,'SCHEDULE ALARMS', 'Y');

     -- Old logs are not required anymore
--    if vcntrecordinglog > 0 then
--      vsql := 'insert into recording_log(RECORDING_LOG_ID,
--                                      RECORDING_ID,
--                                      LOG_MESSAGE,
--                                      SUB_COMPONENT_STATUS,
--                                      SUB_COMPONENT_TYPE,
--                                      SUB_COMPONENT_ID,
--                                      CREATED_BY,
--                                      UPDATED_BY,
--                                      CREATE_DATE,
--                                      UPDATE_DATE)
--                select trl.recording_log_id,trl.recording_id,trl.log_message,trl.SUB_COMPONENT_STATUS,trl.SUB_COMPONENT_TYPE,trl.SUB_COMPONENT_ID,trl.created_by,trl.updated_by,trl.create_date,trl.update_date
--                from temp_recording_log trl join recording r on (trl.recording_id = r.recording_id)
--                where trl.recording_log_id is not null';
--      CREATEOBJECT(vsql,'Insert into recording from temp_rec_rules');
--    end if;

    vsql := 'insert into playback_window_restrictions (
      RESTRICTION_ID,
      START_TIME,
      DURATION,
      RESTRICTIONS,
      TYPE,
      CREATED_BY,
      UPDATED_BY,
      CREATE_DATE,
      UPDATE_DATE,
      RIGHT_ID,
      RECORDING_ID)
      select playback_window_restr_seq.nextval,
        pwr.start_time,
        pwr.duration,
        pwr.restrictions,
        pwr.type,
        ''SYSTEM'',
        ''SYSTEM'',
        current_timestamp,
        current_timestamp,
        null,
        pwr.recording_id
      from
        (select distinct pwr.start_time,
              pwr.duration,
              pwr.restrictions,
              pwr.type,
              trr.recording_id
        from playback_window_restrictions pwr, temp_rec_rules trr
        where trr.right_id = pwr.right_id
        and trr.recording_id is not null
        and not exists (select 1 from playback_window_restrictions pwrinner where pwrinner.recording_id = trr.recording_id and pwrinner.type = pwr.type)
        ) pwr';
    CREATEOBJECT(vsql,'Insert into playback_window_restr');
   
    vsql := 'insert into recording_Queue(
        RECORDING_ID,
        SCHEDULE_ID,
        POOL_ID,
        RM_ID,
        SCHEDULE_START_TIME,
        SCHEDULE_END_TIME,
        SCHEDULE_DURATION,
        STATION_ID,
        PROGRAM_ID,
        HD_TV,
        CC,
        REPEAT,
        RM_NAME,
        RM_IP_ADDRESS,
        RM_PORT,
        EPISODE_TITLE,
        STATION_OFFSET,
        HEADEND_NAME,
        CONTRACT_ID,
        YEAR_PROGRAM,
        ASSET_ID,
        MOVIE_ASSET_ID,
        TITLE_ASSET_ID,
        POSTER_ASSET_ID,
        POSTER_IMAGE_NAME,
        POSTER_DESCRIPTION,
        PROGRAM_TITLE,
        PROGRAM_DESCRIPTION,
        OFFSET_START_TIME,
        OFFSET_END_TIME,
        RECORDING_MODE,
        PRIORITY,
        PROVIDER_QA_CONTACT,
        CATALOG_ENTRY,
        BILLING_ID,
        MPAA_RATING,
        TV_RATING,
        PROVIDER_ID,
        PROVIDER_NAME,
        PRODUCT,
        CATEGORY,
        AUDIO_TYPE,
        MAX_COPIES,
        PREVIEW_PERIOD,
        REDUNDANCY_COUNT,
        LICENCE_START_DATE,
        LICENSE_END_DATE,
        REQUEST_TYPE,
        VOD_START_TIME,
        VOD_DURATION,
        VOD_RESTRICTIONS,
        RTI_START_TIME,
        RTI_DURATION,
        RTI_RESTRICTIONS,
        QC_START_TIME,
        QC_DURATION,
        QC_RESTRICTIONS,
        ADVISORIES,
        CREATED_BY,
        UPDATED_BY,
        CREATE_DATE,
        UPDATE_DATE)
        select rec_queue.* FROM
        (SELECT
        trr.recording_id,
        trr.schedule_id,
        trr.pool_id,
        trr.rm_id,
        trr.SCHEDULE_START_TIME,
        trr.SCHEDULE_END_TIME,
        trr.SCHEDULE_DURATION,
        trr.station_id,
        trr.program_id,
        trr.hd_tv,
        trr.cc,
        trr.repeat,
        rm.name,
        rm.ip_address,
        rm.port,
        trr.EPISODE_TITLE,
        trr.rl_station_offset,
        h.headend_location,
        trr.contract_id,
        trr.YEAR_PROGRAM,
        rec.asset_id,
        rec.MOVIE_ASSET_ID,
        rec.TITLE_ASSET_ID,
        rec.POSTER_ASSET_ID,
        rec.POSTER_IMAGE_NAME,
        rec.POSTER_DESCRIPTION,
        trr.PROGRAM_TITLE,
        trr.PROGRAM_DESCRIPTION,
        trr.OFFSET_START_TIME,
        trr.OFFSET_END_TIME,
        trr.RECORDING_MODE,
        trr.PRIORITY,
        trr.PROVIDER_QA_CONTACT,
        trr.CATALOG_ENTRY,
        trr.BILLING_ID,
        trr.MPAA_RATING,
        trr.TV_RATING,
        trr.PROVIDER_ID,
        trr.PROVIDER_NAME,
        trr.PRODUCT,
        trr.CATEGORY,
        trr.AUDIO_TYPE,
        trr.MAX_COPIES,
        trr.PREVIEW_PERIOD,
        trr.REDUNDANCY_COUNT,
        trr.LICENCE_START_DATE,
        trr.LICENSE_END_DATE,
        '||const_recording_UPDATE_REQUEST||',
        pwr_vod.start_time as vod_start_time,
        pwr_vod.duration as vod_duration,
        pwr_vod.restrictions as vod_restrictions,
        pwr_rti.start_time as rti_start_time,
        pwr_rti.duration as rti_duration,
        pwr_rti.restrictions as rti_restrictions,
        pwr_qc.start_time as qc_start_time,
        pwr_qc.duration as qc_duration,
        pwr_qc.restrictions as qc_restrictions,
        trr.advisories,
        ''SYSTEM'' as created_by,
        ''SYSTEM'' as updated_by,
        sysdate as create_date,
        sysdate as update_date
        FROM temp_rec_rules trr
        join recording rec on (trr.recording_id = rec.recording_id)
        join rtc_pool rp on (rec.pool_id = rp.pool_id)
        join rm on (rp.rm_id = rm.rm_id)
        join headend h on (h.headend_id = rm.headend_id)
        left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_vod on (pwr_vod.recording_id = trr.recording_id and pwr_vod.type=''VOD'')
        left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_qc on (pwr_qc.recording_id = trr.recording_id and pwr_qc.type=''QC'')
        left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_rti on (pwr_rti.recording_id = trr.recording_id and pwr_rti.type=''RTI'')
        order by rec.adjusted_start_time, trr.PRIORITY) rec_queue';
    CREATEOBJECT(vsql,'Insert into Recording_queue');

    -- new HA change
      insertRecQueue;
      
    if vrights_rules = 'RULES' then
      vsql := 'merge into recording_rec_rules_map rrrm
                using ( select distinct trr.recording_id as recording_id,trr.rule_id as rule_id
                        from temp_rec_rules trr join recording r on (r.recording_id = trr.recording_id)
                       ) recrulesjoin
       on (recrulesjoin.recording_id = rrrm.recording_id and recrulesjoin.rule_id = rrrm.rule_id)
       when not matched then
       insert  (recording_id,rule_id,created_by,updated_by,create_date,update_date)
       values (recrulesjoin.recording_id,recrulesjoin.rule_id,''SYSTEM'',''SYSTEM'',current_timestamp,current_timestamp)';
      CREATEOBJECT(vsql,'Merge Recording_rec_rules_map');
    end if;

   if vrights_rules = 'RIGHTS' then  -- condition for Rights

         -- insert trial use fields for each recording
         insert into rec_trial_use_fields
             ( TRIAL_USE_FIELD_ID,
              ASSET_TYPE,
              NAME,
              VALUE_TYPE,
              VALUE,
              MAPPING_ID,
              RECORDING_ID,
              DESCRIPTION,
              CREATED_BY,
              UPDATED_BY,
              CREATE_DATE,
              UPDATE_DATE
             )
            select REC_TRIAL_USE_FIELD_SEQ.nextval, rtuf.asset_type, rtuf.name, rtuf.value_type, rtuf.value, rtuf.mapping_id, trr.recording_id, rtuf.description, 'SYSTEM', 'SYSTEM', current_timestamp, current_timestamp
            from right_trial_use_fields rtuf join temp_rec_rules trr on (rtuf.right_id = trr.right_id and trr.recording_id is not null);

            -- Auto Approve Rights based recordings alarm activity logs
        vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''',

				'''||approve_aa_rights_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.SCHEDULE_START_TIME||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign,
				'''||vevent_name||''',null,'''||system_p_v||''','''||system_p_v||'''
              from (
                      select distinct SCHEDULE_ID,CALL_SIGN,  schedule_start_time at time zone to_char (current_timestamp, ''TZR'')  as schedule_start_time ,
                      PROGRAM_TITLE , rm.name as rm_name, rm.rm_id from TEMP_REC_RULES trr, station s, right r, rtc_pool rp, rm
                      WHERE
                      trr.station_id = s.station_id
                      and trr.pool_id = rp.pool_id
                      and rp.rm_id = rm.rm_id
                      and r.right_id = trr.right_id_used_for_approval
                      and  r.auto_approve = 1 and r.dynexpr is  null and trr.recording_id is not null
                   ) tr';

        vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
        CREATEOBJECT(vsql,'Auto Approve Rights based recordings alarm activity logs');



            -- EPG Driven Recordings based on the REQUEST_RECORD alarm activity logs
        vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''',
					'''||approve_epg_rights_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.SCHEDULE_START_TIME||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign,

              '''||vevent_name||''',null,'''||system_p_v||''','''||system_p_v||'''
              from (  select distinct SCHEDULE_ID,CALL_SIGN,  schedule_start_time at time zone to_char (current_timestamp, ''TZR'')  as schedule_start_time ,
                      PROGRAM_TITLE , rm.name as rm_name, rm.rm_id  from TEMP_REC_RULES trr, station s, right r, rtc_pool rp, rm
                      WHERE
                      trr.station_id = s.station_id
                      and trr.pool_id = rp.pool_id
                      and rp.rm_id = rm.rm_id
                      and upper(trr.request_record) = ''Y'' and r.right_id = trr.right_id_used_for_approval
                      and  r.auto_approve = 1 and r.dynexpr is not null and trr.recording_id is not null) tr';

        vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
        CREATEOBJECT(vsql,'Flag, REQUEST_RECORD based recording logs');

     elsif vrights_rules = 'RULES' then -- condition for Rules

        -- insert trial use fields for each recording
        insert into rec_trial_use_fields
            ( TRIAL_USE_FIELD_ID,
              ASSET_TYPE,
              NAME,
              VALUE_TYPE,
              VALUE,
              MAPPING_ID,
              RECORDING_ID,
              DESCRIPTION,
              CREATED_BY,
              UPDATED_BY,
              CREATE_DATE,
              UPDATE_DATE
             )
            select REC_TRIAL_USE_FIELD_SEQ.nextval, rtuf.asset_type, rtuf.name, rtuf.value_type, rtuf.value, rtuf.mapping_id, trr.recording_id, rtuf.description, 'SYSTEM', 'SYSTEM', current_timestamp, current_timestamp
            from rule_trial_use_fields rtuf join temp_rec_rules trr on (rtuf.rule_id = trr.rule_id and trr.recording_id is not null);


                      -- Auto Approve Rules based recordings alarm activity logs
        vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''',
              	'''||approve_aa_rules_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.SCHEDULE_START_TIME||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign,
              '''||vevent_name||''',null,'''||system_p_v||''','''||system_p_v||'''
              from (
                    select distinct SCHEDULE_ID,CALL_SIGN,PROGRAM_ID,  schedule_start_time at time zone to_char (current_timestamp, ''TZR'') as schedule_start_time ,
                    PROGRAM_TITLE , rm.name as rm_name, rm.rm_id   from TEMP_REC_RULES trr, station s, recording_rules rr, rtc_pool rp, rm
                     WHERE
                     trr.station_id = s.station_id
                      and trr.pool_id = rp.pool_id
                      and rp.rm_id = rm.rm_id
                     and trr.rule_id = rr.rule_id
                      and  rr.auto_approve = 1 and rr.dynexpr is  null and trr.recording_id is not null
                    ) tr';

        vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
        CREATEOBJECT(vsql,'Auto Approve Rules based recording logs');


                      -- EPG Driven Recordings based on the REQUEST_RECORD alarm activity logs
        vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''',
              	'''||approve_epg_rules_p_v||' '||rm_name_p_v||'=''||tr.rm_name||'' '||id_p_v||'=''||tr.rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.SCHEDULE_START_TIME||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign,
                '''||vevent_name||''',null,'''||system_p_v||''','''||system_p_v||'''
              from (
                    select distinct SCHEDULE_ID,CALL_SIGN,PROGRAM_ID,  schedule_start_time at time zone to_char (current_timestamp, ''TZR'') as schedule_start_time  ,
                    PROGRAM_TITLE , rm.name as rm_name, rm.rm_id  from TEMP_REC_RULES trr, station s, recording_rules rr, rtc_pool rp, rm
                     WHERE trr.station_id = s.station_id
                       and trr.pool_id = rp.pool_id
                      and rp.rm_id = rm.rm_id
                     and upper(trr.request_record) = ''Y'' and trr.rule_id = rr.rule_id
                      and  rr.auto_approve = 1 and rr.dynexpr is not null and trr.recording_id is not null
                    ) tr';

        vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
        CREATEOBJECT(vsql,'Flag, REQUEST_RECORD based recording logs');

     end if;

    epgguidedatalogging('End of '||vrights_rules_msg||' processing. EndTime = '||current_timestamp);
    alarmactivitylogging(const_ALARM_TYPE_LOG,const_ALARM_SEVERITY_INFO,replace(replace(replace(getpropertyvalue('auto_approve_rules_rights_process_ended'), '{0}',vrights_rules_msg), '{1}', vepg_download_msg), '{2}',current_timestamp), null, vevent_name);

  end rulesprocess;

   procedure delete_leftover_files(v_filename varchar2) is

	  type vtabdeletefilename is table of varchar2(4000) INDEX BY PLS_INTEGER;

	  vdeletefilenames vtabdeletefilename;

			begin

			  select any_path

			  bulk collect into vdeletefilenames

			  from resource_view

			  where any_path like '/public/rcs/%'

			  and any_path != '/public/rcs/'||v_filename;



			  if vdeletefilenames.count > 0 then

			     FOR indx IN 1 .. vdeletefilenames.COUNT

			     LOOP

			      dbms_xdb.deleteresource(vdeletefilenames(indx),dbms_xdb.delete_recursive_force);

			     END LOOP;

			  end if;

			end;

  procedure processepgdata(v_filename varchar2) is
  begin
    vfilename := v_filename;
    init();
    delete_leftover_files(v_filename);
    loadtstv(v_filename);
    mergetstv();
    commit;
    dbms_xdb.deleteresource('/public/rcs/'||vfilename,dbms_xdb.delete_recursive_force);
    vfilename := null;
    commit;
  exception
    when others then
      -- Error Code 11 = EPG Download Abnormal Error
      dbms_xdb.deleteresource('/public/rcs/'||vfilename,dbms_xdb.delete_recursive_force);
      commit;
      alarmactivitylogging(const_ALARM_TYPE_ALARM,const_ALARM_SEVERITY_CRITICAL,replace(replace(replace(getpropertyvalue('abnormal_epg_download_failed'), '{0}',vfilename), '{1}', regexp_replace(sqlerrm,'[[:punct:]]',' ')), '{2}',current_timestamp),11);
      vfilename := null;
      raise;

  end processepgdata;


  /* Yet to be created - Murtaza*/
  procedure addrmstoright(rm_comma_separated_values VARCHAR2, in_right_id NUMBER) is
  begin
    null;
  end addrmstoright;

  procedure removermsfromright(rm_comma_separated_values VARCHAR2, in_right_id NUMBER, record_count out number ) is
      contract_id Contract.contract_id%type;
      vsql varchar2(32000);
      temp_table_name varchar2(30) := 'temp_rec_right_map';
      temp_count number := 0;

   begin
      RECORDABLESRECRESOURCELOCK();

    -- check for any existing Recordings for these RMs

      vsql := ' select  count(1) as record_count  from recording rec join rtc_pool rp on (rp.pool_id = rec.pool_id) join rm on (rp.rm_id = rm.rm_id)

                            where  rec.status != '||const_recording_DELETED ||'
                                  and
                                  exists (
                                            select 1 from (select schedule_id, right_id, rm_id,  count(1) over ( partition by schedule_id, rm_id) as
                                            counter from recordable_rights_map )  temp where temp.counter =1 and temp.rm_id in ( '||rm_comma_separated_values||' )
                                            and temp.right_id= ' || in_right_id ||'  and temp.schedule_id = rec.schedule_id and rm.rm_id = temp.rm_id
                                         ) ';

      execute immediate vsql into record_count;

      if ( record_count = 0 ) then
        -- Delete any existing recordables belong to the RMs and Right combination
        vsql := ' delete from recordable_rights_map where right_id = '||in_right_id|| ' and rm_id in ( '||rm_comma_separated_values|| ' ) ';
        CREATEOBJECT(vsql,'deleting the recordables for the removed RMs for the given Right', 'Y');
      end if;
   end removermsfromright;

  procedure removermlineup (in_rm_id number, in_station_id varchar2) is


  begin
         RESOURCELOCK('RECORDABLES_RESOURCE_LOCK');
        -- Delete corresponding recorables of  current and future  deleted recordings when the station is deleted from rm_line
     delete
     from recordable_rights_map map1
     where (map1.right_id, map1.schedule_id, map1.rm_id) in
     (select rrmap.right_id, rrmap.schedule_id , rrmap.rm_id from recordable_rights_map rrmap, schedule sc, recording rec, rtc_pool rtcpool,  rm_lineup rmlineup
     where rec.status =const_recording_DELETED
          and sc.end_time + numtodsinterval(nvl(rec.offset_end_time,0),'SECOND') + numtodsinterval(nvl(rmlineup.offset,0),'SECOND') >= current_timestamp
          and  sc.station_id = in_station_id and rrmap.rm_id= in_rm_id and  rrmap.schedule_id = sc.schedule_id and rec.schedule_id = sc.schedule_id
          and rec.pool_id = rtcpool.pool_id  and rtcpool.rm_id = rrmap.rm_id and rmlineup.rm_id= rrmap.rm_id and rmlineup.station_id = sc.station_id);

      -- Delete corresponding recordables of deleted station from rm_linup
     delete from recordable_rights_map where (right_id, schedule_id, rm_id) in ( select rrmap.right_id, rrmap.schedule_id, rrmap.rm_id from recordable_rights_map rrmap, schedule sc
      where  sc.station_id =in_station_id and rrmap.rm_id=in_rm_id and
     rrmap.schedule_id = sc.schedule_id);

     -- Update the schedule to non-recorable when there are no supporting rights because of station deletion in the rm_linep
     update schedule sc set recordable = DEFAULT where sc.recordable = 1  and not exists (select 1 from recordable_rights_map rrmap where rrmap.schedule_id = sc.schedule_id);



  end removermlineup;

  procedure processrmsdeletionfromright(in_right_id number) is
        contract_id Contract.contract_id%type;
      vsql varchar2(32000);
      temp_table_name varchar2(30) := 'temp_rec_right_map';
      temp_count number := 0;
      rec_del_sub_p_v backend_application_resources.value%type := getpropertyvalue('recording_delete_submission_succeeded');
      rm_name_p_v  backend_application_resources.value%type := getpropertyvalue('rm_name');
      id_p_v  backend_application_resources.value%type := getpropertyvalue('id');
      schedule_start_time_p_v  backend_application_resources.value%type := getpropertyvalue('schedule_start_time');
      program_title_p_v  backend_application_resources.value%type := getpropertyvalue('program_title');
      station_call_sign_p_v  backend_application_resources.value%type := getpropertyvalue('station_call_sign');
      rcs_p_v backend_application_resources.value%type := getpropertyvalue('rcs');
      system_p_v backend_application_resources.value%type := getpropertyvalue('system');

   begin
      RECORDABLESRECRESOURCELOCK();
        -- lock recordinglog table exclusively
      lock table recording_log  in exclusive mode wait 5;

      -- get the contract
      select r.contract_id into contract_id from Right r where r.right_id=in_right_id;


      select count(1) into temp_count from user_tables where upper(table_name)=upper(temp_table_name);

          if (temp_count = 1) then
             vsql := 'drop table   ' ||temp_table_name||'  purge';
            autonomoustransaction(vsql);
          end if;


  -- Creating  a temp table with exclusive recording for this right and rm combination
    vsql := ' create table   ' ||temp_table_name||'    nologging as
            select rec.recording_id,
                  rec.schedule_id,
                  rec.pool_id,
                  rp.rm_id,
                  sch.start_time as SCHEDULE_START_TIME ,
                  sch.end_time as SCHEDULE_END_TIME,
                  sch.duration as SCHEDULE_DURATION ,
                  sch.station_id,
                  sch.program_id,
                  sch.hd_tv,
                  sch.cc,
                  sch.repeat,
                  rm.name as RM_NAME,
                  rm.ip_address as RM_IP_ADDRESS,
                  rm.port as RM_PORT,
                  prg.episode_title,
                  rl.offset as STATION_OFFSET,
                  h.headend_location,
                  cast('||contract_id||' as number) contract_id,
                  prg.year_program,
                  rec.asset_id,
                  rec.title_asset_id,
                  rec.movie_asset_id,
                  rec.POSTER_ASSET_ID,
                  rec.POSTER_IMAGE_NAME,
        		  rec.POSTER_DESCRIPTION,
                  prg.title as PROGRAM_TITLE,
                  rec.program_description,
                  rec.offset_start_time,
                  rec.offset_end_time,
                  rec.recording_mode,
                  rec.priority,
                  cast(0 as number) as state,
                  rec.provider_qa_contact,
                  rec.catalog_entry,
                  rec.billing_id,
           		  rec.mpaa_rating,
                  rec.tv_rating,
                  rec.provider_id,
                  rec.provider_name,
                  rec.product,
                  rec.category,
                  rec.audio_type,
                  rec.max_copies,
                  rec.preview_period,
                  rec.REDUNDANCY_COUNT,
                  rec.licence_start_date,
                  rec.license_end_date,
                  cast('||const_recording_DELETE_REQUEST||' as number) as request_type,
                  pwr_vod.start_time as vod_start_time,
                  pwr_vod.duration as vod_duration,
                  pwr_vod.restrictions as vod_restrictions,
                  pwr_rti.start_time as rti_start_time ,
                  pwr_rti.duration as rti_duration,
                  pwr_rti.restrictions as rti_restrictions,
                  pwr_qc.start_time as qc_start_time ,
                  pwr_qc.duration as qc_duration,
                  pwr_qc.restrictions as qc_restrictions,
                  ''SYSTEM'' as created_by,
                  ''SYSTEM'' as updated_by,
                  current_timestamp as create_date,
                  current_timestamp as update_date,
                   rtrim(case when prg.advisory_desc1 is not null then prg.advisory_desc1||''|'' end ||
                         case when prg.advisory_desc2 is not null then prg.advisory_desc2||''|'' end ||
                         case when prg.advisory_desc3 is not null then prg.advisory_desc3||''|'' end ||
                         case when prg.advisory_desc4 is not null then prg.advisory_desc4||''|'' end ||
                         case when prg.advisory_desc5 is not null then prg.advisory_desc5||''|'' end ||
                         case when prg.advisory_desc6 is not null then prg.advisory_desc6||''|'' end, ''|'') as advisories


                  from

                  recording rec
                  join (select * from schedule sc where sc.schedule_id in (select schedule_id from (select schedule_id, right_id, count(1) over ( partition by schedule_id, rm_id) as counter from recordable_rights_map )  temp where temp.counter =1  and temp.right_id='||in_right_id||' )) sch on  sch.schedule_id = rec.schedule_id
                  join program prg on (sch.program_id = prg.program_id)
                  join rtc_pool rp on (rec.pool_id = rp.pool_id)
                  join rm on (rm.rm_id = rp.rm_id )
                  join rm_lineup rl on (rl.rm_id = rm.rm_id and rl.station_id = sch.station_id and rl.headend_id = rm.headend_id)
                  join headend h on (h.headend_id = rm.headend_id)
                  left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_vod on (pwr_vod.recording_id = rec.recording_id and pwr_vod.type=''VOD'')
                  left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_rti on (pwr_rti.recording_id = rec.recording_id and pwr_rti.type=''RTI'')
                  left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_qc on (pwr_qc.recording_id = rec.recording_id and pwr_qc.type=''QC'')
                   where rm.rm_id in (select rm_id from right_rm_map rrm where rrm.right_id = '||in_right_id||'  )
                   and rec.adjusted_end_time >= current_timestamp
                   and rec.status not in ('||const_recording_DELETED||', '||const_recording_SUCCESS||', '||const_recording_FAILED||','||const_recording_PRTL_SUCCESS||' )
                  ';

         autonomoustransaction(vsql);

       -- insert Alarm activity log for the current/future scheduled deleted recordings

      vsql :=' insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY, CREATE_DATE, UPDATE_DATE)

      select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ACTIVITY||','||const_ALARM_SEVERITY_INFO||','''||rcs_p_v||''' as component,
                           '''||rec_del_sub_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||start_time||'','||program_title_p_v||' = ''||title||'','||station_call_sign_p_v||'= ''||call_sign as description,
                        '''||const_EventNameKey_RECORDINGS||''',null,
                          '''||system_p_v||''' as created_by,'''||system_p_v||''' as updated_by, current_timestamp, current_timestamp
                        from (
                                select prg.title as title, s.start_time at time zone to_char (current_timestamp, ''TZR'') as start_time,

                                st.call_sign  as call_sign,
                                rm.name as rm_name,
                                rm.rm_id
                                from  ' ||temp_table_name||'   trrm
                                join recording rec on (trrm.recording_id = rec.recording_id)
                                join rtc_pool rp on (rp.pool_id = trrm.pool_id)
                                join rm on (rm.rm_id = rp.rm_id)
                                join schedule s on (trrm.schedule_id = s.schedule_id)

                                join rcs.program prg on (prg.program_id = s.program_id)
                                join station st on (st.station_id = s.station_id)

                            )';
     CREATEOBJECT(vsql,'inserting the alarm activity log  for deleted recordings', 'Y');

    -- Insert entries for all future and current recordings
 
               
      vsql := ' insert into recording_queue(
                    RECORDING_ID,
                    SCHEDULE_ID,
                    POOL_ID,
                    RM_ID,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    STATION_ID,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    REPEAT,
                    RM_NAME,
                    RM_IP_ADDRESS,
                    RM_PORT,
                    EPISODE_TITLE,
                    STATION_OFFSET,
                    HEADEND_NAME,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    ASSET_ID,
                    TITLE_ASSET_ID,
                    MOVIE_ASSET_ID,
                    POSTER_ASSET_ID,
                    POSTER_IMAGE_NAME,
        			POSTER_DESCRIPTION,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    STATE,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REQUEST_TYPE,
                    VOD_START_TIME,
                    VOD_DURATION,
                    VOD_RESTRICTIONS,
                    RTI_START_TIME,
                    RTI_DURATION,
                    RTI_RESTRICTIONS,
                    QC_START_TIME,
                    QC_DURATION,
                    QC_RESTRICTIONS,
                    CREATED_BY,
                    UPDATED_BY,
                    CREATE_DATE,
                    UPDATE_DATE,
                    advisories)
                    select rec_queue.* FROM
                    (SELECT distinct
                    trrm.RECORDING_ID,
                    trrm.SCHEDULE_ID,
                    trrm.POOL_ID,
                    trrm.RM_ID,
                    SCHEDULE_START_TIME,
                    SCHEDULE_END_TIME,
                    SCHEDULE_DURATION,
                    STATION_ID,
                    PROGRAM_ID,
                    HD_TV,
                    CC,
                    REPEAT,
                    RM_NAME,
                    RM_IP_ADDRESS,
                    RM_PORT,
                    EPISODE_TITLE,
                    STATION_OFFSET,
                    HEADEND_LOCATION,
                    CONTRACT_ID,
                    YEAR_PROGRAM,
                    trrm.ASSET_ID,
                    trrm.TITLE_ASSET_ID,
                    trrm.MOVIE_ASSET_ID,
                    trrm.POSTER_ASSET_ID,
                    trrm.POSTER_IMAGE_NAME,
        			trrm.POSTER_DESCRIPTION,
                    PROGRAM_TITLE,
                    PROGRAM_DESCRIPTION,
                    OFFSET_START_TIME,
                    OFFSET_END_TIME,
                    RECORDING_MODE,
                    PRIORITY,
                    STATE,
                    PROVIDER_QA_CONTACT,
                    CATALOG_ENTRY,
                    BILLING_ID,
                    MPAA_RATING,
                    TV_RATING,
                    PROVIDER_ID,
                    PROVIDER_NAME,
                    PRODUCT,
                    CATEGORY,
                    AUDIO_TYPE,
                    MAX_COPIES,
                    PREVIEW_PERIOD,
                    REDUNDANCY_COUNT,
                    LICENCE_START_DATE,
                    LICENSE_END_DATE,
                    REQUEST_TYPE,
                    VOD_START_TIME,
                    VOD_DURATION,
                    VOD_RESTRICTIONS,
                    RTI_START_TIME,
                    RTI_DURATION,
                    RTI_RESTRICTIONS,
                    QC_START_TIME,
                    QC_DURATION,
                    QC_RESTRICTIONS,
                    trrm.CREATED_BY,
                    trrm.UPDATED_BY,
                    trrm.CREATE_DATE,
                    trrm.UPDATE_DATE,
                    advisories

                    from   ' ||temp_table_name||'  trrm
                     where
                  	trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') >= current_timestamp
                    order by (trrm.SCHEDULE_END_TIME + numtodsinterval(nvl(trrm.offset_end_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'')), trrm.PRIORITY) rec_queue';

           CREATEOBJECT(vsql,'inserting Recording_queue records for future deleted recordings', 'Y');
                    
          -- new HA change
            insertRecQueue;

           -- delete recordable_rights_map entries for the above recordings

           vsql := 'delete from recordable_rights_map rrm
                          where rrm.right_id ='||in_right_id||'
                                and exists (select 1 from ' ||temp_table_name||'  trrm where trrm.schedule_id = rrm.schedule_id and trrm.rm_id = rrm.rm_id)';

           CREATEOBJECT(vsql,'Deleting recordables of the recordings that are exclusive for the given right');


         /** Update Recording Status as deleted  **/

          vsql := ' update recording rec set status='||const_recording_DELETED||' , status_code='||const_rec_status_code_DELETED||' where rec.recording_id in ( select recording_id from ' ||temp_table_name||' ) ';
          CREATEOBJECT(vsql,'Updating Recording status to be deleted ');

  end   processrmsdeletionfromright;


   procedure deleteright(in_right_id number) is
    rms_comma_separated varchar2(1000) :='';
     counter number(3) := 0 ;
  begin
      RECORDABLESRECRESOURCELOCK();

       -- lock recordinglog table exclusively
      lock table recording_log  in exclusive mode wait 5;

        -- Delete all the exclusive recordings existed for the RMs that are asssociated with the given Right
        processrmsdeletionfromright(in_right_id);

        -- process non-exclusive recordings existed for the RMs that are associated with the given Right
        processrightdeletion(in_right_id);

		 -- deleting the recordables for the future deleted recordings for the given right
		      delete from recordable_rights_map rrm where rrm.right_id = in_right_id
		             and exists ( select 1 from recording rec join rtc_pool rp on (rec.pool_id = rp.pool_id)
		                                             join  rm on (rm.rm_id = rp.rm_id) where rec.adjusted_end_time > current_timestamp and rec.status = const_recording_DELETED and rec.schedule_id = rrm.schedule_id and rm.rm_id = rrm.rm_id);


		      -- delete recordables that start time is  after the preprosing interval
		      delete from recordable_rights_map rrm where rrm.right_id = in_right_id

		                and  ( rrm.rm_id, rrm.schedule_id) in
		      (select  rm.rm_id, rec.schedule_id from recording rec,  rtc_pool rtcpool, rm
		                         where  rec.status not in (const_recording_DELETED, const_recording_SUCCESS, const_recording_FAILED, const_recording_PRTL_SUCCESS)
		                        and rec.adjusted_start_time > current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0),'SECOND')
		                        and rec.pool_id = rtcpool.pool_id
		                       and rm.rm_id = rtcpool.rm_id
		      );


		      -- All future recordables that do not have any recordings
		      delete from recordable_rights_map rrm where rrm.right_id = in_right_id
		             and exists (select 1 from schedule sch where sch.start_time > current_timestamp and sch.schedule_id = rrm.schedule_id)
		             and not exists ( select 1 from recording rec join rtc_pool rp on (rec.pool_id = rp.pool_id)
		                                             join  rm on (rm.rm_id = rp.rm_id) where rec.adjusted_end_time > current_timestamp and rec.schedule_id = rrm.schedule_id and rm.rm_id = rrm.rm_id);


          -- Update all invalid recordings status to be deleted
         update  recording rec1  set status =const_recording_DELETED , status_code=const_rec_status_code_DELETED   where exists (select recording_id
          from
          recording rec join

          ( select distinct schedule_id from recording
            minus
          select DISTINCT schedule_id from recordable_rights_map) sch  on (sch.schedule_id = rec.schedule_id)

          where  rec1.recording_id  = rec.recording_id );

      -- Update the Recordable to non_recordable(0) in the Schedule table for exclusive Recordables for this Right
      update schedule s  set recordable = default where  s.recordable = 1 and    not exists (select 1 from recordable_rights_map rrmap where rrmap.schedule_id = s.schedule_id);

      -- Update the Right's status to deleted(2)
       update right set state = const_STATE_DELETED  where right_id = in_right_id;

      -- Run auto Approve Rights
      rulesprocess('RIGHTS');

   end deleteright;

    procedure processrightdeletion(in_right_id number) is
    vsql varchar2(32000);
    oldest_right_recordings_count number := 0;
    temp_count number := 0;
    rm_name_p_v  backend_application_resources.value%type := getpropertyvalue('rm_name');
    id_p_v  backend_application_resources.value%type := getpropertyvalue('id');
    schedule_start_time_p_v  backend_application_resources.value%type := getpropertyvalue('schedule_start_time');
    program_title_p_v  backend_application_resources.value%type := getpropertyvalue('program_title');
    station_call_sign_p_v  backend_application_resources.value%type := getpropertyvalue('station_call_sign');
    rcs_p_v backend_application_resources.value%type := getpropertyvalue('rcs');
    system_p_v backend_application_resources.value%type := getpropertyvalue('system');
    rights_offset_err_p_v backend_application_resources.value%type := getpropertyvalue('rights_offset_error');
    rights_offset_war_p_v backend_application_resources.value%type := getpropertyvalue('rights_offset_warning');
    rec_del_sub_p_v backend_application_resources.value%type := getpropertyvalue('recording_delete_submission_succeeded');
    right_title_p_v backend_application_resources.value%type := getpropertyvalue('right_title');


  begin

  /**
      Delete temp tables information

  */

  delete from temp_oldest_right_recs;
  delete from temp_older_right_recordables;
  delete from temp_recs_after_right_deletion;

   -- Get future  non-exclusive recordings for the given right after deletion of it's exclusive recordings

    insert into  TEMP_OLDEST_RIGHT_RECS
         (select  sc1.schedule_id,rec.recording_id,rtcpool.pool_id, rmlineup.rm_id, rmlineup.headend_id, rmlineup.lineup_id, sc1.right_id ,
                  rec.adjusted_start_time as adjusted_schedule_start_time,
                  rec.adjusted_end_time as adjusted_schedule_end_time
                  from recording rec,  rtc_pool rtcpool, rm_lineup rmlineup , rm ,
                            ( select schedule_id , right_id, rm_id from ( select row_number() over (partition by rrm.schedule_id, rrm.rm_id order by r.create_date ) as rownumber,
                              rrm.schedule_id, rrm.rm_id,  r.right_id
                              from recordable_rights_map rrm , right r where
                              r.right_id = rrm.right_id and r.state=1) rrmouter where rrmouter.rownumber =1
                           )
                        sc1,
                       schedule sc
      								  where sc1.right_id = in_right_id and rec.status not in (const_recording_DELETED, const_recording_SUCCESS, const_recording_FAILED, const_recording_PRTL_SUCCESS)
                        and rec.adjusted_start_time > current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0),'SECOND')
                        and sc.schedule_id = sc1.schedule_id and   rec.schedule_id = sc.schedule_id
                        and rec.pool_id = rtcpool.pool_id
                        and rmlineup.station_id = sc.station_id and rmlineup.rm_id = rtcpool.rm_id
                        and rm.rm_id = rtcpool.rm_id
           );

      select count(1) into oldest_right_recordings_count  from temp_oldest_right_recs;

      -- If count > 0 , then process  rest of the block
    if (oldest_right_recordings_count > 0 ) then
          -- Get all the covered recordables under the next oldest right for the affected recordings
          INSERT INTO temp_older_right_recordables (schedule_id, right_id, rm_id)
            (SELECT DISTINCT SCHEDULE_ID, RIGHT_ID, rm_id  FROM ( select schedule_id , right_id, rm_id from ( select row_number() over (partition by rrm.schedule_id, rrm.rm_id order by r.create_date ) as rownumber,
                        rrm.schedule_id, rrm.rm_id, r.right_id
                        from recordable_rights_map rrm , right r, TEMP_OLDEST_RIGHT_RECS torr  where
                            r.right_id != in_right_id and
                            torr.schedule_id = rrm.schedule_id and
                             r.right_id = rrm.right_id and r.state=1) rrmouter where rrmouter.rownumber = 1)
            );

              -- Future Recordings whose RMs are covered by the second oldest right, then update the corresponding recordings
              insert into  TEMP_RECS_AFTER_RIGHT_DELETION
              (    SELECT distinct
                     TRECS.schedule_id,
                     TRECS.recording_id,
                     TRECS.pool_id,
                     TRECS.rm_id,
                     TRECS.headend_id,
                     TRECS.rm_lineup_id,
                     TRECS.OLDEST_RIGHT_ID ,
                     TRECORDABLES.right_id as older_right_id,
                    cast(1 as number) as toupdate,
                    TRECS.adjusted_schedule_start_time,
                    TRECS.adjusted_schedule_end_time
                  from  TEMP_OLDEST_RIGHT_RECS TRECS join TEMP_OLDER_RIGHT_RECORDABLES TRECORDABLES on (TRECS.schedule_id = TRECORDABLES.schedule_id and TRECS.rm_id = TRECORDABLES.rm_id)
                        join right on (TRECORDABLES.right_id = right.right_id)
                        join rm on (rm.rm_id = TRECS.rm_id )
                        join schedule sc on (sc.schedule_id = trecs.schedule_id)
                        join rm_lineup rl on (rl.station_id= sc.station_id and rl.rm_id = trecs.rm_id)

                   WHERE  TRECS.adjusted_schedule_start_time > current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0),'SECOND') AND
                          sc.start_time+ numtodsinterval(nvl(right.offset_start_time,0), 'SECOND') + numtodsinterval(nvl(rl.offset,0), 'SECOND')
                           > current_timestamp + numtodsinterval(nvl(rm.pre_provisioning_interval,0),'SECOND')
                );

              -- future Recordings whose RMs are not covered by the second oldest right, then delete the corresponding recordings
--
--              insert into  TEMP_RECS_AFTER_RIGHT_DELETION
--              (    SELECT distinct
--                     TRECS.schedule_id,
--                     TRECS.recording_id,
--                     TRECS.pool_id,
--                     TRECS.rm_id,
--                     TRECS.headend_id,
--                      TRECS.rm_lineup_id,
--                     TRECS.OLDEST_RIGHT_ID ,
--                     TRECORDABLES.right_id as older_right_id,
--                    cast(0 as number) as toupdate,
--                     TRECS.adjusted_schedule_start_time,
--                    TRECS.adjusted_schedule_end_time
--                  from  TEMP_OLDEST_RIGHT_RECS TRECS, TEMP_OLDER_RIGHT_RECORDABLES TRECORDABLES
--
--                   WHERE  (TRECS.adjusted_schedule_start_time > CURRENT_TIMESTAMP or TRECS.adjusted_schedule_end_time < CURRENT_TIMESTAMP) AND  TRECS.schedule_id = TRECORDABLES.schedule_id and
--                          TRECS.rm_id NOT in (select rm_id from right_rm_map where right_id = TRECORDABLES.right_id)
--               ) ;


            /** Process the old recordings first, delete recordings and corresponding mappings **/

                   -- insert Alarm activity log for the deleted recordings
            temp_count := 0;
            select count (1) into temp_count from TEMP_RECS_AFTER_RIGHT_DELETION trard  where trard.adjusted_schedule_end_time < current_timestamp and trard.toupdate = 0;

            if (temp_count > 0 ) then  -- old recordings count  start

               -- Delete from TEMP_RECS_AFTER_RIGHT_DELETION
                  delete from TEMP_RECS_AFTER_RIGHT_DELETION where adjusted_schedule_end_time < current_timestamp  and toupdate = 0;

           end if;  --old recordings count  end


            /** Future recordings whose are RMs are not covered by next oldest right,

            create recording_queue entries, and delete recording_logs entries.

            **/

          temp_count := 0;  --- Future recordings count for both to be updated and deleted
          select count(1) into temp_count from TEMP_RECS_AFTER_RIGHT_DELETION trard where trard.adjusted_schedule_start_time > current_timestamp;

          if (temp_count > 0 ) then  -- start of both tobedeleted and tobeupdated future recordings

                        -- Both tobedeleted and tobeupdated future recordings
					 autonomoustransaction(' truncate table temp_old_right_rec_map ');
                     insert into temp_old_right_rec_map
                      		(RECORDING_ID,SCHEDULE_ID,POOL_ID,RM_ID,SCHEDULE_START_TIME,SCHEDULE_END_TIME,SCHEDULE_DURATION,STATION_ID,PROGRAM_ID,HD_TV,CC,
                      			REPEAT,RM_NAME,RM_IP_ADDRESS,RM_PORT,EPISODE_TITLE,STATION_OFFSET,HEADEND_LOCATION,CONTRACT_ID,YEAR_PROGRAM,
                      			ASSET_ID,TITLE_ASSET_ID,MOVIE_ASSET_ID,POSTER_ASSET_ID,POSTER_IMAGE_NAME,POSTER_DESCRIPTION,PROGRAM_TITLE,PROGRAM_DESCRIPTION,OFFSET_START_TIME,OFFSET_END_TIME,RECORDING_MODE,PRIORITY,
                      			STATE,PROVIDER_QA_CONTACT,CATALOG_ENTRY,BILLING_ID,MPAA_RATING,TV_RATING,PROVIDER_ID,PROVIDER_NAME,PRODUCT,CATEGORY,AUDIO_TYPE,
                      			MAX_COPIES,PREVIEW_PERIOD,LICENCE_START_DATE,LICENSE_END_DATE,REQUEST_TYPE,VOD_START_TIME,VOD_DURATION,VOD_RESTRICTIONS,
                      			RTI_START_TIME,RTI_DURATION,RTI_RESTRICTIONS,QC_START_TIME,QC_DURATION,QC_RESTRICTIONS,CREATED_BY,UPDATED_BY,
                      			CREATE_DATE,UPDATE_DATE,TOUPDATE,ADVISORIES,REDUNDANCY_COUNT)
                                 select distinct rec.recording_id,
                                    rec.schedule_id,
                                    rec.pool_id,
                                    rp.rm_id,
                                    sch.start_time as SCHEDULE_START_TIME ,
                                    sch.end_time as SCHEDULE_END_TIME,
                                    sch.duration as SCHEDULE_DURATION ,
                                    sch.station_id,
                                    sch.program_id,
                                    sch.hd_tv,
                                    sch.cc,
                                    sch.repeat,
                                    rm.name as RM_NAME,
                                    rm.ip_address as RM_IP_ADDRESS,
                                    rm.port as RM_PORT,
                                    prg.episode_title,
                                    rl.offset as STATION_OFFSET,
                                    h.headend_location,
                                    contract.contract_id,
                                    prg.year_program,
                                    rec.asset_id,
                                    rec.title_asset_id,
                                    rec.movie_asset_id,
                                    rec.POSTER_ASSET_ID,
                                    rec.POSTER_IMAGE_NAME,
        							rec.POSTER_DESCRIPTION,
                                    prg.title as PROGRAM_TITLE,
                                    rec.program_description,
                                    rec.offset_start_time,
                                    rec.offset_end_time,
                                    rec.recording_mode,
                                    rec.priority,
                                    cast(0 as number) as state,
                                    rec.provider_qa_contact,
                                    rec.catalog_entry,
                                    rec.billing_id,
                                    rec.mpaa_rating,
                                    rec.tv_rating,
                                    rec.provider_id,
                                    rec.provider_name,
                                    rec.product,
                                    rec.category,
                                    rec.audio_type,
                                    rec.max_copies,
                                    rec.preview_period,
                                    rec.licence_start_date,
                                    rec.license_end_date,
                                    decode(trard.toupdate , 0 ,const_recording_DELETE_REQUEST,1,const_recording_UPDATE_REQUEST) as request_type,
                                    pwr_vod.start_time as vod_start_time,
                                    pwr_vod.duration as vod_duration,
                                    pwr_vod.restrictions as vod_restrictions,
                                    pwr_rti.start_time as rti_start_time ,
                                    pwr_rti.duration as rti_duration,
                                    pwr_rti.restrictions as rti_restrictions,
                                    pwr_qc.start_time as qc_start_time ,
                                    pwr_qc.duration as qc_duration,
                                    pwr_qc.restrictions as qc_restrictions,
                                    'SYSTEM' as created_by,
                                    'SYSTEM' as updated_by,
                                    current_timestamp as create_date,
                                    current_timestamp as update_date,
                                    trard.toupdate,
                                     rtrim(case when prg.advisory_desc1 is not null then prg.advisory_desc1||'|' end ||
                                     case when prg.advisory_desc2 is not null then prg.advisory_desc2||'|' end ||
                                     case when prg.advisory_desc3 is not null then prg.advisory_desc3||'|' end ||
                                     case when prg.advisory_desc4 is not null then prg.advisory_desc4||'|' end ||
                                     case when prg.advisory_desc5 is not null then prg.advisory_desc5||'|' end ||
                                     case when prg.advisory_desc6 is not null then prg.advisory_desc6||'|' end, '|') as advisories,
                                     rec.redundancy_count

                                    from

                                    TEMP_RECS_AFTER_RIGHT_DELETION trard
                                    join  recording rec on (trard.recording_id = rec.recording_id and trard.pool_id =rec.pool_id )
                                    join schedule sch on (trard.schedule_id = sch.schedule_id)
                                    join program prg on (sch.program_id = prg.program_id)
                                    join rtc_pool rp on (trard.pool_id = rp.pool_id)
                                    join rm on ( rm.rm_id = trard.rm_id )
                                    join rm_lineup rl on (rl.lineup_id =trard.rm_lineup_id and  rl.rm_id = rm.rm_id and rl.station_id = sch.station_id and rl.headend_id = rm.headend_id)
                                    join headend h on (h.headend_id = trard.headend_id )
                                    join right on (trard.oldest_right_id = right.right_id)
                                    join contract on (contract.contract_id = right.contract_id)
                                    left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_vod on (pwr_vod.recording_id = rec.recording_id and pwr_vod.type='VOD')
                                    left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_rti on (pwr_rti.recording_id = rec.recording_id and pwr_rti.type='RTI')
                                    left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_qc on (pwr_qc.recording_id = rec.recording_id and pwr_qc.type='QC')

                                    where trard.adjusted_schedule_start_time > current_timestamp  ;

                      temp_count := 0 ;
                      -- To be deleted future recordings count
                      select count(1) into temp_count from TEMP_RECS_AFTER_RIGHT_DELETION trard where trard.adjusted_schedule_start_time > current_timestamp and  trard.TOUPDATE = 0 ;

                      if (temp_count > 0 ) then

                                      insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY, CREATE_DATE, UPDATE_DATE)

                                     ( select alarm_activitylog_seq.nextval,const_ALARM_TYPE_ACTIVITY,const_ALARM_SEVERITY_INFO,''''||rcs_p_v||'''' as component,
                                     					                   rec_del_sub_p_v||' '||rm_name_p_v||'='||rm_name||' '||id_p_v||'='||rm_id||','||schedule_start_time_p_v||' = '||start_time||','||program_title_p_v||' = '||title||','||station_call_sign_p_v||'= '||call_sign as description,


                                                          const_EventNameKey_RECORDINGS as event_name,null,
                                                          ''''||system_p_v||'''' as created_by,''''||system_p_v||'''' as updated_by, current_timestamp, current_timestamp
                                                        from
                                                            (select prg.title as title, s.start_time at time zone to_char(current_timestamp, 'TZR') as start_time,
                                                            st.call_sign as call_sign,
                                                            rm.name as rm_name,
                                                            rm.rm_id

                                                            from
                                                            TEMP_RECS_AFTER_RIGHT_DELETION    trrm
                                                            join recording rec on (trrm.recording_id = rec.recording_id)
                                                            join rtc_pool rp on (rp.pool_id = trrm.pool_id)
                                                            join rm on (rm.rm_id = rp.rm_id)
                                                            join schedule s on (trrm.schedule_id = s.schedule_id)
                                                            join rcs.program prg on (prg.program_id = s.program_id)
                                                            join station st on (st.station_id = s.station_id)

                                                        where trrm.adjusted_schedule_start_time > current_timestamp and  trrm.TOUPDATE = 0)
                                      );

                                     -- to be deleted future recordings
       
          
              
                  vsql := 'insert into recording_queue(
                                                    RECORDING_ID,
                                                    SCHEDULE_ID,
                                                    POOL_ID,
                                                    RM_ID,
                                                    SCHEDULE_START_TIME,
                                                    SCHEDULE_END_TIME,
                                                    SCHEDULE_DURATION,
                                                    STATION_ID,
                                                    PROGRAM_ID,
                                                    HD_TV,
                                                    CC,
                                                    REPEAT,
                                                    RM_NAME,
                                                    RM_IP_ADDRESS,
                                                    RM_PORT,
                                                    EPISODE_TITLE,
                                                    STATION_OFFSET,
                                                    HEADEND_NAME,
                                                    CONTRACT_ID,
                                                    YEAR_PROGRAM,
                                                    ASSET_ID,
                                                    TITLE_ASSET_ID,
                                                    MOVIE_ASSET_ID,
                                                    POSTER_ASSET_ID,
                                                    POSTER_IMAGE_NAME,
        											                      POSTER_DESCRIPTION,
                                                    PROGRAM_TITLE,
                                                    PROGRAM_DESCRIPTION,
                                                    OFFSET_START_TIME,
                                                    OFFSET_END_TIME,
                                                    RECORDING_MODE,
                                                    PRIORITY,
                                                    STATE,
                                                    PROVIDER_QA_CONTACT,
                                                    CATALOG_ENTRY,
                                                    BILLING_ID,
                                                    MPAA_RATING,
                                                    TV_RATING,
                                                    PROVIDER_ID,
                                                    PROVIDER_NAME,
                                                    PRODUCT,
                                                    CATEGORY,
                                                    AUDIO_TYPE,
                                                    MAX_COPIES,
                                                    PREVIEW_PERIOD,
                                                    REDUNDANCY_COUNT,
                                                    LICENCE_START_DATE,
                                                    LICENSE_END_DATE,
                                                    REQUEST_TYPE,
                                                    VOD_START_TIME,
                                                    VOD_DURATION,
                                                    VOD_RESTRICTIONS,
                                                    RTI_START_TIME,
                                                    RTI_DURATION,
                                                    RTI_RESTRICTIONS,
                                                    QC_START_TIME,
                                                    QC_DURATION,
                                                    QC_RESTRICTIONS,
                                                    CREATED_BY,
                                                    UPDATED_BY,
                                                    CREATE_DATE,
                                                    UPDATE_DATE,
                                                    advisories) 
                                                    select rec_queue.* FROM
                                                    (SELECT
                                                    trrm.RECORDING_ID,
                                                    trrm.SCHEDULE_ID,
                                                    trrm.POOL_ID,
                                                    trrm.RM_ID,
                                                    SCHEDULE_START_TIME,
                                                    SCHEDULE_END_TIME,
                                                    SCHEDULE_DURATION,
                                                    STATION_ID,
                                                    PROGRAM_ID,
                                                    HD_TV,
                                                    CC,
                                                    REPEAT,
                                                    RM_NAME,
                                                    RM_IP_ADDRESS,
                                                    RM_PORT,
                                                    EPISODE_TITLE,
                                                    STATION_OFFSET,
                                                    HEADEND_LOCATION,
                                                    CONTRACT_ID,
                                                    YEAR_PROGRAM,
                                                    trrm.ASSET_ID,
                                                    trrm.TITLE_ASSET_ID,
                                                    trrm.MOVIE_ASSET_ID,
                                                    trrm.POSTER_ASSET_ID,
                                                    trrm.POSTER_IMAGE_NAME,
        											                      trrm.POSTER_DESCRIPTION,
                                                    trrm.PROGRAM_TITLE,
                                                    trrm.PROGRAM_DESCRIPTION,
                                                    trrm.OFFSET_START_TIME,
                                                    trrm.OFFSET_END_TIME,
                                                    trrm.RECORDING_MODE,
                                                    trrm.PRIORITY,
                                                    trrm.STATE,
                                                    trrm.PROVIDER_QA_CONTACT,
                                                    trrm.CATALOG_ENTRY,
                                                    trrm.BILLING_ID,
                                                    trrm.MPAA_RATING,
                                                    trrm.TV_RATING,
                                                    trrm.PROVIDER_ID,
                                                    trrm.PROVIDER_NAME,
                                                    trrm.PRODUCT,
                                                    trrm.CATEGORY,
                                                    trrm.AUDIO_TYPE,
                                                    trrm.MAX_COPIES,
                                                    trrm.PREVIEW_PERIOD,
                                                    trrm.REDUNDANCY_COUNT,
                                                    trrm.LICENCE_START_DATE,
                                                    trrm.LICENSE_END_DATE,
                                                    REQUEST_TYPE,
                                                    VOD_START_TIME,
                                                    VOD_DURATION,
                                                    VOD_RESTRICTIONS,
                                                    RTI_START_TIME,
                                                    RTI_DURATION,
                                                    RTI_RESTRICTIONS,
                                                    QC_START_TIME,
                                                    QC_DURATION,
                                                    QC_RESTRICTIONS,
                                                    trrm.CREATED_BY,
                                                    trrm.UPDATED_BY,
                                                    trrm.CREATE_DATE,
                                                    trrm.UPDATE_DATE,
                                                    advisories
                                                    from   temp_old_right_rec_map  trrm
                                                    join recording rec on (trrm.recording_id = rec.recording_id  and trrm.toupdate = 0  )
                                                    join rtc_pool rp on (rec.pool_id = rp.pool_id) where
                                                    trrm.rm_id = rp.rm_id and
                                                    trrm.SCHEDULE_start_TIME + numtodsinterval(nvl(trrm.offset_start_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') >= current_timestamp
                                                    order by rec.adjusted_start_time, trrm.PRIORITY) rec_queue' ;
                                                     CREATEOBJECT(vsql,'RECORDING Queue Insert');                                                    

             -- new HA change
                insertRecQueue;


                                      -- Delete deleted recordings from TEMP_RECS_AFTER_RIGHT_DELETION
                                      delete from TEMP_RECS_AFTER_RIGHT_DELETION  trrm  where trrm.adjusted_schedule_start_time > current_timestamp and  trrm.TOUPDATE = 0;

                                      update recording rec set status=const_recording_DELETED , status_code=const_rec_status_code_DELETED where recording_id in (select recording_id from TEMP_RECS_AFTER_RIGHT_DELETION  trrm  where trrm.adjusted_schedule_start_time > current_timestamp and  trrm.TOUPDATE = 0);

                          end if ;


                         /**  Starting   delete all moved schedules from future to past due to older right's offset values  **/



                        insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY, CREATE_DATE, UPDATE_DATE)

                       ( select alarm_activitylog_seq.nextval,const_ALARM_TYPE_ACTIVITY,const_ALARM_SEVERITY_INFO,''''||rcs_p_v||'''' as component,

                                            rec_del_sub_p_v||' '||rm_name_p_v||'='||rm_name||' '||id_p_v||'='||rm_id||','||schedule_start_time_p_v||' = '||start_time||','||program_title_p_v||' = '||title||','||station_call_sign_p_v||'= '||call_sign as description,
                                            const_EventNameKey_RECORDINGS as event_name,null,
                                            ''''||system_p_v||'''' as created_by,''''||system_p_v||'''' as updated_by, current_timestamp, current_timestamp
                                          from (select prg.title as title, s.start_time at time zone to_char(current_timestamp, 'TZR') as start_time,
                                              st.call_sign as call_sign,
                                              rm.name as rm_name,
                                              rm.rm_id
                                              from
                                              TEMP_RECS_AFTER_RIGHT_DELETION    trrm
                                              join recording rec on (trrm.recording_id = rec.recording_id)
                                              join rtc_pool rp on (rp.pool_id = trrm.pool_id)
                                              join rm on (rm.rm_id = rp.rm_id)
                                              join schedule s on (trrm.schedule_id = s.schedule_id)
                                              join rcs.program prg on (prg.program_id = s.program_id)
                                              join station st on (st.station_id = s.station_id)
                                              join right on (trrm.older_right_id = right.right_id)
                                               join rm_lineup rl on (rl.lineup_id =trrm.rm_lineup_id and  rl.rm_id = trrm.rm_id and rl.station_id = s.station_id and rl.headend_id = trrm.headend_id)
                                          where trrm.adjusted_schedule_start_time > current_timestamp and trrm.toupdate = 1
                                           and
                                           s.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND') + numtodsinterval(nvl(rl.OFFSET,0),'SECOND') < current_timestamp)
                        );



                           -- create new entries in Recording Queue for  schedules  whose start time falls in the past
                           -- due to older right offsets

                            vsql := 'insert into recording_queue(
                                      RECORDING_ID,
                                      SCHEDULE_ID,
                                      POOL_ID,
                                      RM_ID,
                                      SCHEDULE_START_TIME,
                                      SCHEDULE_END_TIME,
                                      SCHEDULE_DURATION,
                                      STATION_ID,
                                      PROGRAM_ID,
                                      HD_TV,
                                      CC,
                                      REPEAT,
                                      RM_NAME,
                                      RM_IP_ADDRESS,
                                      RM_PORT,
                                      EPISODE_TITLE,
                                      STATION_OFFSET,
                                      HEADEND_NAME,
                                      CONTRACT_ID,
                                      YEAR_PROGRAM,
                                      ASSET_ID,
                                      TITLE_ASSET_ID,
                                      MOVIE_ASSET_ID,
                                      POSTER_ASSET_ID,
                                      POSTER_IMAGE_NAME,
        							                POSTER_DESCRIPTION,
                                      PROGRAM_TITLE,
                                      PROGRAM_DESCRIPTION,
                                      OFFSET_START_TIME,
                                      OFFSET_END_TIME,
                                      RECORDING_MODE,
                                      PRIORITY,
                                      STATE,
                                      PROVIDER_QA_CONTACT,
                                      CATALOG_ENTRY,
                                      BILLING_ID,
                                      MPAA_RATING,
                                      TV_RATING,
                                      PROVIDER_ID,
                                      PROVIDER_NAME,
                                      PRODUCT,
                                      CATEGORY,
                                      AUDIO_TYPE,
                                      MAX_COPIES,
                                      PREVIEW_PERIOD,
                                      REDUNDANCY_COUNT,
                                      LICENCE_START_DATE,
                                      LICENSE_END_DATE,
                                      REQUEST_TYPE,
                                      VOD_START_TIME,
                                      VOD_DURATION,
                                      VOD_RESTRICTIONS,
                                      RTI_START_TIME,
                                      RTI_DURATION,
                                      RTI_RESTRICTIONS,
                                      QC_START_TIME,
                                      QC_DURATION,
                                      QC_RESTRICTIONS,
                                      CREATED_BY,
                                      UPDATED_BY,
                                      CREATE_DATE,
                                      UPDATE_DATE,
                                      advisories) 
                                      select rec_queue.* FROM
                                      (SELECT
                                      trrm.RECORDING_ID,
                                      trrm.SCHEDULE_ID,
                                      trrm.POOL_ID,
                                      trrm.RM_ID,
                                      trrm.SCHEDULE_START_TIME,
                                      trrm.SCHEDULE_END_TIME,
                                      trrm.SCHEDULE_DURATION,
                                      trrm.STATION_ID,
                                      trrm.PROGRAM_ID,
                                      trrm.HD_TV,
                                      trrm.CC,
                                      trrm.REPEAT,
                                      trrm.RM_NAME,
                                      trrm.RM_IP_ADDRESS,
                                      trrm.RM_PORT,
                                      trrm.EPISODE_TITLE,
                                      trrm.STATION_OFFSET,
                                      trrm.HEADEND_LOCATION,
                                      trrm.CONTRACT_ID,
                                      trrm.YEAR_PROGRAM,
                                      trrm.ASSET_ID,
                                      trrm.TITLE_ASSET_ID,
                                      trrm.MOVIE_ASSET_ID,
                                      trrm.POSTER_ASSET_ID,
                                      trrm.POSTER_IMAGE_NAME,
        						              	  trrm.POSTER_DESCRIPTION,
                                      trrm.PROGRAM_TITLE,
                                      trrm.PROGRAM_DESCRIPTION,
                                      trrm.OFFSET_START_TIME,
                                      trrm.OFFSET_END_TIME,
                                      trrm.RECORDING_MODE,
                                      trrm.PRIORITY,
                                      trrm.STATE,
                                      trrm.PROVIDER_QA_CONTACT,
                                      trrm.CATALOG_ENTRY,
                                      trrm.BILLING_ID,
                                      trrm.MPAA_RATING,
                                      trrm.TV_RATING,
                                      trrm.PROVIDER_ID,
                                      trrm.PROVIDER_NAME,
                                      trrm.PRODUCT,
                                      trrm.CATEGORY,
                                      trrm.AUDIO_TYPE,
                                      trrm.MAX_COPIES,
                                      trrm.PREVIEW_PERIOD,
                                      trrm.REDUNDANCY_COUNT,
                                      trrm.LICENCE_START_DATE,
                                      trrm.LICENSE_END_DATE,
                                      cast(const_recording_DELETE_REQUEST as number) as request_type,
                                      VOD_START_TIME,
                                      VOD_DURATION,
                                      VOD_RESTRICTIONS,
                                      RTI_START_TIME,
                                      RTI_DURATION,
                                      RTI_RESTRICTIONS,
                                      QC_START_TIME,
                                      QC_DURATION,
                                      QC_RESTRICTIONS,
                                      trrm.CREATED_BY,
                                      trrm.UPDATED_BY,
                                      trrm.CREATE_DATE,
                                      trrm.UPDATE_DATE,
                                     advisories

                                      from  temp_old_right_rec_map  trrm
                                      join TEMP_RECS_AFTER_RIGHT_DELETION  trard on (trard.recording_id = trrm.recording_id)
                                      join schedule s on (trard.schedule_id = s.schedule_id)
                                      join station st on (st.station_id = s.station_id)
                                      join right on (trard.older_right_id = right.right_id)
                                      join recording rec on (trrm.recording_id = rec.recording_id   and trard.toupdate = 1  )
                                      join rm_lineup rl on (rl.lineup_id =trard.rm_lineup_id and  rl.rm_id = trard.rm_id and rl.station_id = s.station_id and rl.headend_id = trard.headend_id)
                                      join rtc_pool rp on (trard.pool_id = rp.pool_id)
                                      where
                                       trard.adjusted_schedule_start_time > current_timestamp and trard.toupdate = 1
                                           and
                                           s.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'') + numtodsinterval(nvl(rl.OFFSET,0),''SECOND'') < current_timestamp
                                      order by rec.adjusted_start_time, trrm.PRIORITY) rec_queue ';
                                      CREATEOBJECT(vsql,'RECORDING Queue Insert');
              -- new HA change
                insertRecQueue;

                        /** Update Recording Status as deleted  **/
                       update recording rec set status=const_recording_DELETED, status_code=const_rec_status_code_DELETED where recording_id in (select trrm.recording_id from TEMP_RECS_AFTER_RIGHT_DELETION    trrm
                                              join TEMP_OLDEST_RIGHT_RECS torr on (trrm.recording_id = torr.recording_id and trrm.pool_id = torr.pool_id)
                                              join schedule s on (trrm.schedule_id = s.schedule_id)
                                              join station st on (st.station_id = s.station_id)
                                              join right on (trrm.older_right_id = right.right_id)
                                              join rm_lineup rl on (rl.lineup_id =trrm.rm_lineup_id and  rl.rm_id = trrm.rm_id and rl.station_id = s.station_id and rl.headend_id = trrm.headend_id)
                                          where trrm.adjusted_schedule_start_time > current_timestamp and trrm.toupdate = 1
                                           and
                                           s.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND') + numtodsinterval(nvl(rl.OFFSET,0),'SECOND') < current_timestamp);

                     -- Delete from temp table
                     delete from TEMP_RECS_AFTER_RIGHT_DELETION where (recording_id, pool_id) in (

                        select trrm.recording_id, trrm.pool_id from TEMP_RECS_AFTER_RIGHT_DELETION    trrm
                                              join TEMP_OLDEST_RIGHT_RECS torr on (trrm.recording_id = torr.recording_id and trrm.pool_id = torr.pool_id)
                                              join schedule s on (trrm.schedule_id = s.schedule_id)
                                              join station st on (st.station_id = s.station_id)
                                              join right on (trrm.older_right_id = right.right_id)
                                               join rm_lineup rl on (rl.lineup_id =trrm.rm_lineup_id and  rl.rm_id = trrm.rm_id and rl.station_id = s.station_id and rl.headend_id = trrm.headend_id)
                                          where trrm.adjusted_schedule_start_time > current_timestamp and trrm.toupdate = 1
                                           and
                                           s.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND') + numtodsinterval(nvl(rl.OFFSET,0),'SECOND') < current_timestamp);


                        /**   Future updatable recordings          **/
                      temp_count := 0 ;
                       select count(1) into temp_count  from TEMP_RECS_AFTER_RIGHT_DELETION trard where
                       trard.adjusted_schedule_start_time > current_timestamp  and trard.toupdate=1;




                       if ( temp_count > 0 ) then

                          -- find the recordings that are out of right's boundaries
                           -- Raise Alarms for the schedules those cannot create any recordings because of the offsets
                        vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_MAJOR||','''||rcs_p_v||''',
									 '''||rights_offset_err_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          				'||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign||'','||right_title_p_v||'= ''||tr.right_title,
                                        '''||const_EventNameKey_RECORDINGS||''' , 14,'''||system_p_v||''','''||system_p_v||'''
                                          from (

                                                  select distinct rm.rm_id as rm_id, rm.name as rm_name, station.call_sign, sc.start_time at time zone to_char (current_timestamp, ''TZR'')  as SCHEDULE_START_TIME,prg.title as PROGRAM_TITLE, right.name as right_title
                                                    from TEMP_RECS_AFTER_RIGHT_DELETION trard join right on ( trard.older_right_id = right.right_id)
                                                         join contract  on  (contract.contract_id = right.contract_id)
                                                          join schedule sc on ( trard.schedule_id = sc.schedule_id)
                                                          join station on (station.station_id = sc.station_id)
                                                          join program prg on (prg.program_id = sc.program_id)
                                                          join rm on (trard.rm_id = rm.rm_id)
                                                          left outer join metadata_config mctv on (mctv.type=''TVRating'' and mctv.name=upper(sc.tv_rating))
                                                          left outer join metadata_config mcmr on (mcmr.type=''MPAARating'' and mcmr.name=upper(prg.mpaa_rating))
                                                          where  trard.toupdate=1
                                                          and   (
                                                                 ( nvl(right.start_date,contract.effective_date) >= sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'')
                                                                    and
                                                                  nvl(right.start_date,contract.effective_date) >= sc.end_time + numtodsinterval(nvl(right.offset_end_time,0),''SECOND'')
                                                                 )
                                                                 OR
                                                                 (
                                                                    nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'')
                                                                      and
                                                                    nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),''SECOND'')
                                                                 )
                                                                 OR
                                                                 (
                                                                    sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'') >= sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),''SECOND'')
                                                                 )
                                                          )

                                      ) tr';

                        vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
                        CREATEOBJECT(vsql,'Offset ALARMS');
          
              
            
                       vsql := 'insert into recording_queue(
                                      RECORDING_ID,
                                      SCHEDULE_ID,
                                      POOL_ID,
                                      RM_ID,
                                      SCHEDULE_START_TIME,
                                      SCHEDULE_END_TIME,
                                      SCHEDULE_DURATION,
                                      STATION_ID,
                                      PROGRAM_ID,
                                      HD_TV,
                                      CC,
                                      REPEAT,
                                      RM_NAME,
                                      RM_IP_ADDRESS,
                                      RM_PORT,
                                      EPISODE_TITLE,
                                      STATION_OFFSET,
                                      HEADEND_NAME,
                                      CONTRACT_ID,
                                      YEAR_PROGRAM,
                                      ASSET_ID,
                                      TITLE_ASSET_ID,
                                      MOVIE_ASSET_ID,
                                      POSTER_ASSET_ID,
                                      POSTER_IMAGE_NAME,
        							                POSTER_DESCRIPTION,
                                      PROGRAM_TITLE,
                                      PROGRAM_DESCRIPTION,
                                      OFFSET_START_TIME,
                                      OFFSET_END_TIME,
                                      RECORDING_MODE,
                                      PRIORITY,
                                      STATE,
                                      PROVIDER_QA_CONTACT,
                                      CATALOG_ENTRY,
                                      BILLING_ID,
                                      MPAA_RATING,
                                      TV_RATING,
                                      PROVIDER_ID,
                                      PROVIDER_NAME,
                                      PRODUCT,
                                      CATEGORY,
                                      AUDIO_TYPE,
                                      MAX_COPIES,
                                      PREVIEW_PERIOD,
                                      REDUNDANCY_COUNT,
                                      LICENCE_START_DATE,
                                      LICENSE_END_DATE,
                                      REQUEST_TYPE,
                                      VOD_START_TIME,
                                      VOD_DURATION,
                                      VOD_RESTRICTIONS,
                                      RTI_START_TIME,
                                      RTI_DURATION,
                                      RTI_RESTRICTIONS,
                                      QC_START_TIME,
                                      QC_DURATION,
                                      QC_RESTRICTIONS,
                                      CREATED_BY,
                                      UPDATED_BY,
                                      CREATE_DATE,
                                      UPDATE_DATE,
                                      advisories)
                                      select rec_queue.* FROM
                                      (SELECT
                                      trrm.RECORDING_ID,
                                      trrm.SCHEDULE_ID,
                                      trrm.POOL_ID,
                                      trrm.RM_ID,
                                      trrm.SCHEDULE_START_TIME,
                                      trrm.SCHEDULE_END_TIME,
                                      trrm.SCHEDULE_DURATION,
                                      trrm.STATION_ID,
                                      trrm.PROGRAM_ID,
                                      trrm.HD_TV,
                                      trrm.CC,
                                      trrm.REPEAT,
                                      trrm.RM_NAME,
                                      trrm.RM_IP_ADDRESS,
                                      trrm.RM_PORT,
                                      trrm.EPISODE_TITLE,
                                      trrm.STATION_OFFSET,
                                      trrm.HEADEND_LOCATION,
                                      trrm.CONTRACT_ID,
                                      trrm.YEAR_PROGRAM,
                                      trrm.ASSET_ID,
                                      trrm.TITLE_ASSET_ID,
                                      trrm.MOVIE_ASSET_ID,
                                      trrm.POSTER_ASSET_ID,
                                      trrm.POSTER_IMAGE_NAME,
        							                trrm.POSTER_DESCRIPTION,
                                      trrm.PROGRAM_TITLE,
                                      trrm.PROGRAM_DESCRIPTION,
                                      trrm.OFFSET_START_TIME,
                                      trrm.OFFSET_END_TIME,
                                      trrm.RECORDING_MODE,
                                      trrm.PRIORITY,
                                      trrm.STATE,
                                      trrm.PROVIDER_QA_CONTACT,
                                      trrm.CATALOG_ENTRY,
                                      trrm.BILLING_ID,
                                      trrm.MPAA_RATING,
                                      trrm.TV_RATING,
                                      trrm.PROVIDER_ID,
                                      trrm.PROVIDER_NAME,
                                      trrm.PRODUCT,
                                      trrm.CATEGORY,
                                      trrm.AUDIO_TYPE,
                                      trrm.MAX_COPIES,
                                      trrm.PREVIEW_PERIOD,
                                      trrm.REDUNDANCY_COUNT,
                                      trrm.LICENCE_START_DATE,
                                      trrm.LICENSE_END_DATE,
                                      cast(const_recording_DELETE_REQUEST as number) as request_type,
                                      VOD_START_TIME,
                                      VOD_DURATION,
                                      VOD_RESTRICTIONS,
                                      RTI_START_TIME,
                                      RTI_DURATION,
                                      RTI_RESTRICTIONS,
                                      QC_START_TIME,
                                      QC_DURATION,
                                      QC_RESTRICTIONS,
                                      trrm.CREATED_BY,
                                      trrm.UPDATED_BY,
                                      trrm.CREATE_DATE,
                                      trrm.UPDATE_DATE,
                                     advisories
                                      from  temp_old_right_rec_map  trrm
                                      join TEMP_RECS_AFTER_RIGHT_DELETION  trard on (trard.recording_id = trrm.recording_id)
                                      join schedule s on (trard.schedule_id = s.schedule_id)
                                      join station st on (st.station_id = s.station_id)
                                      join right on (trard.older_right_id = right.right_id)
                                      join contract on (contract.contract_id = right.contract_id)
                                      join recording rec on (trrm.recording_id = rec.recording_id   and trard.toupdate = 1  )
                                      join rm_lineup rl on (rl.lineup_id =trard.rm_lineup_id and  rl.rm_id = trard.rm_id and rl.station_id = s.station_id and rl.headend_id = trard.headend_id)
                                      join rtc_pool rp on (trard.pool_id = rp.pool_id)
                                      where
                                       trard.adjusted_schedule_start_time > current_timestamp and trard.toupdate = 1
                                           and
                                           s.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'') + numtodsinterval(nvl(rl.OFFSET,0),''SECOND'') < current_timestamp
                                            and   (
                                                                         ( nvl(right.start_date,contract.effective_date) >= s.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'')
                                                                            and
                                                                          nvl(right.start_date,contract.effective_date) >= s.end_time + numtodsinterval(nvl(right.offset_end_time,0),''SECOND'')
                                                                         )
                                                                         OR
                                                                         (
                                                                            nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= s.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'')
                                                                              and
                                                                            nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= s.END_time + numtodsinterval(nvl(right.offset_END_time,0),''SECOND'')
                                                                         )
                                                                         OR
                                                                         (
                                                                            s.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'') >= s.END_time + numtodsinterval(nvl(right.offset_END_time,0),''SECOND'')
                                                                         )
                                                 )

                                      order by rec.adjusted_start_time, trrm.PRIORITY)) rec_queue ';
                            CREATEOBJECT(vsql,'RECORDING Queue Insert ');
                   
              -- new HA change
                insertRecQueue;

                      -- update recording status
                      update recording rec set status=const_recording_DELETED, status_code=const_rec_status_code_DELETED where recording_id in
                              (select recording_id from   TEMP_RECS_AFTER_RIGHT_DELETION trard join right on ( trard.older_right_id = right.right_id)
                                                                 join contract  on  (contract.contract_id = right.contract_id)
                                                                  join schedule sc on ( trard.schedule_id = sc.schedule_id)
                                                                  join station on (station.station_id = sc.station_id)
                                                                  join program prg on (prg.program_id = sc.program_id)
                                                                  left outer join metadata_config mctv on (mctv.type='TVRating' and mctv.name=upper(sc.tv_rating))
                                                                  left outer join metadata_config mcmr on (mcmr.type='MPAARating' and mcmr.name=upper(prg.mpaa_rating))
                                                                  where  trard.toupdate=1
                                                                  and   (
                                                                         ( nvl(right.start_date,contract.effective_date) >= sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND')
                                                                            and
                                                                          nvl(right.start_date,contract.effective_date) >= sc.end_time + numtodsinterval(nvl(right.offset_end_time,0),'SECOND')
                                                                         )
                                                                         OR
                                                                         (
                                                                            nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND')
                                                                              and
                                                                            nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),'SECOND')
                                                                         )
                                                                          OR
                                                                         (
                                                                            sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND') >= sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),'SECOND')
                                                                         )
                                                                  )
                            );


                        delete from TEMP_RECS_AFTER_RIGHT_DELETION where  recording_id in
                           (select recording_id from   TEMP_RECS_AFTER_RIGHT_DELETION trard join right on ( trard.older_right_id = right.right_id)
                                                                     join contract  on  (contract.contract_id = right.contract_id)
                                                                      join schedule sc on ( trard.schedule_id = sc.schedule_id)
                                                                      join station on (station.station_id = sc.station_id)
                                                                      join program prg on (prg.program_id = sc.program_id)
                                                                      left outer join metadata_config mctv on (mctv.type='TVRating' and mctv.name=upper(sc.tv_rating))
                                                                      left outer join metadata_config mcmr on (mcmr.type='MPAARating' and mcmr.name=upper(prg.mpaa_rating))
                                                                      where  trard.toupdate=1
                                                                      and   (
                                                                             ( nvl(right.start_date,contract.effective_date) >= sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND')
                                                                                and
                                                                              nvl(right.start_date,contract.effective_date) >= sc.end_time + numtodsinterval(nvl(right.offset_end_time,0),'SECOND')
                                                                             )
                                                                             OR
                                                                             (
                                                                                nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND')
                                                                                  and
                                                                                nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) <= sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),'SECOND')
                                                                             )
                                                                              OR
                                                                            (
                                                                              sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND') >= sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),'SECOND')
                                                                            )
                                                                      )
                                                                    );



    vsql := 'select alarm_activitylog_seq.nextval,'||const_ALARM_TYPE_ALARM||','||const_ALARM_SEVERITY_WARNING||','''||rcs_p_v||''',
				'''||rights_offset_war_p_v||' '||rm_name_p_v||'=''||rm_name||'' '||id_p_v||'=''||rm_id||'',
                          '||schedule_start_time_p_v||' = ''||tr.schedule_start_time||'','||program_title_p_v||' = ''||tr.PROGRAM_TITLE||'','||station_call_sign_p_v||'= ''||tr.call_sign||'','||right_title_p_v||'= ''||tr.right_title,
                  '''||const_EventNameKey_RECORDINGS||''' , null,'''||system_p_v||''','''||system_p_v||'''
                      from (
                              select distinct rm.rm_id as rm_id, rm.name as rm_name, station.call_sign, sc.start_time  at time zone to_char (current_timestamp, ''TZR'')  as SCHEDULE_START_TIME,prg.title as PROGRAM_TITLE, right.name as right_title
                                                    from TEMP_RECS_AFTER_RIGHT_DELETION trard join right on ( trard.older_right_id = right.right_id)
                                                         join contract  on  (contract.contract_id = right.contract_id)
                                                          join schedule sc on ( trard.schedule_id = sc.schedule_id)
                                                          join station on (station.station_id = sc.station_id)
                                                          join program prg on (prg.program_id = sc.program_id)
                                                          join rm on (trard.rm_id = rm.rm_id)
                                                          left outer join metadata_config mctv on (mctv.type=''TVRating'' and mctv.name=upper(sc.tv_rating))
                                                          left outer join metadata_config mcmr on (mcmr.type=''MPAARating'' and mcmr.name=upper(prg.mpaa_rating))
                                                          where  trard.toupdate=1
                                                          and   (
                                                                  nvl(right.start_date,contract.effective_date) > sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),''SECOND'')
                                                                  OR
                                                                  nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),''SECOND'')
                                                                 )

      ) tr';

    vsql := 'insert into alarm_activitylog (ID,TYPE,SEVERITY,COMPONENT,DESCRIPTION,EVENT_NAME,ERROR_CODE,CREATED_BY,UPDATED_BY) ' || vsql ;
    CREATEOBJECT(vsql,'Offset Warnings');



                                -- update recording table using the next oldest right information

                                 merge into recording rec using
                                     (select distinct
                                     trard.recording_id,
                                     case
                                      when nvl(right.start_date,contract.effective_date) > sc.start_time + numtodsinterval(nvl(right.offset_start_time,0),'SECOND') then
                                           -1 * (
                                          to_number(extract(second from (sc.start_time - nvl(right.start_date,contract.effective_date) ))) +
                                          to_number(extract(minute from (sc.start_time - nvl(right.start_date,contract.effective_date) ))) * 60 +
                                          to_number(extract(hour from (sc.start_time - nvl(right.start_date,contract.effective_date) )))   * 60 * 60 +
                                          to_number(extract(day from (sc.start_time - nvl(right.start_date,contract.effective_date) )))  * 60 * 60* 24)
                                     else   right.offset_start_time
                                     end as new_offset_start_time,

                                     case
                                      when nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) < sc.END_time + numtodsinterval(nvl(right.offset_END_time,0),'SECOND') then

                                          to_number(extract(second from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - sc.end_time  ))) +
                                          to_number(extract(minute from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - sc.end_time  ))) * 60 +
                                          to_number(extract(hour from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - sc.end_time  )))   * 60 * 60 +
                                          to_number(extract(day from ( nvl(RIGHT.END_DATE,CONTRACT.EXPIRY_DATE) - sc.end_time  )))  * 60 * 60* 24
                                    else right.offset_END_time
                                    end as new_offset_end_time,
                                      right.PRIORITY,
                                     right.recording_mode,
                                      right.PROVIDER_QA_CONTACT,
                                      right.CATALOG_ENTRY,
                                     right.BILLING_ID,
                                    case when mcmr.name is null then right.mpaa_rating else mcmr.name end as MPAA_RATING,
                                    case when mctv.name is null then right.tv_rating else mctv.name end as TV_RATING,
                                     right.PROVIDER_ID,
                                     right.PROVIDER_NAME,
                                    right.PRODUCT,
                                     right.CATEGORY,
                                    right.AUDIO_TYPE,
                                     right.MAX_COPIES,
                                    right.PREVIEW_PERIOD,
                                    right.LICENSE_RELATIVE_START,
                                    right.LICENSE_RELATIVE_DURATION,
                                    sc.start_time,
                                    sc.end_time

                                   from  TEMP_RECS_AFTER_RIGHT_DELETION trard join right on ( trard.older_right_id = right.right_id)
                                           join contract  on  (contract.contract_id = right.contract_id)
                                            join schedule sc on ( trard.schedule_id = sc.schedule_id)
                                            join station on (station.station_id = sc.station_id)
                                            join program prg on (prg.program_id = sc.program_id)
                                            left outer join metadata_config mctv on (mctv.type='TVRating' and mctv.name=upper(sc.tv_rating))
                                            left outer join metadata_config mcmr on (mcmr.type='MPAARating' and mcmr.name=upper(prg.mpaa_rating))
                                            where  trard.toupdate=1
                                            )  rec_right_info

                                on ( rec_right_info.recording_id = rec.recording_id)

                                when matched then
                                update set

                                  rec.offset_start_time  = rec_right_info.new_offset_start_time,
                                  rec.offset_end_time = rec_right_info.new_offset_end_time,
                                  rec.PRIORITY = rec_right_info.PRIORITY,
                                  rec.recording_mode = rec_right_info.recording_mode,
                                  rec.PROVIDER_QA_CONTACT = rec_right_info.PROVIDER_QA_CONTACT,
                                  rec.CATALOG_ENTRY = rec_right_info.CATALOG_ENTRY,
                                 rec.BILLING_ID = rec_right_info.BILLING_ID,
                                 rec.MPAA_RATING = rec_right_info.MPAA_RATING,
                                 rec.TV_RATING = rec_right_info.TV_RATING,
                                rec.PROVIDER_ID = rec_right_info.PROVIDER_ID,
                                rec.PROVIDER_NAME = rec_right_info.PROVIDER_NAME,
                                rec.PRODUCT = rec_right_info.PRODUCT,
                                rec.CATEGORY = rec_right_info.CATEGORY,
                                rec.AUDIO_TYPE = rec_right_info.AUDIO_TYPE,
                                rec.MAX_COPIES = rec_right_info.MAX_COPIES,
                               rec.PREVIEW_PERIOD = rec_right_info.PREVIEW_PERIOD,
                               rec.LICENCE_START_DATE = rec_right_info.start_time + numtodsinterval(rec_right_info.license_relative_start,'SECOND'),
                               rec.LICENSE_END_DATE = rec_right_info.start_time + numtodsinterval(rec_right_info.license_relative_start,'SECOND') + numtodsinterval(rec_right_info.license_relative_duration,'SECOND');

                                -- delete playback window restrictions for these recordings
                               delete from playback_window_restrictions pwr where exists (select 1 from
                               TEMP_RECS_AFTER_RIGHT_DELETION trard   where pwr.recording_id = trard.recording_id and
                               trard.toupdate=1);

                               -- Delete Trial Use Fields for these Recordings and re-insert new Trial Use Fields from the next oldest Right.

                               delete from rec_trial_use_fields rtuf  where exists (select 1 from
                               TEMP_RECS_AFTER_RIGHT_DELETION trard  where rtuf.recording_id = trard.recording_id and
                                 trard.toupdate=1);

                              -- Insert
                              insert into rec_trial_use_fields
                               ( TRIAL_USE_FIELD_ID,
                                ASSET_TYPE,
                                NAME,
                                VALUE_TYPE,
                                VALUE,
                                MAPPING_ID,
                                RECORDING_ID,
                                DESCRIPTION,
                                CREATED_BY,
                                UPDATED_BY,
                                CREATE_DATE,
                                UPDATE_DATE
                               )
                               select REC_TRIAL_USE_FIELD_SEQ.nextval, rtuf.asset_type, rtuf.name, rtuf.value_type, rtuf.value, rtuf.mapping_id, trard.recording_id, rtuf.description, 'SYSTEM', 'SYSTEM', current_timestamp, current_timestamp
                               from TEMP_RECS_AFTER_RIGHT_DELETION trard
                                            join right_trial_use_fields rtuf on (trard.older_right_id = rtuf.right_id)
                                            where  trard.toupdate=1;

                               -- Update the Recording entries adjusted start time and adjusted end time for the affected recordings

                               update recording rec
                               set (rec.adjusted_start_time, rec.adjusted_end_time) =

                                    (
                                      select distinct sc.start_time+ numtodsinterval(nvl(rec.offset_start_time,0), 'SECOND') + numtodsinterval(nvl(rl.offset,0), 'SECOND') ,
                                        sc.end_time + numtodsinterval(nvl(rec.offset_end_time,0), 'SECOND') + numtodsinterval(nvl(rl.offset,0), 'SECOND')
                                          from
                                           temp_old_right_rec_map torrm,
                                           schedule sc,
                                           rtc_pool rp,
                                           rm_lineup rl,
                                           rm,
                                           TEMP_RECS_AFTER_RIGHT_DELETION trard
                                        where
                                            torrm.toupdate=1
                                            and torrm.recording_id =rec.recording_id
                                            and torrm.schedule_id = sc.schedule_id
                                            and rp.pool_id = torrm.pool_id
                                            and rec.pool_id = rp.pool_id
                                            and rm.rm_id = rp.rm_id
                                            and trard.recording_id = torrm.recording_id
                                            and rm.headend_id = rl.headend_id  and rp.rm_id = rl.rm_id and sc.station_id = rl.station_id


                                  )


                             where exists (select 1 from temp_old_right_rec_map torrm
                              join TEMP_RECS_AFTER_RIGHT_DELETION trard on (torrm.recording_id = trard.recording_id)
                                              where torrm.toupdate =1 and  torrm.recording_id = rec.recording_id

                                            );




                                -- insert new playbackwindow restrictions for using the next oldest right
                              insert into playback_window_restrictions
                              ( RESTRICTION_ID,
                                START_TIME,
                                DURATION,
                                RESTRICTIONS,
                                TYPE,
                                CREATED_BY,
                                UPDATED_BY,
                                CREATE_DATE,
                                UPDATE_DATE,
                                RIGHT_ID,
                                RECORDING_ID)

                                select
                                playback_window_restr_seq.nextval,
                                pwr.start_time,
                                pwr.duration,
                                pwr.restrictions,
                                pwr.type,
                                'SYSTEM',
                                'SYSTEM',
                                current_timestamp,
                                current_timestamp,
                                 null,
                                 trard.recording_id

                                from TEMP_RECS_AFTER_RIGHT_DELETION trard
                                            join playback_window_restrictions pwr on (pwr.right_id = trard.older_right_id)

                                            where  trard.toupdate=1;

                              -- create new entries in Recording Queue
   

             
                                  vsql := 'insert into recording_queue(
                                              RECORDING_ID,
                                              SCHEDULE_ID,
                                              POOL_ID,
                                              RM_ID,
                                              SCHEDULE_START_TIME,
                                              SCHEDULE_END_TIME,
                                              SCHEDULE_DURATION,
                                              STATION_ID,
                                              PROGRAM_ID,
                                              HD_TV,
                                              CC,
                                              REPEAT,
                                              RM_NAME,
                                              RM_IP_ADDRESS,
                                              RM_PORT,
                                              EPISODE_TITLE,
                                              STATION_OFFSET,
                                              HEADEND_NAME,
                                              CONTRACT_ID,
                                              YEAR_PROGRAM,
                                              ASSET_ID,
                                              TITLE_ASSET_ID,
                                              MOVIE_ASSET_ID,
                                              POSTER_ASSET_ID,
                                              POSTER_IMAGE_NAME,
                                              POSTER_DESCRIPTION,
                                              PROGRAM_TITLE,
                                              PROGRAM_DESCRIPTION,
                                              OFFSET_START_TIME,
                                              OFFSET_END_TIME,
                                              RECORDING_MODE,
                                              PRIORITY,
                                              STATE,
                                              PROVIDER_QA_CONTACT,
                                              CATALOG_ENTRY,
                                              BILLING_ID,
                                              MPAA_RATING,
                                              TV_RATING,
                                              PROVIDER_ID,
                                              PROVIDER_NAME,
                                              PRODUCT,
                                              CATEGORY,
                                              AUDIO_TYPE,
                                              MAX_COPIES,
                                              PREVIEW_PERIOD,
                                              REDUNDANCY_COUNT,
                                              LICENCE_START_DATE,
                                              LICENSE_END_DATE,
                                              REQUEST_TYPE,
                                              VOD_START_TIME,
                                              VOD_DURATION,
                                              VOD_RESTRICTIONS,
                                              RTI_START_TIME,
                                              RTI_DURATION,
                                              RTI_RESTRICTIONS,
                                              QC_START_TIME,
                                              QC_DURATION,
                                              QC_RESTRICTIONS,
                                              CREATED_BY,
                                              UPDATED_BY,
                                              CREATE_DATE,
                                              UPDATE_DATE,
                                              advisories)
                                              select rec_queue.* FROM
                                              (SELECT
                                              trrm.RECORDING_ID,
                                              trrm.SCHEDULE_ID,
                                              trrm.POOL_ID,
                                              trrm.RM_ID,
                                              trrm.SCHEDULE_START_TIME,
                                              trrm.SCHEDULE_END_TIME,
                                              trrm.SCHEDULE_DURATION,
                                              trrm.STATION_ID,
                                              trrm.PROGRAM_ID,
                                              trrm.HD_TV,
                                              trrm.CC,
                                              trrm.REPEAT,
                                              trrm.RM_NAME,
                                              trrm.RM_IP_ADDRESS,
                                              trrm.RM_PORT,
                                              trrm.EPISODE_TITLE,
                                              trrm.STATION_OFFSET,
                                              trrm.HEADEND_LOCATION,
                                              trrm.CONTRACT_ID,
                                              trrm.YEAR_PROGRAM,
                                              trrm.ASSET_ID,
                                              trrm.TITLE_ASSET_ID,
                                              trrm.MOVIE_ASSET_ID,
                                              trrm.POSTER_ASSET_ID,
                                              trrm.POSTER_IMAGE_NAME,
        									  trrm.POSTER_DESCRIPTION,
                                              trrm.PROGRAM_TITLE,
                                              trrm.PROGRAM_DESCRIPTION,
                                              rec.OFFSET_START_TIME,
                                              rec.OFFSET_END_TIME,
                                              rec.RECORDING_MODE,
                                              rec.PRIORITY,
                                              trrm.STATE,
                                              rec.PROVIDER_QA_CONTACT,
                                              rec.CATALOG_ENTRY,
                                              rec.BILLING_ID,
                                              rec.MPAA_RATING,
                                              rec.TV_RATING,
                                              rec.PROVIDER_ID,
                                              rec.PROVIDER_NAME,
                                              rec.PRODUCT,
                                              rec.CATEGORY,
                                              rec.AUDIO_TYPE,
                                              rec.MAX_COPIES,
                                              rec.PREVIEW_PERIOD,
                                              rec.REDUNDANCY_COUNT,
                                              rec.LICENCE_START_DATE,
                                              rec.LICENSE_END_DATE,
                                              trrm.REQUEST_TYPE,
                                              pwr_vod.start_time as vod_start_time,
                                              pwr_vod.duration as vod_duration,
                                              pwr_vod.restrictions as vod_restrictions,
                                              pwr_rti.start_time as rti_start_time,
                                              pwr_rti.duration as rti_duration,
                                              pwr_rti.restrictions as rti_restrictions,
                                              pwr_qc.start_time as qc_start_time,
                                              pwr_qc.duration as qc_duration,
                                              pwr_qc.restrictions as qc_restrictions,
                                              trrm.CREATED_BY,
                                              trrm.UPDATED_BY,
                                              trrm.CREATE_DATE,
                                              trrm.UPDATE_DATE,

                                              advisories

                                              from   temp_old_right_rec_map trrm
                                              join TEMP_RECS_AFTER_RIGHT_DELETION trard on (trrm.recording_id = trard.recording_id)
                                              join recording rec on (rec.recording_id = trrm.recording_id and trrm.toupdate = 1 )
                                              join rtc_pool rp on (rec.pool_id = rp.pool_id)
                                              left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_vod on (pwr_vod.recording_id = trrm.recording_id and pwr_vod.type=''VOD'')
                                              left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_qc on (pwr_qc.recording_id = trrm.recording_id and pwr_qc.type=''QC'')
                                              left outer join PLAYBACK_WINDOW_RESTRICTIONS pwr_rti on (pwr_rti.recording_id = trrm.recording_id and pwr_rti.type=''RTI'')
                                              where
                                              trrm.rm_id = rp.rm_id and
                                              trrm.SCHEDULE_START_TIME + numtodsinterval(nvl(trrm.offset_start_time,0),''SECOND'') + numtodsinterval(nvl(trrm.STATION_OFFSET,0),''SECOND'') > current_timestamp
                                              order by rec.adjusted_start_time, trrm.PRIORITY) rec_queue' ;
                                              CREATEOBJECT(vsql,'RECORDING Queue Insert');
                                  
              -- new HA change
                insertRecQueue;
                              end if;  /** End of Future recordings update **/

                  end if;  --- END of both tobedeleted and tobeupdated future recordings



          /**  Ending   delete all moved schedules from future to past due to right's offset values  **/

      end if;  /** End of Oldest Right's recoding count condition */


  end processrightdeletion;


   procedure processrecordables(in_contract_id number) is
   contract_effective_date Contract.effective_date%type;
   contract_expiry_date Contract.expiry_date%type;
   begin

        RESOURCELOCK('RECORDABLES_RESOURCE_LOCK');
      if (in_contract_id > 0 ) then
      select effective_date, expiry_date into contract_effective_date, contract_expiry_date from contract where contract_id = in_contract_id;
       delete from recordable_rights_map rrm
          where (right_id, schedule_id, rm_id) in
            (
                    select contract_recorables.right_id, contract_recorables.schedule_id, contract_recorables.rm_id  from ( select schedule_id, right_id, rm_id from recordable_rights_map rrm
                     where exists (select 1 from right where right.contract_id =in_contract_id and  right.right_id = rrm.right_id) )
                     contract_recorables
                      join (select schedule_id  from schedule sc
                            where (sc.end_time > contract_expiry_date at TIME ZONE to_char(sc.end_time, 'TZR') or  sc.start_time < contract_effective_date at TIME ZONE to_char(sc.start_time, 'TZR'))
                           )contract_effected_schedules
                      on (contract_effected_schedules.schedule_id = contract_recorables.schedule_id)
             );
      end if;

      -- Change schedule recorable status to default for all non-recordables
      update schedule set recordable = default where recordable = 1
        and not exists (select 1 from recordable_rights_map  rrm where rrm.schedule_id = schedule.schedule_id);

   end processrecordables;

  procedure runschemastatistics is
  begin
    dbms_stats.gather_schema_stats(ownname=>'RCS',cascade=>TRUE);
  end runschemastatistics;

  procedure deleterecording(in_recording_id number) is
  vsql varchar2(32767);
  begin
    delete from REC_TRIAL_USE_FIELDS rtf where rtf.recording_id = in_recording_id;
    --CREATEOBJECT(vsql,'delete REC_TRIAL_USE_FIELDS');

   delete from RECORDING_REC_RULES_MAP rrrm  where rrrm.recording_id = in_recording_id;
   -- CREATEOBJECT(vsql,'delete RECORDING_REC_RULES_MAP');

    delete from PLAYBACK_WINDOW_RESTRICTIONS pwr where pwr.recording_id = in_recording_id;
    --CREATEOBJECT(vsql,'delete PLAYBACK_WINDOW_RESTRICTIONS');

   delete from RECORDING_LOG rl where rl.recording_id = in_recording_id;
    --CREATEOBJECT(vsql,'delete RECORDING_LOG');

   delete from recording rr where rr.recording_id = in_recording_id;
   -- CREATEOBJECT(vsql,'delete recording');

  end deleterecording;
     
  procedure insertRecQueue is

  TYPE t_rq_id 		   is table of temp_rec_queue_list.recording_queue_id%TYPE;
  b_rq_id		   t_rq_id;
    
  wk_parent_rm_id          recording_queue.rm_id%type;
  wk_RECORDING_ID          recording_queue.recording_id%TYPE;
  wk_SCHEDULE_ID           recording_queue.schedule_id%TYPE;
  wk_SCHEDULE_START_TIME   recording_queue.schedule_start_time%TYPE;
  wk_SCHEDULE_END_TIME     recording_queue.schedule_end_time%TYPE;
  wk_SCHEDULE_DURATION     recording_queue.schedule_duration%TYPE; 
  wk_STATION_ID            recording_queue.station_id%TYPE;
  wk_PROGRAM_ID            recording_queue.program_id%TYPE;
  wk_HD_TV                 recording_queue.HD_TV%type;
  wk_CC                    recording_queue.CC%TYPE;
  wk_REPEAT                recording_queue.REPEAT%TYPE;
  wk_EPISODE_TITLE         recording_queue.EPISODE_TITLE%TYPE;
  wk_STATION_OFFSET        recording_queue.STATION_OFFSET%TYPE;
  wk_HEADEND_NAME          recording_queue.HEADEND_NAME%TYPE;
  wk_CONTRACT_ID           recording_queue.CONTRACT_ID%TYPE;
  wk_YEAR_PROGRAM          recording_queue.YEAR_PROGRAM%TYPE;
  wk_ASSET_ID              recording_queue.ASSET_ID%TYPE;
  wk_MOVIE_ASSET_ID        recording_queue.MOVIE_ASSET_ID%TYPE;
  wk_TITLE_ASSET_ID        recording_queue.TITLE_ASSET_ID%TYPE;
  wk_POSTER_ASSET_ID       recording_queue.POSTER_ASSET_ID%TYPE;
  wk_POSTER_IMAGE_NAME     recording_queue.POSTER_IMAGE_NAME%TYPE;
  wk_POSTER_DESCRIPTION    recording_queue.POSTER_DESCRIPTION%TYPE;
  wk_PROGRAM_TITLE         recording_queue.PROGRAM_TITLE%TYPE;
  wk_PROGRAM_DESCRIPTION   recording_queue.PROGRAM_DESCRIPTION%TYPE;
  wk_OFFSET_START_TIME     recording_queue.OFFSET_START_TIME%TYPE;
  wk_OFFSET_END_TIME       recording_queue.OFFSET_END_TIME%TYPE;
  wk_RECORDING_MODE        recording_queue.RECORDING_MODE%TYPE;
  wk_PRIORITY              recording_queue.PRIORITY%TYPE;
  wk_PROVIDER_QA_CONTACT   recording_queue.PROVIDER_QA_CONTACT%TYPE;
  wk_CATALOG_ENTRY         recording_queue.CATALOG_ENTRY%TYPE;
  wk_BILLING_ID            recording_queue.BILLING_ID%TYPE;
  wk_MPAA_RATING           recording_queue.MPAA_RATING%TYPE;
  wk_TV_RATING             recording_queue.TV_RATING%TYPE;
  wk_PROVIDER_ID           recording_queue.PROVIDER_ID%TYPE;
  wk_PROVIDER_NAME         recording_queue.PROVIDER_NAME%TYPE; 
  wk_PRODUCT               recording_queue.PRODUCT%TYPE;
  wk_CATEGORY              recording_queue.CATEGORY%TYPE;
  wk_AUDIO_TYPE            recording_queue.AUDIO_TYPE%TYPE;            
  wk_max_copies            recording_queue.max_copies%TYPE;            
  wk_preview_period        recording_queue.preview_period%TYPE;              
  wk_REDUNDANCY_COUNT      recording_queue.REDUNDANCY_COUNT%TYPE;                
  wk_LICENCE_START_DATE    recording_queue.LICENCE_START_DATE%TYPE;            
  wk_LICENSE_END_DATE      recording_queue.LICENSE_END_DATE%TYPE;            
  wk_REQUEST_TYPE          recording_queue.REQUEST_TYPE%TYPE;            
  wk_VOD_START_TIME        recording_queue.VOD_START_TIME%TYPE;            
  wk_VOD_DURATION          recording_queue.VOD_DURATION%TYPE;            
  wk_VOD_RESTRICTIONS      recording_queue.VOD_RESTRICTIONS%TYPE;            
  wk_RTI_START_TIME        recording_queue.RTI_START_TIME%TYPE;            
  wk_RTI_DURATION          recording_queue.RTI_DURATION%TYPE;            
  wk_RTI_RESTRICTIONS      recording_queue.RTI_RESTRICTIONS%TYPE;            
  wk_QC_START_TIME         recording_queue.QC_START_TIME%TYPE;            
  wk_QC_DURATION           recording_queue.QC_DURATION%TYPE;            
  wk_QC_RESTRICTIONS       recording_queue.QC_RESTRICTIONS%TYPE;            
  wk_ADVISORIES            recording_queue.ADVISORIES%TYPE;            
  wk_CREATED_BY            recording_queue.CREATED_BY%TYPE;            
  wk_UPDATED_BY            recording_queue.UPDATED_BY%TYPE;           

  wk_redundant_rm_id recording_queue.rm_id%type;
  wk_rm_name recording_queue.rm_name%type;
  wk_rm_ipaddress recording_queue.rm_ip_address%type;
  wk_port recording_queue.rm_port%type;
  wk_pool_id recording_queue.pool_id%type;

  

	cursor c1 is
		select r.RM_ID,r.NAME,r.IP_ADDRESS,r.port,p.POOL_ID
		  from RM r,RTC_POOL p
		  where r.parent_rm_id =  wk_parent_rm_id
      and p.rm_id=r.rm_id;
  begin
  SELECT recording_queue_id
     BULK COLLECT INTO b_rq_id
     FROM temp_rec_queue_list;

   
if(b_rq_id.count > 0) then
  FOR rq IN b_rq_id.FIRST .. b_rq_id.LAST
  LOOP
  
  SELECT  RECORDING_ID,RM_ID,SCHEDULE_ID,SCHEDULE_START_TIME,SCHEDULE_END_TIME, SCHEDULE_DURATION,STATION_ID,
          PROGRAM_ID,HD_TV,CC,REPEAT,EPISODE_TITLE,STATION_OFFSET,HEADEND_NAME,CONTRACT_ID,
          YEAR_PROGRAM,ASSET_ID,MOVIE_ASSET_ID,TITLE_ASSET_ID,POSTER_ASSET_ID,POSTER_IMAGE_NAME,POSTER_DESCRIPTION,
          PROGRAM_TITLE,PROGRAM_DESCRIPTION,OFFSET_START_TIME,OFFSET_END_TIME,RECORDING_MODE,PRIORITY,PROVIDER_QA_CONTACT,
          CATALOG_ENTRY,BILLING_ID,MPAA_RATING,TV_RATING,PROVIDER_ID,PROVIDER_NAME,PRODUCT,CATEGORY,AUDIO_TYPE,max_copies,
          preview_period,REDUNDANCY_COUNT,LICENCE_START_DATE,LICENSE_END_DATE,REQUEST_TYPE,VOD_START_TIME,VOD_DURATION,
          VOD_RESTRICTIONS,RTI_START_TIME,RTI_DURATION,RTI_RESTRICTIONS,QC_START_TIME,QC_DURATION,QC_RESTRICTIONS,
          ADVISORIES,CREATED_BY,UPDATED_BY 
   INTO  wk_RECORDING_ID,wk_parent_rm_id,wk_SCHEDULE_ID,wk_SCHEDULE_START_TIME,wk_SCHEDULE_END_TIME,wk_SCHEDULE_DURATION,wk_STATION_ID,            
          wk_PROGRAM_ID,wk_HD_TV,wk_CC,wk_REPEAT,wk_EPISODE_TITLE,wk_STATION_OFFSET,wk_HEADEND_NAME,wk_CONTRACT_ID,           
          wk_YEAR_PROGRAM,wk_ASSET_ID,wk_MOVIE_ASSET_ID,wk_TITLE_ASSET_ID,wk_POSTER_ASSET_ID,wk_POSTER_IMAGE_NAME,    
          wk_POSTER_DESCRIPTION,wk_PROGRAM_TITLE,wk_PROGRAM_DESCRIPTION,wk_OFFSET_START_TIME,wk_OFFSET_END_TIME,
          wk_RECORDING_MODE,wk_PRIORITY,wk_PROVIDER_QA_CONTACT,wk_CATALOG_ENTRY,wk_BILLING_ID,wk_MPAA_RATING,wk_TV_RATING,             
          wk_PROVIDER_ID,wk_PROVIDER_NAME,wk_PRODUCT,wk_CATEGORY,wk_AUDIO_TYPE,wk_max_copies,wk_preview_period,                     
          wk_REDUNDANCY_COUNT,wk_LICENCE_START_DATE,wk_LICENSE_END_DATE,wk_REQUEST_TYPE,wk_VOD_START_TIME,wk_VOD_DURATION,                     
          wk_VOD_RESTRICTIONS,wk_RTI_START_TIME,wk_RTI_DURATION,wk_RTI_RESTRICTIONS,wk_QC_START_TIME,wk_QC_DURATION,                      
          wk_QC_RESTRICTIONS,wk_ADVISORIES,wk_CREATED_BY,wk_UPDATED_BY 
  From    recording_queue
  where   recording_queue_id= b_rq_id(rq);
          

  
  open c1;
    loop
	    fetch c1 into wk_redundant_rm_id,
                    wk_rm_name,
                    wk_rm_ipaddress,
                    wk_port,
                    wk_pool_id;
      IF c1%NOTFOUND
      	THEN
         	CLOSE C1;
         	EXIT;
      END IF;
      			   -- Create Recording
             
  
   
        -- vsql :=  '
        INSERT INTO recording_queue (
         RECORDING_QUEUE_ID,
         RECORDING_ID,
         SCHEDULE_ID,
         POOL_ID,
         RM_ID,
         SCHEDULE_START_TIME,
         SCHEDULE_END_TIME,
         SCHEDULE_DURATION,
         STATION_ID,
         PROGRAM_ID,
         HD_TV,
         CC,
         REPEAT,
         RM_NAME,
         RM_IP_ADDRESS,
         RM_PORT,
         EPISODE_TITLE,
         STATION_OFFSET,
         HEADEND_NAME,
         CONTRACT_ID,
         YEAR_PROGRAM,
         ASSET_ID,
         MOVIE_ASSET_ID,
         TITLE_ASSET_ID,
         POSTER_ASSET_ID,
         POSTER_IMAGE_NAME,
         POSTER_DESCRIPTION,
         PROGRAM_TITLE,
         PROGRAM_DESCRIPTION,
         OFFSET_START_TIME,
         OFFSET_END_TIME,
         RECORDING_MODE,
         PRIORITY,
         PROVIDER_QA_CONTACT,
         CATALOG_ENTRY,
         BILLING_ID,
         MPAA_RATING,
         TV_RATING,
         PROVIDER_ID,
         PROVIDER_NAME,
         PRODUCT,
         CATEGORY,
         AUDIO_TYPE,
         max_copies,
         preview_period,
         REDUNDANCY_COUNT,
         LICENCE_START_DATE,
         LICENSE_END_DATE,
         REQUEST_TYPE,
         VOD_START_TIME,
         VOD_DURATION,
         VOD_RESTRICTIONS,
         RTI_START_TIME,
         RTI_DURATION,
         RTI_RESTRICTIONS,
         QC_START_TIME,
         QC_DURATION,
         QC_RESTRICTIONS,
         ADVISORIES,
         CREATED_BY,
         UPDATED_BY)
         values (
         recording_queue_seq.nextval,
         wk_RECORDING_ID,
         wk_SCHEDULE_ID,
         wk_pool_id,
         wk_redundant_rm_id,
         wk_SCHEDULE_START_TIME,
         wk_SCHEDULE_END_TIME,
         wk_SCHEDULE_DURATION,
         wk_STATION_ID,            
         wk_PROGRAM_ID,
         wk_HD_TV,
         wk_CC,
         wk_REPEAT,
         wk_rm_name,
         wk_rm_ipaddress,
         wk_port,
         wk_EPISODE_TITLE,
         wk_STATION_OFFSET,
         wk_HEADEND_NAME,
         wk_CONTRACT_ID,           
         wk_YEAR_PROGRAM,
         wk_ASSET_ID,
         wk_MOVIE_ASSET_ID,
         wk_TITLE_ASSET_ID,
         wk_POSTER_ASSET_ID,
         wk_POSTER_IMAGE_NAME,    
         wk_POSTER_DESCRIPTION,
         wk_PROGRAM_TITLE,
         wk_PROGRAM_DESCRIPTION,
         wk_OFFSET_START_TIME,
         wk_OFFSET_END_TIME,
         wk_RECORDING_MODE,
         wk_PRIORITY,
         wk_PROVIDER_QA_CONTACT,
         wk_CATALOG_ENTRY,
         wk_BILLING_ID,
         wk_MPAA_RATING,
         wk_TV_RATING,             
         wk_PROVIDER_ID,
         wk_PROVIDER_NAME,
         wk_PRODUCT,
         wk_CATEGORY,
         wk_AUDIO_TYPE,
         wk_max_copies,
         wk_preview_period,                     
         wk_REDUNDANCY_COUNT,
         wk_LICENCE_START_DATE,
         wk_LICENSE_END_DATE,
         wk_REQUEST_TYPE,
         wk_VOD_START_TIME,
         wk_VOD_DURATION,                     
         wk_VOD_RESTRICTIONS,
         wk_RTI_START_TIME,
         wk_RTI_DURATION,
         wk_RTI_RESTRICTIONS,
         wk_QC_START_TIME,
         wk_QC_DURATION,                      
         wk_QC_RESTRICTIONS,
         wk_ADVISORIES,
         wk_CREATED_BY,
         wk_UPDATED_BY);
       --  CREATEOBJECT(vsql,'RECORDING Queue Insert for Mirror RMs');
            
  end loop;
  end loop;
  vsql := 'TRUNCATE TABLE temp_rec_queue_list'; 
  CREATEOBJECT(vsql,'clear temp_rec_queue_list table');
end if;
  end insertRecQueue;
   
END PROCESSTSTV;
/
exit
