#!/bin/bash
################################################
#
# Author: Martin Holm <martin.holm@ericsson.com
#
# File and command output collection script
# for CMS migration between SLES and RHEL
#
################################################
archive=/home/upgrade/sles-backup
doFiles=false
doCommands=false
quiet=false
force=false
remove=false

function usage ()  {
	echo
	echo "Syntax: $(basename $0) [-f file] [-c file] [-h] [-F] [-R] [-q]"
	echo
	echo "    -f file     File containing the file list you want to save."
	echo "    -c file     File containing the output file and commands of command output"
	echo "                you want to save."
	echo "    -h          This."
	echo "    -F          Force collection, even if files exists. Might create multiple"
	echo "                copies of the same files."
	echo "    -R          Remove old files in the archive directory before collecting"
	echo "    -q          Quiet. Do not show any warnings."
	echo
}

function CheckCommand () {
        local file=$1
        shift
	basecommand=$1
        command=$@
        if [ -z "${command}" ]; then
                Warn "Syntax: $FUNCNAME <file> <command>"
                return 1
        elif [ -z $(which ${basecommand} 2>/dev/null) ]; then
		Warn "Command '${basecommand}' not found"
		return 1
	elif [ ! -x $(which ${basecommand} 2>/dev/null) ]; then
		Warn "Command '${basecommand}' is not executable"
		return 1
	else
                return 0
        fi
}

function CheckFile () {
        local file=$1
	# check if we got a file to check
	if [ -n $file ]; then
		# check if file exists
        	if [ -f $file ]; then
                	return 0
        	elif [ -r $file ]; then
			Warn "File $file is not readable"
			return 1
		else
               		Warn "File $file does not exist"
                	return 1
        	fi
	fi
	# if no file, return false
	return 1
}

function Die () {
	echo $@
	exit 1
}

function GenFileName () {
	local file=$archive/$1
	if [ ! -f $file ]; then
		echo $file
		return
	fi
	for i in {1..10}; do
		if [ ! -f ${file}_${i} ]; then
			echo ${file}_${i}
			return
		fi
	done
	Die "Fatal error: Unable to generate suffixes for filename $file"
}

function Warn () {
	if [ $quiet == "false" ]; then
		echo $@
	fi
}


while getopts ":c:f:o:hRFq" opt; do
	case $opt in
		c)
			cfile=$OPTARG
			doCommands=true
			;;
		f)
			ffile=$OPTARG
			doFiles=true
			;;
		o)
			outfile=$OPTARG
			;;
		h)
			usage
			exit 1
			;;
		F)
			force=true
			;;
		R)
			remove=true
			force=true
			;;
		q)
			quiet=true
			;;
		\?)
			echo "Invalid option: -$OPTARG"
			usage
			exit 1
			;;
	esac
		
done

# check if our output folder exists. if not, try to create it
test -d $archive || mkdir -p $archive || (Die "Could not create directory $archive")
# check if we already have files in our archive
if [ $(ls $archive|wc -l) -gt 0 ]; then
	if [ $remove == "true" ]; then
		Warn "Removing older files"
		rm -f $archive/*
	fi
	if [ $force == "false" ]; then
		echo
		echo "You already have files in $archive"
		echo "If you want to proceed anyway, run this script with -F or -R options"
		echo
		echo "For more information, see: $(basename $0) -h"
		echo
		exit 1
	fi
fi
		

if [ $doFiles == "true" ]; then
	if [ -z $ffile ]; then
		Die "No file supplied for -f."
	fi
	if [ ! -f $ffile ]; then
		Die "File '$ffile' does not exist"
	fi
	grep -v "^#" $ffile|(
		while read line; do
			filename=$line
			for filename2 in $filename; do
				CheckFile $filename2
				if [ $? -eq 0 ]; then
					cp $filename2 $(GenFileName $(basename $filename2))
				fi
			done
		done
	)
fi
if [ $doCommands == "true" ]; then
	if [ -z $cfile ]; then
		Die "No file supplied for -c."
	fi
	if [ ! -f $cfile ]; then
		Die "File '$cfile' does not exist"
	fi
	grep -v "^#" $cfile|(
		while read line; do
			thisFile=$(echo $line|cut -d ' ' -f1)
			thisCommand=$(echo $line|cut -d ' ' -f2-)
			CheckCommand $thisFile $thisCommand
			if [ $? -eq 0 ]; then
				$thisCommand > $(GenFileName $(basename $thisFile))
			fi
		done
	)
fi



exit 0

